/* @(#) im_open: vips front end to file open functions
ARGS: filename and mode which is one of:
"r" read by mmap (im_mmapin)
"rw" read and write by mmap (im_mmapinrw)
"w" write for write (im_writeline)
"t" for temporary memory image
"p" for partial memory image
RETURNS: IMAGE pointer or 0 on error
Copyright: Kirk Martinez 29/5/92

Modified:
 * 8/8/94 JC
 *	- ANSIfied
 *	- im_open_local added
 * 16/8/94 JC
 *	- im_malloc() added
 * 22/11/94 JC & KM
 *	- tiff added
 * 1/2/95 JC
 *	- tiff removed again!
 *	- now just im_istiff(), im_tiff2vips() and im_vips2tiff()
 * 	- applications have responsibility for making the translation
 * 26/3/96 JC
 *	- im_open_local() now closes on close, not evalend
 * 14/11/96 JC
 *	- tiff and jpeg added again
 * 	- open for read used im_istiff() and im_isjpeg() to switch
 *	- open for write looks at suffix
 * 23/4/97 JC
 *	- im_strdup() now allows NULL IMAGE parameter
 *	- subsample parameter added to im_tiff2vips()
 * 29/10/98 JC
 *	- byte-swap stuff added
 * 16/6/99 JC
 *	- 8x byte swap added for double/double complex
 *	- better error message if file does not exist
 *	- ignore case when testing suffix for save
 *	- fix im_mmapinrw() return test to stop coredump in edvips if
 *	  unwritable
 * 2/11/99 JC
 *	- malloc/strdup stuff moved to memory.c
 * 5/8/00 JC
 *	- fixes for im_vips2tiff() changes
 * 13/10/00 JC
 *	- ooops, missing 'else'
 * 22/11/00 JC
 * 	- ppm read/write added
 * 12/2/01 JC
 * 	- im__image_sanity() added
 * 16/5/01 JC
 *	- now allows RW for non-native byte order, provided it's an 8-bit
 *	  image
 * 11/7/01 JC
 *	- im_tiff2vips() no longer has subsample option
 * 25/3/02 JC
 *	- better im_open() error message
 * 12/12/02 JC
 *	- sanity no longer returns errors, just warns
 * 28/12/02 HB
 *     - Added PNG support
 * 6/5/03 JC
 *	- added im_open_header() (from header.c)
 * 22/5/03 JC
 *	- im__skip_dir() knows about ':' in filenames for vips2tiff etc.
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

#ifdef _MSC_VER
#define strcasecmp(a,b) _stricmp(a,b)
#endif

/* Suffix sets.
 */
static const char *im__suffix_vips[] = {
	"v", "",
	NULL
};
static const char *im__suffix_tiff[] = {
	"tif", "tiff",
	NULL
};
static const char *im__suffix_jpeg[] = {
	"jpeg", "jpg", "jfif", "jpe",
	NULL
};
static const char *im__suffix_ppm[] = {
	"ppm", "pbm", "pgm", 
	NULL
};
static const char *im__suffix_png[] = {
	"png", 
	NULL
};

/* Skip any leading path stuff. Horrible: if this is a filename which came
 * from win32 and we're a *nix machine, it'll have '\\' not '/' as the
 * separator :-(
 *
 * Try to fudge this ... if the file doesn't contain any of our native
 * separators, look for the opposite one as well. If there are none of those
 * either, just return the filename.
 *
 * Extra wrinkle: the filename can be something for im_vips2tiff etc., ie, it
 * can contain output format specifiers (eg. "/home/john/me.jpg:75", or
 * "c:\fred.jpg"). We have to skip the leading ':' on win32 (it's part of the
 * path), but not skip the ':' that separates the filename from the format
 * specification. Remember, the format specifiers can contain '/' too.
 */
const char *
im__skip_dir( const char *filename )
{
        const char *p, *q, *r;

	const char native_dir_sep = IM_DIR_SEP;
	const char non_native_dir_sep = native_dir_sep == '/' ? '\\' : '/';

	/* First, skip any leading "C:" stuff.
	 */
	p = filename;
	if( strlen( p ) > 1 && 
		isalpha( p[0] ) && p[1] == ':' )
		p += 2;

	/* Find the end of the part of p that could be the filename. 
	 */
	if( !(q = strchr( p, ':' )) )
		q = p + strlen( p ) + 1;

	/* Search back for the first native dir sep, or failing that, the first
	 * non-native dir sep.
	 */
	for( r = q; r > p && r[-1] != native_dir_sep; r-- )
		;
	if( r == p )
		for( r = q; r > p && r[-1] != non_native_dir_sep; r-- )
			;

        return( r );
}

/* Split filename into name / mode components. name and mode should both be
 * FILENAME_IM_MAX chars.
 */
void
im__filename_split( const char *path, char *name, char *mode )
{
        const char *p;
        char *q;

        im_strncpy( name, path, FILENAME_MAX );
	p = im__skip_dir( name );
        if( (q = strchr( p, ':' )) ) {
                strcpy( mode, q + 1 );
                *q = '\0';
        }
        else
                strcpy( mode, "" );
}

/* Extract suffix from filename, ignoring any mode string. Suffix should be
 * FILENAME_MAX chars.
 */
void
im__filename_suffix( const char *path, char *suffix )
{
	char name[FILENAME_MAX];
	char mode[FILENAME_MAX];
        char *p;

	im__filename_split( path, name, mode );
        if( (p = strrchr( name, '.' )) ) 
                strcpy( suffix, p + 1 );
        else
                strcpy( suffix, "" );
}

/* Does a filename have one of a set of suffixes. Ignore case.
 */
int
im__filename_suffix_match( const char *path, const char *suffixes[] )
{
	char suffix[FILENAME_MAX];
	const char **p;

	im__filename_suffix( path, suffix );
	for( p = suffixes; *p; p++ )
		if( strcasecmp( suffix, *p ) == 0 )
			return( 1 );

	return( 0 );
}

/* Test string for prefix.
 */
int
im__isprefix( const char *s1, const char *s2 )
{
        return( strncmp( s1, s2, strlen( s1 ) ) == 0 );
}

/* And postfix.
 */
int
im__ispostfix( const char *s1, const char *s2 )
{
        int l1 = strlen( s1 );
        int l2 = strlen( s2 );

        if( l2 > l1 )
                return( 0 );

        return( strcmp( s1 + l1 - l2, s2 ) == 0 );
}

/* p points to the start of a buffer ... move it on through the buffer (ready
 * for the next call), and return the current option (or NULL for option
 * missing).
 */
char *
im__getnextoption( char **in )
{
        char *p = *in;
        char *q = p;

        if( !p || !*p )
                return( NULL );

        if( (p = strchr( p, ',' )) ) {
                /* Another option follows this one .. set up to pick that out
                 * next time.
                 */
                *p = '\0';
                *in = p + 1;
        }
        else {
                /* This is the last one.
                 */
                *in = NULL;
        }

        if( strlen( q ) > 0 )
                return( q );
        else
                return( NULL );
}

/* Get a suboption string, or NULL.
 */
char *
im__getsuboption( const char *buf )
{
        char *p;

        if( (p = strchr( buf, ':' )) )
                return( p + 1 );

        return( NULL );
}

/* Swap pairs of bytes.
 */
static void
swap2_gen( PEL *in, PEL *out, int width, IMAGE *im )
{	
	int x;
	int sz = IM_IMAGE_SIZEOF_PEL( im ) * width;	/* Bytes in buffer */

	for( x = 0; x < sz; x += 2 ) {
		out[x] = in[x + 1];
		out[x + 1] = in[x];
	}
}

/* 2-ary byteswapper.
 */
static int 
swap2( IMAGE *in, IMAGE *out )
{
	/* Check args.
	 */
        if( im_piocheck( in, out ) )
		return( -1 );
	if( in->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_swap2: in must be uncoded" );
		return( -1 );
	}
	if( in->BandFmt != IM_BANDFMT_SHORT && 
		in->BandFmt != IM_BANDFMT_USHORT ) {
		im_errormsg( "im_swap2: 16-bit images only" );
		return( -1 );
	}

	/* Prepare output header.
	 */
	if( im_cp_desc( out, in ) )
		return( -1 );

	/* Generate!
	 */
	if( im_wrapone( in, out, (im_wrapone_fn) swap2_gen, in, NULL ) )
		return( -1 );

	return( 0 );
}

/* Swap 4- of bytes.
 */
static void
swap4_gen( PEL *in, PEL *out, int width, IMAGE *im )
{	
	int x;
	int sz = IM_IMAGE_SIZEOF_PEL( im ) * width;	/* Bytes in buffer */

	for( x = 0; x < sz; x += 4 ) {
		out[x] = in[x + 3];
		out[x + 1] = in[x + 2];
		out[x + 2] = in[x + 1];
		out[x + 3] = in[x];
	}
}

/* 4-ary byteswapper.
 */
static int 
swap4( IMAGE *in, IMAGE *out )
{
	/* Check args.
	 */
        if( im_piocheck( in, out ) )
		return( -1 );
	if( in->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_swap4: in must be uncoded" );
		return( -1 );
	}
	if( in->BandFmt != IM_BANDFMT_INT && 
		in->BandFmt != IM_BANDFMT_UINT &&
		in->BandFmt != IM_BANDFMT_FLOAT && 
		in->BandFmt != IM_BANDFMT_COMPLEX ) {
		im_errormsg( "im_swap4: 4-swap images only" );
		return( -1 );
	}

	/* Prepare output header.
	 */
	if( im_cp_desc( out, in ) )
		return( -1 );

	/* Generate!
	 */
	if( im_wrapone( in, out, (im_wrapone_fn) swap4_gen, in, NULL ) )
		return( -1 );

	return( 0 );
}

/* Swap 8- of bytes.
 */
static void
swap8_gen( PEL *in, PEL *out, int width, IMAGE *im )
{	
	int x;
	int sz = IM_IMAGE_SIZEOF_PEL( im ) * width;	/* Bytes in buffer */

	for( x = 0; x < sz; x += 8 ) {
		out[x] = in[x + 7];
		out[x + 1] = in[x + 6];
		out[x + 2] = in[x + 5];
		out[x + 3] = in[x + 4];
		out[x + 4] = in[x + 3];
		out[x + 5] = in[x + 2];
		out[x + 6] = in[x + 1];
		out[x + 7] = in[x];
	}
}

/* 8-ary byteswapper.
 */
static int 
swap8( IMAGE *in, IMAGE *out )
{
	/* Check args.
	 */
        if( im_piocheck( in, out ) )
		return( -1 );
	if( in->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_swap4: in must be uncoded" );
		return( -1 );
	}
	if( in->BandFmt != IM_BANDFMT_DOUBLE && 
		in->BandFmt != IM_BANDFMT_DPCOMPLEX ) {
		im_errormsg( "im_swap8: 8-swap images only" );
		return( -1 );
	}

	/* Prepare output header.
	 */
	if( im_cp_desc( out, in ) )
		return( -1 );

	/* Generate!
	 */
	if( im_wrapone( in, out, (im_wrapone_fn) swap8_gen, in, NULL ) )
		return( -1 );

	return( 0 );
}

/* Open a VIPS image and byte-swap the image data if necessary.
 */
static IMAGE *
read_vips( const char *filename )
{
	IMAGE *im, *im2;

	if( !(im = im_init( filename )) )
		return( NULL );
	if( im_openin( im ) ) {
		im_close( im );
		return( NULL );
	}

	/* Already in native format?
	 */
	if( im_isMSBfirst( im ) == im_amiMSBfirst() ) 
		return( im );

	/* Not native ... but maybe does not need swapping? 
	 */
	if( im->Coding == IM_CODING_LABQ )
		return( im );
	if( im->Coding != IM_CODING_NONE ) {
		im_close( im );
		im_errormsg( "im_open: unknown coding type" );
		return( NULL );
	}
	if( im->BandFmt == IM_BANDFMT_CHAR || im->BandFmt == IM_BANDFMT_UCHAR )
		return( im );
	
	/* Needs swapping :( make a little pipeline up to do this for us.
	 */
	if( !(im2 = im_open( filename, "p" )) )
		return( NULL );
        if( im_add_close_callback( im2, 
		(im_callback_fn) im_close, im, NULL ) ) {
                im_close( im );
                im_close( im2 );
                return( NULL );
        }
	switch( im->BandFmt ) {
        case IM_BANDFMT_SHORT: 		swap2( im, im2 ); break; 
        case IM_BANDFMT_USHORT: 	swap2( im, im2 ); break; 
        case IM_BANDFMT_INT: 		swap4( im, im2 ); break; 
        case IM_BANDFMT_UINT: 		swap4( im, im2 ); break; 
        case IM_BANDFMT_FLOAT: 		swap4( im, im2 ); break; 
        case IM_BANDFMT_DOUBLE:		swap8( im, im2 ); break; 
        case IM_BANDFMT_COMPLEX:	swap4( im, im2 ); break;
        case IM_BANDFMT_DPCOMPLEX:	swap8( im, im2 ); break;
	default:
		im_close( im2 );
		im_errormsg( "im_open: internal error!" );
		return( NULL );
	}

	return( im2 );
}

/* Delayed save: if we write to TIFF or to JPEG format, actually do the write
 * to a "p" and on evalend do im_vips2tiff() or whatever. Track save
 * parameters here.
 */
typedef struct {
	int (*save_fn)();	/* Save function */
	IMAGE *im;		/* Image to save */
	char *filename;		/* Save args */
} SaveBlock;

/* From evalend callback: invoke a delayed save.
 */
static int
invoke_sb( SaveBlock *sb )
{
	if( sb->save_fn( sb->im, sb->filename ) )
		return( -1 );

	return( 0 );
}

/* Attach a SaveBlock to an image.
 */
static int
attach_sb( IMAGE *out, int (*save_fn)(), const char *filename )
{
	SaveBlock *sb = IM_NEW( out, SaveBlock );

	if( !sb )
		return( -1 );
	sb->im = out;
	sb->save_fn = save_fn;
	sb->filename = im_strdup( out, filename );

	if( im_add_evalend_callback( out, 
		(im_callback_fn) invoke_sb, (void *) sb, NULL ) )
		return( -1 );

	return( 0 );
}

IMAGE *
im_open( const char *filename, const char *mode )
{
	IMAGE *im;

	switch( mode[0] ) {
        case 'r':
		/* Check for other formats.
		 */
		if( !im_existsf( "%s", filename ) ) {
			im_errormsg( "im_open: \"%s\" is not readable",
				filename );
			im = NULL;
		}
		else if( im_istiff( filename ) ) {
			/* Make a partial for im_tiff2vips to write to.
			 */
			if( !(im = im_open( "output of im_tiff2vips", "p" )) )
				return( NULL );
			if( im_tiff2vips( filename, im ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else if( im_isjpeg( filename ) ) {
			if( !(im = im_open( "output of im_jpeg2vips", "p" )) )
				return( NULL );
			if( im_jpeg2vips( filename, im ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else if( im_isppm( filename ) ) {
			if( !(im = im_open( "output of im_ppm2vips", "p" )) )
				return( NULL );
			if( im_ppm2vips( filename, im ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else if( im_ispng( filename ) ) {
			if( !(im = im_open( "output of im_png2vips", "p" )) )
				return( NULL );
			if( im_png2vips( filename, im ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else if( im_isvips( filename ) ) {
			if( mode[1] == 'w' ) {
				/* Has to be native format for >8 bits.
				 */
				if( !(im = im_init( filename )) )
					return( NULL );
				if( im_openinrw( im ) ) {
					im_close( im );
					return( NULL );
				}
				if( im->Bbits != IM_BBITS_BYTE &&
					im_isMSBfirst( im ) != 
						im_amiMSBfirst() ) {
					im_close( im );
					im_errormsg( "im_open: open for read-"
						"write for native format "
						"images only" );
					return( NULL );
				}
			}
			else 
				im = read_vips( filename );
		}
		else if( im_ismagick( filename ) ) {
			if( !(im = im_open( "output of im_magick2vips", "p" )) )
				return( NULL );
			if( im_magick2vips( filename, im ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else {
			im_errormsg( "im_open: \"%s\" is not "
				"a supported format", filename );
			im = NULL;
		}

        	break;

	case 'w':
		/* Look at the suffix for format write.
		 */
		if( im__filename_suffix_match( filename, im__suffix_vips ) ) 
			im = im_openout( filename );
		else if( im__filename_suffix_match( filename, 
			im__suffix_tiff ) ) {
			/* TIFF write. Save to a partial, and on evalend
			 * im_vips2tiff from that.
			 */
			if( !(im = im_open( "im_open:vips2tiff:1", "p" )) )
				return( NULL );
			if( attach_sb( im, im_vips2tiff, filename ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else if( im__filename_suffix_match( filename, 
			im__suffix_jpeg ) ) {
			/* JPEG write. 
			 */
			if( !(im = im_open( "im_open:vips2jpeg:1", "p" )) )
				return( NULL );
			if( attach_sb( im, im_vips2jpeg, filename ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else if( im__filename_suffix_match( filename, 
			im__suffix_ppm ) ) {
			/* ppm write. 
			 */
			if( !(im = im_open( "im_open:vips2ppm:1", "p" )) )
				return( NULL );
			if( attach_sb( im, im_vips2ppm, filename ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else if( im__filename_suffix_match( filename, 
			im__suffix_png ) ) {
			/* png write. 
			 */
			if( !(im = im_open( "im_open:vips2png:1", "p" )) )
				return( NULL );
			if( attach_sb( im, im_vips2png, filename ) ) {
				im_close( im );
				return( NULL );
			}
		}
		else {
			im_errormsg( "im_open: unknown suffix for save" );
			return( NULL );
		}
        	break;

        case 't':
                im = im_setbuf( filename );
                break;

        case 'p':
                im = im_partial( filename );
                break;

	default:
		/* Getting here means bad mode
		 */
		im_errormsg( "im_open: bad mode \"%s\"", mode );
		return( NULL );
        }

	return( im );
}

/* Make something local to an image descriptor ... pass in a constructor
 * and a destructor, plus three args.
 */
void *
im_local( IMAGE *im, 
	im_construct_fn cons, im_callback_fn dest, void *a, void *b, void *c )
{
	void *obj;

	if( !im ) {
		im_errormsg( "im_local: NULL image descriptor" );
		return( NULL );
	}

        if( !(obj = cons( a, b, c )) )
                return( NULL );
        if( im_add_close_callback( im, (im_callback_fn) dest, obj, a ) ) {
                dest( obj, a );
                return( NULL );
        }
 
        return( obj );
}

/* Make an array of things local to a descriptor ... eg. make 6 local temp
 * images.
 */
int
im_local_array( IMAGE *im, void **out, int n,
	im_construct_fn cons, im_callback_fn dest, void *a, void *b, void *c )
{
	int i;

	for( i = 0; i < n; i++ )
		if( !(out[i] = im_local( im, cons, dest, a, b, c )) )
			return( -1 );

	return( 0 );
}

/* Test an image for sanity. We could add many more tests here.
 */
static const char *
image_sanity( IMAGE *im )
{
	if( !im ) 
		return( "NULL descriptor" );
	if( !im->filename ) 
		return( "NULL filename" );

	if( im->Xsize < -1 || im->Ysize < -1 || im->Bands < -1 ) 
		return( "bad dimensions" );
	if( im->BandFmt < -1 || im->BandFmt > IM_BANDFMT_DPCOMPLEX ||
		(im->Coding != -1 &&
			im->Coding != IM_CODING_NONE && 
			im->Coding != IM_CODING_LABQ) ||
		im->Type < -1 || im->Type > IM_TYPE_FOURIER ||
		im->dtype > IM_PARTIAL || im->dhint > IM_ANY ) 
		return( "bad enum value" );
	if( im->Xres < 0 || im->Xres < 0 ) 
		return( "bad resolution" );

	return( NULL );
}

int 
im_image_sanity( IMAGE *im )
{
	const char *msg;

	if( (msg = image_sanity( im )) ) {
		im_warning( "im_image_sanity: \"%s\" %s",
			im ? (im->filename ? im->filename : "") : "", msg );
		im_printdesc( im );
	}

	return( 0 );
}

typedef int (*open_fn)( const char *filename, IMAGE *im );

static IMAGE *
open_header_sub( open_fn fn, const char *filename )
{
	IMAGE *im;

	if( !(im = im_open( "open_header_sub", "p" )) || fn( filename, im ) ) {
		im_close( im );
		return( NULL );
	}

	return( im );
}

IMAGE *
im_open_header( const char *file )
{
	IMAGE *im;

	if( !im_existsf( "%s", file ) ) {
		im_errormsg( "im_open_header: file \"%s\" does not exist", 
			file );
		return( NULL );
	}

	if( im_istiff( file ) ) {
		if( !(im = open_header_sub( im_tiff2vips_header, file )) ) 
			return( NULL );
	}
	else if( im_isjpeg( file ) ) {
		if( !(im = open_header_sub( im_jpeg2vips_header, file )) ) 
			return( NULL );
	}
	else if( im_isppm( file ) ) {
		if( !(im = open_header_sub( im_ppm2vips_header, file )) ) 
			return( NULL );
	}
	else if( im_ispng( file ) ) {
		if( !(im = open_header_sub( im_png2vips_header, file )) ) 
			return( NULL );
	}
	else if( im_isvips( file ) ) {
		if( !(im = im_init( file )) )
			return( NULL );
		if( im__read_header( im ) ) {
			im_close( im );
			return( NULL );
		}
	}
	else if( im_ismagick( file ) ) {
		if( !(im = open_header_sub( im_magick2vips_header, file )) ) 
			return( NULL );
	}
	else {
		im_errormsg( "im_open_header: \"%s\": unknown image format", 
			file );
		return( NULL );
	}

	return( im );
}
