/* Linked list library.
 * 
 * list.c: linked list functions.
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <vips/vips.h>
#include <vips/util.h>
#include <vips/rect.h>
#include <vips/list.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Like strtok(), but better. Give a string and a list of break characters;
 * write a '\0' into the string over the first break character and return a
 * pointer to the next non-break character. If there are no break characters,
 * then return a pointer to the end of the string. If passed a pointer to an
 * empty string or NULL, then return NULL.
 */
char *
im__break_token( char *str, char *brk )
{
	char *p;

	/* Is the string empty? If yes, return NULL immediately.
	 */
	if( !str || !*str )
		return( NULL );

	/* Skip initial break characters.
	 */
	p = str + strspn( str, brk );

	/* Search for the first break character after the token.
	 */
	p += strcspn( p, brk );

	/* Is there string left?
	 */
	if( *p ) {
		/* Write in an end-of-string mark and return the start of the
		 * next token.
		 */
		*p++ = '\0';
		p += strspn( p, brk );
	}

	return( p );
}

/* Return the length of a list.
 */
int
im_list_len( List *l )
{	
	int n;

	for( n = 0; l; n++, l = tl(l) )
		;

	return( n );
}

/* Find the position of an entry in a list. -1 for not there. Index from 0.
 */
/*VARARGS1*/
int
im_list_pos( List *l, void *t )
{	
	int n;

	for( n = 0; l; n++, l = tl(l) )
		if( hd(l) == t ) 
			return( n );

	im_errormsg( "im_list_pos: not found" );
	return( -1 );
}

/* Non-zero if list contains member. Does not set im_errormsg on not there!
 */
/*VARARGS1*/
int
im_list_member( List *l, void *t )
{	
	int n;

	for( n = 0; l; n++, l = tl(l) )
		if( hd(l) == t ) 
			return( -1 );

	return( 0 );
}

/* Return the object at index n in a list. Index from 0. NULL for index out 
 * of range.
 */
/*VARARGS1*/
void *
im_list_index( List *l, int n )
{	
	if( n < 0 ) {
		im_errormsg( "im_list_index: negative index" );
		return( NULL );
	}

	for( ; l; n--, l = tl(l) )
		if( n == 0 )
			return( hd(l) );

	im_errormsg( "im_list_index: index too large" );
	return( NULL );
}

/* Map a function over a list. Allow extra args .. they must be
 * all 32 bits. If the function returns NULL, apply it to the next in the
 * list. If it returns a value, we return that value.
 * 
 * The function may im_list_remove the list element it is being applied to; 
 * make sure we don't refer to it after we call it.
 */
/*VARARGS1*/
void *
im_list_map( List *l, im_list_map_fn fn, void *a, void *b )
{	
	void *r;
	List *nxt;

	for( ; l; l = nxt ) {
		nxt = tl(l);

		if( (r = fn( hd(l), a, b )) )
			return( r );
	}

	return( NULL );
}

/* Like im_list_map, but traverse the list in reverse order. Much slower and
 * more expensive! Emergencies only.
 */
void *
im_list_map_rev( List *l, im_list_map_fn fn, void *a, void *b )
{	
	void *r, *ths;

	if( !l )
		/* Empty list .. easy!
		 */
		return( NULL );
	else {
		ths = hd(l);

		/* Try the rest of the list first.
		 */
		if( (r = im_list_map_rev( tl(l), fn, a, b )) )
			return( r );

		/* Do this element.
		 */
		return( fn( ths, a, b ) );
	}
}

/* Useful test function for map. Have we found something?
 */
void *
im_list_eq( void *a, void *b )
{		
	if( a == b )
		return( a );
	else
		return( NULL );
}

/* Fold up a list.
 */
void *
im_list_fold( List *l, void *start, im_list_fold_fn fn, void *a, void *b )
{	
	void *c = start;
	List *nxt;

	for( ; l; l = nxt ) {
		nxt = tl(l);

		c = fn( hd(l), c, a, b );
	}

	return( c );
}

/* 'Fix' a list with a function. We repeatedly map the function over the list
 * until it returns NULL for all elements. This is useful if the function we 
 * are mapping changes the list, perhaps removing or adding an element. The
 * function one fixes the list with should return non-NULL if it makes a
 * change of this sort, NULL otherwise. Since the list may be changing, pass a
 * pointer to the base.
 */
void
im_list_fix( List **base, im_list_map_fn fn, void *a, void *b )
{	
	while( im_list_map( *base, fn, a, b ) )
		;
}

/* Add a new element to front of a list. Side effect - so pass pointer to 
 * old head.
 */
/*VARARGS1*/
int
im_list_add( List **base, void *new )
{	
	List *n;

	if( !(n = IM_NEW( NULL, List )) )
		return( -1 );

	hd(n) = new;
	tl(n) = *base;
	*base = n;

	return( 0 );
}

/* Insert a new element in a list before an object already there. 
 */
int
im_list_insert( List **base, void *new, void *old )
{
	for( ; *base; base = &tl(*base) )
		if( hd(*base) == old ) {
			/* Found!
			 */
			if( im_list_add( base, new ) )
				return( -1 );
			
			return( 0 );
		}

	im_errormsg( "im_list_insert: not found" );
	return( -1 );
}

/* Append a new element to the end of a list. Again, pointer to old head.
 */
/*VARARGS1*/
int
im_list_append( List **base, void *new )
{
	List **n;

	for( n = base; *n; n = &tl( *n ) )
		;

	return( im_list_add( n, new ) );
}

/* Remove an item from a List. Side-effect, so pass pointer to old head.
 */
/*VARARGS1*/
int
im_list_remove( List **base, void *t )
{	
	List **n;
	List *f;

	for( n = base; *n; n = &tl( *n ) )
		if( hd( *n ) == t ) {
			f = *n;
			*n = tl( *n );
			im_free( f );

			return( 0 );
		}

	im_errormsg( "im_list_remove: not found" );
	return( -1 );
}

/* Free a list. We free the nodes one by one, applying a user freeing
 * function to each first. Pass pointer to head. If the user function is NULL,
 * then just free List nodes.
 */
/*VARARGS1*/
void
im_list_free( List **base, im_list_free_fn fn, void *a, void *b )
{	
	List *l, *nxt;

	for( l = *base; l; l = nxt  ) {
		nxt = tl(l);

		if( fn ) 
			fn( hd(l), a, b );
		im_free( l );
	}

	*base = NULL;
}
