/* Support for thread stuff.
 * 
 * JC & KM 9/5/94
 * Modified:
 * 28/11/94 JC
 *	- return(0) missing from tidy_thread_info()
 * 4/8/99 RP JC
 *	- reorganised for POSIX
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>

#include <vips/vips.h>
#include <vips/util.h>
#include <vips/list.h>
#include <vips/rect.h>
#include <vips/region.h>
#include <vips/thread.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Comment out to stop debg msg.
#define DEBUG_IO
 */

/* Mutex lock and unlock on an image.
 */
void
im_lock( im_lock_t *l )
{
#ifdef HAVE_THREADS
#ifdef SOLARIS_THREADS 
	if( mutex_lock( (mutex_t *) l ) ) 
#elif HAVE_PTHREAD
	if( pthread_mutex_lock( (pthread_mutex_t *) l ) )
#endif
		error_exit( "panic: unable to lock" );
#endif /*HAVE_THREADS*/
}

void
im_unlock( im_lock_t *l )
{
#ifdef HAVE_THREADS
#ifdef SOLARIS_THREADS 
	if( mutex_unlock( (mutex_t *) l ) ) 
#elif HAVE_PTHREAD
	if( pthread_mutex_unlock( (pthread_mutex_t *) l ) )
#endif
		error_exit( "panic: unable to unlock" );
#endif /*HAVE_THREADS*/
}

void
im_lock_init( im_lock_t *l )
{
#ifdef HAVE_THREADS
#ifdef SOLARIS_THREADS 
	if( mutex_init( (mutex_t *) l, USYNC_THREAD, NULL ) ) 
#elif HAVE_PTHREAD
	if( pthread_mutex_init( (pthread_mutex_t *) l, NULL ) )
#endif
		error_exit( "panic: unable to init lock" );
#endif /*HAVE_THREADS*/
}

void
im_lock_destroy( im_lock_t *l )
{
#ifdef HAVE_THREADS
#ifdef SOLARIS_THREADS 
	if( mutex_destroy( (mutex_t *) l ) ) 
#elif HAVE_PTHREAD
	if( pthread_mutex_destroy( (pthread_mutex_t *) l ) )
#endif
		error_exit( "panic: unable to destroy lock" );
#endif /*HAVE_THREADS*/
}

/* Semaphore implementaion.
 */
void
im_semaphore_init( im_semaphore_t *s, int v, char *name )
{
	s->v = v;
	s->name = name;

	im_lock_init( &s->mutex );

#ifdef HAVE_THREADS
#ifdef HAVE_PTHREAD
	if( pthread_cond_init( &s->cond, NULL ) )
#elif SOLARIS_THREADS
	if( cond_init( &s->cond, NULL, NULL ) )
#endif
		error_exit( "panic: unable to init semaphore condition" );
#endif /*HAVE_THREADS*/
}

void
im_semaphore_destroy( im_semaphore_t *s )
{
	im_lock_destroy( &s->mutex );

#ifdef HAVE_THREADS
#ifdef HAVE_PTHREAD
	if( pthread_cond_destroy( &s->cond ) )
#elif SOLARIS_THREADS
	if( cond_destroy( &s->cond ) )
#endif
		error_exit( "panic: unable to destroy semaphore condition" );
#endif /*HAVE_THREADS*/
}

/* Add n to the semaphore and signal any threads that are blocked waiting 
 * a change.
 */
int
im_semaphore_upn( im_semaphore_t *s, int n )
{
	int value_after_op;

	im_lock( &s->mutex );
	s->v += n;
	value_after_op = s->v;
	im_unlock( &s->mutex );

#ifdef HAVE_THREADS
#ifdef HAVE_PTHREAD
	if( pthread_cond_signal( &s->cond ) )
#elif SOLARIS_THREADS
	if( cond_signal( &s->cond ) )
#endif
		error_exit( "panic: unable to signal semaphore condition" );
#endif /*HAVE_THREADS*/

#ifdef DEBUG_IO
	printf( "im_semaphore_upn(\"%s\",%d) = %d\n", 
		s->name, n, value_after_op );
	if( value_after_op > 1 )
		im_errormsg( "up over 1!" );
#endif /*DEBUG_IO*/

	return( value_after_op );
}

/* Increment the semaphore.
 */
int
im_semaphore_up( im_semaphore_t *s )
{
	return( im_semaphore_upn( s, 1 ) );
}

/* Wait for sem>n, then subtract n.
 */
int
im_semaphore_downn( im_semaphore_t *s, int n )
{
	int value_after_op;

	im_lock( &s->mutex );
#ifdef HAVE_THREADS
	while( s->v < n )
#ifdef HAVE_PTHREAD
		if( pthread_cond_wait( &s->cond, &s->mutex ) )
#elif SOLARIS_THREADS
		if( cond_wait( &s->cond, &s->mutex ) )
#endif
			error_exit( "panic: im_semaphore_down()" );
#endif /*HAVE_THREADS*/
	s->v -= n;
	value_after_op = s->v;
	im_unlock( &s->mutex );

#ifdef DEBUG_IO
	printf( "im_semaphore_downn(\"%s\",%d): %d\n", 
		s->name, n, value_after_op );
#endif /*DEBUG_IO*/

	return( value_after_op );
}

/* Wait for sem>0, then decrement.
 */
int
im_semaphore_down( im_semaphore_t *s )
{
	return( im_semaphore_downn( s, 1 ) );
}

/* Very basic thread layer.
 */
void
im_thread_create( void *(*start_func)( void * ), void *arg, im_threadid_t *id )
{
#ifdef HAVE_THREADS
#ifdef HAVE_PTHREAD
	if( pthread_create( (pthread_t *) id, NULL, start_func, arg ) ) {
#elif SOLARIS_THREADS
        if( thr_create( NULL, 0, start_func, arg, 0, (thread_t *) id ) ) {
#endif
		im_errormsg_system( errno, "im_thread_create() failed" );
                error_exit( "did you forget to link with the thread library?" );
	}
#endif /*HAVE_THREADS*/
}

/* Pick up the return value from a thread.
 */
void
im_thread_join( im_threadid_t id, void **ret )
{
#ifdef HAVE_THREADS
#ifdef HAVE_PTHREAD
	if( pthread_join( (pthread_t) id, ret ) )
#elif SOLARIS_THREADS
	if( thr_join( (thread_t) id, NULL, ret ) )
#endif 
		error_exit( "panic: im_thread_join() failed" );
#endif /*HAVE_THREADS*/
}
