/*
   Copyright (c) 2010 Sebastian Trueg <trueg@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "converterjob.h"
#include "virtuosocontroller.h"
#include "nepomukstuff.h"
#include "isql.h"

#include <KLocale>
#include <KStandardDirs>
#include <KDebug>
#include <KUrl>
#include <kio/netaccess.h>
#include <kio/global.h>
#include <kio/copyjob.h>

#include <QtCore/QDir>
#include <QtCore/QDateTime>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>


namespace {
    KUrl::List allFilesInFolder( const QString& path ) {
        QDir dir( path );
        QStringList files = dir.entryList( QDir::Files );
        KUrl::List urls;
        foreach( const QString& file, files ) {
            urls << KUrl( dir.absoluteFilePath( file ) );
        }
        return urls;
    }
}


ConverterJob::ConverterJob( QObject* parent )
    : KJob( parent ),
      m_dumpTmpDir( Nepomuk::storagePath() ),
      m_backup( true ),
      m_keepDump( false )
{
    m_virtuosoController = new Soprano::VirtuosoController();
    m_virtuosoController->setDirsAllowed( QStringList() << KUrl( m_dumpTmpDir.name() ).path(KUrl::RemoveTrailingSlash) );
}


ConverterJob::~ConverterJob()
{
    delete m_virtuosoController;
}


void ConverterJob::setBackup( bool backup )
{
    m_backup = backup;
}


void ConverterJob::setKeepDump( bool keep )
{
    m_keepDump = keep;
}


void ConverterJob::start()
{
    if ( QDBusConnection::sessionBus().interface()->isServiceRegistered( "org.kde.NepomukStorage" ) ) {
        setErrorText( i18n( "Please disable Nepomuk before starting the converter." ) );
        emitResult();
        return;
    }


    m_dumpTmpDir.setAutoRemove( !m_keepDump );

    if ( m_backup ) {
        m_backupPath = Nepomuk::storagePath() + "/backup/" + QDateTime::currentDateTime().toString( "yyyy-MM-ddThh-mm-ss" );
        emit infoMessage( this, i18n( "Creating backup in <filename>%1</filename>...", m_backupPath ) );
        KStandardDirs::makeDir( m_backupPath );
        KIO::Job* job = KIO::copy( allFilesInFolder( Nepomuk::storagePath() ), m_backupPath );
        if ( !KIO::NetAccess::synchronousRun( job, 0 ) ) {
            setErrorText( i18n( "Failed to create backup (%1).", job->errorText() ) );
            emitResult();
            return;
        }
    }

    emit infoMessage( this, i18n( "Starting Virtuoso version 5..." ) );
    if ( !startVirtuoso( 5 ) ) {
        emitResult();
        return;
    }

    connect( ISql::fireAndForgetCommand( m_virtuosoController->usedPort(), createDumpGraphs() ),
             SIGNAL( finished( bool ) ),
             this, SLOT( slotCreateDumpGraphsDone( bool ) ) );
}


void ConverterJob::slotCreateDumpGraphsDone( bool success )
{
    kDebug() << success;

    if ( !success ) {
        setErrorText( i18n( "Failed to create stored procedure (%1)", QLatin1String( "dump_graphs" ) ) );
        emitResult();
        return;
    }

    // start the dump
    emit infoMessage( this, i18n( "Dumping V5 database to %1...", m_dumpTmpDir.name() ) );

    connect( ISql::fireAndForgetCommand( m_virtuosoController->usedPort(), QString( "dump_graphs('%1');" ).arg( m_dumpTmpDir.name() ) ),
             SIGNAL( finished( bool ) ),
             this, SLOT( slotDumpGraphsDone( bool ) ) );
}


void ConverterJob::slotDumpGraphsDone( bool success )
{
    kDebug() << success;

    // shut down V5
    emit infoMessage( this, i18n( "Shutting down Virtuoso version 5..." ) );
    m_virtuosoController->shutdown();

    // handle error
    if ( !success ) {
        setErrorText( i18n( "Failed to dump all graphs in the database to %1", m_dumpTmpDir.name() ) );
        cleanupAndEmitResult();
        return;
    }

    // delete all the old data
    emit infoMessage( this, i18n( "Deleting Virtuoso V5 database files..." ) );
    removeNepomukDb();

    // start V6
    emit infoMessage( this, i18n( "Starting Virtuoso version 6..." ) );
    if ( !startVirtuoso( 6 ) ) {
        cleanupAndEmitResult();
        return;
    }

    // enabled full-text index
    connect( ISql::fireAndForgetCommand( m_virtuosoController->usedPort(), "DB.DBA.RDF_OBJ_FT_RULE_ADD( null, null, 'nepomuk' ); DB.DBA.VT_BATCH_UPDATE ('DB.DBA.RDF_OBJ', 'OFF', null);" ),
             SIGNAL( finished( bool ) ),
             this, SLOT( slotFulltextIndexingStateUpdated( bool ) ) );
}


void ConverterJob::slotFulltextIndexingStateUpdated( bool )
{
    // create load_graphs procedure
    connect( ISql::fireAndForgetCommand( m_virtuosoController->usedPort(), createLoadGraphs() ),
             SIGNAL( finished( bool ) ),
             this, SLOT( slotCreateLoadGraphsDone( bool ) ) );
}


void ConverterJob::slotCreateLoadGraphsDone( bool success )
{
    kDebug() << success;

    // handle error
    if ( !success ) {
        setErrorText( i18n( "Failed to create stored procedure (%1)", QLatin1String( "load_graphs" ) ) );
        cleanupAndEmitResult();
        return;
    }

    // load all graphs
    emit infoMessage( this, i18n( "Importing dump from %1 into V6 database.", m_dumpTmpDir.name() ) );
    connect( ISql::fireAndForgetCommand( m_virtuosoController->usedPort(), QString( "load_graphs('%1');" ).arg( m_dumpTmpDir.name() ) ),
             SIGNAL( finished( bool ) ),
             this, SLOT( slotLoadGraphsDone( bool ) ) );
}


void ConverterJob::slotLoadGraphsDone( bool success )
{
    kDebug() << success;

    // handle error
    if ( !success ) {
        setErrorText( i18n( "Failed to import database dump." ) );
        cleanupAndEmitResult();
        return;
    }

    // remove stored procedure
    connect( ISql::fireAndForgetCommand( m_virtuosoController->usedPort(), QString( "drop procedure load_graphs;" ) ),
             SIGNAL( finished( bool ) ),
             this, SLOT( slotDumpLoadGraphsDone( bool ) ) );
}


void ConverterJob::slotDumpLoadGraphsDone( bool success )
{
    kDebug() << success;

    if ( m_backup ) {
        emit infoMessage( this, i18n( "Removing backup data." ) );
        KIO::NetAccess::del( m_backupPath, 0 );
    }

    cleanupAndEmitResult();
}


void ConverterJob::cleanupAndEmitResult()
{
    // shut down Virtuoso
    if ( m_virtuosoController->isRunning() )
        m_virtuosoController->shutdown();

    if ( !errorText().isEmpty() &&
         m_backup ) {
        restoreBackup();
    }

    // delete tmpDir
    if ( m_keepDump ) {
        emit infoMessage( this, i18n( "Keeping V5 dump in <filename>%1</filename>", m_dumpTmpDir.name() ) );
    }
    else {
        emit infoMessage( this, i18n( "Removing temp data." ) );
        KIO::NetAccess::del( m_dumpTmpDir.name(), 0 );
    }

    // error is set before
    emitResult();
}


void ConverterJob::restoreBackup()
{
    emit infoMessage( this, i18n( "Conversion failed. Restoring backup..." ) );
    removeNepomukDb();
    KIO::Job* job = KIO::copy( allFilesInFolder( m_backupPath ), Nepomuk::storagePath() );
    if ( !KIO::NetAccess::synchronousRun( job, 0 ) ) {
        setErrorText( i18n( "Failed to restore backup (%1).", job->errorText() ) );
    }
}


void ConverterJob::removeNepomukDb()
{
    foreach( const KUrl& file, allFilesInFolder( Nepomuk::storagePath() ) ) {
        KIO::NetAccess::del( file, 0 );
    }
}


bool ConverterJob::startVirtuoso( int version )
{
    kDebug() << version;

    const QString nepomukDir = Nepomuk::storagePath();
    QString virtuosoPath;

    if ( version == 6 ) {
        virtuosoPath = Soprano::VirtuosoController::locateVirtuosoBinary();
    }
    else {
        virtuosoPath = KStandardDirs::locate( "data", "virtuosoconverter/v5/virtuoso-t" );
    }

    return m_virtuosoController->start( virtuosoPath, nepomukDir );
}


// static
QString ConverterJob::createDumpGraphs()
{
    QFile f( KStandardDirs::locate( "data", "virtuosoconverter/dump_graphs.sql" ) );
    f.open( QIODevice::ReadOnly );
    return f.readAll();
}


// static
QString ConverterJob::createLoadGraphs()
{
    QFile f( KStandardDirs::locate( "data", "virtuosoconverter/load_graphs.sql" ) );
    f.open( QIODevice::ReadOnly );
    return f.readAll();
}

#include "converterjob.moc"
