/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file Line.hpp
 */


#ifndef LINE_HPP
#define LINE_HPP

#include <string>
#include <vector>
#include <iostream>
#include <fstream>

#include "TokenSource.hpp"



/*! \class Line Line.hpp "../source/src/parser/Line.hpp"
 *  \brief Line object stand for paje format syntax unit .
 */

class Line{

private:
    std::vector<std::string> _tokens;
 
    TokenSource::TokenSource _source; // TokenSource namespaced for windows support
 
    bool _is_eof;
    unsigned int _line_count;

    void clear();

public:
    /*!
     *  \brief Constructor for the line
     */
    Line();

    /*!
     *  \brief Constructor for the line
     *  \param l A reference of the line to be copied
     */
    Line(Line &l);

    /*!
     *  \brief  Constructor for the line
     *  \param  filename : a filename
     */
    Line(std::string &filename);
    
    /*!
     *  \brief Destructor
     *  Destroy the line
     */
    ~Line();
    
    /*!
     *  \fn starts_with(const std::string &s) const
     *  \param s : a string
     *  \return true if the line starts with the param, false else
     */
    bool starts_with(const std::string &s) const;

    /*!
     *  \fn is_eof() const
     *  \brief test if file ended
     *  \return true if more lines are available
     */
    bool is_eof() const;

    /*!
     *  \fn newline()
     *  \brief step to the next line of the file, a call to eof is required before newline()
     *  
     */
    void newline();


    /*!
     *  \fn operator==(const Line &l) const
     *  \brief true if the lines contains the same items
     *  \param l the line we want to check if it is equal to this
     *  \return true if the lines are equals
     *  
     */
    bool operator==(const Line &l) const;

    /*!
     *  \fn item(unsigned int i, std::string &e) const
     *  \brief the ith token in the line
     *  \param i the number of the token we want
     *  \param e the string where we store the name of the token
     *  \return true if the string is filled, false if no item exists at this index.
     */
    bool item(unsigned int i, std::string &e) const;

    /*!
     * \fn print() const
     * \brief print the line
     *  Useful for debug.
     * 
     */
    void print() const;

    /*!
     *  \fn length() const
     *  \brief number of token
     *  \return the number of tokens in the line.
     */
    unsigned int length() const;
    
    /*!
     *  \fn get_line_count() const
     *  \brief read line amount
     *  \return the number of lines already read.
     *
     */
    unsigned int get_line_count() const;
  
    /*!
     *  \fn    open(const char *filename)
     *  \brief open a file
     *  \param filename valid filename
     */
     void open(const char *filename);

};

#endif // LINE_HPP
