/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file interface_console.cpp
 *\brief This is the console interface C source code.
 */


#include <queue>
#include <string>
#include <iostream>
#include <fstream>
#include <list>
#include <map>
#include <vector>
#include <stack>
/* -- */
#include <QObject>
/* -- */
#include "common/common.hpp"
#include "common/info.hpp"
#include "common/Errors.hpp"
#include "common/Tools.hpp"
/* -- */


//#include "render/render.hpp"
#include "render/Render_svg.hpp"
#include "render/Render.hpp"
/* -- */
#include "trace/values/Values.hpp"
#include "trace/tree/Interval.hpp"
#include "trace/tree/Node.hpp"
#include "trace/tree/BinaryTree.hpp"
#include "trace/EntityValue.hpp"
#include "trace/EntityTypes.hpp"
#include "trace/Entitys.hpp"
//#include "trace/tree/Interval.hpp"
#include "trace/Trace.hpp"
#include "trace/DrawTree.hpp"
#include "trace/DrawTrace.hpp"
/* -- */
#include "parser/File.hpp"
#include "parser/Parser.hpp"
#include "parser/ParserPaje.hpp"
#ifdef WITH_OTF
#include <otf.h>
#include "parser/ParserOTF.hpp"
#endif //WITH_OTF
#include "parser/ParserVite.hpp"
/* -- */
#include "statistics/Stats_window.hpp"
/* -- */
#include "interface/resource.hpp"
#include "interface/Interface_graphic.hpp"
#include "core/Core.hpp"
#include "interface/parsing_thread.hpp"
/* -- */
#ifdef WITH_VBO
#include "render/Render_alternate.hpp"
#else
#include "render/Render_opengl.hpp"
#endif


using namespace std;

#define message *Message::get_instance() << "(" << __FILE__ << " l." << __LINE__ << "): "

/***********************************
 *
 *
 *
 * Constructor and destructor.
 *
 *
 *
 **********************************/

Core::Core(int argc, char ** argv){
    
    
    bool useGUI;/* if window interface can be displayed */
    //  QString current_path;


    app = new QApplication(argc, argv);/* create the Qt application */        
 
    


    //  glutInit(&argc, argv);/* use for OpenGL text */

    /* Qt uses the default system encoding for QString (used when opening a file) */
    QTextCodec::setCodecForCStrings(QTextCodec::codecForLocale()); 


    useGUI = true;
    
  
    //    cerr << "Core debug: " << QDir::currentPath().toStdString().c_str() << " " << argv[0] << endl;
    


    /*  if (!useGUI){
        cerr << QObject::tr("Error: Graphical window cannot be displayed.").toStdString() << endl;
        exit(EXIT_FAILURE);
        }*/
    
    _main_window = NULL;
    _render_opengl =  NULL;
    _trace = NULL;
    
    /* Init of the times */
    _time_start = 0;
    _time_end = 0;

    _run_env[0] = new QString(QDir::currentPath().toStdString().c_str());
    _run_env[1] = new QString(argv[0]);
    
    Message::set_interface(this);/* define which interface will receive messages */
    

    _state = get_state(argc, argv);

    /*
     * Suppose that no window will be displayed.
     */
    _is_window_displayed = false;

    /*
     * No trace loaded
     */
    _is_trace_loaded = false;
    
    launch_action(_state);
}


Core::~Core(){
    
    delete app;
    if(_trace != NULL){
        delete _trace;
    }
    if (_run_env[0]!=NULL)
        delete _run_env[0];

    if (_run_env[1]!=NULL)
        delete _run_env[1];

    /* Qt desallocates _main_window and _render_opengl automatically */
    
    //free(_run_env[0]);
    Message::kill();

}



/***********************************
 *
 *
 *
 * Running function.
 *
 *
 *
 **********************************/

bool Core::draw_trace(const string & filename, const int format){

    Parser *parser;

    DrawTrace drawing_ogl;
    
#ifdef WITH_VBO
    Render<Render_alternate> render(_render_opengl);
#else
    Render<Render_opengl> render(_render_opengl);
#endif


    QTime time_elapsed;
    ostringstream buf_txt;
    int time_buf;
    time_buf = 0;


    // Get the parser in function of the extension of the file
    const unsigned int position_of_dot = filename.find_last_of('.');
    if (position_of_dot != string::npos) {
        if(filename.substr(position_of_dot) == ".trace") {
            parser = new ParserPaje();
        }
#ifdef WITH_OTF
        else if(filename.substr(position_of_dot) == ".otf") {
            parser = new ParserOTF();
        }
#endif //WITH_OTF
        else if(filename.substr(position_of_dot) == ".ept") {
            parser = new ParserVite();
        }
        else {
            Error::set(Error::_BAD_FILE_EXTENSION, Error::_WARNING);
            parser = new ParserPaje();        
        }
    }
    else {
        Error::set(Error::_BAD_FILE_EXTENSION, Error::_WARNING);
        parser = new ParserPaje();        
    }


    parser->set_file_to_parse(filename);
    
    QApplication::setOverrideCursor(Qt::WaitCursor);

    if (_DRAW_SVG == format)
        _render_svg = new Render_svg(&_path_to_export);// svg.init(_path_to_export.c_str());
        

            
    if (NULL == _trace) { /* no trace is loaded, parse the file */

        /* Init data */
        Info::Entity::x_min          = 0;
        Info::Entity::x_max          = 0;
        Info::Render::_x_min_visible = 0;
        Info::Render::_x_max_visible = 0;

        _trace = new Trace();
        
        // Init of the thread
        parsing_thread thread;
        thread.init(parser, _trace, filename);

        int loaded = 0;
        float loaded_f = 0.0f;/* floating value of the loading file state. (between 0 and 1) */
        
        if(_main_window != NULL) { // If we have a window we show a progress bar
            _progress_dialog = new QProgressDialog(QObject::tr("Parsing"), QObject::tr("Cancel"), 0, 100, _main_window);
            _progress_dialog->setWindowTitle(QObject::tr("Loading of ")+QString::fromStdString(filename));
            _progress_dialog->show();
            
            _main_window->setDisabled(true);
            _progress_dialog->setDisabled(false); // to be able to cancel while parsing
        }

        thread.start();
        time_elapsed.start();
        buf_txt.str("");
        
        while(!(parser->is_end_of_parsing() && thread.isFinished())) {
            loaded_f = parser->get_size_loaded();/* value between 0 and 1. Should be multiplied by 100 to obtain a percentage */
            loaded = (int)(loaded_f*100.0f);

#ifdef WIN32
	  Sleep(1000);
#else
	  sleep(1); // We wait 1 second
#endif
          cout << QObject::tr("Loading of the trace : ").toStdString() << loaded << "%" ;


          if (loaded_f>0.0f){
                /* divided by to have in second 1000 since time_elapsed.elapsed() returns ms */
              time_buf = (int)(time_elapsed.elapsed() * (1.0 / loaded_f - 1.0) / 1000);

                if (time_buf>=3600){/* convert second in hour and min */
                    buf_txt << "Parsing... Remaining: " <<  time_buf/3600 << " h " << (time_buf%3600)/60 << " min " << time_buf%60 << " s";
                }
                else if (time_buf>=60){/* convert second in min */
                    buf_txt << "Parsing... Remaining: " << time_buf/60 << " min " << time_buf%60 << " s";
                }
                else{
                    buf_txt << "Parsing... Remaining: " << time_buf << " s";
                }
               
                cout << "  ~  " << buf_txt.str(); 

                if (  (NULL!=_main_window)  &&  (NULL!=_progress_dialog)  )
                    _progress_dialog->setLabelText(QString( buf_txt.str().c_str()));

                buf_txt.str("");
          }

          cout << endl;


          if(_main_window != NULL) { // If we have a window we show a progress bar
                update_progress_bar(loaded);
                QApplication::processEvents();
            
                if(_progress_dialog->wasCanceled()) {
                    cout << QObject::tr("Canceled at ").toStdString() << loaded << "%" << endl;
                    parser->set_canceled();
                    *Message::get_instance() << QObject::tr("The trace opening was canceled by the user at ").toStdString() << loaded << "%" << Message::ende;
                   
                    break; // Quit the loop
                }
            }
        }
        
        Error::print_numbers();
        Error::flush("log.txt");
        
        if(_main_window != NULL) { // If we have a window we show a progress bar
            delete _progress_dialog;
            _main_window->setDisabled (false);
        }
        // Wait for the end of the thread
        while(!thread.wait()) {
        }
                
        delete parser;       
    }
    else if ( _file_opened != filename) {/* just check if execution is normal */
        *Message::get_instance() << "Try to use file: " << filename << " instead of a previous parsed file: " << _file_opened << Message::ende;
    }

    if (Info::Render::_x_min_visible == Info::Render::_x_max_visible){// first time
        _trace->set_interval_constrained(new Interval(0,_trace->get_max_date()));
    }else{
        _trace->set_interval_constrained(new Interval(Info::Render::_x_min_visible, Info::Render::_x_max_visible));
    }
         

    if (_DRAW_OPENGL == format) {
        drawing_ogl.build(&render, _trace);
        _render_opengl->build();
        _render_opengl->updateGL();
        _render_opengl->refresh_scroll_bars();
        _file_opened = filename;/* store filename for a future export */
    }
    else if (_DRAW_SVG == format) {

        Element_pos buf_min, buf_max;
        
        /* Store current view */
        buf_min = Info::Entity::x_min;
        buf_max = Info::Entity::x_max;

        /* Change current view to fit the user zoom */
        Info::Render::_x_min_visible = _time_start;

        if(_time_end == 0.) {
            Info::Render::_x_max_visible = _trace->get_max_date().get_value();
            Info::Entity::x_max = Info::Render::_x_max_visible;
        }
        else {
                Info::Render::_x_max_visible = _time_end;
        }
        
        if(Info::Entity::x_max >= Info::Render::_x_min_visible) {
            Info::Entity::x_min = Info::Render::_x_min_visible;
        }
        if(_time_end > _time_start && _time_end <= Info::Entity::x_max) {
            Info::Entity::x_max = _time_end;
        }
//         /* Change current view to fit the user zoom */
//         if(_state == _STATE_EXPORT_FILE_IN_INTERVAL) { // Command line
//             if(Info::Entity::x_max >= _time_start) {
//                 Info::Entity::x_min = _time_start;
//             }
//             if(_time_end > _time_start && _time_end <= Info::Entity::x_max) {
//                 Info::Entity::x_max = _time_end;
//             }
//         }
//         else { // Export launched by the graphical interface
//             Info::Entity::x_min = Info::Render::_x_min_visible;
//             Info::Entity::x_max = Info::Render::_x_max_visible;
//         }

        Render<Render_svg> render(_render_svg);
        drawing_ogl.build(&render, _trace);
  
        delete _render_svg;/* Release the svg render */

        /* Restore previous view */
        Info::Entity::x_min = buf_min;
        Info::Entity::x_max = buf_max;
        
        // svg.end();
    }else{/* error */
        *Message::get_instance() << "No kind of render recognized" << Message::ende;
        return false;
    }
    //   std::cout << _trace->get_max_date().get_value() << __FILE__ << __LINE__ << std::endl;
       
    
    _is_trace_loaded = true;
    QApplication::restoreOverrideCursor();  

    return true;
}


int Core::run(){

    /*
     * If a window is displayed, enter in the Qt event loop.
     */
    if (_is_window_displayed){
        return app->exec();
    }
    else{/* else, quit the application */
        return EXIT_SUCCESS;
    }
}




/***********************************
 *
 *
 *
 * The command line parameter processing functions.
 *
 *
 *
 **********************************/

int Core::get_state(int argc, char** argv){
 

    /*The following statics must be initialised in case their are not overrided by a command line request*/
    /*  Svg::set_height_factor (20);
    Svg::set_wide_factor (100);
    Svg::set_interval(0, 0);
    Svg::set_accuracy(0.1);
    Svg::set_scale_frequency(50);*/

    if(argc == 1){
        // just the name of the program, launch the window interface
        return _STATE_LAUNCH_GRAPHICAL_INTERFACE;
    }

    int state = _STATE_DISPLAY_HELP; 


    for(int i = 1 ; i < argc ; i ++) {

        if((string)argv[i] == "-h") {
            // display the help message
            return _STATE_DISPLAY_HELP;
        }
        else if((string)argv[i] == "-f" || (string)argv[i] == "-a") {
            // We want to open the file which follows this argument
            i ++;
            if(i < argc) {
                _file_opened = argv[i];
                state = _STATE_OPEN_FILE;
            }
            else {
                cerr << QObject::tr("no filename to open").toStdString() << endl;
                return _STATE_UNKNOWN;
            }
        }
        else if((string)argv[i] == "-e") {
            // We want to export the file which follows this argument
            i ++;
            if(i < argc) {
                _path_to_export = argv[i];
                state = _STATE_EXPORT_FILE;
            }
            else{
                cerr << QObject::tr("no filename for export").toStdString() << endl;
                return _STATE_UNKNOWN;
            }
        }
        else if((string)argv[i] == "-t") {
            // We want to export the file which follows this argument
            extract_times(argv[++ i]);
            if(_time_end == -1 || _time_start == -1){
                cerr << QObject::tr("One of the time do not exist").toStdString() << endl;
                state = _STATE_DISPLAY_HELP;
            }
            else if (_time_end < _time_start && _time_end != 0.){
                cerr << QObject::tr("The end time is lower than the start one!").toStdString() << endl;
                state = _STATE_DISPLAY_HELP;
            }
            state |= _STATE_IN_AN_INTERVAL;
        }
         else if(argv[i][0] == '-') {
             // This is here where we put new options !! (for filters for example)

        //     if((string)argv[i] == "-epsilon") {
        //         i ++;
        //         if(i > argc) {
        //             cerr << QObject::tr("need another argument for epsilon").toStdString() << endl;
        //             return _STATE_UNKNOWN;
        //         }
        //         else {
        //             double accuracy = convert_to_double(argv[i]);
        //             if(accuracy < 0.) {
        //                 cerr << QObject::tr("need a positive accuracy for epsilon").toStdString() << endl;
        //                 return _STATE_UNKNOWN;
        //             }
        //             else {
        //                 //  Svg::set_accuracy(accuracy);
        //             }
        //         }
        //     }
        //     else if((string)argv[i] == "-wide") {
        //         i ++;
        //         if (i > argc) {
        //             cerr << "need another argument for wide" << endl;
        //             return _STATE_UNKNOWN;
        //         }
        //         double wide = convert_to_double(argv[i]);
        //         if(wide < 0.) {
        //             cerr << "need a positive width for the export" << endl;
        //             return _STATE_UNKNOWN;
        //         }
        //         else {
        //             //  Svg::set_wide_factor(wide);
        //         }
        //     }
        //     else if((string)argv[i] == "-grow") {
        //         i ++;
        //         if (i > argc) {
        //             cerr << "need another argument for grow" << endl;
        //             return _STATE_UNKNOWN;
        //         }
        //         double grow = convert_to_double(argv[i]);
        //         if(grow < 0.) {
        //             cerr << "need a positive grown for the export" << endl;
        //             return _STATE_UNKNOWN;
        //         }
        //         else {
        //             //  Svg::set_height_factor(grow);
        //         }
        //     }
        //     else if((string)argv[i] == "-scale") {
        //         i ++;
        //         if (i > argc) {
        //             cerr << "need another argument for scale" << endl;
        //             return _STATE_UNKNOWN;
        //         }
        //         double scale = convert_to_double(argv[i]);
        //         if(scale < 0.) {
        //             cerr << "need a positive scale for the export" << endl;
        //             return _STATE_UNKNOWN;
        //         }
        //         else {
        //             //  Svg::set_scale_frequency(scale);
        //         }
        //     }
        //     else 
             if((string)argv[i] == "-interval") {
              
                if (!(i +2 < argc)) {
                    cerr << "interval argc" << endl;
                    return _STATE_UNKNOWN;
                }
                if (argv[i+1][0] == '-') {
                    cerr << "interval -" << endl;
                    return _STATE_UNKNOWN;
                }
                if (!(argv[i+2][0] != '-'))  {
                    cerr << "interval -" << endl;
                    return _STATE_UNKNOWN;
                }
	   
                double t1 = atof(argv[i+1]);
                double t2 = atof(argv[i+2]);
	   
                if (t1 >= t2) {
                    cerr << "interval neg" << endl;
                    return _STATE_UNKNOWN;
                }

                // Svg::set_interval(t1, t2);
                i += 2;
            }
            else {
                cerr << "the argument "<< argv[i] << " is unknown." << endl;
                return _STATE_DISPLAY_HELP;
            }
	  
        }
        else{ // This is a file to open
            _file_opened = argv[i];
            state = _STATE_OPEN_FILE;
        }
    }
    return state;
}

void Core::extract_times(const char *name) {
    bool has_time_start = false;
    bool has_time_end = false;
    string temp = name;
    if(name[0] == '['){
        has_time_start = true;
    }
    if(name[strlen(name)-1] == ']') {
        has_time_end = true;
    }
            
    if(has_time_start && has_time_end) {
        string start = temp.substr(temp.find('[')+1, temp.find(']')-1);
        temp = temp.substr(temp.find(']')+1);
        string end = temp.substr(temp.find('[')+1, temp.find(']')-2);
        _time_start = convert_to_double(start);
        _time_end = convert_to_double(end);
    }
    else if(has_time_start) {
        _time_start = convert_to_double(temp.substr(temp.find('[')+1, temp.find(']')-1));
    }
    else if(has_time_end) {
        _time_end = convert_to_double(temp.substr(temp.find('[')+1, temp.find(']')-2));
    }
    else{
        _time_start = 0;
        _time_end = 0;
    }
}

void Core::launch_action(int state, void* arg) {

    DrawTrace buf;
    QGLFormat format(QGL::HasOverlay);

    switch(state) {

    case _STATE_DISPLAY_HELP :
        display_help();
        break;
        
    case _STATE_LAUNCH_GRAPHICAL_INTERFACE :
    case _STATE_OPEN_FILE:


        if ( _STATE_OPEN_FILE == state)
            message << QObject::tr("Opening the file: ").toStdString ()+_file_opened << Message::endi;
        
        _main_window = new Interface_graphic (this);/* launch the window interface */
        Message::set_interface (_main_window);/* define which interface will receive messages */

        //   format.setOverlay(true);
#ifdef WITH_VBO
        _render_opengl = new Render_alternate (this, _main_window, format);
#else
        _render_opengl = new Render_opengl (this, _main_window, format);
#endif

        if ( NULL == _render_opengl){
            message <<  QObject::tr("Cannot allocate memory for an OpengGL instance").toStdString () << Message::ende;
            break;
        }

        if (NULL == _render_opengl->context()){
            message <<  QObject::tr("Cannot allocate an OpengGL context").toStdString () << Message::ende;
            break;
        }
        
        if (false == _render_opengl->isValid()){
            message <<  QObject::tr("Invalid context: OpenGL is not supported on your system").toStdString () << Message::ende;
         }


        // if (false == _render_opengl->format().hasOverlay ()){
        //     /* If no overlay was created, check the original format */
 
        //     if (false == _render_opengl->context()->requestedFormat().hasOverlay())
        //         message <<  QObject::tr("No overlay format originally created").toStdString () << Message::endw;
        //     else
        //         message <<  QObject::tr("An overlay format was originally asked, but was not created").toStdString () << Message::endw;
        // }

        _main_window->bind_render_area((QGLWidget*)_render_opengl);

        if ( _STATE_OPEN_FILE == state){
            _main_window->opening_file(_file_opened);/* Must be called after binding the render area to the main window */
            
            if(false==draw_trace(_file_opened, _DRAW_OPENGL))
                message <<  QObject::tr("Draw trace failed").toStdString () << Message::ende;
        }

        _is_window_displayed = true;
        break;

    case _STATE_OPEN_FILE_IN_AN_INTERVAL:
        
        launch_action(_STATE_OPEN_FILE, NULL); 
        
        break;

    case _STATE_RELEASE_RENDER_AREA:
        if (_render_opengl->unbuild()==false)
            message << "Close file : an error occured with trace releasing." << Message::ende;
        _file_opened.clear();

        if (false == _is_trace_loaded){
            *Message::get_instance() << "Try to release a render area whereas no file was loaded" << Message::ende;
        }else{
             _is_trace_loaded = false;
        }

        if (NULL == _trace){
            *Message::get_instance() << "Try to release a render area whereas no trace is loaded" << Message::ende;
        }else{
            delete _trace;
            _trace = NULL;
        }

       
        _render_opengl->updateGL();

        /* Release all data */
        Info::release_all();

        break;

        
    case _STATE_EXPORT_FILE:
        if (_file_opened.empty()){
            *Message::get_instance() << "Please to previously open a trace." << Message::endw;
            return;
        }

        *Message::get_instance() << "export of " << _file_opened << " to " << _path_to_export << Message::endi;
        
        draw_trace(_file_opened, _DRAW_SVG);
       
        break;

    case _STATE_EXPORT_FILE_IN_INTERVAL:
        cout << "export of " << _file_opened << " to " << _path_to_export << " between ";
        if(_time_start == 0){
            cout << "the beginning of the trace to ";
        }
        else{
            cout << _time_start << " seconds to ";
        }
        if(_time_end != 0){
            cout << _time_end << " seconds." << endl;
        }
        else{
            cout << "the end of the trace.";
        }
        
        // Svg::set_interval(_time_start, _time_end);
        draw_trace(_file_opened, _DRAW_SVG);
        
        break;

    case _STATE_RENDER_AREA_CHANGE_TRANSLATE:
        _render_opengl->change_translate( *((int*)arg) );
        break;
	
    case _STATE_RENDER_AREA_CHANGE_SCALE:
        _render_opengl->change_scale( *((Element_pos*)arg) );
        break;

    case _STATE_RENDER_AREA_CHANGE_SCALE_Y:
        _render_opengl->change_scale_y( *((Element_pos*)arg) );
        break;

    case _STATE_RENDER_AREA_CHANGE_CONTAINER_SCALE:
        _render_opengl->change_scale_container_state( *((int*)arg) );
        break;

    case _STATE_RENDER_AREA_REPLACE_SCALE:
        _render_opengl->replace_scale( *((Element_pos*)arg) );
        _render_opengl->replace_scale_y( *((Element_pos*)arg) );
        break;

    case _STATE_RENDER_AREA_REPLACE_TRANSLATE:
        _render_opengl->replace_translate( *((Element_pos*)arg) );
        break;

    case _STATE_RENDER_AREA_REPLACE_TRANSLATE_Y:
        _render_opengl->replace_translate_y( *((Element_pos*)arg) );
        break;

    case _STATE_RENDER_AREA_REGISTERED_TRANSLATE:
        _render_opengl->registered_translate( *((int*)arg) );
        break;
	
    case _STATE_AJUST_SCROLL_BARS:
	//	cerr << "console: x_max " << ((Element_pos*)arg)[0] << " | y_max " << ((Element_pos*)arg)[1] << endl;
	_main_window->set_scroll_bars_length( ((Element_pos*)arg)[0], ((Element_pos*)arg)[1] );
	break;
	
    case _STATE_REFRESH_SCROLL_BARS:
	_main_window->linking_scroll_bars( ((Element_pos*)arg)[0], ((Element_pos*)arg)[1] );
	break;
	
    case _STATE_ZOOM_BOX_VALUE:
        _main_window->change_zoom_box_value( *((int*)arg) );
        break;

    case _STATE_RENDER_DISPLAY_INFORMATION:
        
        if(_trace != NULL){
            buf.display_information(_trace, Info::Render::_info_x, Info::Render::_info_y, Info::Render::_info_accurate);
        }
        break;

    case _STATE_RENDER_UPDATE:
        _render_opengl->updateGL();
        break;

    default:/* like _STATE_UNKNOWN */
        display_help();
        warning(string("Cannot determine the arguments past. Please check the correct syntax."));
       
    }
}


void Core::display_help(){
    cout << endl 
         << "Usage: vite [OPTION...] [[-f] inputfile] [-e outputfile] " << endl
         << endl
         << "  -h                 display this help                  " << endl
         << "  -f inputfile       open the inputfile                 " << endl
         << "  -a                 display all the inputfile (Default)" << endl
         << "  -t [T0]:[T1]       display the interval [T0:T1] (Default: T0 = 0 and T1 = Tmax)" << endl 
         << "  -e outputfile      export the trace file in the svg format to outpufile"        << endl
	 << endl;
}




/***********************************
 *
 *
 *
 * Informative message functions.
 *
 *
 *
 **********************************/
 
void Core::error(const string &s) const{
    cerr << "ERROR: " << s <<endl;
}


void Core::warning(const string &s) const{
    cerr << "Warning: "<< s <<endl;
}

void Core::information(const string &s) const{
    cerr << s <<endl;
}

const string Core::get_filename() const{
    return _file_opened;
}

void Core::set_path_to_export(const string& path){
    _path_to_export = path;
}

const QString** Core::get_runenv() const{
    return (const QString**)_run_env;
}

void Core::update_progress_bar(const int loaded){
    _progress_dialog->setValue(loaded);
    _progress_dialog->update();
}

void Core::set_min_value_for_export(const double d) {
    _time_start = d;
}

void Core::set_max_value_for_export(const double d) {
    _time_end = d;
}

Trace *Core::get_trace() const {
    return _trace;
}


void Core::export_variable(Variable *var, string filename) {
    const list<pair<Date, Double> > *variable_values = var->get_values();

    ofstream file(filename.c_str(), ios::out | ios::trunc);

    if(file) {
        double first_value = 0.;
        double second_value = 0.;
        const double min = var->get_min().get_value();
        const double max = var->get_max().get_value();
        
        for(list<pair<Date, Double> >::const_iterator value = variable_values->begin();
            value != variable_values->end();
            value++) {
            first_value = (*value).first.get_value();
            second_value =((*value).second.get_value()-min)/(max-min) ;
            file << first_value << "\t" << second_value << endl;
        }
    
        file.close();
    }
    else { 
        // Error
        _main_window->error("Unable to open " + filename + " in order to export the counter");
    }
}
