/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015, 2016 by Kay F. Jahnke                     */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/// \file splinus.cc
///
/// \brief compare a periodic b-spline with a sine
/// 
/// This is a simple example using a periodic b-spline
/// over just two values: 1 and -1. This spline is used to approximate
/// a sine function. You pass the spline's desired degree on the command
/// line. Next you enter a number (interpreted as degrees) and the program
/// will output the sine and the 'splinus' of the given angle.
/// As you can see when playing with higher degrees, the higher the spline's
/// degree, the closer the match with the sine. So apart from serving as a
/// very simple demonstration of using a 1D periodic b-spline, it teaches us
/// that a periodic b-spline can approximate a sine.
/// To show off, we use long double as the spline's data type.
/// This program also shows a bit of functional programming magic, putting
/// together the 'splinus' functor from several of vspline's functional
/// bulding blocks.
///
/// compile with: clang++ -pthread -O3 -std=c++11 splinus.cc -o splinus

#include <assert.h>
#include <vspline/vspline.h>

int main ( int argc , char * argv[] )
{
  assert ( argc > 1 ) ;
  
  int degree = std::atoi ( argv[1] ) ;
  
  assert ( degree >= 0 && degree < 25 ) ;
  
  // create the bspline object
  
  typedef vspline::bspline < long double , 1 > spline_type ;
  
  spline_type bsp ( 2 ,                   // two values
                    degree ,              // degree as per command line
                    vspline::PERIODIC ,   // periodic boundary conditions
                    vspline::BRACED ,     // implicit scheme, bracing coeffs
                    -1 ,                  // horizon (unused w. BRACED)
                    0.0 ) ;               // no tolerance
          
  // the bspline object's 'core' is a MultiArrayView to the knot point
  // data, which we set one by one for this simple example:
  
  bsp.core[0] = 1.0L ;
  bsp.core[1] = -1.0L ;
  
  // now we prefilter the data
  
  bsp.prefilter() ;
  
  // we build 'splinus' as a functional construct. Inside the brace,
  // we 'chain' several vspline::unary_functors:
  // - a 'domain' which scales and shifts input to the spline's range
  //   we use the same values as for the gate: the gate has mapped
  //   the input to [ 90 , 450 ], and this is now translated to
  //   spline coordinates [ 0 , 2 ] by the domain.
  // - a periodic gate, mapping out-of-range values into the range.
  //   we want a range from 0.0 to 2.0 here.
  // - a b-spline evaluator calculating our result value
  //   from the result of chaining the two previous operations.
  //   so the evaluator receives values in the range of [ 0 , 2 ].
  // this 'chain' is wrapped into a vspline::callable object so that
  // we can conveniently access it by just calling it.

//   auto splinus
//   = vspline::grok
//     (   vspline::domain ( bsp , 90.0L , 450.0L )
//       + vspline::periodic ( 0.0L , 2.0L )
//       + vspline::evaluator < long double , long double > ( bsp ) ) ;

//   auto splinus
//   = vspline::grok
//     (   vspline::domain ( bsp , 90.0L , 450.0L )
//       + vspline::make_safe_evaluator < spline_type , long double > ( bsp ) ) ;

  auto splinus = vspline::grok
    (   vspline::domain ( bsp , 90.0L , 450.0L )
      + vspline::make_safe_evaluator < spline_type , long double > ( bsp )
    ) ;

  // alternatively we can use this construct. This will work just about
  // the same, but has a potential flaw: If arithmetic imprecision should
  // land the output of the periodic gate just slightly below 90.0, the
  // domain may produce a value just below 0.0, resulting in an out-of-bounds
  // access. So the construct above is preferable.
  // Just to demonstrate that vspline::grok aso produces an object that
  // can be used with function call syntax, we use vspline::grok here instead
  // of vspline::callable.
      
  auto splinus2
  = vspline::grok
    (   vspline::periodic ( 90.0L , 450.0L )
      + vspline::domain ( bsp , 90.0L , 450.0L )
      + vspline::evaluator < long double , long double > ( bsp ) ) ;

  // now we let the user enter arbitrary angles, calculate the sine
  // and the 'splinus' and print the result and difference:

  while ( true )
  {
    std::cout << " > " ;
    long double x ;
    std::cin >> x ;                       // get an angle
    long double xs = x * M_PI / 180.0L ;  // note: sin() uses radians 
    
    // finally we can produce both results. Note how we can use periodic_ev,
    // the combination of gate and evaluator, like an ordinary function.

    std::cout << "sin(" << x << ") = "
              << sin ( xs ) << std::endl
              << "splinus(" << x << ") = "
              << splinus ( x ) << std::endl
              << "splinus2(" << x << ") = "
              << splinus2 ( x ) << std::endl
              << "difference sin/splinus: "
              << sin ( xs ) - splinus ( x ) << std::endl
              << "difference sin/splinus2: "
              << sin ( xs ) - splinus2 ( x ) << std::endl
              << "difference splinus/splinus2: "
              << splinus2 ( x ) - splinus ( x ) << std::endl ;

  }
}
