/* Vuls - Vulnerability Scanner
Copyright (C) 2016  Future Architect, Inc. Japan.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

package scan

import (
	"reflect"
	"testing"

	"github.com/future-architect/vuls/config"
)

func TestParseDockerPs(t *testing.T) {

	var test = struct {
		in       string
		expected []config.Container
	}{
		`c7ca0992415a romantic_goldberg ubuntu:14.04.5
f570ae647edc agitated_lovelace centos:latest`,
		[]config.Container{
			{
				ContainerID: "c7ca0992415a",
				Name:        "romantic_goldberg",
				Image:       "ubuntu:14.04.5",
			},
			{
				ContainerID: "f570ae647edc",
				Name:        "agitated_lovelace",
				Image:       "centos:latest",
			},
		},
	}

	r := newRedhat(config.ServerInfo{})
	actual, err := r.parseDockerPs(test.in)
	if err != nil {
		t.Errorf("Error occurred. in: %s, err: %s", test.in, err)
		return
	}
	for i, e := range test.expected {
		if !reflect.DeepEqual(e, actual[i]) {
			t.Errorf("expected %v, actual %v", e, actual[i])
		}
	}
}

func TestParseLxdPs(t *testing.T) {

	var test = struct {
		in       string
		expected []config.Container
	}{
		`+-------+
| NAME  |
+-------+
| test1 |
+-------+
| test2 |
+-------+`,
		[]config.Container{
			{
				ContainerID: "test1",
				Name:        "test1",
			},
			{
				ContainerID: "test2",
				Name:        "test2",
			},
		},
	}

	r := newRedhat(config.ServerInfo{})
	actual, err := r.parseLxdPs(test.in)
	if err != nil {
		t.Errorf("Error occurred. in: %s, err: %s", test.in, err)
		return
	}
	for i, e := range test.expected {
		if !reflect.DeepEqual(e, actual[i]) {
			t.Errorf("expected %v, actual %v", e, actual[i])
		}
	}
}

func TestParseIp(t *testing.T) {

	var test = struct {
		in        string
		expected4 []string
		expected6 []string
	}{
		in: `1: lo: <LOOPBACK,UP,LOWER_UP> mtu 65536 qdisc noqueue state UNKNOWN \    link/loopback 00:00:00:00:00:00 brd 00:00:00:00:00:00
1: lo    inet 127.0.0.1/8 scope host lo
1: lo    inet6 ::1/128 scope host \       valid_lft forever preferred_lft forever
2: eth0: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 1500 qdisc pfifo_fast state UP qlen 1000\    link/ether 52:54:00:2a:86:4c brd ff:ff:ff:ff:ff:ff
2: eth0    inet 10.0.2.15/24 brd 10.0.2.255 scope global eth0
2: eth0    inet6 fe80::5054:ff:fe2a:864c/64 scope link \       valid_lft forever preferred_lft forever
3: eth1: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 1500 qdisc pfifo_fast state UP qlen 1000\    link/ether 08:00:27:36:76:60 brd ff:ff:ff:ff:ff:ff
3: eth1    inet 192.168.33.11/24 brd 192.168.33.255 scope global eth1
3: eth1    inet6 2001:db8::68/64 scope link \       valid_lft forever preferred_lft forever `,
		expected4: []string{"10.0.2.15", "192.168.33.11"},
		expected6: []string{"2001:db8::68"},
	}

	r := newRedhat(config.ServerInfo{})
	actual4, actual6 := r.parseIP(test.in)
	if !reflect.DeepEqual(test.expected4, actual4) {
		t.Errorf("expected %v, actual %v", test.expected4, actual4)
	}
	if !reflect.DeepEqual(test.expected6, actual6) {
		t.Errorf("expected %v, actual %v", test.expected6, actual6)
	}
}

func TestIsAwsInstanceID(t *testing.T) {
	var tests = []struct {
		in       string
		expected bool
	}{
		{"i-1234567a", true},
		{"i-1234567890abcdef0", true},
		{"i-1234567890abcdef0000000", true},
		{"e-1234567890abcdef0", false},
		{"i-1234567890abcdef0 foo bar", false},
		{"no data", false},
	}

	r := newRedhat(config.ServerInfo{})
	for _, tt := range tests {
		actual := r.isAwsInstanceID(tt.in)
		if tt.expected != actual {
			t.Errorf("expected %t, actual %t, str: %s", tt.expected, actual, tt.in)
		}
	}
}
