#include "branchobj.h"
#include "texteditor.h"
#include "mapeditor.h"
#include "mainwindow.h"

extern TextEditor *textEditor;
extern Main *mainWindow;
extern FlagRowObj *standardFlagsDefault;
extern QAction *actionEditOpenURL;


/////////////////////////////////////////////////////////////////
// BranchObj
/////////////////////////////////////////////////////////////////

BranchObj* BranchObj::itLast=NULL;


BranchObj::BranchObj () :OrnamentedObj()
{
//    cout << "Const BranchObj ()\n";
    setParObj (this);	
    init();
    depth=-1;
}

BranchObj::BranchObj (QCanvas* c):OrnamentedObj (c)
{
//    cout << "Const BranchObj (c)  called from MapCenterObj (c)\n";
	parObj=NULL;
    canvas=c;
}

BranchObj::BranchObj (QCanvas* c, LinkableMapObj* p):OrnamentedObj (c)
{
//    cout << "Const BranchObj (c,p)\n";
    canvas=c;
    setParObj (p);	
    depth=p->getDepth()+1;
	if (depth==1)
		// Calc angle to mapCenter if I am a mainbranch
		// needed for reordering the mainbranches clockwise 
		// around mapcenter 
		angle=getAngle (QPoint ((int)(x() - parObj->getChildPos().x() ), 
								(int)(y() - parObj->getChildPos().y() ) ) );
    init();
}

BranchObj::~BranchObj ()
{
//	cout << "Destr BranchObj of "<<this<<endl;
	// Check, if this branch was the last child to be deleted
	// If so, unset the scrolled flags

	BranchObj *po=(BranchObj*)(parObj);
	BranchObj *bo;
	if (po)
	{
		bo=((BranchObj*)(parObj))->getLastBranch();
		if (!bo) po->unScroll();
	}
	clear();
}

bool BranchObj::operator< ( const BranchObj & other )
{
    return  angle < other.angle;
}

bool BranchObj::operator== ( const BranchObj & other )
{
    return angle == other.angle;
}

int BranchObjPtrList::compareItems ( QPtrCollection::Item i, QPtrCollection::Item j)
{
	// Make sure PtrList::find works
	if (i==j) return 0;

	if ( ((BranchObj*)(i))->angle > ((BranchObj*)(j))->angle )
		return 1;
	else
		return -1;
}

void BranchObj::init () 
{
    branch.setAutoDelete (false);
    floatimage.setAutoDelete (true);
    xlink.setAutoDelete (false);

	if (parObj)
	{
		absPos=getRandPos();
		absPos+=parObj->getChildPos();
	}

    lastSelectedBranch=-1;

    setChildObj(this);

	scrolled=false;
	tmpUnscrolled=false;

	url="";
	vymLink="";
}

void BranchObj::copy (BranchObj* other)
{
    OrnamentedObj::copy(other);

	branch.clear();
    BranchObj* b;
    for (b=other->branch.first(); b;b=other->branch.next() ) 
		// Make deep copy of b
		// Because addBranch again calls copy for the childs,
		// Those will get a deep copy, too
		addBranch(b);	

	FloatImageObj *fi;
	for (fi=other->floatimage.first(); fi;fi=other->floatimage.next() )
		addFloatImage (fi);

	scrolled=other->scrolled;
	tmpUnscrolled=other->tmpUnscrolled;
	setVisibility (other->visible);

	url=other->url;
	vymLink=other->vymLink;

	angle=other->angle;

    positionBBox();
}

void BranchObj::clear() 
{
	floatimage.clear();
	while (!xlink.isEmpty())
		deleteXLink (xlink.first() );

	BranchObj *bo;
	while (!branch.isEmpty())
	{
		bo=branch.first();
		branch.removeFirst();
		delete (bo);
	}
}

int BranchObj::getNum()
{
	if (parObj)
		return ((BranchObj*)(parObj))->getNum ((BranchObj*)(this));
	else
		return 0;
}

int BranchObj::getNum(BranchObj *bo)
{
	// keep current pointer in branch, 
	// otherwise save might fail
	int cur=branch.at();
	int ind=branch.findRef (bo);
	branch.at(cur);
	return ind;
}

int BranchObj::getFloatImageNum(FloatImageObj *fio)
{
	return floatimage.findRef (fio);
}

int BranchObj::countBranches()
{
	return branch.count();
}

int BranchObj::countFloatImages()
{
	return floatimage.count();
}

int BranchObj::countXLinks()
{
	return xlink.count();
}

void BranchObj::setParObjTmp(LinkableMapObj* lmo, QPoint m, int off)
{
	// Temporary link to lmo
	// m is position of mouse pointer 
	// offset 0: default 1: below lmo   -1 above lmo  (if possible)


	BranchObj* o=(BranchObj*)(lmo);
	if (!parObjTmpBuf) 
		parObjTmpBuf=parObj;

	// ignore mapcenter and mainbranch
	if (lmo->getDepth()<2) off=0;
	if (off==0)
	{
		link2ParPos=false;
	//	parObj=o;
	}	
	else
	{	
		link2ParPos=true;
	//	if (off>0)
	//		parObj=o->getParObj();
	//	else	
	//		parObj=o->getParObj();
	//	parObj=o;
	}		
	parObj=o;

	depth=parObj->getDepth()+1;

	// setLinkStyle calls updateLink, only set it once
	if (style!=getDefLinkStyle() ) setLinkStyle (getDefLinkStyle());

	// Move temporary to new position at destination
	// Usually the positioning would be done by reposition(),
	// but then also the destination branch would "Jump" around...
	// Better just do it approximately
	if (depth==1)
	{	// new parent is the mapcenter itself

		QPoint p= normalise ( QPoint (m.x() - o->getChildPos().x(),
									  m.y() - o->getChildPos().y() ));
		if (p.x()<0) p.setX( p.x()-bbox.width() );
		move2RelPos (p);
	} else
	{	
		int y;
		if (off==0)
		{
			// new parent is just a branch, link to it
			QRect t=o->getBBoxSizeWithChilds();
			if (o->getLastBranch())
				y=t.y() + t.height() ;
			else
				y=t.y();

		} else
		{
			if (off<0)
				// we want to link above lmo
				y=o->y() - height() + 5;
			else	
				// we want to link below lmo
				// Bottom of sel should be 5 pixels above
				// the bottom of the branch _below_ the target:
				// Don't try to find that branch, guess 12 pixels
				y=o->getChildPos().y()  -height() + 12; 
		}	
		if (o->getOrientation()==OrientLeftOfCenter)
			move ( o->getChildPos().x() - linkwidth, y );
		else	
			move (o->getChildPos().x() + linkwidth, y );
	}	

	// updateLink is called implicitly in move
	reposition();	// FIXME shouldn't be this a request?
}

void BranchObj::unsetParObjTmp()
{
	if (parObjTmpBuf) 
	{
		link2ParPos=false;
		parObj=parObjTmpBuf;
		parObjTmpBuf=NULL;
		depth=parObj->getDepth()+1;
		setLinkStyle (getDefLinkStyle() );
		updateLink();
	}		
}

void BranchObj::unScroll()
{
	if (tmpUnscrolled) resetTmpUnscroll();
	if (scrolled) toggleScroll();
}

void BranchObj::toggleScroll()
{
	BranchObj *bo;
	if (scrolled)
	{
		scrolled=false;
		systemFlags->deactivate("scrolledright");
		for (bo=branch.first(); bo; bo=branch.next() )
		{
			bo->setVisibility(true);
		}
	} else
	{
		scrolled=true;
		systemFlags->activate("scrolledright");
		for (bo=branch.first(); bo; bo=branch.next() )
		{
			bo->setVisibility(false);
		}
	}
	calcBBoxSize();
	positionBBox();	
	move (absPos.x(), absPos.y() );
	forceReposition();
}

bool BranchObj::isScrolled()
{
	return scrolled;
}

bool BranchObj::hasScrolledParent(BranchObj *start)
{
	// Calls parents recursivly to
	// find out, if we are scrolled at all.
	// But ignore myself, just look at parents.

	if (this !=start && scrolled) return true;

	BranchObj* bo=(BranchObj*)(parObj);
	if (bo) 
		return bo->hasScrolledParent(start);
	else
		return false;
}

void BranchObj::tmpUnscroll()
{
	// Unscroll parent (recursivly)
	BranchObj* bo=(BranchObj*)(parObj);
	if (bo) bo->tmpUnscroll();
		
	// Unscroll myself
	if (scrolled)
	{
		tmpUnscrolled=true;
		systemFlags->activate("tmpUnscrolledright");
		toggleScroll();
	}	
}

void BranchObj::resetTmpUnscroll()
{
	// Unscroll parent (recursivly)
	BranchObj* bo=(BranchObj*)(parObj);
	if (bo)
		bo->resetTmpUnscroll();
		
	// Unscroll myself
	if (tmpUnscrolled)
	{
		tmpUnscrolled=false;
		systemFlags->deactivate("tmpUnscrolledright");
		toggleScroll();
	}	
}

void BranchObj::setVisibility(bool v, int toDepth)
{
    if (depth <= toDepth)
    {
		frame->setVisibility(v);
		heading->setVisibility(v);
		systemFlags->setVisibility(v);
		standardFlags->setVisibility(v);
		LinkableMapObj::setVisibility (v);
		
		if (!scrolled && (depth < toDepth))
		{
			// Now go recursivly through all childs
			BranchObj* b;
			for (b=branch.first(); b;b=branch.next() ) 
				b->setVisibility (v,toDepth);	
			FloatImageObj *fio;
			for (fio=floatimage.first(); fio; fio=floatimage.next())
				fio->setVisibility (v);
			XLinkObj* xlo;
			for (xlo=xlink.first(); xlo;xlo=xlink.next() ) 
				xlo->setVisibility ();	
		}
    } // depth <= toDepth	
	requestReposition();
}	

void BranchObj::setVisibility(bool v)
{
    setVisibility (v,MAX_DEPTH);
}


void BranchObj::setLinkColor ()
{
	// Overloaded from LinkableMapObj
	// BranchObj can use color of heading

	if (mapEditor->getLinkColorHint()==HeadingColor)
		LinkableMapObj::setLinkColor (heading->getColor() );
	else	
		LinkableMapObj::setLinkColor ();
}

void BranchObj::setColor (QColor col, bool colorChilds)
{
    heading->setColor(col);
	setLinkColor();
    if (colorChilds) 
    {
		BranchObj *bo;
		for (bo=branch.first(); bo; bo=branch.next() )
			bo->setColor(col,colorChilds);
    }	
}

QColor BranchObj::getColor()
{
	return heading->getColor();
}

BranchObj* BranchObj::first()
{
	itLast=NULL;	
	return this; 
}
	
BranchObj* BranchObj::next()
{
	BranchObj *lmo;
	BranchObj *bo=branch.first();
	BranchObj *po=(BranchObj*)(parObj);

	if (!itLast)
	{	// We are just beginning at the mapCenter
		if (bo) 
		{
			itLast=this;
			return bo;
		}	
		else
		{
			itLast=NULL;
			return NULL;
		}	
	}

	if (itLast==parObj)
	{	// We come from above
		if (bo)
		{
			// there are childs, go there
			itLast=this;
			return bo;
		}	
		else
		{	// no childs, try to go up again
			if (po)
			{
				// go up
				itLast=this;
				lmo=po->next();
				itLast=this;
				return lmo;

			}	
			else
			{
				// can't go up, I am mapCenter
				itLast=NULL;
				return NULL;
			}	
		}
	}

	// Try to find last child, we came from, in my own childs
	bool searching=true;
	while (bo && searching)
	{
		if (itLast==bo) searching=false;
		bo=branch.next();
	}
	if (!searching)
	{	// found lastLMO in my childs
		if (bo)
		{
			// found a brother of lastLMO 
			itLast=this;
			return bo;
		}	
		else
		{
			if (po)
			{
				// go up
				itLast=this;
				lmo=po->next();
				itLast=this;
				return lmo;
			}
			else
			{
				// can't go up, I am mapCenter
				itLast=NULL;
				return NULL;
			}	
		}
	}

	// couldn't find last child, it must be a nephew of mine
	bo=branch.first();
	if (bo)
	{
		// proceed with my first child
		itLast=this;	
		return bo;
	}	
	else
	{
		// or go back to my parents
		if (po)
		{
			// go up
			itLast=this;
			lmo=po->next();
			itLast=this;
			return lmo;
		}	
		else
		{
			// can't go up, I am mapCenter
			itLast=NULL;
			return NULL;
		}	
	}	
}

BranchObj* BranchObj::getLastIterator()
{
	return itLast;
}

void BranchObj::setLastIterator(BranchObj* it)
{
	itLast=it;
}


void BranchObj::move (double x, double y)
{
	OrnamentedObj::move (x,y);
    positionBBox();
}

void BranchObj::move (QPoint p)
{
	move (p.x(), p.y());
}

void BranchObj::moveBy (double x, double y)
{
	OrnamentedObj::moveBy (x,y);
    positionBBox();
    BranchObj* b;
    for (b=branch.first(); b;b=branch.next() ) 
		b->moveBy (x,y);
}
	
void BranchObj::moveBy (QPoint p)
{
	moveBy (p.x(), p.y());
}


void BranchObj::positionBBox()
{
	// FIXME testing (optimization)
	/*
	QString h=getHeading();
	if (!h.isEmpty())
		cout << "BO::positionBBox("<<h<<")\n";
	else	
		cout << "BO::positionBBox (noHeading)\n";
	*/	

    heading->positionBBox();
	systemFlags->positionBBox();
	standardFlags->positionBBox();
	// It seems that setting x,y also affects width,height
	int w_old=bbox.width();
	int h_old=bbox.height();
    bbox.setX (absPos.x() );
	bbox.setY (absPos.y() );
	bbox.setWidth(w_old);
	bbox.setHeight(h_old);
	
	setSelBox();

	// set the frame
	frame->setRect(QRect(bbox.x(),bbox.y(),bbox.width(),bbox.height() ) );

	// Update links to other branches
	XLinkObj *xlo;
    for (xlo=xlink.first(); xlo; xlo=xlink.next() )
		xlo->updateXLink();
}

void BranchObj::calcBBoxSize()
{
    QSize heading_r=heading->getSize();
    int heading_w=static_cast <int> (heading_r.width() );
    int heading_h=static_cast <int> (heading_r.height() );
    QSize sysflags_r=systemFlags->getSize();
	int sysflags_h=sysflags_r.height();
	int sysflags_w=sysflags_r.width();
    QSize stanflags_r=standardFlags->getSize();
	int stanflags_h=stanflags_r.height();
	int stanflags_w=stanflags_r.width();
    int w;
    int h;

	// set width to sum of all widths
	w=heading_w + sysflags_w + stanflags_w;
	// set height to maximum needed height
	h=max (sysflags_h,stanflags_h);
	h=max (h,heading_h);

    w+=frame->getBorder();
    h+=frame->getBorder();
    bbox.setSize (QSize (w,h));
}

LinkableMapObj* BranchObj::findMapObj(QPoint p, LinkableMapObj* excludeLMO)
{
	// Search branches
    BranchObj *b;
    LinkableMapObj *lmo;
    for (b=branch.first(); b; b=branch.next() )
    {	
		lmo=b->findMapObj(p, excludeLMO);
		if (lmo != NULL) return lmo;
    }
	
	// Search myself
    if (inBBox (p) && (this != excludeLMO) && isVisibleObj() ) 
		return this;

	// Search float images
	FloatImageObj *foi;
    for (foi=floatimage.first(); foi; foi=floatimage.next() )
		if (foi->inBBox(p) && 
			(foi != excludeLMO) && 
			foi->getParObj()!= excludeLMO &&
			foi->isVisibleObj() 
		) return foi;

    return NULL;
}

void BranchObj::setHeading(QString s)
{
	/*FIXME
    // Adjusting font size
    QFont font=heading->getFont();
	font.setPointSize(getDefHeadingSize() );
    heading->setFont(font);
	*/
    heading->setText(s);	// set new heading
	calcBBoxSize();			// recalculate bbox
    positionBBox();			// rearrange contents
	requestReposition();
}

void BranchObj::setURL(QString s)
{
	url=s;
	if (!url.isEmpty())
		systemFlags->activate("url");
	else	
		systemFlags->deactivate("url");
	calcBBoxSize();			// recalculate bbox
    positionBBox();			// rearrange contents
	forceReposition();
}

QString BranchObj::getURL()
{
	return url;
}

void BranchObj::setVymLink(QString s)
{
	if (!s.isEmpty())
	{
		// We need the relative (from loading) 
		// or absolute path (from User event)
		// and build the absolute path.
		// Note: If we have relative, use path of
		// current map to build absolute path
		QDir d(s);
		if (!d.path().startsWith ("/"))
		{
			QString p=mapEditor->getDestPath();
			int i=p.findRev("/",-1);
			d.setPath(p.left(i)+"/"+s);
			d.convertToAbs();
		}
		vymLink=d.path();
		systemFlags->activate("vymLink");
	}	
	else	
	{
		systemFlags->deactivate("vymLink");
		vymLink="";
	}	
	calcBBoxSize();			// recalculate bbox
    positionBBox();			// rearrange contents
	forceReposition();
}

QString BranchObj::getVymLink()
{
	return vymLink;
}

QString BranchObj::saveToDir (const QString &tmpdir,const QString &prefix, const QPoint& offset)
{
    QString s,a;
	QString scrolledAttr;
	if (scrolled) 
		scrolledAttr=attribut ("scrolled","yes");
	else
		scrolledAttr="";

	QString posAttr;
	if (depth<2) posAttr=
		attribut("absPosX",QString().setNum(absPos.x(),10)) +
		attribut("absPosY",QString().setNum(absPos.y(),10)); 
	else
		posAttr="";

	QString linkAttr=getLinkAttr();

	QString urlAttr;
	if (!url.isEmpty())
		urlAttr=attribut ("url",url);

	QString vymLinkAttr;
	if (!vymLink.isEmpty())
		vymLinkAttr=attribut ("vymLink",convertToRel(mapEditor->getDestPath(),vymLink) );

	QString frameAttr;
	if (frame->getFrameType()!=NoFrame)
		frameAttr=attribut ("frameType",frame->getFrameTypeName());
	else
		frameAttr="";

	// save area, if not scrolled
	QString areaAttr;
	if (!((BranchObj*)(parObj))->isScrolled() )
	{
		areaAttr=
			attribut("x1",QString().setNum(absPos.x()-offset.x(),10)) +
			attribut("y1",QString().setNum(absPos.y()-offset.y(),10)) +
			attribut("x2",QString().setNum(absPos.x()+width()-offset.x(),10)) +
			attribut("y2",QString().setNum(absPos.y()+height()-offset.y(),10));

	} else
		areaAttr="";
	
    s=beginElement ("branch" +scrolledAttr +posAttr +linkAttr +urlAttr +vymLinkAttr +frameAttr +areaAttr );
    incIndent();

	// save heading
    s+=valueElement("heading", getHeading(),
		attribut ("textColor",QColor(heading->getColor()).name()));

	// save names of flags set
	s+=standardFlags->saveToDir(tmpdir,prefix,0);
	
	// save note
	if (!note.isEmpty() )
		s+=note.saveToDir();
	
	// Save branches
    BranchObj *bo;
    for (bo=branch.first(); bo; bo=branch.next() )
		s+=bo->saveToDir(tmpdir,prefix,offset);

	// Save FloatImages
	FloatImageObj *fio;
	for (fio=floatimage.first(); fio; fio=floatimage.next() )
		s+=fio->saveToDir (tmpdir,prefix);

	// Save XLinks
	XLinkObj *xlo;
    for (xlo=xlink.first(); xlo; xlo=xlink.next() )
		s+=xlo->saveToDir();

    decIndent();
    s+=endElement   ("branch");
    return s;
}

void BranchObj::addXLink (XLinkObj *xlo)
{
	xlink.append (xlo);
	
}

void BranchObj::removeXLinkRef (XLinkObj *xlo)
{
	xlink.remove (xlo);
}

void BranchObj::deleteXLink(XLinkObj *xlo)
{
	xlo->deactivate();
	if (!xlo->isUsed()) delete (xlo);
}

void BranchObj::deleteXLinkAt (int i)
{
	XLinkObj *xlo=xlink.at(i);
	xlo->deactivate();
	if (!xlo->isUsed()) delete(xlo);
}

XLinkObj* BranchObj::XLinkAt (int i)
{
	return xlink.at(i);
}

int BranchObj::countXLink()
{
	return xlink.count();
}

BranchObj* BranchObj::XLinkTargetAt (int i)
{
	if (xlink.at(i))
		return xlink.at(i)->otherBranch (this);
	else
		return NULL;
}

LinkableMapObj* BranchObj::addFloatImage ()
{
	FloatImageObj *newfi=new FloatImageObj (canvas,this);
	floatimage.append (newfi);
	if (hasScrolledParent(this) )
		newfi->setVisibility (false);
	else	
		newfi->setVisibility(visible);
	requestReposition();
	return newfi;
}

LinkableMapObj* BranchObj::addFloatImage (FloatImageObj *fio)
{
	FloatImageObj *newfi=new FloatImageObj (canvas,this);
	floatimage.append (newfi);
	newfi->copy (fio);
	if (hasScrolledParent(this) )
		newfi->setVisibility (false);
	else	
		newfi->setVisibility(visible);
	requestReposition();
	return newfi;
}

FloatImageObj* BranchObj::getFirstFloatImage ()
{
    return floatimage.first();
}

FloatImageObj* BranchObj::getLastFloatImage ()
{
    return floatimage.last();
}

FloatImageObj* BranchObj::getFloatImageNum (const uint &i)
{
    return floatimage.at(i);
}

void BranchObj::removeFloatImage (FloatImageObj *fio)
{
	floatimage.remove (fio);
	requestReposition();
}

void BranchObj::savePosInAngle ()
{
	// Save position in angle
    BranchObj *b;
	int i=0;
    for (b=branch.first(); b; b=branch.next() )
	{
		b->angle=i;
		i++;
	}
}

void BranchObj::setDefAttr (BranchModification mod)
{
	int fontsize;
	switch (depth)
	{
		case 0: fontsize=16; break;
		case 1: fontsize=12; break;
		default: fontsize=10; break;
	}	

	setLinkColor ();
	setLinkStyle(getDefLinkStyle());
	QFont font("Sans Serif,8,-1,5,50,0,0,0,0,0");
	font.setPointSize(fontsize);
	heading->setFont(font );

	if (mod==NewBranch)
		setColor (((BranchObj*)(parObj))->getColor(),false);
	
	calcBBoxSize();
}

BranchObj* BranchObj::addBranch()
{
    BranchObj* newbo=new BranchObj(canvas,this);
    branch.append (newbo);
    newbo->setParObj(this);
	newbo->setDefAttr(NewBranch);
    newbo->setHeading ("new");
	if (scrolled)
		newbo->setVisibility (false);
	else	
		newbo->setVisibility(visible);
	newbo->updateLink();	
	requestReposition();
	return newbo;
}

BranchObj* BranchObj::addBranch(BranchObj* bo)
{
    BranchObj* newbo=new BranchObj(canvas,this);
    branch.append (newbo);
    newbo->copy(bo);
    newbo->setParObj(this);
	newbo->setDefAttr(MovedBranch);
	if (scrolled)
		newbo->setVisibility (false);
	else	
		newbo->setVisibility(bo->visible);
	newbo->updateLink();	
	requestReposition();
	return newbo;
}

BranchObj* BranchObj::addBranchPtr(BranchObj* bo)
{
	branch.append (bo);
	bo->setParObj (this);
	bo->depth=depth+1;
	bo->setDefAttr(MovedBranch);
	if (scrolled) tmpUnscroll();
	setLastSelectedBranch (bo);
	return bo;
}

BranchObj* BranchObj::insertBranch(int pos)
{
	savePosInAngle();
	// Add new bo and resort branches
	BranchObj *newbo=addBranch ();
	newbo->angle=pos-0.5;
	branch.sort();
	return newbo;
}

BranchObj* BranchObj::insertBranch(BranchObj* bo, int pos)
{
	savePosInAngle();
	// Add new bo and resort branches
	bo->angle=pos-0.5;
	BranchObj *newbo=addBranch (bo);
	branch.sort();
	return newbo;
}

BranchObj* BranchObj::insertBranchPtr (BranchObj* bo, int pos)
{
	savePosInAngle();
	// Add new bo and resort branches
	bo->angle=pos-0.5;
	branch.append (bo);
	bo->setParObj (this);
	bo->depth=depth+1;
	bo->setDefAttr (MovedBranch);
	if (scrolled) tmpUnscroll();
	setLastSelectedBranch (bo);
	branch.sort();
	return bo;
}

void BranchObj::removeBranchHere(BranchObj* borem)
{
	// This removes the branch bo from list, but 
	// inserts its childs at the place of bo
	BranchObj *bo;
	bo=borem->getLastBranch();
	int pos=borem->getNum();
	while (bo)
	{
		bo->moveBranchTo (this,pos+1);
		bo=borem->getLastBranch();
	}	
	removeBranch (borem);
}

void BranchObj::removeChilds()
{
	clear();
}

void BranchObj::removeBranch(BranchObj* bo)
{
    // if bo is not in branch remove returns false, we
    // don't care...
	
    if (branch.remove (bo))
		delete (bo);
	else
		qWarning ("BranchObj::removeBranch tried to remove non existing branch?!\n");
	requestReposition();
}

void BranchObj::removeBranchPtr(BranchObj* bo)
{
	branch.remove (bo);
	requestReposition();
}

void BranchObj::setLastSelectedBranch (BranchObj* bo)
{
    lastSelectedBranch=branch.find(bo);
}

BranchObj* BranchObj::getLastSelectedBranch ()
{
    if (lastSelectedBranch>=0) 
	{
		BranchObj* bo=branch.at(lastSelectedBranch);
		if (bo) return bo;
    }	
    return branch.first();
}

BranchObj* BranchObj::getFirstBranch ()
{
    return branch.first();
}

BranchObj* BranchObj::getLastBranch ()
{
    return branch.last();
}

BranchObj* BranchObj::getBranchNum (const uint &i)
{
    return branch.at(i);
}


BranchObj* BranchObj::moveBranchUp(BranchObj* bo1) // move a branch up (modify myself)
{
	savePosInAngle();
    int i=branch.find(bo1);
    if (i>0) 
	{	// -1 if bo1 not found 
		branch.at(i)->angle--;
		branch.at(i-1)->angle++;
		branch.sort();
		return branch.at(i-1);
	} else
		return branch.at(i);
}

BranchObj* BranchObj::moveBranchDown(BranchObj* bo1)
{
	savePosInAngle();
    int i=branch.find(bo1);
	int j;
	if (branch.next())
	{
		j = branch.at();
		branch.at(i)->angle++;
		branch.at(j)->angle--;
		branch.sort();
		return branch.at(j);
	} else
		return branch.at(i);
}

BranchObj* BranchObj::moveBranchTo (BranchObj* dst, int pos)
{
	// Find current parent and 
	// remove pointer to myself there
	if (!dst) return NULL;
	BranchObj *par=(BranchObj*)(parObj);
	if (par)
		par->removeBranchPtr (this);
	else
		return NULL;

	// Create new pointer to myself at dst
	if (pos<0||dst->getDepth()==0)
	{	
		// links myself as last branch at dst
		dst->addBranchPtr (this);
		updateLink();
		return this;
	} else
	{
		// inserts me at pos in parent of dst
		if (par)
		{
			BranchObj *bo=dst->insertBranchPtr (this,pos);
			bo->setDefAttr(MovedBranch);
			updateLink();
			return bo;

		} else
			return NULL;
	}	
}

void BranchObj::alignRelativeTo (QPoint ref)
{
/* FIXME testing
	if (!getHeading().isEmpty())
		cout << "BO::alignRelTo "<<getHeading()<<endl;
	else	
		cout << "BO::alignRelTo  ???"<<endl;
	cout << "  d="<<depth<<endl;
*/	
	int th = bboxTotal.height();	

	// If I am the mapcenter or a mainbranch, reposition heading
	if (depth<2)
	{
		move (absPos.x(),absPos.y());
		if (depth==1)
		{
			// Calc angle to mapCenter if I am a mainbranch
			// needed for reordering the mainbranches clockwise 
			// around mapcenter 
			angle=getAngle (QPoint ((int)(x() - parObj->getChildPos().x() ), 
									(int)(y() - parObj->getChildPos().y() ) ) );
		}	
	} 
	else
    {
		// Align myself depending on orientation and parent, but
		// only if I am not the mainbranch or mapcenter itself
		switch (orientation) 
		{
			case OrientLeftOfCenter:
				move (ref.x()-bbox.width(), ref.y() + (th-bbox.height())/2 );
			break;
			case OrientRightOfCenter:	
				move (ref.x(), ref.y() + (th-bbox.height())/2 );
			break;
			default:
				cout <<"LMO::alignRelativeTo: oops, no orientation given...\n";
			break;
		}		
    }		

	FloatImageObj *fio;
    for (fio=floatimage.first(); fio; fio=floatimage.next() )
		fio->reposition();

	if (scrolled) return;

    // Set reference point for alignment of childs
    QPoint ref2;
    if (orientation==OrientLeftOfCenter)
		ref2.setX(childPos.x() - linkwidth);
    else	
		ref2.setX(childPos.x() + linkwidth);

	if (depth==1)
		ref2.setY(absPos.y()-(bboxTotal.height()-bbox.height())/2);
	else	
		ref2.setY(ref.y() );	

    // Align the childs depending on reference point 
    BranchObj *b;
    for (b=branch.first(); b; b=branch.next() )
    {	
		b->alignRelativeTo (ref2);
		ref2.setY(ref2.y() + b->getBBoxSizeWithChilds().height() );
    }
}


void BranchObj::reposition()
{	
/* FIXME testing
	if (!getHeading().isEmpty())
		cout << "BO::reposition  "<<getHeading()<<endl;
	else	
		cout << "BO::reposition  ???"<<endl;
*/		
	if (depth==0)
	{
		// only calculate the sizes once. If the deepest LMO 
		// changes its height,
		// all upper LMOs have to change, too.
		calcBBoxSizeWithChilds();
	    alignRelativeTo ( QPoint (absPos.x(),
			absPos.y()-(bboxTotal.height()-bbox.height())/2) );
		branch.sort();	
	} else
	{
		// This is only important for moving branches:
		// For editing a branch it isn't called...
	    alignRelativeTo ( QPoint (absPos.x(),
							absPos.y()-(bboxTotal.height()-bbox.height())/2) );
	}
}


QRect BranchObj::getTotalBBox()
{
	QRect r=bbox;

	if (scrolled) return r;

	BranchObj* b;
	for (b=branch.first();b ;b=branch.next() )
		r=addBBox(b->getTotalBBox(),r);

	FloatImageObj* fio;
	for (fio=floatimage.first();fio ;fio=floatimage.next() )
		r=addBBox(fio->getTotalBBox(),r);
		
	return r;
}

QRect BranchObj::getBBoxSizeWithChilds()
{
	return bboxTotal;
}

void BranchObj::calcBBoxSizeWithChilds()
{
	// This is called only from reposition and
	// and only for mapcenter. So it won't be
	// called more than once for a single user 
	// action
	
	// Calculate size of LMO including all childs (to align them later)

	bboxTotal.setX(bbox.x() );
	bboxTotal.setY(bbox.y() );

	// if branch is scrolled, ignore childs, but still consider floatimages
	if (scrolled)
	{
		bboxTotal.setWidth (bbox.width());
		bboxTotal.setHeight(bbox.height());
		return;
	}
	
	QRect r(0,0,0,0);
	QRect br;
	// Now calculate recursivly
	// sum of heights 
	// maximum of widths 
	// minimum of y
	BranchObj* b;
	for (b=branch.first();b ;b=branch.next() )
	{
		b->calcBBoxSizeWithChilds();
		br=b->getBBoxSizeWithChilds();
		r.setWidth( max (br.width(), r.width() ));
		r.setHeight(br.height() + r.height() );
		if (br.y()<bboxTotal.y()) bboxTotal.setY(br.y());
	}
	// Add myself and also
	// add width of link to sum if necessary
	if (branch.isEmpty())
		bboxTotal.setWidth (bbox.width() + r.width() );
	else	
		bboxTotal.setWidth (bbox.width() + r.width() + linkwidth);
	bboxTotal.setHeight(max (r.height(),  bbox.height() ) );
//	frame->setRect(QRect(bbox.x(),bbox.y(),bbox.width(),bbox.height() ) );
}

void BranchObj::select()
{
	// set Text in Editor	
	textEditor->setText(note.getNote() );
	QString fnh=note.getFilenameHint();
	if (fnh!="")
		textEditor->setFilenameHint(note.getFilenameHint() );
	else	
		textEditor->setFilenameHint(getHeading() );
	textEditor->setFontHint (note.getFontHint() );

    LinkableMapObj::select();
	// Tell parent that I am selected now:
	BranchObj* po=(BranchObj*)(parObj);
    if (po)	// TODO	    Try to get rid of this cast...
        po->setLastSelectedBranch(this);
		
	// temporary unscroll, if we have scrolled parents somewhere
	if (parObj) ((BranchObj*)(parObj))->tmpUnscroll();

	// Show URL and link in statusbar
	QString status;
	if (!url.isEmpty()) status+="URL: "+url+"  ";
	if (!vymLink.isEmpty()) status+="Link: "+vymLink;
	if (!status.isEmpty()) mainWindow->statusMessage (status);

	// Update Toolbar
	standardFlags->updateToolbar();

	// Update Browserbutton
	if (!url.isEmpty())
		actionEditOpenURL->setEnabled (true);
	else	
		actionEditOpenURL->setEnabled (false);

	// Update actions in mapeditor
	mapEditor->updateActions();
}

void BranchObj::unselect()
{
	LinkableMapObj::unselect();
	// Delete any messages like vymLink in StatusBar
	mainWindow->statusMessage ("");

	// save note from editor and set flag
	// text is done by updateNoteFlag(), just save
	// filename here
	note.setFilenameHint (textEditor->getFilename());

	// reset temporary unscroll, if we have scrolled parents somewhere
	if (parObj) ((BranchObj*)(parObj))->resetTmpUnscroll();

	// Erase content of editor 
	textEditor->setInactive();

	// unselect all buttons in toolbar
	standardFlagsDefault->updateToolbar();
}

QString BranchObj::getSelectString()
{
	QString s;
	if (parObj)
	{
		if (depth==1)
			s= "bo:" + QString("%1").arg(getNum());
		else	
			s= ((BranchObj*)(parObj))->getSelectString() + ",bo:" + QString("%1").arg(getNum());
	} else
		s="mc:";
	return s;
}

