#include "frameobj.h"

#include <QColor>
#include <QDebug>

/////////////////////////////////////////////////////////////////
// FrameObj
/////////////////////////////////////////////////////////////////
FrameObj::FrameObj() : MapObj()
{
//    cout << "Const FrameObj ()\n";
    init ();
}

FrameObj::FrameObj(QGraphicsScene *s) :MapObj(s)
{
//    cout << "Const FrameObj\n";
    init ();
}

FrameObj::~FrameObj()
{
    clear();
}

void FrameObj::init()
{
    type=NoFrame;
    padding=10;
    borderWidth=1;
    penColor=QColor (Qt::black);
    brushColor=QColor (Qt::white);
    includeChildren=false;
}

void FrameObj::clear()
{
    switch (type)
    {
	case NoFrame:
	    break;
	case Rectangle:
	    delete rectFrame;
	    break;
	case Ellipse:
	    delete ellipseFrame;
	    break;
	case Cloud:
	    delete pathFrame;
	    break;
    }
    type=NoFrame;
    padding=0;
}

void FrameObj::move(double x, double y)
{
    switch (type)
    {
	case NoFrame:
	    break;
	case Rectangle:
	    rectFrame->setPos (x,y);
	    break;
	case Ellipse:
	    ellipseFrame->setPos (x,y);
	case Cloud:
	    pathFrame->setPos (x,y);
	    break;
    }
}

void FrameObj::moveBy(double x, double y)
{
    MapObj::moveBy (x,y);
}

void FrameObj::positionBBox()
{
}

void FrameObj::calcBBoxSize()
{
}

float FrameObj::roof (float x)
{
    if (x<=0.5)
	return  x;
    else
	return 1-x;
}

void FrameObj::setRect(const QRectF &r)
{
    bbox=r;
    switch (type)
	{
	case NoFrame:
	    break;
	case Rectangle:
	    rectFrame->setRect (QRectF(bbox.x(),bbox.y(),bbox.width(),bbox.height() ));
	    break;
	case Ellipse:
	    ellipseFrame->setRect (QRectF(bbox.x(),bbox.y(),bbox.width(),bbox.height() ));
	    break;
	case Cloud:
	    QPointF tl=bbox.topLeft();
	    QPointF tr=bbox.topRight();
	    QPointF bl=bbox.bottomLeft();
	    QPainterPath path;
	    path.moveTo (tl);

	    float w=bbox.width();   // width
	    float h=bbox.height();  // height
	    int n=w/40;		    // number of intervalls
	    float d=w/n;	    // width of interwall

	    // Top path
	    for (float i=0; i<n; i++)
	    {
		//qDebug()<<"i="<<i<<"  n="<<n<<"  r="<<roof ((i+1)/n)<<"  r2="<<roof((i+0.5)/n);
		path.cubicTo (
		    tl.x() + i*d,     tl.y()- 100*roof ((i+0.5)/n) , 
		    tl.x() + (i+1)*d, tl.y()- 100*roof ((i+0.5)/n) , 
		    tl.x() + (i+1)*d + 20*roof ((i+1)/n), tl.y()- 50*roof((i+1)/n) );
	    }
	    // Right path
	    n=h/20;
	    d=h/n;
	    for (float i=0; i<n; i++)
	    {
		//qDebug()<<"i="<<i<<"  n="<<n<<"  r="<<roof ((i+1)/n)<<"  r2="<<roof((i+0.5)/n);
		path.cubicTo (
		    tr.x()+ 100*roof ((i+0.5)/n)        , tr.y() + i*d,
		    tr.x()+ 100*roof ((i+0.5)/n)        , tr.y() + (i+1)*d,
		    tr.x() + 60*roof ((i+1)/n)          , tr.y() + (i+1)*d );
	    }
	    n=w/60;
	    d=w/n;
	    // Bottom path
	    for (float i=n; i>0; i--)
	    {
		//qDebug()<<"i="<<i<<"  n="<<n<<"  r="<<roof ((i+1)/n)<<"  r2="<<roof((i+0.5)/n);
		path.cubicTo (
		    bl.x() + i*d,  bl.y()+ 100*roof ((i-0.5)/n) , 
		    bl.x() + (i-1)*d,      bl.y()+ 100*roof ((i-0.5)/n) , 
		    bl.x() + (i-1)*d + 20*roof ((i-1)/n), bl.y()+ 50*roof((i-1)/n) );
	    }
	    // Left path
	    n=h/20;
	    d=h/n;
	    for (float i=n; i>0; i--)
	    {
		//qDebug()<<"i="<<i<<"  n="<<n<<"  r="<<roof ((i+1)/n)<<"  r2="<<roof((i+0.5)/n);
		path.cubicTo (
		    tl.x()- 100*roof ((i-0.5)/n)        , tr.y() + i*d,
		    tl.x()- 100*roof ((i-0.5)/n)        , tr.y() + (i-1)*d,
		    tl.x()-  60*roof ((i-1)/n)          , tr.y() + (i-1)*d );
	    }
	    pathFrame->setPath(path);
	    break;
    }
}

void FrameObj::setPadding (const int &i)
{
    padding=i;
    repaint();
}

int FrameObj::getPadding()
{
    if (type==NoFrame) 
	return 0;
    else    
	return padding;
}

void FrameObj::setBorderWidth (const int &i)
{
    borderWidth=i;
    repaint();
}

int FrameObj::getBorderWidth()
{
    return borderWidth;
}

FrameObj::FrameType FrameObj::getFrameType()
{
    return type;
}

FrameObj::FrameType FrameObj::getFrameType(const QString &s)
{
    if (s=="Rectangle")
	return Rectangle;
    else if (s=="Ellipse")
	return Ellipse;
    else if (s=="Cloud")
	return Cloud;
    return NoFrame;	
}

QString FrameObj::getFrameTypeName()
{
    switch (type)
    {
	case Rectangle:
	    return "Rectangle";
	    break;
	case Ellipse:
	    return "Ellipse";
	    break;
	case Cloud:
	    return "Cloud";
	    break;
	default:
	    return "NoFrame";
    }
}

void FrameObj::setFrameType(const FrameType &t)
{
    if (t!=type)
    {
    clear();
    type=t;
    switch (type)
    {
	case NoFrame:
	    break;
	case Rectangle:
	    rectFrame = scene->addRect(QRectF(0,0,0,0), QPen(penColor), brushColor);
	    rectFrame->setZValue(Z_INIT);
	    rectFrame->show();
	    break;
	case Ellipse:
	    ellipseFrame = scene->addEllipse(QRectF(0,0,0,0), QPen(penColor), brushColor);
	    ellipseFrame->setZValue(Z_INIT);
	    ellipseFrame->show();
	case Cloud:
	    QPainterPath path;
	    pathFrame = scene->addPath(path, QPen(penColor), brushColor);
	    pathFrame->setZValue(Z_INIT);
	    pathFrame->show();
	    break;
	}
    }
    setVisibility (visible);
}

void FrameObj::setFrameType(const QString &t)
{
    if (t=="Rectangle")
	FrameObj::setFrameType (Rectangle);
    else if (t=="Ellipse")  
	FrameObj::setFrameType (Ellipse);
    else if (t=="Cloud")  
	FrameObj::setFrameType (Cloud);
    else    
	FrameObj::setFrameType (NoFrame);
}

void FrameObj::setPenColor (QColor col)
{
    penColor=col;
    repaint();
}

QColor FrameObj::getPenColor ()
{
    return penColor;
}

void FrameObj::setBrushColor (QColor col)
{
    brushColor=col;
    repaint();
}

QColor FrameObj::getBrushColor ()
{
    return brushColor;
}

void FrameObj::setFrameIncludeChildren(bool b)
{
    includeChildren=b;
}

bool FrameObj::getFrameIncludeChildren()
{
    return includeChildren;
}

void FrameObj::repaint()
{
    //  qDebug()<<"FO:repaint tI="<<treeItem;
    //	qDebug()<<"              "<<treeItem->getHeading();
    QPen pen;
    pen.setColor (penColor);
    pen.setWidth (borderWidth);
    QBrush brush (brushColor);
    switch (type)
    {
	case Rectangle:
	    rectFrame->setPen   (pen);
	    rectFrame->setBrush (brush);
	    break;
	case Ellipse:
	    ellipseFrame->setPen   (pen);
	    ellipseFrame->setBrush (brush);
	    break;
	case Cloud:
	    pathFrame->setPen   (pen);
	    pathFrame->setBrush (brush);
	    break;
	default:
	    break;
    }
}

void FrameObj::setZValue (double z)
{
    switch (type)
    {
	case NoFrame:
	    break;
	case Rectangle:
	    rectFrame->setZValue (z);
	    break;
	case Ellipse:
	    ellipseFrame->setZValue (z);
	    break;
	case Cloud:
	    pathFrame->setZValue (z);
	    break;
    }
}

void FrameObj::setVisibility (bool v)
{
    MapObj::setVisibility(v);
    switch (type)
    {
	case NoFrame:
	    break;
	case Rectangle:
	    if (visible)
		rectFrame->show();
	    else    
		rectFrame->hide();
	    break;
	case Ellipse:
	    if (visible)
		ellipseFrame->show();
	    else    
		ellipseFrame->hide();
	    break;
	case Cloud:
	    if (visible)
		pathFrame->show();
	    else    
		pathFrame->hide();
	    break;
    }
}

QString FrameObj::saveToDir ()
{
    if (type==NoFrame) return QString();
    QString frameTypeAttr=attribut ("frameType",getFrameTypeName());
    QString penColAttr=attribut ("penColor",penColor.name() );
    QString brushColAttr=attribut ("brushColor",brushColor.name() );
    QString paddingAttr=attribut ("padding",QString::number (padding) );
    QString borderWidthAttr=attribut ("borderWidth",QString::number (borderWidth) );
    QString incChildren;
    if (includeChildren)
	incChildren=attribut ("includeChildren","true");
    return singleElement (
	"frame",frameTypeAttr + 
	penColAttr + 
	brushColAttr +
	paddingAttr +
	borderWidthAttr +
	incChildren);
}

