/*-----------------------------------------------------------------------------
 *  FILE: SearchDialog.cc
 *
 *      Copyright(c) 2006 Gareth Foster.
 *
 *      Dialog derivation that implements Wallpaper Tray search tool.
 *
 *-----------------------------------------------------------------------------
 */

#include "SearchDialog.hh"
#include <stdexcept>
#include <iostream>
#include <glib/gi18n.h>
#include <boost/filesystem/exception.hpp>
#include <boost/version.hpp>

//------------------------------------------------------------------------
/**
<SearchDialog::SearchDialog>
@brief Create a new search dialog.

@param _pWindow A pointer required to allow creation of this dialog from glade xml.
@param _pGlade A reference to a pointer required to allow creation of this dialog from glade xml.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
SearchDialog::SearchDialog
(
	BaseObjectType *							_pWindow,
	const Glib::RefPtr<Gnome::Glade::Xml> &		_pGlade
) :
	Gtk::Dialog(_pWindow),
	m_pGlade(_pGlade)
{
	// set up the gconf client
	m_pGconfClient = Gnome::Conf::Client::get_default_client();

	// grab the main druid
	m_pGlade->get_widget("druid_search", m_pDruid);

	if(m_pDruid != NULL)
	{
		// hook up the cancel and finish buttons
		m_pDruid->signal_cancel().connect(sigc::mem_fun(*this, &SearchDialog::OnButtonCancelClick));
	}// end if

	// grab the finish druid page
	m_pGlade->get_widget("druidpagefinish", m_pDruidPageFinish);
	if(m_pDruidPageFinish != NULL)
	{
		// hook up the cancel and finish buttons	
		m_pDruidPageFinish->signal_finish().connect(sigc::mem_fun(*this, &SearchDialog::OnButtonApplyClick));
	}// end if

	// grab the finish druid page
	m_pGlade->get_widget("druidpagestandard_search_string", m_pDruidPageStdSearchString);
	if(m_pDruidPageStdSearchString != NULL)
	{
		// hook up the cancel and finish buttons	
		m_pDruidPageStdSearchString->signal_next().connect(sigc::mem_fun(*this, &SearchDialog::OnButtonNextClick));
	}// end if

	// grab the finish druid page
	m_pGlade->get_widget("druidpagestart", m_pDruidPageStdStart);

	// grab the tree view
	m_pGlade->get_widget("treeview_results", m_pTreeView);

	if(m_pTreeView == NULL)
	{
		throw std::runtime_error("SearchDialog::SearchDialog() : failed to get widget from glade");
	}// end if

	// set up the list store and model etc
	m_pListStore = Gtk::ListStore::create(m_Model);

	m_pTreeView->set_model(m_pListStore);
	m_pTreeView->append_column(_("Directories"), m_Model.m_FilePath);

	m_pGlade->get_widget("radiobutton_select", m_pRadioButtonSelect);
	m_pGlade->get_widget("radiobutton_delete", m_pRadioButtonDelete);
	m_pGlade->get_widget("entry_search_text", m_pEntrySearchString);

	// give the window an icon
	set_icon_from_file(GNOMEICONDIR"/wp_tray-applet.png");

	show_all_children();
}// end SearchDialog::SearchDialog

//------------------------------------------------------------------------
/**
<SearchDialog::~SearchDialog>
@brief Destroy a search dialog.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
SearchDialog::~SearchDialog
(
)
{
	// ...
}// end SearchDialog::~SearchDialog

//------------------------------------------------------------------------
/**
<SearchDialog::OnButtonNextClick>
@brief Respond to a user clicking through the druid.

@param _refDruid A reference to the clicked button.
@return Whether we handled this event.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
bool SearchDialog::OnButtonNextClick
(
	Gtk::Widget &								_refWidget
)
{
	std::list<Glib::ustring> lsWpDir;
	std::list<Glib::ustring> lsResults;

	lsWpDir = m_pGconfClient->get_string_list("/apps/wp_tray/dir_list");

	boost::regex regex;
	Glib::ustring szRString;

	try
	{
	    szRString += ".*";
	    szRString += m_pEntrySearchString->get_text();
	    szRString += ".*";
	    regex = boost::regex(szRString.c_str(), boost::regex::perl | boost::regex::icase);
	}// end try
	catch (...)
	{
	    Gtk::MessageDialog dialog	(
				_("Incorrect regular expression.\nPlease use Perl-style format."),
				false,
				Gtk::MESSAGE_ERROR,
				Gtk::BUTTONS_OK,
				true
			);

	    dialog.run();

	    return true;
	}// end catch

	// search each directory
	for(std::list<Glib::ustring>::iterator it = lsWpDir.begin(); it != lsWpDir.end(); ++ it)
	{
		// do the boost regex search
		// TODO: C string conversion is 'orrible :/
		SearchDialog::WallpaperSearch(fs::path(*it), regex, lsResults);
	}// end for

	if(lsResults.size() == 0)
	{
	    Gtk::MessageDialog dialog	(
				_("\nNo wallpapers found."),
				false,
				Gtk::MESSAGE_INFO,
				Gtk::BUTTONS_OK,
				true
			);

	    dialog.run();

	    return true;
	}//end if

	// clear list store
	m_pListStore->clear();

	// append results to the list store
	for(std::list<Glib::ustring>::iterator it = lsResults.begin(); it != lsResults.end(); ++ it)
	{
		Gtk::TreeModel::iterator it_new = m_pListStore->append();

		(*it_new)[m_Model.m_FilePath] = (*it);
	}// end for

	return false;
}// end SearchDialog::OnButtonNextClick

//------------------------------------------------------------------------
/**
<SearchDialog::OnButtonNextClick>
@brief Respond to a user clicking through the druid.

@param _refWidget A pointer to the apply button.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void SearchDialog::OnButtonApplyClick
(
	Gtk::Widget &								_refWidget
)
{
	Glib::RefPtr<Gtk::TreeSelection> p_TreeSelection = m_pTreeView->get_selection();

	Gtk::TreeModel::iterator it = p_TreeSelection->get_selected();
	
	// if anything is selected
	if(it)
	{
		Glib::ustring szValue;

		it->get_value(0, szValue);

		if(m_pRadioButtonDelete->get_active() == true)
		{
			if(szValue.length() == 0)
				return;

			// attempt to remove the file
			try
			{
				fs::path pathWp(szValue.c_str(), fs::native);

				if(fs::exists(pathWp) == false)
				{
					return;
				}// end if

				fs::remove(pathWp);
			}// end try
			catch(fs::filesystem_error e)
			{
				return;
			}// end catch

			if(m_pGconfClient->get_string("/desktop/gnome/background/picture_filename") == szValue)
			{
				// TODO: set new wp ???
			}// end if
		}// end if
		else
		if(m_pRadioButtonSelect->get_active() == true && szValue.length() > 0)
		{
			// set new in gconf
			m_pGconfClient->set("/desktop/gnome/background/picture_filename", szValue);
		}// end if
	}// end if

	SearchDialog::hide();
}// end SearchDialog::OnButtonApplyClick

//------------------------------------------------------------------------
/**
<SearchDialog::OnButtonCancelClick>
@brief Hide the dialog when cancel is clicked.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void SearchDialog::OnButtonCancelClick
(
)
{
	SearchDialog::hide();
}// end SearchDialog::OnButtonCancelClick

//------------------------------------------------------------------------
/**
<SearchDialog::OnShow>
@brief Set up the dialog when it appears.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void SearchDialog::OnShow
(
)
{
	// clear list store
	m_pListStore->clear();

	// clear search box
	m_pEntrySearchString->set_text("");

	// TODO: this doesn't work
	m_pDruid->set_page(*m_pDruidPageStdStart);

	Gtk::Dialog::on_show();
}// end SearchDialog::OnShow

//------------------------------------------------------------------------
/**
<SearchDialog::WallpaperSearch>
@brief Used to actually search the users wallpaper collection.

@param _pathCurrent,
@param _refRegex,
@param _reflsResults
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void SearchDialog::WallpaperSearch
(
	fs::path									_pathCurrent,
	const boost::regex &						_refRegex,
	std::list<Glib::ustring> &					_reflsResults
)
{
	fs::directory_iterator itEnd;

	for(fs::directory_iterator itDir(_pathCurrent); itDir != itEnd; ++ itDir)
	{
		try
		{
			if(fs::is_directory(*itDir))
			{
				// recurse
				SearchDialog::WallpaperSearch(*itDir, _refRegex, _reflsResults);
			}// end if
			else
			{
				if(boost::regex_match(itDir->leaf(), _refRegex))
				{
#if BOOST_VERSION > 103301
					_reflsResults.push_back(itDir->path().directory_string());
#else
					_reflsResults.push_back((*itDir).native_directory_string());
#endif
				}// end if
			}// end else
		}// end try
		catch(const std::exception & ex)
		{
			std::cerr << "SearchDialog::WallpaperSearch() : " << itDir->leaf() << " " << ex.what() << std::endl;
		}// end catch
	}// end for
}// end SearchDialog::WallpaperSearch
