/*-----------------------------------------------------------------------------
 *  FILE: Wallpapers.cc
 *
 *      Copyright(c) 2006 Gareth Foster.
 *
 *      Dialog derivation that implements Wallpaper Tray search tool.
 *
 *-----------------------------------------------------------------------------
 */

#include "Wallpapers.hh"
#include <iostream>
#include <cstdlib>
#include <boost/filesystem/exception.hpp>
#include <boost/version.hpp>
#include <glibmm.h>
#include "GnomeWPList.hh"
#include "Applet.hh"

bool fnNameCheckYes(const std::string &) { return true; }

//------------------------------------------------------------------------
/**
<Wallpapers::Wallpapers>
@brief Create a new wallpaper manager class.

@param _pApplet A pointer to the containing panel applet.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
Wallpapers::Wallpapers
(
	Applet *				_pApplet
) :
	m_regexImageCheck("(.*).jpg|(.*).jpeg|(.*).png|(.*).gif|(.*).bmp|(.*).tiff", boost::regex::icase),
	m_pApplet(_pApplet),
	m_Notify(_pApplet, this)
{
	boost::filesystem::path::default_name_check(fnNameCheckYes);
	
	// set up the gconf client
	m_pGconfClient = Gnome::Conf::Client::get_default_client();
	
	m_pGconfClient->add_dir("/apps/wp_tray");

	// seed random number generator
	std::srand(std::time(NULL));

	// create wallpaper change timeout
	if(m_pGconfClient->get_bool("/apps/wp_tray/b_timeout") == true)
	{
		// set the wallpaper timeout
		m_sigcConnection = Glib::signal_timeout().connect	(
				sigc::mem_fun(this, &Wallpapers::SetRandomDelayed),
				m_pGconfClient->get_int("/apps/wp_tray/n_timeout") * 60 * 1000
			);
	}// end if

	// listen to changes to our keys
	m_pGconfClient->notify_add	(
			"/apps/wp_tray/b_timeout",
			sigc::mem_fun(*this, &Wallpapers::RegenerateTimeout)
		);
	m_pGconfClient->notify_add	(
			"/apps/wp_tray/n_timeout",
			sigc::mem_fun(*this, &Wallpapers::RegenerateTimeout)
		);
}// end Wallpapers::Wallpapers

//------------------------------------------------------------------------
/**
<Wallpapers::~Wallpapers>
@brief Destroy a wallpaper manager class.

@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
Wallpapers::~Wallpapers
(
)
{
	// ...
}// end Wallpapers::~Wallpapers


//------------------------------------------------------------------------
/**
<Wallpapers::SetRandom>
@brief Set a random wallpaper from the user configured directories.

@param _bImediate Whether or not the change should wait for user confirmation.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
bool Wallpapers::SetRandom
(
	bool					_bImediate
)
{
	std::size_t nFileCount(0);
	std::size_t	nStaticCount(0);

	// grab all the directories
	std::list<Glib::ustring> lsWpDir = m_pGconfClient->get_string_list("/apps/wp_tray/dir_list");

	for(std::list<Glib::ustring>::iterator it = lsWpDir.begin(); it != lsWpDir.end(); ++ it)
	{
		// recurse each path to count wallpapers
		Wallpapers::CountDirectoryEntries(it->c_str(), nFileCount);
	}// end for

	nStaticCount = Wallpapers::CountStatics();

	if(nFileCount == 0 && nStaticCount == 0)
	{
		return false;
	}// end if

	// get random image index
	std::size_t nRandomIndex = (std::rand() % (nFileCount + nStaticCount)) + 1;

	if(nRandomIndex <= nFileCount)
	{
		for(std::list<Glib::ustring>::iterator it = lsWpDir.begin(); it != lsWpDir.end(); ++ it)
		{
			// recurse each path to get random wallpaper
			Wallpapers::GetDirectoryEntryAtIndex(it->c_str(), nRandomIndex, m_szFile);
		}// end for
	}// end if
	else
	{
		m_szFile = Wallpapers::GetStaticAtIndex(nRandomIndex % nStaticCount);
	}// end else

	// set new in gconf
	if(m_szFile.length() > 0)
	{
		m_pApplet->RenderThumbnail(m_szFile);

		if(_bImediate == true || m_pGconfClient->get_bool("/apps/wp_tray/b_notifications") == false)
		{
			m_pGconfClient->set("/desktop/gnome/background/picture_filename", m_szFile);

			m_pApplet->SetThumbnailOnPanel();
		}// end if
		else
		{
			m_Notify.WallpaperChanged(m_szFile);
		}// end else
	}// end if
	
	return true;
}// end Wallpapers::set_random

//------------------------------------------------------------------------
/**
<Wallpapers::SetRandomDelayed>
@brief Have a wallpaper set, first giving the user the chance to block the change using libnotify.

@return Whether the wallpaper change was successful.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
bool Wallpapers::SetRandomDelayed
(
)
{
	return Wallpapers::SetRandom(false);
}// end Wallpapers::SetRandomImmediate

//------------------------------------------------------------------------
/**
<Wallpapers::GetCurrent>
@brief Gets the location of the current wallpaper on disk.

@return The location of the current wallpaper on disk.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
Glib::ustring Wallpapers::GetCurrent
(
)
{
	return m_pGconfClient->get_string("/desktop/gnome/background/picture_filename");
}// end Wallpapers::get_current

//------------------------------------------------------------------------
/**
<Wallpapers::DeleteCurrent>
@brief Delete the current wallpaper from disk.

@return Whether or not the wallpaper was deleted.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
bool Wallpapers::DeleteCurrent
(
)
{
	Glib::ustring szWpPath(m_pGconfClient->get_string("/desktop/gnome/background/picture_filename"));

	if(szWpPath.length() == 0)
	{
		return false;
	}// end if

	// attempt to remove the file
	try
	{
		fs::path pathWp(szWpPath.c_str(), fs::native);

		if(fs::exists(pathWp) == false)
		{
			return false;
		}// end if

		fs::remove(pathWp);
	}// end try
	catch(fs::filesystem_error e)
	{
		return false;
	}// end catch

	return Wallpapers::SetRandom(true);
}// end Wallpapers::DeleteCurrent

//------------------------------------------------------------------------
/**
<Wallpapers::AcceptChange>
@brief Private method called by LibNotify to signal users wishes on a wallpaper change.

@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
void Wallpapers::AcceptChange
(
)
{
	m_pGconfClient->set("/desktop/gnome/background/picture_filename", m_szFile);

	m_pApplet->SetThumbnailOnPanel();
}// end Wallpapers::AcceptChange

//------------------------------------------------------------------------
/**
<Wallpapers::RejectChange>
@brief Private method called by LibNotify to signal users wishes on a wallpaper change.

@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
void Wallpapers::RejectChange
(
)
{
	// do nowt
}// end Wallpapers::RejectChange

//------------------------------------------------------------------------
/**
<Wallpapers::DeleteChange>
@brief Private method called by LibNotify to signal users wishes on a wallpaper change.

@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
void Wallpapers::DeleteChange
(
)
{
	m_pGconfClient->set("/desktop/gnome/background/picture_filename", m_szFile);

	Wallpapers::DeleteCurrent();
}// end Wallpapers::DeleteChange

//------------------------------------------------------------------------
/**
<Wallpapers::CountDirectoryEntries>
@brief Recursive method to count the files in all the directories the user is monitoring.

@param _pathCurrent Boost path giving the current directory/path in recursion.
@param _nFileCount Number of files the algorithm has hit so far.
@return The file count.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
std::size_t Wallpapers::CountDirectoryEntries
(
	fs::path				_pathCurrent,
	std::size_t &			_nFileCount
)
{
	fs::directory_iterator itEnd;

	for(fs::directory_iterator it(_pathCurrent); it != itEnd; ++ it)
	{
		try
		{
			// is it a directory and not hidden?
			if(fs::is_directory(*it) && it->leaf()[0] != '.')
			{
				if(fs::symbolic_link_exists(*it))
				{
					if(m_pGconfClient->get_bool("/apps/wp_tray/b_follow_links") == true)
					{
						// recurse
						Wallpapers::CountDirectoryEntries(*it, _nFileCount);
					}// end if
				}// end if
				else
				{
					// recurse
					Wallpapers::CountDirectoryEntries(*it, _nFileCount);
				}// end else
			}// end if
			else
			{
				if(!m_pGconfClient->get_bool("/apps/wp_tray/b_img_check") || boost::regex_match(it->leaf(), m_regexImageCheck))
				{
					++ _nFileCount;
				}// end if
			}// end else
		}// end try
		catch(const std::exception & ex)
		{
			std::cerr << "Wallpapers::CountEntriesInDirectory() : " << ex.what() << std::endl;
		}// end catch
	}// end for

	return _nFileCount;
}// end Wallpapers::CountDirectoryEntries

//------------------------------------------------------------------------
/**
<Wallpapers::CountDirectoryEntries>
@brief Recursive method, treats all monitored folders as one, and finds the file at _nFileIndex.

@param _pathCurrent Boost path giving the current directory/path in recursion.
@param _nFileCount Number of files the algorithm has yet to pass.
@param _szFile An out parameter that gives the resulting file name.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
void Wallpapers::GetDirectoryEntryAtIndex
(
	fs::path				_pathCurrent,
	std::size_t &			_nFileIndex,
	Glib::ustring &			_szFile
)
{
	fs::directory_iterator itEnd;

	if(_nFileIndex == 0)
	{
		return;
	}// end if

	for(fs::directory_iterator it(_pathCurrent); it != itEnd && _nFileIndex != 0; ++ it)
	{
		try
		{
			if(fs::is_directory(*it))
			{
				if(fs::symbolic_link_exists(*it))
				{
					if(m_pGconfClient->get_bool("/apps/wp_tray/b_follow_links") == true)
					{
						// recurse
						Wallpapers::GetDirectoryEntryAtIndex(*it, _nFileIndex, _szFile);
					}// end if
				}// end if
				else
				{
					// recurse
					Wallpapers::GetDirectoryEntryAtIndex(*it, _nFileIndex, _szFile);
				}// end else
			}// end if
			else
			{
				if(!m_pGconfClient->get_bool("/apps/wp_tray/b_img_check") || boost::regex_match(it->leaf(), m_regexImageCheck))
				{
					if(-- _nFileIndex == 0)
					{
#if BOOST_VERSION > 103301
						_szFile = it->path().directory_string();
#else
						_szFile = (*it).native_directory_string();
#endif
					}// end if
				}// end if
			}// end else
		}// end try
		catch(const std::exception & ex)
		{
			std::cerr << "Wallpapers::GetDirectoryEntryAtIndex() : " << it->leaf() << " " << ex.what() << std::endl;
		}// end catch
	}// end for
}// end Wallpapers::GetDirectoryEntryAtIndex

//------------------------------------------------------------------------
/**
<Wallpapers::CountStatics>
@brief Gives the number of useable wallpapers held by the Gnome Wallpaper tool.

@return The number of useable wallpapers held by the Gnome Wallpaper tool.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
std::size_t Wallpapers::CountStatics
(
)
{
	// contentious, but opting here to rescan the xml for gnome wp list every time
	GnomeWPList gnomeWPList;

	return gnomeWPList.GetFileList().size();
}// end Wallpapers::CountStatics

//------------------------------------------------------------------------
/**
<Wallpapers::CountStatics>
@brief Gives the number of useable wallpapers held by the Gnome Wallpaper tool.

@return The number of useable wallpapers held by the Gnome Wallpaper tool.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
std::string Wallpapers::GetStaticAtIndex
(
	std::size_t				_nFileIndex
)
{
	GnomeWPList gnomeWPList;

	const std::list<Glib::ustring> & reflsFileList = gnomeWPList.GetFileList();

	std::list<Glib::ustring>::const_iterator it = reflsFileList.begin(); 

	while(it != reflsFileList.end())
	{
		if(-- _nFileIndex == 0)
		{
			return *it;
		}// end if

		++ it;
	}// end while

	return "";
}// end Wallpapers::GetStaticAtIndex

//------------------------------------------------------------------------
/**
<Wallpapers::RegenerateTimeout>
@brief Called when the user changes the wallpaper change timeout, to re-establish the glib callback.

@param _nCnxn The connection id associated with the gconf key.
@param _gconfEntry The gconf entry that has changed.
@date 12-06-2006 GAF Written
**/
//------------------------------------------------------------------------
void Wallpapers::RegenerateTimeout
(
	guint					_nCnxn,
	Gnome::Conf::Entry		_gconfEntry
)
{
	// delete old timeout!
	m_sigcConnection.disconnect();

	// (re)create wallpaper change timeout
	if(m_pGconfClient->get_bool("/apps/wp_tray/b_timeout") == true)
	{
		// set the wallpaper timeout
		m_sigcConnection = Glib::signal_timeout().connect	(
				sigc::mem_fun(this, &Wallpapers::SetRandomDelayed),
				m_pGconfClient->get_int("/apps/wp_tray/n_timeout") * 60 * 1000
			);
	}// end if
}// end Wallpapers::regenerate_timeout
