// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@ececs.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: PingObject.cpp
// 
//---------------------------------------------------------------------------
#include <string>
#include <clutils/StringUtilities.h>
#include "PingEvent.h"
#include "PingObject.h"
#include "PingObjectState.h"
#include <warped/SimulationManager.h>
#include <warped/warped.h>
#include <warped/IntVTime.h>

PingObject::PingObject( int myId,
			const string &initDestObjectName,
			const int initNumberOfBalls,
			bool master,
			bool initRandomDelays ):
   myObjectName( getName( myId )), 
   myDestObjectName( initDestObjectName ), 
   numBalls( initNumberOfBalls ),
   sendTo( 0 ),
   isMaster( master ),
   randomDelays( initRandomDelays ){}

PingObject::~PingObject(){
  deallocateState(getState());
}

void
PingObject::initialize(){
  sendTo = getObjectHandle(myDestObjectName);
  // object_0 kick starts the simulation
  if( isMaster ){
    startBall();
  }
}

void
PingObject::startBall(){
  PingObjectState* myState = static_cast<PingObjectState*>(getState());
  myState->ballStarted();
  sendEvent( getName() );
}

void
PingObject::sendEvent( const string &owner ){
  //we want to send another event
  IntVTime recvTime = dynamic_cast<const IntVTime&> (getSimulationTime()) + 1;
  Event *newEvent =  new PingEvent( getSimulationTime(),
				    recvTime,
				    getName(),
				    sendTo->getName(),
				    owner );
  sendTo->receiveEvent( newEvent );
  PingObjectState* myState = static_cast<PingObjectState*>(getState());
  myState->ballSent();
}

void
PingObject::executeProcess(){
  PingObjectState *myState = static_cast<PingObjectState*>(getState());
  ASSERT(myState != 0);

  // If we're a master and we've started less balls than we're supposed to
  // have we'll send a new one on.
  if( isMaster ){
    if( myState->getNumStarted() < numBalls ){
      startBall();
    }
    // Else we're finished.
  }

  ASSERT( haveMoreEvents() == true );
  while(haveMoreEvents() == true){ 
    //we got an event
    //let's get the event
    const PingEvent *eventReceived = dynamic_cast<const PingEvent *>(getEvent());
    ASSERT( eventReceived != 0 );
    myState->ballReceived();
    if( randomDelays ){
      // We want reproducibility
      srandom( 42 );
      long maxCount = (long)((double)random()/(double)RAND_MAX)*10000000;
      double x = 1237.0;
      for( long i = 0; i < maxCount; i++ ){
	x = x / 3.3;
      }
      // Just to hopefully keep the compiler from optimizing this out.
      lastX = x;
    }
    if( isMaster ){
      // Is this our ball?
      if( eventReceived->getOwner() != getName() ){
	// Then we'll send it on.
	sendEvent( eventReceived->getOwner() );
      }
      // Else we sent a new ball above if we needed to.
    }
    else{
      sendEvent( eventReceived->getOwner() );
    }
  }
}

void
PingObject::finalize(){
  SEVERITY severity = NOTE;
  //simulation is over 
  //let's see how we did  
  PingObjectState* myState = static_cast<PingObjectState*>(getState());
  ASSERT(myState != NULL);
  
  string msg = myObjectName + " " + myState->getSummaryString() + "\n";
  
  reportError( msg, severity );
}


State*
PingObject::allocateState() {
  return new PingObjectState();
}

void
PingObject::deallocateState( const State *state ){
   // delete state
   // HINT: you could insert this in a free pool of states
   delete state;
}

void
PingObject::reclaimEvent(const Event *event){
   // delete event
   // HINT: you could insert this in a free pool of event
   delete event;
}

string 
PingObject::getName( int forId ){
  return "PingPongObject" + intToString(forId);
}
