--  2001, 2002 Peter Thiemann
module CGIInternals (module CGIInternals, getPathInfo)
{-- interface
  (ask		-- WithHTML x CGI a -> CGI ()
  ,tell		-- CGIOutput a => a -> CGI ()
  ,htell 	-- WithHTML x IO () -> CGI ()
  ,run		-- CGI () -> IO ()
  ,runWithHook 	-- ([String] -> CGI ()) -> CGI () -> IO ()
  )
--}
where

import Prelude hiding (head,div,span)
import qualified Prelude
import qualified List
import Maybe
import Monad
import System
import IO

import Auxiliary
import qualified Base64
import CookieIO
import CGIMonad
import CGIOutput
import CGITypes
import EventHandlers
import Fields
import Frames
import qualified HTMLWrapper as H hiding (map)
import Images
import JavaScript
import qualified RFC2279
import qualified RFC2397
import RawCGIInternal hiding (CGIEnv (..))
import qualified URLCoding

import CGIConfig

-- |Safe embedding of an 'IO' action into the 'CGI' monad. 
io :: (Read a, Show a) => IO a -> CGI a
io ioa =
  once (unsafe_io ioa)

-- |Runs a 'CGI' state transformer forever. Its specification is just
--
-- > feedback f x = f x >>= feedback f
-- 
-- However, 'feedback' is more efficient because it avoids the 
-- buildup of long interaction logs by cutting back every time just before 'f' 
-- is invoked. Unfortunately, it's useless due to typing restrictions.
feedback :: (Read a, Show a) => (a -> CGI a) -> a -> CGI a
feedback f x = 
  CGI (\ cgistate ->
  case inparm cgistate of
    -- if we've got a value in the log,
    -- ignore the parameter x and just use the logged value
    PAR_MARK _ : PAR_RESULT code : rest ->
      case reads code of
	(result, ""):_ ->
	  return (result
	         ,cgistate { inparm = rest })
	_ ->
	  reportError "Result unreadable"
	  	(H.text_S "Cannot read " >> H.text (show code)) cgistate
    [] ->
      let stid = stateID cgistate 
	  newparm = PAR_MARK stid
	  newvalue = PAR_RESULT (show x)
	  in
      do (nextx, cgistate') <- 
	   unCGI (f x) cgistate { outparm = newvalue : newparm : outparm cgistate
				, stateID = nextstid (nextstid stid newparm) newvalue }
	 let (oldstid, outparm') = popToMark $ outparm cgistate'
	 unCGI (feedback f nextx)
	       cgistate { inparm =  inparm cgistate'
			, outparm = outparm'
			, stateID = oldstid
			, cookieMap = cookieMap cgistate' 
			, cookiesToSend = cookiesToSend cgistate' }
    _ -> 
      reportError "Out of sync" H.empty cgistate)

-- |Control operator for the 'CGI' monad. Its specification is
--
-- > callWithCurrentHistory (\backto x -> action x >>= backto) x
-- > ==
-- > action x >>= callWithCurrentHistory (\backto x -> action x >>= backto)
-- 
-- However, 'callWithCurrentHistory' is more efficient because it avoids the 
-- buildup of long interaction logs by cutting back every time just before
-- 'action' gets executed. 
callWithCurrentHistory :: (Read a, Show a) =>
			  ((a -> CGI ()) -> a -> CGI ()) -> a -> CGI ()
callWithCurrentHistory g x = 
  CGI (\ cgistate ->
  let
    prompt stid x =
      CGI (\ cgistate' ->
	let outparm' = popTo stid (outparm cgistate') 
	in
	unCGI (callWithCurrentHistory g x) 
	      cgistate { inparm =  inparm cgistate'
		       , outparm = outparm'
		       , stateID = stid
		       , cookieMap = cookieMap cgistate' 
		       , cookiesToSend = cookiesToSend cgistate' } )
    popTo stid [] = error "popTo did not find its mark"
    popTo stid (PAR_MARK stid' : rest) | stid' == stid = rest
    popTo stid (_ : rest) = popTo stid rest
  in
  case inparm cgistate of
    -- if we've got a value in the log,
    -- ignore the parameter x and just use the logged value
    PAR_MARK stid : PAR_RESULT code : rest ->
      case reads code of
	(result, ""):_ ->
	  unCGI (g (prompt stid) result) cgistate { inparm = rest }
	_ ->
	  reportError "Result unreadable"
	  	(H.text_S "Cannot read " >> H.text (show code)) cgistate
    [] ->
      let stid = stateID cgistate 
	  newmark = PAR_MARK stid
	  newvalue = PAR_RESULT (show x)
	  in
      unCGI (g (prompt stid) x)
	cgistate { outparm = newvalue : newmark : outparm cgistate
		 , stateID = nextstid (nextstid stid newmark) newvalue }
    _ -> 
      reportError "Out of sync" H.empty cgistate)

-- |Brackets a 'CGI' action so that only its result is visible. Improves
-- efficiency by not executing the bracketed action after it has been performed
-- once. Use this for avoiding the inefficient buildup of long interaction logs.
once :: (Read a, Show a) => CGI a -> CGI a
once cgi = 
  CGI (\ cgistate ->
  case inparm cgistate of
    PAR_RESULT code : rest ->
      case reads code of
	(result, ""):_ ->
	  return (result
	         ,cgistate { inparm = rest })
	_ ->
	  reportError "Result unreadable"
	  	(H.text_S "Cannot read " >> H.text (show code)) cgistate
    [] ->
      let stid = stateID cgistate 
	  newparm = PAR_MARK stid in
      unCGI cgi cgistate { outparm = newparm : outparm cgistate
			 , stateID = nextstid stid newparm }
      >>= finish cgistate
    PAR_MARK _ : rest -> 
      unCGI cgi cgistate { inparm = rest } >>= finish cgistate
    _ -> 
      reportError "Out of sync" H.empty cgistate)
  where finish cgistate (v, cgistate') =
 	  let (oldstid, outparm') = popToMark $ outparm cgistate'
	      newparm = PAR_RESULT (show v)
	  in
	  return (v
	      	 ,cgistate { inparm = inparm cgistate'
			   , outparm = newparm : outparm'
			   , stateID = nextstid oldstid newparm
			   , cookieMap = cookieMap cgistate' 
			   , cookiesToSend = cookiesToSend cgistate' })

-- |Repeats a 'CGI' action without saving its state so that the size of the
-- interaction log remains constant.
forever :: CGI () -> CGI ()
forever cgi = 
  CGI (\ cgistate ->
  case inparm cgistate of
    [] ->
      let stid = stateID cgistate 
	  newparm = PAR_MARK stid in
      unCGI cgi cgistate { outparm = newparm : outparm cgistate
			 , stateID = nextstid stid newparm }
      >>= 
      const (reportError "Black hole" H.empty cgistate)
    PAR_MARK _ : rest -> 
      unCGI cgi cgistate { inparm = rest } >>= finish (mcount cgistate)
    _ -> 
      reportError "Out of sync" H.empty cgistate)
  where finish previousMcount (v, cgistate') =
	  let (oldstid, outparm') = popToMark $ outparm cgistate'
	      newparm_ignored = PAR_RESULT (show v)
	      newparm = PAR_MARK oldstid
	  in
	  unCGI cgi 
	    cgistate' { inparm = []
		      , outparm = newparm : outparm'
		      , stateID = nextstid oldstid newparm
		      , mcount = previousMcount }

-- |Unsafe variant of 'once':  returns the computed value only the first time
-- and returns a default value in all later invocations.
-- [deprecated]
onceAndThen :: (Read a, Show a) => a -> CGI a -> CGI a
onceAndThen a cgi = 
  CGI (\ cgistate ->
  case inparm cgistate of
    PAR_IGNORED : rest ->
      return (a
	     ,cgistate { inparm = rest })
    [] ->
      unCGI cgi cgistate { outparm = PAR_MARK (stateID cgistate) : outparm cgistate }
      >>= finish
    PAR_MARK _ : rest -> 
      unCGI cgi cgistate { inparm = rest } >>= finish
    _ -> 
      reportError "Out of sync" H.empty cgistate)
  where popToMark [] = []
	popToMark (PAR_MARK _: rest) = rest
	popToMark (_: rest) = popToMark rest
	finish (v, cgistate') =
 	  let out = outparm cgistate' in
	  return (v
	      	 ,cgistate' { outparm = PAR_IGNORED : popToMark out })

-- internal helper function
popToMark [] = error "popToMark did not find MARK"
popToMark (PAR_MARK v: rest) = (v, rest)
popToMark (_: rest) = popToMark rest

-- |Directly lifts the 'IO' monad into the 'CGI' monad. This is generally unsafe
-- and should be avoided. Use 'io' instead.
unsafe_io :: IO a -> CGI a
unsafe_io = lift

-- |Takes a monadic value that constructs a HTML page and delivers this
-- page to the browser. This page may contain forms and input widgets.
ask :: H.WithHTML x CGI a -> CGI ()
ask ma =
  do CGI (\cgistate -> 
       return ((), cgistate { pageInfo = initialPageInfo cgistate }))
     setAction tell
     elem <- H.build_document ma 
     CGI $
       \cgistate -> let
	 pi = pageInfo cgistate
	 atable = actionTable pi
	 mbnds = bindings pi
	 msubmitter = mbnds >>= assocParm subVar
	 maction = msubmitter >>= \x -> lookup x atable
	 nextState = nextCGIState cgistate
	 defsubmission = listToMaybe (reverse atable)
	 defaction = maybe (unCGI . tell) 
			   (const (maybe (unCGI . tellError "Unspecified action")
					 snd
					 defsubmission))
			   mbnds
	 go = (maybe defaction id maction) elem nextState
	 oldgo = (nextaction pi elem) nextState
       in
	 go

-- |Like 'ask', but passes the constructed HTML page to the @elementAction@
-- parameter. This function may send the page via Email or store it into a
-- file. Anyone loading this page in a browser can resume the interaction.
askOffline :: H.WithHTML x CGI a -> (H.Element -> IO ()) -> CGI ()
askOffline ma elementAction =
  do CGI (\cgistate -> return ((), cgistate { pageInfo = initialPageInfo cgistate }))
     setAction tell
     elem <- H.build_document ma 
     CGI $
       \cgistate ->
       case bindings (pageInfo cgistate) of
         Nothing ->
	   elementAction elem >> return ((), cgistate)
	 Just _ ->
	   (nextaction (pageInfo cgistate) elem) (nextCGIState cgistate)

-- |Turns a 'CGI' action into an 'IO' action. Used to turn the main 'CGI' action
-- into the @main@ function of the program. Typical use:
-- 
-- > main = run mainCGI
run :: CGI () -> IO ()
run =
  runWithOptions []

-- |Turns a 'CGI' action into an 'IO' action. Used to turn the main 'CGI' action
-- into the @main@ function of the program. Takes additional low-level
-- options. Typical use: 
-- 
-- > main = runWithOptions [] mainCGI
runWithOptions :: CGIOptions -> CGI () -> IO ()
runWithOptions options =
  runInternal options (fallbackTranslator Nothing)

-- |Variant of 'run' where an additional argument @cgigen@ specifies an action
-- taken when the script is invoked with a non-empty query string as in
-- @script-name?query-string@ 
runWithHook :: CGIOptions -> ([String] -> CGI ()) -> CGI () -> IO ()
runWithHook options cgigen =
  runInternal options (fallbackTranslator (Just cgigen))

runInternal options cgigen cgiProg =
  start options $ makeServletInternal cgigen cgiProg

-- |Transform a CGI action into a servlet suitable for running from Marlow's web
-- server.
makeServlet :: CGI () -> CGIProgram
makeServlet cgiProg = 
  makeServletInternal (fallbackTranslator Nothing) cgiProg

-- |Like 'makeServlet' with additional CGI generator as in 'runWithHook'.
makeServletWithHook :: ([String] -> CGI ()) -> CGI () -> CGIProgram
makeServletWithHook cgigen cgiProg =
  makeServletInternal (fallbackTranslator $ Just cgigen) cgiProg

makeServletInternal cgigen (CGI cgi) = \ info decoded_parameters ->
  let maybecgiparm = assocParm "=CGI=parm=" decoded_parameters in
  let maybecgistid = assocParm "=CGI=stid=" decoded_parameters in
  do oldparm <- case maybecgiparm of
       Just cgiparm ->
         liftM read $ liftM RFC2279.decode $ decode $ Base64.decode' $ cgiparm
       Nothing -> 
         return []
     key <- generateKey
     let newparm 
          | null decoded_parameters = []
	  | otherwise = (PAR_VALUES $ dropSpecialParameters decoded_parameters)
	  		: oldparm
	 oldstid
	  | null decoded_parameters || isNothing maybecgistid = initialStateID
	  | otherwise = fromJust maybecgistid
	 cgistate = CGIState { inparm = reverse newparm
		             , outparm = newparm
			     , stateID = oldstid
			     , cgiInfo = info
			     , mcount = 0
			     , pageInfo = (initialPageInfo cgistate) {inFrame = 0}
			     , encoder = makeEncoder key
			     , cookieMap = map decodeCookie (cgiCookies info)
			     , cookiesToSend = []
			     }
	 args = cgiArgs info
     -- writeDebugFile "/tmp/CGIOLDPARM" (show oldparm)
     -- writeDebugFile "/tmp/CGINEWPARM" (show newparm)
     if null args
	 then cgi cgistate >> return ()
	 else unCGI (cgigen $ args) cgistate >> return ()
     exitWith ExitSuccess

-- ======================================================================
-- internal references

-- |Create a hyperlink to internal entity.
makeRef :: Monad m
	=> String		-- ^internal name of entity
	-> H.WithHTML x m ()	-- ^body of the reference
	-> CGI (H.WithHTML y m ())
makeRef fileName elems =
  CGI (\cgistate ->
  let fileURL = url cgistate ++ '?' : fileName in
  return (hlink (URL fileURL) elems, cgistate))

-- |Create a popup hyperlink to internal entity.
makePopupRef :: 
           String		-- ^name of popup window
	-> String		-- ^internal name of entity
	-> H.HTMLCons x y CGI ()
makePopupRef name fileName elems =
  do baseUrl <- H.lift getUrl
     let fileURL = baseUrl ++ '?' : fileName
     popuplink name (URL fileURL) elems

-- |Create hyperlink to internal entity @\/path?name@.
makeA :: String -> String -> HTMLField x y ()
makeA path name elems =
  do url <- H.lift getUrl
     let fullurl = url ++ '/': path ++ '?' : name
     hlink (URL fullurl) elems

-- ======================================================================
-- input fields & forms
-- 
data VALID = VALID
data INVALID = INVALID

data InputField a x = 
     InputField { ifName :: String
		, ifToken :: CGIFieldName  
     		, ifFty :: String
		, ifString :: Maybe String
		, ifValue :: Maybe a
		, ifRaw :: CGIParameters
		, ifBound :: Bool	    -- True if form submitted
		}

-- |create a virtual input field from the concatenation of two input fields
concatFields :: (Reason c, Read c)
	     => InputField c INVALID -> InputField Text INVALID
  	     -> InputField c INVALID
concatFields ifa ifb = 
  concatFieldsWith g ifa [ifb]
  where g sa [sb] = sa ++ sb

-- |Create a virtual input field from the result of applying a function to two
-- input fields. Parsing is applied to the result of the function call.
concatFieldsWith :: (Reason c, Read c) 
		 => (String -> [String] -> String)
		 -> InputField c INVALID 
		 -> [InputField Text INVALID]
                 -> InputField c INVALID
concatFieldsWith trans ifa ifbs =
  let newString = do stra <- ifString ifa
		     strbs <- mapM ifString ifbs
		     return (trans stra strbs)
      newValue  = do s <- newString
		     maybeRead s
  in 
  InputField { ifName = ifName ifa	-- ++ '|' : ifName ifb
	     , ifToken = ifToken ifa
	     , ifFty = ifFty ifa
	     , ifString = newString
	     , ifValue = newValue
	     , ifRaw = ifRaw ifa
	     , ifBound = ifBound ifa && all ifBound ifbs
	     }

-- |Combine the values of separately parsed fields
combineFieldsWith2 f2 if1 if2 =
  InputField { ifName = ifName if1	-- ++ '|' : ifName if2
	     , ifToken = ifToken if1
	     , ifFty = ifFty if1
	     , ifString = Nothing
	     , ifValue = liftM2 f2 (ifValue if1) (ifValue if2)
	     , ifRaw = ifRaw if1
	     , ifBound = ifBound if1 && ifBound if2
	     }

-- transition code
name = ifName
string = ifString
valueInputField inf =
  case ifValue inf of
    Nothing -> error ("InputField { " ++
		      "ifName = " ++ show (ifName inf) ++ ", " ++
		      "ifString = " ++ show (ifString inf) ++ ", " ++
		      "ifBound = " ++ show (ifBound inf) ++
		      " }")
    Just vl -> vl
-- raw = ifRaw
-- transition code end

feither :: (a -> b) -> (c -> d) -> Either a c -> Either b d
feither f g (Left a) = Left (f a)
feither f g (Right b) = Right (g b)

-- to expose less of the implementation, the following type could be 
-- propagate :: MonadPlus err => Either (err x) a -> .. Either (err x) (a, b)
propagate :: Either [err] a -> Either [err] b -> Either [err] (a,b)
propagate (Right a) (Right b) = Right (a, b)
propagate (Right a) (Left bss) = Left bss
propagate (Left ass) (Right b) = Left ass
propagate (Left ass) (Left bss) = Left (mplus ass bss)

data ValidationError =
     ValidationError { veName :: String			    -- name of erroneous field
		     , veToken :: CGIFieldName		    -- token of erroneous field
		     , veString :: Maybe String		    -- value of erroneous field
		     }

validateInputField inf =
    case ifValue inf of 
      Nothing | ifBound inf -> 
        Left [ValidationError (ifName inf) (ifToken inf) (ifString inf)]
      _ ->
        Right InputField{ ifName = ifName inf
			, ifToken = ifToken inf
			, ifFty  = ifFty inf
			, ifString = ifString inf
			, ifValue = ifValue inf
			, ifRaw = ifRaw inf
			, ifBound = ifBound inf
			}

-- internal
data InputType = TEXT | PASSWORD | CHECKBOX |
    RADIO | SUBMIT | RESET |
    FILE | HIDDEN | IMAGE | BUTTON
    deriving (Eq)

instance Show InputType where
  show TEXT = "text"
  show PASSWORD = "password"
  show CHECKBOX = "checkbox"
  show RADIO = "radio"
  show SUBMIT = "submit"
  show RESET = "reset"
  show FILE = "file"
  show HIDDEN = "hidden"
  show IMAGE = "image"
  show BUTTON = "button"

textual :: InputType -> Bool
textual TEXT = True
textual PASSWORD = True
textual FILE = True
textual _ = False

-- |Every input widget maps the content generator for the widget (which may
-- produce HTML elements or attributes) to the content generator of the widget. 
type HTMLField x y a = H.WithHTML x CGI () -> H.WithHTML y CGI a

-- |Creates a reset button that clears all fields of a form.
resetField :: HTMLField x y (InputField () INVALID)
resetField = 
  genericField RESET (const (Just ()))

-- |Creates a submit button. Unsafe. Internal use only.
submitField :: CGI () -> HTMLField x y ()
submitField action =
  internalSubmitField False (Right action)

defaultSubmitField :: CGI () -> HTMLField x y ()
defaultSubmitField action =
  internalSubmitField True (Right action)

internalSubmitField isDefault what attrs =
  do sf <- genericField SUBMIT (const (Just ())) attrs
     attachAction (ifName sf) isDefault what

attachAction fname isDefault what =
  do pageInfo <- H.lift getInfo
     let mbnds = bindings pageInfo
	 localAction = 
	   case what of
	     Right action ->
	       let act = resetFrame >> action in
	       do H.lift (setAction (const act))
		  H.lift (registerAction fname (const act))
	     Left sts ->
	       let names = map veName sts
		   values = map (fromMaybe "" . veString) sts
	       in
	       do H.lift (setAction tell)			    -- CHANGE THIS
		  H.lift (registerAction fname tell)
		  H.lift (setFaulty $ zip names values)
{-- 
     lift (unsafe_io (do appendFile "/tmp/CGIMBNDS" ("\nlooking for "++name sf++"\n")
			 appendFile "/tmp/CGIMBNDS" (show mbnds)))
--}
     case mbnds >>= assocParm subVar of
       Nothing ->
	 return ()
       Just submitter ->
	 if submitter == fname || submitter == "" && isDefault 
	 then localAction
	 else return ()
	 
{-	 
     case do bnds <- maybeToList mbnds
	     's' : fns <-  fieldNames bnds
	     return ()
          of
       [] | isDefault && isJust mbnds -> 
         localAction
       _ ->
         case mbnds >>= assocParm fname of
           Nothing ->
             return ()
	   Just _ ->
             localAction
-}

-- |Creates an input field that submits the field to the browser when data is
-- entered into this field. 
activeInputField :: (Reason a, Read a)
	=> (a -> CGI ())	-- ^Function that maps input data to a CGI action.
	-> HTMLField x y ()
activeInputField actionFun attrs =
  activateI actionFun inputField attrs

-- |Attach a CGI action to the value returned by the input field. Activation
-- means that data is submitted as soon as it is entered.
activateI :: (a -> CGI ()) -> HTMLField x y (InputField a INVALID) -> HTMLField x y ()
activateI actionFun inputField attrs =
  do invalid_inf <- inputField (do attrs
				   onChange $ "WASHSubmit(this.name);")
     let r = validateInputField invalid_inf
	 rv = either Left (Right . valueInputField) r
     when (ifBound invalid_inf) $
       activateInternal actionFun (ifName invalid_inf) rv

activateInternal actionFun name what =
  case what of
    Right val -> 
      let act = resetFrame >> actionFun val in
      do H.lift (setAction (const act))
	 H.lift (registerAction name (const act))
    Left sts ->
      let names = map veName sts
	  values = map (fromMaybe "" . veString) sts
      in
      do H.lift (setAction tell)			    -- CHANGE THIS
	 H.lift (setFaulty $ zip names values)
	 H.lift (registerAction name tell)

-- |Create a textual input field. Return type can be *anything* in class 'Reason'
-- and 'Read'.
inputField :: (Reason a, Read a) => HTMLField x y (InputField a INVALID)
inputField = 
  genericField TEXT (maybeRead . fromMaybe "")

maybeRead :: Read a => String -> Maybe a
maybeRead s = g (reads s)
  where g ((a,""):_) = Just a
	g _ = Nothing

-- |Create a textual input field that returns the string entered. (Avoids having
-- to put quotes around a string.)
textInputField :: HTMLField x y (InputField String INVALID)
textInputField =
  genericField TEXT id

-- |Creates a textual input field that takes a custom validation function.
checkedTextInputField :: (Maybe String -> Maybe String)
			 -> HTMLField x y (InputField String INVALID)
checkedTextInputField g attrs =
  genericField TEXT g attrs

-- |Like 'inputField' but the characters are not echoed on the screen.
passwordInputField :: (Reason a, Read a) => HTMLField x y (InputField a INVALID)
passwordInputField =
  genericField PASSWORD (maybeRead . fromMaybe "")

-- |Creates a checkbox. Returns 'True' if box was checked.
checkboxInputField :: HTMLField x y (InputField Bool INVALID)
checkboxInputField =
  genericField CHECKBOX g
  where g Nothing = Just False
	g (Just _) = Just True

-- |Creates a file input field. Returns a temporary 'FileReference'. The
-- 'fileReferenceName' of the result is *not* guaranteed to be persistent. The
-- application is responsible for filing it away at a safe place.
fileInputField :: HTMLField x y (InputField FileReference INVALID)
fileInputField attrs =
  H.lift (setEnctype "multipart/form-data") >>
  genericField FILE (maybeRead . fromMaybe "") attrs

-- |Creates a file input field. Like 'fileInputField' but has an additional
-- parameter for additional validation of the input.
checkedFileInputField :: (Maybe FileReference -> Maybe FileReference) 
			  -> HTMLField x y (InputField FileReference INVALID)
checkedFileInputField filter attrs =
  H.lift (setEnctype "multipart/form-data") >>
  genericField FILE (filter . maybeRead . fromMaybe "") attrs

instance Reason FileReference where
  reason _ = "FileReference"

genericField :: Reason a => InputType -> (Maybe String -> Maybe a) -> HTMLField x y (InputField a INVALID)
genericField inputType decode fieldAttrs =
  let isSUBMIT = inputType == SUBMIT 
      isFILE   = inputType == FILE
  in
  do fieldName'' <- H.lift nextName
     let fieldName' = show fieldName''
	 fieldName | isSUBMIT = 's' : tail fieldName' 
	 	   | otherwise = fieldName'
	 fieldType = show inputType
	 checked = textual inputType
     info <- H.lift getInfo
     (explanation, result) <-
       H.input_T
	     (do H.attr_SS "type" fieldType
		 fieldAttrs
		 H.attr_SD "name" fieldName
		 when isSUBMIT $
		   onClick ("this.form."++subVar++".value=this.name; return true")
		 attrs <- H.get_attrs
		 let [nameAttr] = [ H.attr_value a 
				  | a <- attrs, H.attr_name a == "name" 
				  ]
		     bds = bindings info
		     maybestring = bds >>= assocParm nameAttr
		     rawvalues = maybeToList bds >>= assocParmR nameAttr
		     mdecoded = decode maybestring
		     decoded = fromJust mdecoded
		     isBound = isJust bds
		     theReason = reason decoded
		     advice = "Enter " ++ prependArticle theReason
		     explanation = theReason ++ " expected"
		 -- H.attr_SD "washtype" (washtype decoded)
		 when checked $ do
		   onMouseOver ("self.status=" ++ jsShow advice ++ "; return true")
		   onMouseOut  ("self.status=''; return true")
		 unless (isFILE || isSUBMIT) $ case maybestring of 
	           Nothing -> H.empty
		   Just str -> H.attr_SD "value" str
		 return (explanation,
		   InputField { ifName = nameAttr
			      , ifToken = fieldName''
     		  	      , ifFty = fieldType
			      , ifString = maybestring
			      , ifValue = mdecoded
			      , ifRaw = rawvalues
			      , ifBound = isBound
			      }))
     H.lift $ addField (ifName result) checked
     when checked $ do
       nothingI <- internalImage nothing explanation
       makeImg nothingI 
         (do H.attr_SS "align" "center"
	     H.attr_SD "name" ('i' : ifName result))
     return result

-- |generates a hyperlink that submits the current form.
internalSubmitLink :: Bool -> Either [ValidationError] (CGI ()) -> H.HTMLCons x y CGI ()
internalSubmitLink isDefault what subs =
  do fieldToken <- H.lift nextName
     let fieldName = show fieldToken
	 atv = "javascript:" ++ "WASHSubmit('"++fieldName++"'); void 0;"
     H.a_T (H.attr_SS "href" atv >> subs)
     attachAction fieldName isDefault what

-- |Create an input field from an image. Returns (x,y)
-- position clicked in the image.
imageField :: Image -> HTMLField x y (InputField (Int, Int) INVALID)
imageField image fieldAttrs =
  do fieldToken <- H.lift nextName
     let fieldName = show fieldToken
     H.input_T (do H.attr_SS "type" "image"
		   H.attr_SD "name" fieldName
		   H.attr_SD "src" (unURL $ imageSRC image)
		   fieldAttrs)
     H.lift $ addField fieldName False
     info <- H.lift getInfo
     return $
       let maybe_xy = 
	    do bds <- bindings info
	       x <- assocParm (fieldName ++ ".x") bds
	       y <- assocParm (fieldName ++ ".y") bds
	       return (x, y)
       in
       InputField { ifName = fieldName
		  , ifToken = fieldToken
       		  , ifFty = "image"
		  , ifString = do (x, y) <- maybe_xy
				  return ("(" ++ x ++ "," ++ y ++ ")")
		  , ifValue  = do (x, y) <- maybe_xy
				  return (read x, read y)
     		  , ifRaw = []
		  , ifBound = isJust (bindings info)
		  }

-- a virtual field that never appears on the screen
data RadioGroup a x =
     RadioGroup { radioName   :: String
		, radioToken  :: CGIFieldName
                , radioString :: Maybe String
		, radioValue  :: Maybe a
		, radioBound  :: Bool
		}

validateRadioGroup rg =
  case radioValue rg of 
    Nothing | radioBound rg -> 
      Left [ValidationError (radioName rg) (radioToken rg) (radioString rg)]
    _ ->
      Right RadioGroup  { radioName = radioName rg
			, radioToken = radioToken rg
			, radioString = radioString rg
			, radioValue = radioValue rg
			, radioBound = radioBound rg
			}

valueRadioGroup rg =
  case radioValue rg of
    Nothing -> error ("RadioGroup { " ++
		      "radioName = " ++ show (radioName rg) ++ ", " ++
		      "radioString = " ++ show (radioString rg) ++ ", " ++
		      "radioBound = " ++ show (radioBound rg) ++
		      " }")
    Just vl -> vl

-- |Create a handle for a new radio group. /This handle is invisible on the screen!/
radioGroup :: Read a => H.WithHTML x CGI (RadioGroup a INVALID)
radioGroup =
  do token <- H.lift nextName
     let fieldName = show token
     info <- H.lift getInfo
     H.lift $ addField fieldName False
     let bds = bindings info
	 maybeString = bds >>= assocParm fieldName
	 maybeVal = maybeString >>= (g . reads . URLCoding.decode)
	 g ((a,""):_) = Just a
	 g _ = Nothing
     return $
       RadioGroup { radioName = fieldName
		  , radioToken = token
		  , radioString = maybeString
		  , radioValue = maybeVal
		  , radioBound = isJust bds
		  }

-- |Create a new radio button and attach it to an existing 'RadioGroup'.
radioButton :: Show a => RadioGroup a INVALID -> a -> HTMLField x y ()
radioButton rg val fieldAttrs =
  H.input_T (do H.attr_SS "type" "radio"
		H.attr_SD "name" (radioName rg)
		H.attr_SD "value" (URLCoding.encode (show val)) 
		fieldAttrs)

-- |Create and place the error indicator for an existing 'RadioGroup'. Becomes
-- visible only if no button of a radio group is pressed.
radioError :: RadioGroup a INVALID -> H.WithHTML x CGI ()
radioError rg = 
  let name = radioName rg in
  do im <- internalImage nothing "Select exactly one button"
     makeImg im (H.attr "align" "center" >> H.attr "name" ('i' : name))

-- buttons

-- |Create a single button.
makeButton :: HTMLField x y (InputField Bool INVALID)
makeButton fieldAttrs =
  let fieldType = "button" in
  do fieldToken <- H.lift nextName
     let fieldName = show fieldToken
     H.input_T (do H.attr_SS "type" fieldType
		   H.attr_SD "name" fieldName 
		   fieldAttrs)
     H.lift $ addField fieldName False
     info <- H.lift getInfo
     let bds = bindings info
	 maybeString = bds >>= assocParm fieldName
	 maybeVal = 
	   case bds of
	     Nothing -> Nothing
	     Just parms ->
	       case maybeString of
	         Nothing -> Just False
		 Just _  -> Just True
     return $
       InputField { ifName = fieldName
		  , ifToken = fieldToken
     		  , ifFty = fieldType
		  , ifString = maybeString
		  , ifValue = maybeVal
		  , ifRaw = []
		  , ifBound = isJust bds
		  }

-- form
-- multiple arguments formed according to CGI 1.1 rev 3 spec
-- http://cgi-spec.golux.com/draft-coar-cgi-v11-03-clean.html#5.0

constructQuery url [] =
  url
constructQuery url args =
  url ++ '?' : concat (List.intersperse "+" (map URLCoding.encode args))

-- |Wraps an HTML form around its arguments. All standard attributes are
-- computed and need not be supplied explicitly.
makeForm :: H.WithHTML x CGI a -> H.WithHTML y CGI ()
makeForm attrs_elems = do
  url <- H.lift getUrl
  args <- H.lift getCGIArgs
  H.script_S 
    (do H.attr_SS "type" "text/javascript"
	H.comment $
	      "\n" ++
	      "var SubmitAction=[];" ++ 
	      "function OnSubmit(){" ++
	      "var r=true;" ++
	      "for(var i=0;i<SubmitAction.length;i++){r=r&&SubmitAction[i]();};" ++ 
	      "return r;};"++
	      "function WASHSubmit(fn){" ++
	      "var ff = document.forms[0];" ++
	      "ff."++subVar++".value=fn;" ++
	      "ff.submit();" ++
	      "}" ++
	      "\n// "
	     )
  jsprog <- H.form_T $
    do H.attr_SD "action" (constructQuery url args)
       H.attr_SS "method" "post"
       onSubmit ("return OnSubmit();")
       attrs_elems
       H.attr_SS "target" "_self"		    -- ensure target attr is present
       -- formname <- liftM show $ H.lift nextName
       info <- H.lift getInfo
       parm <- H.lift getParm
       stid <- H.lift getStateID
       encoder <- H.lift getEncoder
       fields <- H.lift getFields
       let ff = faultyfields info
	   realparm | null ff   = parm
                  | otherwise = tail parm
       H.input_S
	 (do H.attr_SS "type" "hidden"
	     H.attr_SS "name" subVar
	     H.attr_SS "value" "")
       H.input_S 
	 (do H.attr_SS "type" "hidden"
	     H.attr_SS "name" "=CGI=parm="
	     H.attr_SD "value"
	       (Base64.encode' $ encoder $ RFC2279.encode $ show $ realparm))
       H.input_S 
	 (do H.attr_SS "type" "hidden"
	     H.attr_SS "name" "=CGI=stid="
	     H.attr_SD "value" (Base64.encode' stid))
       let iFields = [iname | (iname@('f':_), True) <- fields]
	   questionURL = url ++ '?' : ffName question
	   hilight =
	    case iFields of
	     [] -> []
	     name':_ ->  ["document.forms[0]." ++ name' ++ ".focus();"
	 		 ,"document.forms[0]." ++ name' ++ ".select();"]
	   markAsFaulty (fname, fvalue) | fname `elem` iFields =
	     "document.i" ++ fname ++ ".src = " ++ jsShow questionURL ++ ";"
	    | otherwise =
	     "popupstr += " 
	     ++ jsShow ("In a previous form, the field " ++ fname ++ 
			" had an unparsable input value: " ++ fvalue ++ "\n")
	     ++ ";"
	   jsprog | null ff = hilight
		| otherwise =
	    "popupstr = '';"
	    : map markAsFaulty ff
	    ++ "if (popupstr != '') alert(popupstr + 'Please go back and reenter!');"
	    : hilight
       when (not (null ff)) $ H.comment_T ("Faultyfields: " ++ show ff)
       -- H.attr_SD "name" formname
       H.attr_SD "enctype" (enctype info)
       return jsprog
  when (not (null jsprog)) $
      H.script_T
	 (do H.attr_SS "type" "text/javascript"
	     H.rawtext_S "<!-- "
	     H.rawtext ('\n' : unlines jsprog)
	     H.rawtext_S "// -->")

-- textarea

-- |Create a text area with a preset string.
makeTextarea :: String -> HTMLField x y (InputField String INVALID)
makeTextarea fieldValue fieldAttrs =
  do token <- H.lift nextName
     info <- H.lift getInfo
     let bds = bindings info
	 mvalue = bds >>= assocParm name
	 name = show token
	 displayValue = fromMaybe fieldValue mvalue
     H.textarea_T (do H.attr_SD "name" name
		      H.text_S displayValue
		      fieldAttrs)
     return $
       InputField { ifName = name
		  , ifToken = token
       		  , ifFty = "textarea"
		  , ifString = mvalue
		  , ifValue = mvalue
		  , ifRaw = maybeToList bds >>= assocParmR name
		  , ifBound = isJust bds
		  }

-- select

selectTags = map (('o':) . show) [(1::Int)..] 

-- |Create a selection box where multiple entries can be selected.
selectMultiple :: Eq a
	=> (a -> String)	-- ^function to display values of type a
	-> [a]			-- ^list of preselected entries
	-> [a]			-- ^list of all possible entries
	-> (Int, Int)		-- ^(min, max) number of fields that must be selected 
	-> HTMLField x y (InputField [a] INVALID)
selectMultiple shw defs opts (minSel, maxSel) attrs =
  do token <- H.lift nextName
     let name = show token
     info <- H.lift getInfo
     let bds = bindings info
	 rawvalues = maybeToList bds >>= assocParmR name
	 g ('o':i:rest) = i /= '0' 
	 g _ = False
	 inputs = filter g $ map fieldContents rawvalues
	 values = inputs >>=
		  (maybeToList . flip List.elemIndex selectTags) >>=
		  (\i -> [opts !! i])
	 len = length inputs
	 mvalue | minSel <= len && len <= maxSel = Just values
	        | otherwise = Nothing
	 revisedDefaults | isJust bds && not (null inputs) = values
	 		 | otherwise = defs
     let makeoption (opt, tag) = 
	   H.option_T (do H.text (shw opt)
			  H.attr_SD "value" tag
			  when (opt `elem` revisedDefaults) $
			       H.attr_SS "selected" "selected")
	 makeChoice :: Int -> String
	 makeChoice n | n == 0 = "no choice"
                   | n == 1 = "1 choice"
		   | n == maxBound = "arbitrary many choices"
		   | otherwise = show n ++ " choices"
	 makeRange :: Int -> Int -> String
	 makeRange lo hi | lo == maxBound = "Arbitrary many choices"
                      | lo == hi = "Exactly " ++ makeChoice lo
                      | otherwise = "Minimum " ++ makeChoice lo ++
		      		    "; maximum " ++ makeChoice hi
     H.select_T
       (do attrs
	   H.attr_SD "name" name
	   when (maxSel > 1) $
		H.attr_SS "multiple" "multiple"
	   when (null defs && minSel > 0) $
		H.option_S (do H.text_S "--"
			       H.attr_SS "value" "o0")
	   mapM_ makeoption (zip opts selectTags))
     im <- internalImage nothing (makeRange minSel maxSel)
     makeImg im (do H.attr_SS "align" "center"
		    H.attr_SD "name" ('i' : name))
     return $ 
       InputField { ifName = name
		  , ifToken = token
       		  , ifFty = "select"
		  , ifString = Nothing -- fmap show value
		  , ifValue = mvalue
		  , ifRaw = rawvalues
		  , ifBound = isJust bds
		  }

-- |Create a selection box where exactly one entry can be selected.
selectSingle :: Eq a
	=> (a -> String)	-- ^function to display values of type a
	-> Maybe a		-- ^optional preselected value
	-> [a]			-- ^list of all possible values
	-> HTMLField x y (InputField a INVALID)
selectSingle shw mdef opts attrs =
  do inf <- selectMultiple shw (maybeToList mdef) opts (1,1) attrs
     return $
       InputField { ifName = ifName inf
		  , ifToken = ifToken inf
       		  , ifFty = "select"
		  , ifString = ifString inf
		  , ifValue = fmap Prelude.head (ifValue inf)
		  , ifRaw = ifRaw inf
		  , ifBound = ifBound inf
		  }

-- |Selection box for elements of a "Bounded" type. Argument is the optional
-- preselected value.
selectBounded :: (Enum a, Bounded a, Read a, Show a, Eq a) =>
		 Maybe a -> HTMLField x y (InputField a INVALID)
selectBounded def =
  selectSingle show def [minBound..maxBound]
     
-- ======================================================================
-- attributes

-- |Create a 'SIZE' attribute from an 'Int'.
fieldSIZE :: Monad m => Int -> H.WithHTML x m ()
fieldSIZE i = H.attr_SD "size" (show i)

-- |Create a 'MAXLENGTH' attribute from an 'Int'.
fieldMAXLENGTH :: Monad m => Int -> H.WithHTML x m ()
fieldMAXLENGTH i = H.attr_SD "maxlength" (show i)

-- |Create a 'VALUE' attribute from any 'Show'able.
fieldVALUE :: (Monad m, Show a) => a -> H.WithHTML x m ()
fieldVALUE a = H.attr_SD "value" (show a)

-- ======================================================================
-- output routines

-- |Terminates script by sending its argument to the browser.
tell :: CGIOutput a => a -> CGI ()
tell a =
  CGI (\cgistate -> 
  -- appendFile "/tmp/CGIFRAME" ("tell/enter\n") >>
  let frameno = inFrame $ pageInfo cgistate in
  -- appendFile "/tmp/CGIFRAME" ("tell #" ++ show frameno ++ "\n") >>
  if frameno == 0 then
  do putCookies cgistate
     cgiPut (cgiHandle $ cgiInfo cgistate) a
     exitWith ExitSuccess
  else
  do let fname = frameFullPath (outparm cgistate) frameno
     h <- openFile fname WriteMode
     cgiPut h a
     hClose h
     return ((), cgistate))

-- |Terminate script by sending a HTML page constructed by monadic argument. 
htell :: H.WithHTML x IO () -> CGI a
htell hma =
  CGI (\cgistate ->
  do putCookies cgistate
     itell (cgiHandle $ cgiInfo cgistate) hma)
     -- never reached


-- Images

data Image =
  Image { imageSRC :: URL
  	, imageALT :: String
	}

-- |Reference to internal image.
internalImage ::
	   FreeForm		-- ^the raw image
	-> String		-- ^alternative text
	-> H.WithHTML x CGI Image
internalImage ff alttext =
  do baseUrl <- H.lift getUrl
     externalImage (URL (baseUrl ++ '?' : ffName ff)) alttext

-- |Reference to internal image via data URL (small images, only).
dataImage ::
	   FreeForm		-- ^the raw image
	-> String		-- ^alternative text
	-> H.WithHTML x CGI Image
dataImage ff alttext =
  let url = URL (RFC2397.encode (ffContentType ff, ffRawContents ff)) in
  externalImage url alttext

-- |Reference to internal image via javascript URL (does not seem to work).
jsImage ::
	   FreeForm		-- ^the raw image
	-> String		-- ^alternative text
	-> H.WithHTML x CGI Image
jsImage ff alttext =
  let url = URL ("javascript:" ++ jsShow (ffRawContents ff)) in
  externalImage url alttext

-- |Reference to image by URL.
externalImage ::
	   URL			-- ^URL of image
	-> String		-- ^alternative text
	-> H.WithHTML x CGI Image
externalImage url alttext =
     return $ Image { imageSRC = url
     		    , imageALT = alttext
		    }

-- |Create an inline image.
makeImg :: Image -> HTMLField x y ()
makeImg image attrs = 
     H.img_T (do H.attr_SD "src" (unURL $ imageSRC image)
		 H.attr_SD "alt" (imageALT image)
		 H.attr_SD "title" (imageALT image)
		 attrs)

fallbackTranslator mCgigen =
  docTranslator [nothing, question] $
  frameTranslator $
  nextTranslator mCgigen

frameTranslator :: ([String] -> CGI ()) -> [String] -> CGI ()
frameTranslator next (name@('F':'R':'A':'M':'E':':':_):_) =
  tell (ResponseFileReference (frameDir ++ name))
frameTranslator next strs =
  next strs

-- |A /translator/ is a function @[String] -> CGI ()@. It takes the query string
-- of the URL (of type @[String]@) and translates it into a CGI
-- action. @docTranslator docs next@ 
-- takes a list of 'FreeForm' documents and a next translator. It tries to
-- select a document by its 'ffName' and falls through to the
-- @next@ translator if no document matches.
docTranslator :: [FreeForm] -> ([String] -> CGI ()) -> [String] -> CGI ()
docTranslator docs next [name] =
  let f (doc : rest) =
        if name == ffName doc then tell doc else f rest
      f [] = next [name]
  in  f docs
docTranslator docs next strs =
  next strs

-- |Terminates a sequence of translators.
lastTranslator :: [String] -> CGI ()
lastTranslator =
  nextTranslator Nothing

nextTranslator Nothing _ =
  tell (Status 404 "Not Found" Nothing)
nextTranslator (Just cgigen) strs =
  cgigen strs

-- 
reportError :: String -> H.WithHTML x IO () -> CGIState -> IO (a, CGIState)
reportError ttl elems cgistate =
  unCGI (htell message) cgistate
  -- never reached
  where message = 
	  H.standardPage ttl (elems >> backLink H.empty)

-- |Link to previous page in browser's history. Uses JavaScript.
backLink :: Monad m => H.HTMLCons x y m ()
backLink attrs =
  hlink (URL "javascript:back()") (H.text_S "Try again..." >> attrs)

-- |Plain Hyperlink from an URL string.
hlink :: Monad m => URL -> H.HTMLCons x y m ()
hlink url subs = 
  H.a_T (H.attr_SD "href" (unURL url) >> subs)

-- |Hyperlink that creates a named popup window from an URL string.
popuplink :: Monad m => String -> URL -> H.HTMLCons x y m ()
popuplink name url subs = 
  let atv = "javascript:window.open(" ++
            jsShow (unURL url) ++ "," ++
	    jsShow name ++
	    "); void(0);" in
  H.a_T (H.attr_SD "href" atv >> subs)

-- |restart application.
restart :: CGI ()
restart = 
  do myurl <- getUrl
     tell (Location $ URL myurl)

-- |Convenient workhorse. Takes the title of a page and a monadic HTML value for
-- the contents of the page. Wraps the contents in a form so that input fields
-- and buttons may be used inside.
standardQuery :: String -> H.WithHTML x CGI a -> CGI ()
standardQuery ttl elems =
  ask (H.standardPage ttl (makeForm elems))

tellError :: String -> H.Element -> CGI a
tellError str elems = 
  htell message 
  where message = 
	  H.standardPage str (backLink H.empty)
-- 
debug message = unsafe_io $
  do putStrLn "content-type: text/plain"
     putStrLn ""
     putStrLn message
     putStrLn "------------------------------------------------------------"
-- 
prependArticle "" = ""
prependArticle xs@(x:_) =
  if x `elem` "aeiouAEIOU" then "an " ++ xs else "a " ++ xs

-- name for the submission variable and form field
subVar = "WASHsub"
