# -*- coding: utf-8 -*-
"""
w2lapp.searchform.py: different search forms

web2ldap - a web-based LDAP Client,
see http://www.web2ldap.de for details

(c) by Michael Stroeder <michael@stroeder.com>

This module is distributed under the terms of the
GPL (GNU GENERAL PUBLIC LICENSE) Version 2
(see http://www.gnu.org/copyleft/gpl.html)

$Id: searchform.py,v 1.62 2012/04/07 13:51:58 michael Exp $
"""

import types,ldap,pyweblib.forms,w2lapp.core,w2lapp.gui,w2lapp.cnf

searchform_mode_text = {
  'adv':'Advanced',
  'base':'Basic',
  'exp':'Expert',
}

search_options = (
  (r'(%s=*%s*)',u'contains'),
  (r'(%s=*)',u'exists'),
  (r'(!(%s=*%s*))',u"doesn't contain"),
  (r'(%s=%s)',u'is'),
  (r'(!(%s=%s))',u'is not'),
  (r'(%s=%s*)',u'begins with'),
  (r'(%s=*%s)',u'ends with'),
  (r'(%s~=%s)',u'sounds like'),
  (r'(%s>=%s)',u'greater equal than'),
  (r'(%s<=%s)',u'lesser equal than')
)


def SearchForm_exp(sid,outf,command,form,ls,dn,sub_schema,Msg,filterstr=''):
  """Output expert search form"""
  # expert search form for using LDAP filters
  filterstr = form.getInputValue('filterstr',[filterstr])[0]
  result = """
  <fieldset>
    <legend>Search filter parameters</legend>
    <dl>
      <dt>LDAP filter string</dt>
      <dd>
        <input name="filterstr" maxlength="400" size="%d" value="%s">
      </dd>
    </dl>
  </fieldset>
    """ % (
      w2lapp.cnf.GetParam(ls,'searchform_filterstr_size',90),
      form.utf2display(filterstr),
    )
  return result # SearchForm_exp()


def SearchForm_base(sid,outf,command,form,ls,dn,sub_schema,Msg):
  """
  Output basic search form based on a HTML template configured
  with host-specific configuration parameter searchform_template
  """
  searchform_template_name = form.getInputValue('searchform_template',['_'])[0]
  searchform_template_cfg = w2lapp.cnf.GetParam(ls,'searchform_template','')
  assert type(searchform_template_cfg)==types.DictType,TypeError("Host-specific parameter 'searchform_template' has invalid type")
  searchform_template = searchform_template_cfg.get(searchform_template_name,None)
  searchform_template_filename = w2lapp.gui.GetVariantFilename(searchform_template,form.accept_language)
  template_str = open(searchform_template_filename,'r').read()
  return template_str # SearchForm_base()


def SearchForm_adv(sid,outf,command,form,ls,dn,sub_schema,Msg):
  """advanced search form with select lists"""

  search_submit = form.getInputValue('search_submit',[u'Search'])[0]

  old_rows = int(form.getInputValue('search_params',['1'])[0])
  rows = old_rows + {'More':1,'Fewer':-1}.get(search_submit,0)
  # Enforce upper/lower bounds
  rows = min(max(1,rows),w2lapp.cnf.misc.max_searchparams)

  # More and Fewer-Buttons
  if rows<w2lapp.cnf.misc.max_searchparams:
    more_button = '<input type="submit" name="search_submit" value="More">'
  else:
    more_button = ''

  if rows>1:
    fewer_button = '<input type="submit" name="search_submit" value="Fewer">'
  else:
    fewer_button = ''

  search_mode = form.getInputValue('search_mode',[ur'(&%s)'])[0]

  search_mode_select = pyweblib.forms.Select(
    'search_mode',
    u'Search mode',1,
    options=[(ur'(&%s)',u'all'),(ur'(|%s)',u'any')],
    default=search_mode
  )
  search_mode_select.setCharset(form.accept_charset)
  search_attrs = [
    unicode(attr_type)
    for attr_type in w2lapp.cnf.GetParam(ls,'search_attrs',('cn','mail','uid','o'))
  ]

  search_option_list = form.getInputValue('search_option',[])
  search_attr_list = form.getInputValue('search_attr',[])
  search_string_list = form.getInputValue('search_string',[])

  search_attr_options = []
  for attr_type in search_attrs:
    attr_type_se = sub_schema.get_obj(ldap.schema.AttributeType,attr_type)
    if attr_type_se and attr_type_se.desc:
      search_attr_options.append((attr_type,attr_type,attr_type_se.desc))
    else:
      search_attr_options.append((attr_type,attr_type,None))
    search_attr_options.sort()

  search_attr_select = pyweblib.forms.Select(
    'search_attr',u'Search attribute type',
    w2lapp.cnf.misc.max_searchparams,
    options=search_attr_options
  )
  search_attr_select.setCharset(form.accept_charset)

  search_fields_html_list = []

  # Output a row of the search form
  for i in range(rows):
    try:
      search_attr_default = search_attr_list[i]
      search_option_default = search_option_list[i]
      search_string_default = search_string_list[i]
    except IndexError:
      search_attr_default = search_attrs[i % len(search_attrs)]
      search_option_default = u''
      search_string_default = u''
    search_fields_html_list.append('\n'.join((
      search_attr_select.inputHTML(default=search_attr_default).encode('ascii'),
      form.field['search_option'].inputHTML(default=search_option_default),
      form.field['search_string'].inputHTML(default=search_string_default),
      '<br>'
    )))

  # Eigentliches Suchformular ausgeben
  result = """
    <input type="hidden" name="searchform_mode" value="adv">
    %s
    <fieldset>
      <legend>Search filter parameters</legend>
      <p>%s %s search parameters</p>
      Match %s of the following.<br>
      %s
    </fieldset>
    """ % (
      form.hiddenFieldHTML('search_params',unicode(rows),u''),
      more_button,fewer_button,
      search_mode_select.inputHTML(),
      '\n'.join(search_fields_html_list),
      )
  return result # SearchForm_adv()


def w2l_SearchForm(
  sid,outf,command,form,ls,dn,
  searchform_mode='base',
  Msg='',
  filterstr='',
  scope=ldap.SCOPE_SUBTREE,
  search_root=None,
):
  """Output a search form"""

  if Msg:
    msg_html = '<p class="ErrorMessage">%s</p>' % (Msg)
  else:
    msg_html = ''

  sub_schema = ls.retrieveSubSchema(dn,w2lapp.cnf.GetParam(ls,'_schema',None))

  searchform_mode = form.getInputValue('searchform_mode',[searchform_mode])[0]
  search_root = form.getInputValue('search_root',[search_root or ls.getSearchRoot(dn)])[0]
  search_root_field = w2lapp.gui.SearchRootField(form,ls,dn,name='search_root')
  search_root_field.setDefault(search_root)

  ContextMenuList = [
    form.applAnchor(
      'searchform',searchform_mode_text[mode],sid,
      [
        ('dn',dn),
        ('searchform_mode',mode),
        ('search_root',search_root),
        ('filterstr',filterstr),
        ('scope',str(scope)),
      ],
    )
    for mode in searchform_mode_text.keys()
    if mode!=searchform_mode
  ]

  searchform_template_cfg = w2lapp.cnf.GetParam(ls,'searchform_template','')
  if type(searchform_template_cfg)==types.DictType:
    for searchform_template_name in searchform_template_cfg.keys():
      if searchform_template_name!='_':
        ContextMenuList.append(form.applAnchor(
          'searchform',searchform_template_name,sid,
          [
            ('dn',dn),
            ('searchform_mode','base'),
            ('searchform_template',searchform_template_name),
            ('search_root',search_root),
            ('filterstr',filterstr),
            ('scope',str(scope)),
          ],
        ))

  if searchform_mode == 'base':
    # base search form with fixed input fields
    inner_searchform_html = SearchForm_base(sid,outf,command,form,ls,dn,sub_schema,Msg)
  elif searchform_mode == 'exp':
    # expert search form with single filter input field
    inner_searchform_html = SearchForm_exp(sid,outf,command,form,ls,dn,sub_schema,Msg,filterstr)
  elif searchform_mode == 'adv':
    # base search form with fixed input fields
    inner_searchform_html = SearchForm_adv(sid,outf,command,form,ls,dn,sub_schema,Msg)

  searchoptions_template_filename = w2lapp.gui.GetVariantFilename(
    w2lapp.cnf.GetParam(ls,'searchoptions_template',None),
    form.accept_language
  )
  searchoptions_template_str = open(searchoptions_template_filename,'r').read()

  w2lapp.gui.TopSection(
    sid,outf,form,ls,dn,
    '%s Search Form' % searchform_mode_text[searchform_mode],
    w2lapp.gui.MainMenu(sid,form,ls,dn),
    context_menu_list=ContextMenuList
  )

  outf.write("""
    <div id="Input" class="Main">
      %s
      <p>
        <input type="submit" name="search_submit" value="Search">
        <input type="reset" value="Reset">
      </p>
      %s
      %s
  %s
  %s
  </form>
  </div>
  """ % (
    form.beginFormHTML('search',sid,'GET'),
    msg_html,
    inner_searchform_html,
    form.hiddenFieldHTML('dn',dn,u''),
    searchoptions_template_str.format(
      search_root_field.inputHTML(),
      form.field['scope'].inputHTML(default=form.getInputValue('scope',[unicode(scope)])[0]),
      form.field['search_resnumber'].inputHTML(default=unicode(w2lapp.cnf.GetParam(ls,'search_resultsperpage',10))),
      form.field['search_lastmod'].inputHTML(default=form.getInputValue('search_lastmod',[unicode(-1)])[0]),
      form.utf2display(form.getInputValue('search_attrs',[''])[0]),
    ),
  ))

  w2lapp.gui.PrintFooter(outf,form)

