/**************************************************************************
* Generic, native, relational database
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ROWBASE_H
#define ROWBASE_H

typedef unsigned int uint;

namespace RDB
{

/**
* A general base class for all table rows.
*/
class Row
{
public:
    /**
    * Default constructor.
    */
    Row() { }

    /**
    * Destructor.
    */
    virtual ~Row() { }

public:
    /**
    * Return the value of the given key.
    *
    * @param dim Index of the key to return.
    *
    * @return Value of the given key.
    */
    int key( uint dim ) const;
};

/**
* A base class template for table rows.
*
* Inherit this class to create table rows with custom properties and methods.
*
* The @p DIMS template parameter is the number of keys in the row.
*
* Example of declaring a row class and a table for storing rows of this type:
*
* @code
* class MyRow : public RowBase<1>
* {
* public:
*     MyRow( int id, const string& name ) : m_name( name )
*     {
*         setKey( 0, id );
*     }
*
*     int id() const { return key( 0 ); }
*
*     const string& name() const { return m_name; }
*
*     void setName( const string& name ) { m_name = name; }
*
* private:
*     string m_name;
* };
*
* typedef SimpleTable<MyRow> MyTable;
* @endcode
*/
template<uint DIMS>
class RowBase : public Row
{
public:
    /**
    * Default constructor.
    */
    RowBase() { }

    /**
    * Destructor.
    */
    ~RowBase() { }

public:
    int key( uint dim ) const { return m_key[ dim ]; }

protected:
    /**
    * Set the value of the given key.
    *
    * Call this method in the inherited class constructor to initialize the keys of the row.
    *
    * @param dim The index of the key.
    *
    * @param key The value of the key.
    */
    void setKey( uint dim, int key ) { m_key[ dim ] = key; }

public:
    /**
    * Number of keys in the row.
    */
    static const uint Dimensions = DIMS;

private:
    int m_key[ DIMS ];
};

inline int Row::key( uint dim ) const
{
    return ( (const RowBase<1>*)this )->key( dim );
}

}

#endif
