/**************************************************************************
* Generic, native, relational database
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include <stdlib.h>

#include "tabledata.h"

using namespace RDB;

TableData::TableData() :
    m_buckets( 0 )
{
}

TableData::~TableData()
{
    delete[] m_buckets;
}

void TableData::initialize( uint dim, uint size )
{
    m_dim = dim;
    m_size = size;
    m_buckets = new Bucket*[ size ];

    clear();
}

void TableData::clear()
{
    for ( uint i = 0; i < m_size; i++ )
        m_buckets[ i ] = 0;
}

TableData::Bucket* TableData::allocBucket( uint dims )
{
    return (Bucket*)malloc( sizeof( Bucket ) + ( dims - 1 ) * sizeof( Link ) );
}

void TableData::freeBucket( Bucket* bucket )
{
    free( bucket );
}

void TableData::linkBucket( Bucket* bucket )
{
    uint hash = bucket->m_row->key( m_dim ) % m_size;
    bucket->m_links[ m_dim ].m_next = m_buckets[ hash ];
    if ( bucket->m_links[ m_dim ].m_next )
        bucket->m_links[ m_dim ].m_next->m_links[ m_dim ].m_prev = bucket;
    bucket->m_links[ m_dim ].m_prev = 0;
    m_buckets[ hash ] = bucket;
}

void TableData::unlinkBucket( Bucket* bucket )
{
    if ( bucket->m_links[ m_dim ].m_prev ) {
        bucket->m_links[ m_dim ].m_prev->m_links[ m_dim ].m_next = bucket->m_links[ m_dim ].m_next;
    } else {
        uint hash = bucket->m_row->key( m_dim ) % m_size;
        m_buckets[ hash ] = bucket->m_links[ m_dim ].m_next;
    }
    if ( bucket->m_links[ m_dim ].m_next )
        bucket->m_links[ m_dim ].m_next->m_links[ m_dim ].m_prev = bucket->m_links[ m_dim ].m_prev;
}

TableData::Bucket* TableData::findBucket( int key ) const
{
    uint hash = key % m_size;
    Bucket* bucket = m_buckets[ hash ];
    while ( bucket ) {
        if ( bucket->m_row->key( m_dim ) == key )
            return bucket;
        bucket = bucket->m_links[ m_dim ].m_next;
    }
    return 0;
}

TableData::Bucket* TableData::findBucket( int key, uint otherDim, int otherKey ) const
{
    uint hash = key % m_size;
    Bucket* bucket = m_buckets[ hash ];
    while ( bucket ) {
        if ( bucket->m_row->key( m_dim ) == key && bucket->m_row->key( otherDim ) == otherKey )
            return bucket;
        bucket = bucket->m_links[ m_dim ].m_next;
    }
    return 0;
}

Row* TableData::find( int key ) const
{
    Bucket* bucket = findBucket( key );
    if ( !bucket )
        return NULL;
    return bucket->m_row;
}

Row* TableData::find( int key, uint otherDim, int otherKey ) const
{
    Bucket* bucket = findBucket( key, otherDim, otherKey );
    if ( !bucket )
        return NULL;
    return bucket->m_row;
}
