/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef VIEWMANAGER_H
#define VIEWMANAGER_H

#include <QObject>
#include <QMap>

#include "views/viewevent.h"

namespace WebIssues
{

class View;
class ViewerWindow;

/**
* Class managing the views of the application.
*
* This class provides methods for opening views of various types,
* except the project view which is embedded in the main window.
*
* If a view of a given type with the same identifier already exists,
* it is activated and another view is not created.
*
* All views are embedded in their own ViewerWindow.
*
* The view manager also provides a method for notifying views when
* some configuration is changed and for closing all views querying
* them before closing.
*
* The instance of this class is available using the dataManager global variable.
* It is created and owned by the ConnectionManager.
*/
class ViewManager : public QObject
{
    Q_OBJECT
public:
    /**
    * Constructor.
    */
    ViewManager();

    /**
    * Destructor.
    */
    ~ViewManager();

public:
    /**
    * Open the types view.
    */
    void openTypesView();

    /**
    * Open the users view.
    */
    void openUsersView();

    /**
    * Open the folder view for the given folder.
    */
    void openFolderView( int folderId );

    /**
    * Open the folder view for the given folder and filter.
    */
    void openFolderView( int folderId, const QString& filter );

    /**
    * Open the issue details view for the given issue.
    */
    void openIssueView( int issueId, int itemId );

    /**
    * Open the members view for the given project.
    */
    void openMembersView( int projectId );

    /**
    * Open the add comment view for the given issue.
    */
    void openCommentView( int issueId );

    /**
    * Add the view to the view manager.
    */
    void addView( View* view );

    /**
    * Query all views to close.
    *
    * @return @c true if views can be closed, @c false otherwise.
    */
    bool queryCloseViews();

    /**
    * Post a view event to all views of the given class.
    *
    * @param className The name of the class or @c NULL to send the event
    * to all views.
    *
    * @param action The action to execute.
    *
    * @param id The additional identifier.
    */
    void postViewEvent( const char* className, ViewEvent::Action action, int id );

    /**
    * Activate the given view.
    *
    * The view's window is raised and activated.
    *
    * @param view The view to activate.
    */
    void activateView( View* view );

    /**
    * Close the given view.
    *
    * The view's window is destroyed.
    *
    * @param view The view to close.
    */
    void closeView( View* view );

private:
    View* openView( const char* className, int id );
    View* findView( const char* className, int id );
    View* createView( const char* className, int id );

private slots:
    void viewDestroyed( QObject* view );

private:
    QMap<View*, ViewerWindow*> m_views;
};

/**
* Global pointer used to access the ViewManager.
*/
extern ViewManager* viewManager;

}

#endif
