/**************************************************************************
* Simple XML-based UI builder for Qt 4
* Copyright (C) 2007 Michał Męciński
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef XMLUI_BUILDER_H
#define XMLUI_BUILDER_H

#include <QObject>
#include <QList>
#include <QMap>

#include "node.h"

class QMainWindow;
class QMenu;
class QToolBar;
class QAction;

/**
* Simple XML-based UI builder for Qt 4.
*
* This is a set of classes which provide a simple Qt 4 replacement
* for the KXMLGUI classes. It is useful for windows which have to combine
* menus and toolbars from various views, components or plug-ins.
*
* The client is an action container combined with an XML file
* which describes the layout of the menu bar, toolbars and context menus.
*
* The builder can merge the layout from several clients and create
* the menus and toolbars.
*/
namespace XmlUi
{

class Node;
class Client;

/**
* The UI builder.
*
* The builder contains one or more registered clients. It merges
* the menus and toolbars defined by these clients into one UI.
*
* The builder is associated with a main window and automatically
* creates the menu bar and toolbars when clients are added or removed.
* It can also create popup menus on demand.
*
* The window should create a builder and add at least one client
* to create the menu bar and toolbars.
*/
class Builder : public QObject
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param window The main window this builder is associated with.
    */
    Builder( QMainWindow* window );

    /**
    * Destructor.
    */
    ~Builder();

public:
    /**
    * Add a client to the builder.
    *
    * The UI is automatically rebuilt. The UI is merged in the same
    * order the clients are added.
    *
    * @param client The client to add.
    */
    void addClient( Client* client );

    /**
    * Remove a client from the builder.
    *
    * The UI is automatically rebuilt.
    *
    * @param client The client to remove.
    */
    void removeClient( Client* client );

    /**
    * Create a context menu.
    *
    * @param id The identifier of the menu.
    *
    * @return The created menu or @c NULL if it wasn't found.
    */
    QMenu* contextMenu( const QString& id );

    /**
    * Rebuild the entire UI.
    */
    void rebuildAll();

signals:
    /**
    * Emitted after the UI is rebuilt.
    */
    void reset();

private:
    Node mergeNodes( const Node& node1, const Node& node2 );

    bool canMergeNodes( const Node& node1, const Node& node2 );

    Node resolveGroups( const Node& node );

    Node normalizeNode( const Node& node );

    void populateMenuBar( const Node& node );

    QToolBar* createToolBar( const Node& node );

    QMenu* createMenu( const Node& node );

    QAction* findAction( const QString& id );

    QString findTitle( const QString& id );

private:
    QMainWindow* m_mainWindow;

    QList<Client*> m_clients;

    Node m_rootNode;

    QList<QToolBar*> m_toolBars;

    QMap<QString, QMenu*> m_contextMenus;

    QList<QToolBar*> m_oldToolBars;
};

}

#endif
