/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "application.h"

#include <QSettings>
#include <QSessionManager>
#include <QMessageBox>
#include <QDir>
#include <QUrl>
#include <QDesktopServices>
#include <QLocale>
#include <QTranslator>
#include <QLibraryInfo>

#include "connectionmanager.h"
#include "viewmanager.h"
#include "mainwindow.h"
#include "configdata.h"
#include "iconloader.h"

Application::Application( int& argc, char** argv ) : QApplication( argc, argv )
{
    Q_INIT_RESOURCE( icons );
    Q_INIT_RESOURCE( resources );

    // NOTE: update these paths after changing them in the project file

    QString path = applicationDirPath();
#if defined( Q_WS_WIN )
    m_manualPath = QDir::cleanPath( path + "/../doc" );
    m_translationsPath = QDir::cleanPath( path + "/../translations" );
#else
    m_manualPath = QDir::cleanPath( path + "/../share/doc/webissues/doc" );
    m_translationsPath = QDir::cleanPath( path + "/../share/webissues/translations" );
#endif

    loadTranslation( "webissues", false );
    loadTranslation( "qt", true );

#if defined( Q_WS_WIN )
    setStyle( "windowsmodernstyle" );
#endif

    setWindowIcon( IconLoader::icon( "webissues" ) );
    setQuitOnLastWindowClosed( false );

    configData = new ConfigData();

    connectionManager = new ConnectionManager();

    m_mainWindow = new MainWindow();

    QSettings* settings = configData->settings();
    settings->beginGroup( "ShutdownState" );
    bool visible = settings->value( "Visible", true ).toBool();
    bool connected = settings->value( "Connected", false ).toBool();
    settings->endGroup();

    if ( !configData->dockWindow() || configData->showAtStartup() == RestoreAlways ||
        ( configData->showAtStartup() == RestoreAuto && visible ) )
        m_mainWindow->show();

    if ( configData->reconnectAtStartup() == RestoreAlways ||
        ( configData->reconnectAtStartup() == RestoreAuto && connected ) )
        m_mainWindow->reconnect();
}

Application::~Application()
{
    QSettings* settings = configData->settings();
    settings->beginGroup( "ShutdownState" );
    settings->setValue( "Visible", m_mainWindow->isVisible() );
    settings->setValue( "Connected", connectionManager->isConnected() );
    settings->endGroup();

    delete m_mainWindow;
    m_mainWindow = NULL;

    delete connectionManager;
    connectionManager = NULL;

    delete configData;
    configData = NULL;
}

void Application::commitData( QSessionManager& manager )
{
#if !defined( QT_NO_SESSIONMANAGER )
    if ( manager.allowsInteraction() ) {
        if ( viewManager && !viewManager->queryCloseViews() )
            manager.cancel();
    }
#endif
}

void Application::about()
{
    QString link = "<a href=\"http://webissues.mimec.org\">webissues.mimec.org</a>";

    QString message;
    message += "<h3>" + tr( "WebIssues %1" ).arg( version() ) + "</h3>";
    message += "<p>" + tr( "Desktop client for WebIssues, an issue tracking and team collaboration system." ) + "</p>";
    message += "<p>" + trUtf8( "Copyright (C) 2006 Michał Męciński" ) + "<br>" + tr( "Copyright (C) 2007-2008 WebIssues Team" ) + "</p>";
    message += "<p>" + tr( "This program is licensed under the GNU General Public License." ) + "</p>";
    message += "<p>" + tr( "Visit %1 for more information." ).arg( link ) + "</p>";

    QMessageBox::about( activeWindow(), tr( "About WebIssues" ), message );
}

QString Application::version() const
{
    return QString( "0.9.3" );
}

QUrl Application::manualIndex() const
{
    return QUrl::fromLocalFile( m_manualPath + "/index.html" );
}

void Application::manual()
{
    QDesktopServices::openUrl( manualIndex() );
}

bool Application::loadTranslation( const QString& name, bool tryQtDir )
{
    QString fullName = name + "_" + QLocale::system().name();

    QTranslator* translator = new QTranslator( this );

    if ( translator->load( fullName, m_translationsPath ) ) {
        installTranslator( translator );
        return true;
    }

    if ( tryQtDir && translator->load( fullName, QLibraryInfo::location( QLibraryInfo::TranslationsPath ) ) ) {
        installTranslator( translator );
        return true;
    }

    delete translator;
    return false;
}
