/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef STATEROWS_H
#define STATEROWS_H

#include <QDateTime>

/**
* The watch state of a folder or issue.
*/
enum WatchState
{
    NotWatched,
    WatchNormal,
    WatchUpdated,
    WatchNew
};

/**
* A row of a folder state table.
*
* This table is used by the DataManager for tracking the state of folders.
*/
class FolderState
{
public:
    /**
    * Constructor.
    */
    FolderState( int folderId );

    /**
    * Destructor.
    */
    ~FolderState();

public:
    /**
    * Return the identifier of the folder.
    */
    int folderId() const { return m_folderId; }

    /**
    * Set the stamp of the last <tt>LIST ISSUES</tt> command.
    */
    void setListStamp( int stamp ) { m_listStamp = stamp; }

    /**
    * Return the stamp of the last <tt>LIST ISSUES</tt> command.
    *
    * This property is used by the DataManager for the incremental update of folders.
    */
    int listStamp() const { return m_listStamp; }

    /**
    * Set the cached state of the folder.
    */
    void setCached( bool cached ) { m_cached = cached; }

    /**
    * Return the cached state of the folder.
    *
    * This property is used by the DataManager for cashing folders in data files.
    */
    bool cached() const { return m_cached; }

    /**
    * Set the watch state of the folder.
    */
    void setWatchState( WatchState state ) { m_watchState = state; }

    /**
    * Return the watch state of the folder.
    *
    * This property is calculated by the dashboard view to indicate folder watch status.
    */
    WatchState watchState() const { return m_watchState; }

private:
    int m_folderId;
    int m_listStamp;
    bool m_cached;
    WatchState m_watchState;
};

/**
* A row of an issue state table.
*
* This table is used by the DataManager for tracking the state of issues.
*/
class IssueState
{
public:
    /**
    * Constructor.
    */
    IssueState( int issueId );

    /**
    * Destructor.
    */
    ~IssueState();

public:
    /**
    * Return the identifier of the issue.
    */
    int issueId() const { return m_issueId; }

    /**
    * Set the stamp of the last <tt>GET DETAILS</tt> command.
    */
    void setDetailsStamp( int stamp ) { m_detailsStamp = stamp; }

    /**
    * Return the stamp of the last <tt>GET DETAILS</tt> command.
    *
    * This property is used by the DataManager for the incremental update of issues.
    */
    int detailsStamp() const { return m_detailsStamp; }

    /**
    * Set the lock count of the issue details.
    */
    void setLockCount( int count ) { m_lockCount = count; }

    /**
    * Return the lock count of the issue details.
    *
    * This property is used by the DataManager for locking issue details.
    */
    int lockCount() const { return m_lockCount; }

    /**
    * Set the time when the issue was last used.
    */
    void setTimeUsed( const QDateTime& time ) { m_usedTime = time; }

    /**
    * Return the time when the issue was last used.
    *
    * This property is used by the DataManager for removing least recently used issue details
    * from memory.
    */
    const QDateTime& timeUsed() const { return m_usedTime; }

    /**
    * Set the watch stamp of the issue.
    */
    void setWatchStamp( int stamp ) { m_watchStamp = stamp; }

    /**
    * Return the watch stamp of the issue.
    *
    * This property is the stamp when the issue was last viewed, used to track the watch
    * state of the issue.
    */
    int watchStamp() const { return m_watchStamp; }

private:
    int m_issueId;
    int m_detailsStamp;
    int m_lockCount;
    QDateTime m_usedTime;
    int m_watchStamp;
};

#endif
