/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "settingsdialog.h"

#include <QLayout>
#include <QLabel>
#include <QTabWidget>
#include <QCheckBox>
#include <QComboBox>
#include <QSpinBox>
#include <QLineEdit>
#include <QIntValidator>
#include <QDialogButtonBox>
#include <QPushButton>
#include <QMessageBox>

#include "configdata.h"
#include "connectionmanager.h"
#include "iconloader.h"

SettingsDialog::SettingsDialog( QWidget* parent ) : QDialog( parent )
{
    setWindowTitle( tr( "Settings" ) );

    QVBoxLayout* mainLayout = new QVBoxLayout( this );

    QTabWidget* tabWidget = new QTabWidget( this );

    QWidget* generalTab = new QWidget( tabWidget );
    QGridLayout* generalLayout = new QGridLayout( generalTab );

    m_dockCheckBox = new QCheckBox( tr( "&Dock main window in system tray" ), generalTab );
    generalLayout->addWidget( m_dockCheckBox, 0, 0, 1, 2 );

    connect( m_dockCheckBox, SIGNAL( stateChanged( int ) ), this, SLOT( dockChanged() ) );

    QFrame* separator = new QFrame( generalTab );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    generalLayout->addWidget( separator, 1, 0, 1, 2 );

    QLabel* showLabel = new QLabel( tr( "&Show main window at startup:" ), generalTab );
    generalLayout->addWidget( showLabel, 2, 0 );

    m_showComboBox = new QComboBox( generalTab );
    m_showComboBox->addItem( tr( "Never" ) );
    m_showComboBox->addItem( tr( "Automatically" ) );
    m_showComboBox->addItem( tr( "Always" ) );
    generalLayout->addWidget( m_showComboBox, 2, 1 );

    showLabel->setBuddy( m_showComboBox );

    QLabel* reconnectLabel = new QLabel( tr( "&Restore last connection at startup:" ), generalTab );
    generalLayout->addWidget( reconnectLabel, 3, 0 );

    m_reconnectComboBox = new QComboBox( generalTab );
    m_reconnectComboBox->addItem( tr( "Never" ) );
    m_reconnectComboBox->addItem( tr( "Automatically" ) );
    m_reconnectComboBox->addItem( tr( "Always" ) );
    generalLayout->addWidget( m_reconnectComboBox, 3, 1 );

    reconnectLabel->setBuddy( m_reconnectComboBox );

#if defined( Q_WS_WIN )
    QFrame* separator2 = new QFrame( generalTab );
    separator2->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    generalLayout->addWidget( separator2, 4, 0, 1, 2 );

    m_autoStartCheckBox = new QCheckBox( tr( "Start &WebIssues when I start Windows" ), generalTab );
    generalLayout->addWidget( m_autoStartCheckBox, 5, 0, 1, 2 );
#endif

    tabWidget->addTab( generalTab, IconLoader::icon( "configure" ), tr( "General" ) );

    QWidget* connectionTab = new QWidget( tabWidget );
    QGridLayout* connectionLayout = new QGridLayout( connectionTab );

    generalLayout->setRowStretch( generalLayout->rowCount(), 1 );
    generalLayout->setColumnStretch( 0, 1 );

    QLabel* proxyLabel = new QLabel( tr( "&Proxy used for Internet connections:" ), connectionTab );
    connectionLayout->addWidget( proxyLabel, 0, 0, 1, 2 );

    m_proxyCombo = new QComboBox( connectionTab );
    m_proxyCombo->addItem( tr( "No Proxy" ) );
    m_proxyCombo->addItem( tr( "SOCKS5 Proxy" ) );
#if ( QT_VERSION >= 0x040300 )
    m_proxyCombo->addItem( tr( "HTTP Proxy" ) );
#endif
    connectionLayout->addWidget( m_proxyCombo, 0, 2 );

    connect( m_proxyCombo, SIGNAL( currentIndexChanged( int ) ), this, SLOT( proxyChanged() ) );

    proxyLabel->setBuddy( m_proxyCombo );

    QLabel* hostLabel = new QLabel( tr( "&Host name:" ), connectionTab );
    connectionLayout->addWidget( hostLabel, 1, 0 );

    m_proxyHostEdit = new QLineEdit( connectionTab );
    connectionLayout->addWidget( m_proxyHostEdit, 1, 1, 1, 2 );

    hostLabel->setBuddy( m_proxyHostEdit );

    QLabel* portLabel = new QLabel( tr( "Port &number:" ), connectionTab );
    connectionLayout->addWidget( portLabel, 2, 0, 1, 2 );

    m_proxyPortEdit = new QLineEdit( connectionTab );
    QIntValidator* portValidator = new QIntValidator( 1, 65535, m_proxyPortEdit );
    m_proxyPortEdit->setValidator( portValidator );
    connectionLayout->addWidget( m_proxyPortEdit, 2, 2 );

    portLabel->setBuddy( m_proxyPortEdit );

    QLabel* userLabel = new QLabel( tr( "&User:" ), connectionTab );
    connectionLayout->addWidget( userLabel, 3, 0 );

    m_proxyUserEdit = new QLineEdit( connectionTab );
    connectionLayout->addWidget( m_proxyUserEdit, 3, 1, 1, 2 );

    userLabel->setBuddy( m_proxyUserEdit );

    QLabel* passwordLabel = new QLabel( tr( "&Password:" ), connectionTab );
    connectionLayout->addWidget( passwordLabel, 4, 0 );

    m_proxyPasswordEdit = new QLineEdit( connectionTab );
    m_proxyPasswordEdit->setEchoMode( QLineEdit::Password );
    connectionLayout->addWidget( m_proxyPasswordEdit, 4, 1, 1, 2 );

    passwordLabel->setBuddy( m_proxyPasswordEdit );

    connectionLayout->setRowStretch( connectionLayout->rowCount(), 1 );
    connectionLayout->setColumnStretch( 1, 1 );

    tabWidget->addTab( connectionTab, IconLoader::icon( "connection" ), tr( "Connection" ) );

    QWidget* advancedTab = new QWidget( tabWidget );
    QGridLayout* advancedLayout = new QGridLayout( advancedTab );

    QLabel* intervalLabel = new QLabel( tr( "I&nterval of periodic data updates:" ), advancedTab );
    advancedLayout->addWidget( intervalLabel, 0, 0 );

    m_intervalSpin = new QSpinBox( advancedTab );
    m_intervalSpin->setRange( 1, 120 );
    m_intervalSpin->setSuffix( tr( " min" ) );
    advancedLayout->addWidget( m_intervalSpin, 0, 1 );

    intervalLabel->setBuddy( m_intervalSpin );

    QFrame* separator3 = new QFrame( advancedTab );
    separator3->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    advancedLayout->addWidget( separator3, 1, 0, 1, 2 );

    QLabel* actionLabel = new QLabel( tr( "&Default action for attachments:" ), advancedTab );
    advancedLayout->addWidget( actionLabel, 2, 0 );

    m_actionComboBox = new QComboBox( advancedTab );
    m_actionComboBox->addItem( tr( "Ask what to do" ) );
    m_actionComboBox->addItem( tr( "Open" ) );
    m_actionComboBox->addItem( tr( "Save As..." ) );
    advancedLayout->addWidget( m_actionComboBox, 2, 1 );

    actionLabel->setBuddy( m_actionComboBox );

    QLabel* cacheLabel = new QLabel( tr( "&Maximum size of attachments cache:" ), advancedTab );
    advancedLayout->addWidget( cacheLabel, 3, 0 );

    m_cacheSpin = new QSpinBox( advancedTab );
    m_cacheSpin->setRange( 1, 200 );
    m_cacheSpin->setSuffix( tr( " MB" ) );
    advancedLayout->addWidget( m_cacheSpin, 3, 1 );

    cacheLabel->setBuddy( m_cacheSpin );

    QLabel* warnLabel = new QLabel( tr( "&Warn when adding attachments larger than:" ), advancedTab );
    advancedLayout->addWidget( warnLabel, 4, 0 );

    m_warnSpin = new QSpinBox( advancedTab );
    m_warnSpin->setRange( 0, 50 );
    m_warnSpin->setSuffix( tr( " MB" ) );
    m_warnSpin->setSpecialValueText( tr( "Off" ) );
    advancedLayout->addWidget( m_warnSpin, 4, 1 );

    warnLabel->setBuddy( m_warnSpin );

    advancedLayout->setRowStretch( 5, 1 );
    advancedLayout->setColumnStretch( 0, 1 );

    tabWidget->addTab( advancedTab, IconLoader::icon( "gear" ), tr( "Advanced" ) );

    mainLayout->addWidget( tabWidget );

    mainLayout->addSpacing( 10 );

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Apply | QDialogButtonBox::Cancel,
        Qt::Horizontal, this );
    mainLayout->addWidget( buttonBox );

    buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    buttonBox->button( QDialogButtonBox::Apply )->setText( tr( "&Apply" ) );
    buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );
    connect( buttonBox->button( QDialogButtonBox::Apply ), SIGNAL( clicked() ), this, SLOT( apply() ) );

    m_dockCheckBox->setChecked( configData->dockWindow() );
    m_reconnectComboBox->setCurrentIndex( configData->reconnectAtStartup() );

    QNetworkProxy::ProxyType proxyType = configData->networkProxy().type();
    if ( proxyType == QNetworkProxy::Socks5Proxy )
        m_proxyCombo->setCurrentIndex( 1 );
#if ( QT_VERSION >= 0x040300 )
    else if ( proxyType == QNetworkProxy::HttpProxy )
        m_proxyCombo->setCurrentIndex( 2 );
#endif

    if ( proxyType != QNetworkProxy::NoProxy ) {
        m_proxyHostEdit->setText( configData->networkProxy().hostName() );
        if ( configData->networkProxy().port() != 0 )
            m_proxyPortEdit->setText( QString::number( configData->networkProxy().port() ) );
        m_proxyUserEdit->setText( configData->networkProxy().user() );
        m_proxyPasswordEdit->setText( configData->networkProxy().password() );
    }

    m_intervalSpin->setValue( configData->updateInterval() );
    m_actionComboBox->setCurrentIndex( configData->defaultAttachmentAction() );
    m_cacheSpin->setValue( configData->attachmentsCacheSize() );
    m_warnSpin->setValue( configData->warnAttachmentSize() );

#if defined( Q_WS_WIN )
    m_autoStartCheckBox->setChecked( configData->checkAutoStart() );
#endif

    dockChanged();
    proxyChanged();
}

SettingsDialog::~SettingsDialog()
{
}

void SettingsDialog::accept()
{
    if ( apply() )
        QDialog::accept();
}

bool SettingsDialog::apply()
{
    if ( m_proxyCombo->currentIndex() > 0 ) {
        if ( m_proxyHostEdit->text().isEmpty() || !m_proxyPortEdit->hasAcceptableInput() ) {
            QMessageBox::warning( this, tr( "Warning" ), tr( "Missing proxy host name or port number." ) );
            return false;
        }
    }

    configData->setDockWindow( m_dockCheckBox->isChecked() );
    configData->setShowAtStartup( (RestoreOption)m_showComboBox->currentIndex() );
    configData->setReconnectAtStartup( (RestoreOption)m_reconnectComboBox->currentIndex() );

    QNetworkProxy::ProxyType proxyType;
    if ( m_proxyCombo->currentIndex() == 1 )
        proxyType = QNetworkProxy::Socks5Proxy;
#if ( QT_VERSION >= 0x040300 )
    else if ( m_proxyCombo->currentIndex() == 2 )
        proxyType = QNetworkProxy::HttpProxy;
#endif
    else
        proxyType = QNetworkProxy::NoProxy;

    QNetworkProxy proxy( proxyType );

    if ( proxyType != QNetworkProxy::NoProxy ) {
        proxy.setHostName( m_proxyHostEdit->text() );
        proxy.setPort( (quint16)m_proxyPortEdit->text().toInt() );
        proxy.setUser( m_proxyUserEdit->text() );
        proxy.setPassword( m_proxyPasswordEdit->text() );
    }

    if ( connectionManager->isConnected() ) {
        if ( !compareProxies( configData->networkProxy(), proxy ) ) {
            QMessageBox::information( this, tr( "Notice" ),
                tr( "Changes in proxy settings will not be applied until you reconnect to the server." ) );
        }
    }

    configData->setNetworkProxy( proxy );

    configData->setUpdateInterval( m_intervalSpin->value() );
    configData->setDefaultAttachmentAction( (AttachmentAction)m_actionComboBox->currentIndex() );
    configData->setAttachmentsCacheSize( m_cacheSpin->value() );
    configData->setWarnAttachmentSize( m_warnSpin->value() );

#if defined( Q_WS_WIN )
    configData->setAutoStart( m_autoStartCheckBox->isChecked() );
#endif

    configData->saveSettings();

    return true;
}

void SettingsDialog::dockChanged()
{
    if ( m_dockCheckBox->isChecked() ) {
        m_showComboBox->setCurrentIndex( configData->showAtStartup() );
        m_showComboBox->setEnabled( true );
    } else {
        m_showComboBox->setCurrentIndex( 2 );
        m_showComboBox->setEnabled( false );
    }
}

void SettingsDialog::proxyChanged()
{
    bool enabled = ( m_proxyCombo->currentIndex() > 0 );
    m_proxyHostEdit->setEnabled( enabled );
    m_proxyPortEdit->setEnabled( enabled );
    m_proxyUserEdit->setEnabled( enabled );
    m_proxyPasswordEdit->setEnabled( enabled );
}

bool SettingsDialog::compareProxies( const QNetworkProxy& lhs, const QNetworkProxy& rhs )
{
    if ( lhs.type() != rhs.type() )
        return false;

    if ( lhs.type() != QNetworkProxy::NoProxy ) {
        if ( lhs.hostName() != rhs.hostName() || lhs.port() != rhs.port() )
            return false;
        if ( lhs.user() != rhs.user() || lhs.password() != rhs.password() )
            return false;
    }

    return true;
}
