/****************************************************************************
* Simple XML-based UI builder for Qt4
* Copyright (C) 2007-2008 Michał Męciński
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*   1. Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*   2. Redistributions in binary form must reproduce the above copyright
*      notice, this list of conditions and the following disclaimer in the
*      documentation and/or other materials provided with the distribution.
*   3. Neither the name of the copyright holder nor the names of the
*      contributors may be used to endorse or promote products derived from
*      this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
****************************************************************************/

#include "builder.h"

#include <QMainWindow>
#include <QMenu>
#include <QMenuBar>
#include <QToolBar>

#include "node.h"
#include "client.h"

using namespace XmlUi;

Builder::Builder( QMainWindow* window ) : QObject( window ),
    m_mainWindow( window )
{
}

Builder::~Builder()
{
    for ( int i = 0; i < m_clients.count(); i++ )
        m_clients.at( i )->m_builder = NULL;
}

void Builder::addClient( Client* client )
{
    if ( client->m_builder == this )
        return;

    if ( client->m_builder )
        client->m_builder->removeClient( client );

    m_clients.append( client );
    client->m_builder = this;

    rebuildAll();
}

void Builder::removeClient( Client* client )
{
    if ( client->m_builder != this )
        return;

    m_clients.removeAt( m_clients.indexOf( client ) );
    client->m_builder = NULL;

    rebuildAll();
}

Node Builder::mergeNodes( const Node& node1, const Node& node2 )
{
    Node result;

    result.setType( node1.type() );
    result.setId( node1.id() );

    QList<Node> children;

    int mergePos = -1;

    // copy nodes from node1 and find the position of the merge marker
    for ( int i = 0; i < node1.children().count(); i++ ) {
        Node child = node1.children().at( i );

        if ( child.type() == Merge )
            mergePos = children.count();
        else
            children.append( child );
    }

    // no merge marker, append node2 at the end
    if ( mergePos < 0 )
        mergePos = children.count();

    int newMergePos = -1;

    // process nodes from node2
    for ( int i = 0; i < node2.children().count(); i++ ) {
        Node child = node2.children().at( i );

        // remember the position of the merge marker
        if ( child.type() == Merge ) {
            newMergePos = mergePos;
            continue;
        }

        bool merged = false;

        // find a matching node from node1
        for ( int j = 0; j < children.count(); j++ ) {
            if ( canMergeNodes( children.at( j ), child ) ) {
                // replace the original node with the recursively merged node
                children.replace( j, mergeNodes( children.at( j ), child ) );
                merged = true;
                break;
            }
        }

        // no matching node to merge with, insert at merge location
        if ( !merged )
            children.insert( mergePos++, child );
    }

    if ( !children.isEmpty() ) {
        // no merge marker in node2, continue merging from current location
        if ( newMergePos < 0 )
            newMergePos = mergePos;

        // insert new merge marker at appropriate position if needed
        if ( newMergePos < children.count() )
            children.insert( newMergePos, Node( Merge ) );

        result.setChildren( children );
    }

    return result;
}

bool Builder::canMergeNodes( const Node& node1, const Node& node2 )
{
    // separators are never merged
    if ( node1.type() == Separator && node2.type() == Separator )
        return false;

    // the menu bar is always merged
    if ( node1.type() == MenuBar && node2.type() == MenuBar )
        return true;

    // merge if both type and id is the same
    return node1.type() == node2.type() && node1.id() == node2.id();
}

Node Builder::resolveGroups( const Node& node )
{
    Node result;

    result.setType( node.type() );
    result.setId( node.id() );

    for ( int i = 0; i < node.children().count(); i++ ) {
        Node child = node.children().at( i );

        // process the element's groups recursively
        if ( child.children().count() > 0 )
            child = resolveGroups( child );

        if ( child.type() == Group ) {
            // replace the group whith its children
            for ( int k = 0; k < child.children().count(); k++ )
                result.addChild( child.children().at( k ) );
        } else {
            result.addChild( child );
        }
    }

    return result;
}

void Builder::rebuildAll()
{
    if ( m_clients.isEmpty() )
        return;

    m_mainWindow->setUpdatesEnabled( false );

    Node node = m_clients.at( 0 )->m_rootNode;

    for ( int i = 1; i < m_clients.count(); i++ )
        node = mergeNodes( node, m_clients.at( i )->m_rootNode );

    m_rootNode = resolveGroups( node );

    m_mainWindow->menuBar()->clear();

    m_oldToolBars = m_toolBars;
    m_toolBars.clear();

    QList<QMenu*> menus = m_contextMenus.values();
    for ( int i = 0; i < menus.count(); i++ )
        menus.at( i )->deleteLater();
    m_contextMenus.clear();

    for ( int i = 0; i < m_rootNode.children().count(); i++ ) {
        Node child = m_rootNode.children().at( i );
        if ( child.type() == MenuBar ) {
            populateMenuBar( child );
        } else if ( child.type() == ToolBar ) {
            createToolBar( child );
        }
    }

    for ( int i = 0; i < m_oldToolBars.count(); i++ ) {
        QToolBar* toolBar = m_oldToolBars.at( i );
        m_mainWindow->removeToolBar( toolBar );
        toolBar->deleteLater();
    }
    m_oldToolBars.clear();

    m_mainWindow->setUpdatesEnabled( true );

    emit reset();
}

void Builder::populateMenuBar( const Node& node )
{
    for ( int i = 0; i < node.children().count(); i++ ) {
        Node child = node.children().at( i );
        if ( child.type() == Menu ) {
            QMenu* menu = createMenu( child );
            if ( menu )
                m_mainWindow->menuBar()->addMenu( menu );
        }
    }
}

QToolBar* Builder::createToolBar( const Node& node )
{
    QToolBar* toolBar = NULL;
    bool isNew = false;
    bool separator = false;

    for ( int i = 0; i < node.children().count(); i++ ) {
        Node child = node.children().at( i );

        if ( child.type() == Separator && toolBar != NULL ) {
            separator = true;
            continue;
        }

        QAction* action = NULL;

        if ( child.type() == Action )
            action = findAction( child.id() );

        if ( !action || !action->isVisible() )
            continue;

        if ( !toolBar ) {
            for ( int i = 0; i < m_oldToolBars.count(); i++ ) {
                if ( m_oldToolBars.at( i )->objectName() == node.id() ) {
                    toolBar = m_oldToolBars.at( i );
                    m_oldToolBars.removeAt( i );
                    toolBar->setVisible( false );
                    toolBar->clear();
                    break;
                }
            }
            if ( !toolBar ) {
                toolBar = new QToolBar( m_mainWindow );
                toolBar->setObjectName( node.id() );
                toolBar->setWindowTitle( findTitle( node.id() ) );
                isNew = true;
            }
            m_toolBars.append( toolBar );
        }

        if ( separator ) {
            toolBar->addSeparator();
            separator = false;
        }

        toolBar->addAction( action );
    }

    if ( isNew ) {
        emit toolBarCreated( toolBar );
        m_mainWindow->addToolBar( toolBar );
    } else if ( toolBar ) {
        toolBar->setVisible( true );
    }

    return toolBar;
}

QMenu* Builder::createMenu( const Node& node )
{
    QMenu* menu = NULL;
    bool separator = false;

    for ( int i = 0; i < node.children().count(); i++ ) {
        Node child = node.children().at( i );

        if ( child.type() == Separator && menu != NULL ) {
            separator = true;
            continue;
        }

        QAction* action = NULL;

        if ( child.type() == Action )
            action = findAction( child.id() );

        if ( child.type() == Menu ) {
            QMenu* subMenu = createMenu( child );
            if ( subMenu )
                action = subMenu->menuAction();
        }

        if ( !action || !action->isVisible() )
            continue;

        if ( !menu ) {
            QString title = findTitle( node.id() );
            menu = new QMenu( title, m_mainWindow );
        }

        if ( separator ) {
            menu->addSeparator();
            separator = false;
        }

        menu->addAction( action );
    }

    return menu;
}

QMenu* Builder::contextMenu( const QString& id )
{
    QMenu* menu = m_contextMenus.value( id, NULL );
    if ( menu )
        return menu;

    for ( int i = 0; i < m_rootNode.children().count(); i++ ) {
        Node child = m_rootNode.children().at( i );
        if ( child.type() == Menu && child.id() == id ) {
            menu = createMenu( child );
            break;
        }
    }

    if ( menu )
        m_contextMenus.insert( id, menu );

    return menu;
}

QAction* Builder::findAction( const QString& id )
{
    for ( int i = m_clients.count() - 1; i >= 0; i-- ) {
        QAction* action = m_clients.at( i )->action( id );
        if ( action )
            return action;
    }

    return NULL;
}

QString Builder::findTitle( const QString& id )
{
    for ( int i = m_clients.count() - 1; i >= 0; i-- ) {
        QString title = m_clients.at( i )->title( id );
        if ( !title.isEmpty() )
            return title;
    }

    return id;
}
