/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef COMMANDMANAGER_H
#define COMMANDMANAGER_H

#include <QObject>
#include <QUrl>
#include <QList>

class QHttp;
class QHttpResponseHeader;

#if defined( HAVE_OPENSSL )
class QSslSocket;
class QSslCertificate;
class QSslCipher;
class QSslError;
#endif

class AbstractBatch;
class Command;
class Reply;
class ReplyLine;
class CookieJar;

/**
* Class for communicating with the WebIssues server.
*
* This class contains a priority queue of AbstractBatch objects which provide
* commands to execute. Only a single command is executed at a time. If there are
* two or more batches, commands from the batch with the highest priority are
* executed first. Processing commands is asynchronous.
*
* The instance of this class is available using the commandManager global variable.
* It is created and owned by the ConnectionManager.
*/
class CommandManager : public QObject
{
    Q_OBJECT
public:
    /**
    * Type of error that occurred while executing the command.
    */
    enum Error
    {
        /** No error. */
        NoError,
        /** The URL passed to the command manager is invalid. */
        InvalidUrl,
        /** The URL passed to the command manager has an unsupported scheme. */
        UnknownUrlScheme,
        /** Connection to the server failed. The errorCode() method returns a QHttp::Error flag. */
        ConnectionError,
        /** The server returned an HTTP error. The errorCode() method returns the HTTP status. */
        HttpError,
        /** No WebIssues server was found at the given URL. */
        InvalidServer,
        /** The server version is not supported. */
        InvalidVersion,
        /** The server returned an invalid response. */
        InvalidResponse,
        /** The server returned a WebIssues error. The errorCode() method returns the error code. */
        WebIssuesError,
        /** The request was aborted. */
        Aborted
    };

    /**
    * Mode of connection with the server.
    */
    enum Mode
    {
        /** No URL or invalid URL specified. */
        InvalidMode,
        /** URL scheme not supported. */
        UnknownMode,
        /** Regular http connection. */
        HttpMode,
        /** Secure https connection. */
        HttpsMode
    };

public:
    /**
    * Default constructor.
    */
    CommandManager();

    /**
    * Destructor.
    */
    ~CommandManager();

public:
    /**
    * Set the URL of the server to send commands to.
    *
    * @param url URL of the server.
    */
    void setServerUrl( const QUrl& url );

    /**
    * Return the URL of the server.
    */
    const QUrl& serverUrl() const { return m_url; }

    /**
    * Return the mode of connection with the server.
    */
    Mode connectionMode() const { return m_mode; }

    /**
    * Append the batch to the execution queue.
    *
    * The CommandManager takes ownership of the batch object.
    *
    * @param batch Batch to execute.
    */
    void execute( AbstractBatch* batch );

    /**
    * Abort processing the batch and remove it from the execution queue.
    *
    * The batch object is deleted immediately.
    *
    * @param batch Batch to abort.
    */
    void abort( AbstractBatch* batch );

    /**
    * Abort all batches and clear the execution queue.
    */
    void abortAll();

    /**
    * Return the type of the last error.
    */
    Error error() const { return m_error; }

    /**
    * Return the code of the last error.
    */
    int errorCode() const { return m_errorCode; }

    /**
    * Return full formatted error message;
    */
    QString errorMessage( const QString& whatFailed );

    /**
    * Return the server version;
    */
    const QString& serverVersion() const { return m_serverVersion; }    

#if defined( HAVE_OPENSSL )
    /**
    * Return server's chain of certificates.
    */
    QList<QSslCertificate> certificateChain();

    /**
    * Return the session's cryptographic cipher.
    */
    QSslCipher sessionCipher();

    /**
    * Set the list of SSL certificate digests for which errors are ignored.
    */
    void setAcceptedDigests( const QList<QByteArray>& digests );

    /**
    * Ignore SSL errors when called from the sslErrors() signal handler.
    */
    void ignoreSslErrors();
#endif // defined( HAVE_OPENSSL )

signals:
#if defined( HAVE_OPENSSL )
    /**
    * Forwards the sslErrors signal from the QSslSocket.
    */
    void sslErrors( const QList<QSslError>& );
#endif // defined( HAVE_OPENSSL )

private:
    void sendSetHostRequest();
    void sendCommandRequest( Command* command );

    bool handleCommandResponse( const QHttpResponseHeader& response );
    bool handleCommandReply( const Reply& reply );

    bool parseReply( const QString& string, Reply& reply );
    bool validateReply( const Reply& reply );

    QString makeSignature( const ReplyLine& line );

    QString quoteString( const QString& string );
    QString unquoteString( const QString& string );

    void setError( Error error, int code = 0, const QString& string = QString() );

private slots:
    void checkPendingCommand();

    void dataSendProgress( int done, int total );
    void dataReadProgress( int done, int total );

    void readyRead( const QHttpResponseHeader& response );

    void requestFinished( int id, bool error );

#if defined( HAVE_OPENSSL )
    void handleSslErrors( const QList<QSslError>& errors );
#endif // defined( HAVE_OPENSSL )

private:
    QHttp* m_http;

#if defined( HAVE_OPENSSL )
    QSslSocket* m_sslSocket;

    QList<QByteArray> m_acceptedDigests;
#endif

    QList<AbstractBatch*> m_batches;

    Mode m_mode;

    QUrl m_url;

    CookieJar* m_cookieJar;

    int m_currentRequest;
    AbstractBatch* m_currentBatch;
    Command* m_currentCommand;

    QString m_protocolVersion;
    QString m_serverVersion;

    Error m_error;
    int m_errorCode;
    QString m_errorString;
};

/**
* Global pointer used to access the CommandManager.
*/
extern CommandManager* commandManager;

#endif
