/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "connectionmanager.h"

#include <QSettings>
#include <QDir>
#include <QApplication>

#if defined( HAVE_OPENSSL )
#include <QSslCertificate>
#endif

#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/attachmentscache.h"
#include "application.h"
#include "viewmanager.h"
#include "configdata.h"

ConnectionManager* connectionManager = NULL;

ConnectionManager::ConnectionManager() :
    m_info( NULL ),
    m_settings( NULL ),
    m_attachmentsCache( NULL )
{
#if defined( HAVE_OPENSSL )
    loadAcceptedDigests();
#endif
}

ConnectionManager::~ConnectionManager()
{
    closeConnection();
}

void ConnectionManager::createCommandManager()
{
    QNetworkProxy::setApplicationProxy( configData->networkProxy() );

    commandManager = new CommandManager();

#if defined( HAVE_OPENSSL )
    commandManager->setAcceptedDigests( m_acceptedDigests );
#endif
}

void ConnectionManager::openConnection( ConnectionInfo* info )
{
    m_info = info;

    QString path = locateDataFile( "server.conf" );

    m_settings = new QSettings( path, QSettings::IniFormat );

    m_attachmentsCache = new AttachmentsCache();

    dataManager = new DataManager();
    viewManager = new ViewManager();
}

void ConnectionManager::delayedCloseConnection()
{
    QMetaObject::invokeMethod( this, "closeConnection", Qt::QueuedConnection );
}

void ConnectionManager::closeConnection()
{
    if ( commandManager != NULL )
        commandManager->abortAll();

    delete viewManager;
    viewManager = NULL;

    delete dataManager;
    dataManager = NULL;

    if ( commandManager != NULL ) {
        commandManager->deleteLater();
        commandManager = NULL;
    }

    delete m_info;
    m_info = NULL;

    delete m_settings;
    m_settings = NULL;

    delete m_attachmentsCache;
    m_attachmentsCache = NULL;
}

QString ConnectionManager::locateDataFile( const QString& name )
{
    QString path = ( (Application*)qApp )->dataPath() + '/' + m_info->serverUuid() + '/' + name;

    checkAccess( path );

    return path;
}

QString ConnectionManager::locateCacheFile( const QString& name )
{
    QString path = ( (Application*)qApp )->cachePath() + '/' + m_info->serverUuid() + '/' + name;

    checkAccess( path );

    return path;
}

QString ConnectionManager::locateStorageFile( const QString& name )
{
    QString path = ( (Application*)qApp )->storagePath() + '/' + m_info->serverUuid() + '/' + name;

    checkAccess( path );

    return path;
}

bool ConnectionManager::checkAccess( const QString& path )
{
    QFileInfo fileInfo( path );

    if ( fileInfo.exists() )
        return fileInfo.isReadable();

    QDir dir = QDir::root();

    QStringList pathParts = path.split( '/', QString::SkipEmptyParts );

    for ( int i = 0; i < pathParts.size() - 1; i++ ) {
        const QString& part = pathParts.at( i );

        if ( dir.cd( part ) )
            continue;

        if ( dir.mkdir( part ) && dir.cd( part ) )
            continue;

        return false;
    }

    return true;
}

void ConnectionManager::exportSettings( QSettings* settings )
{
    copyAllKeys( settings, m_settings );

    settings->setValue( "ServerUuid", m_info->serverUuid() );
}

bool ConnectionManager::canImportSettings( const QSettings* settings )
{
    return settings->value( "ServerUuid", QString() ).toString() == m_info->serverUuid();
}

void ConnectionManager::importSettings( const QSettings* settings )
{
    m_settings->clear();

    copyAllKeys( m_settings, settings );
}

void ConnectionManager::copyAllKeys( QSettings* dest, const QSettings* src )
{
    QStringList keys = src->allKeys();

    for ( int i = 0; i < keys.count(); i++ ) {
        if ( keys[ i ] == QLatin1String( "ServerUuid" ) )
            continue;
        dest->setValue( keys[ i ], src->value( keys[ i ] ) );
    }
}

#if defined( HAVE_OPENSSL )

void ConnectionManager::loadAcceptedDigests()
{
    m_acceptedDigests.clear();

    QSettings* settings = configData->settings();

    settings->beginGroup( "AcceptedDigests" );

    int count = settings->value( "Count", 0 ).toInt();
    for ( int i = 0; i < count; i++ ) {
        QString hex = settings->value( QString( "Item%1" ).arg( i + 1 ) ).toString();
        m_acceptedDigests.append( QByteArray::fromHex( hex.toAscii() ) );
    }

    settings->endGroup();
}

void ConnectionManager::addAcceptedCertificate( const QSslCertificate& certificate )
{
    QByteArray digest = certificate.digest( QCryptographicHash::Sha1 );

    if ( !m_acceptedDigests.contains( digest ) ) {
        m_acceptedDigests.append( digest );

        QSettings* settings = configData->settings();

        settings->beginGroup( "AcceptedDigests" );

        int count = m_acceptedDigests.count();
        settings->setValue( "Count", count );
        settings->setValue( QString( "Item%1" ).arg( count ), QString( digest.toHex() ) );

        settings->endGroup();
    }
}

#endif // defined( HAVE_OPENSSL )
