/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "issuedialogs.h"

#include <QLayout>
#include <QLabel>
#include <QRadioButton>
#include <QButtonGroup>
#include <QLineEdit>
#include <QFileInfo>

#include "attributes/abstractvalueeditor.h"
#include "attributes/valueeditorfactory.h"
#include "attributes/definitioninfo.h"
#include "data/datamanager.h"
#include "data/issuebatch.h"
#include "models/tablemodelshelper.h"
#include "models/columnconditionsettings.h"
#include "iconloader.h"

IssueDialog::IssueDialog( QWidget* parent ) : CommandDialog( parent )
{
}

IssueDialog::~IssueDialog()
{
}

bool IssueDialog::issueAutoNotify( int issueId )
{
    if ( !dataManager->isIssueNotify( issueId ) ) {
        const IssueRow* issue = dataManager->issues()->find( issueId );
        if ( issue != NULL )
            return folderAutoNotify( issue->folderId() );
    }
    return false;
}

bool IssueDialog::folderAutoNotify( int folderId )
{
    const FolderRow* folder = dataManager->folders()->find( folderId );
    if ( folder != NULL ) {
        ColumnConditionSettings settings;
        settings.openIssueFilters( folder->typeId() );
        return settings.autoNotify( folderId );
    }
    return false;
}

AddIssueDialog::AddIssueDialog( int folderId, QWidget* parent ) : IssueDialog( parent ),
    m_folderId( folderId )
{
    const FolderRow* folder = dataManager->folders()->find( folderId );
    QString name = folder ? folder->name() : QString();

    setWindowTitle( tr( "Add Issue" ) );
    setPrompt( tr( "Create a new issue in folder <b>%1</b>:" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "issue-new", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( tr( "New Issue" ), this );
    m_nameEdit->setMaxLength( 80 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

AddIssueDialog::~AddIssueDialog()
{
}

void AddIssueDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( name.isEmpty() ) {
        showWarning( tr( "Issue name cannot be empty" ) );
        return;
    }

    IssueBatch* batch = new IssueBatch( m_folderId, name );
    batch->setAutoNotify( folderAutoNotify( m_folderId ) );

    executeBatch( batch );
}

bool AddIssueDialog::batchSuccessful( AbstractBatch* batch )
{
    m_issueId = ( (IssueBatch*)batch )->issueId();

    return true;
}

RenameIssueDialog::RenameIssueDialog( int issueId, QWidget* parent ) : IssueDialog( parent ),
    m_issueId( issueId )
{
    const IssueRow* issue = dataManager->issues()->find( issueId );
    m_oldName = issue ? issue->name() : QString();

    setWindowTitle( tr( "Rename Issue" ) );
    setPrompt( tr( "Enter the new name of issue <b>%1</b>:" ).arg( m_oldName ) );
    setPromptPixmap( IconLoader::pixmap( "edit-rename", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( m_oldName, this );
    m_nameEdit->setMaxLength( 80 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

RenameIssueDialog::~RenameIssueDialog()
{
}

void RenameIssueDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( name.isEmpty() ) {
        showWarning( tr( "Issue name cannot be empty" ) );
        return;
    }

    if ( name == m_oldName ) {
        QDialog::accept();
        return;
    }

    IssueBatch* batch = new IssueBatch( m_issueId );
    batch->renameIssue( name );
    batch->setAutoNotify( issueAutoNotify( m_issueId ) );

    executeBatch( batch );
}

SetValueDialog::SetValueDialog( int issueId, int attributeId, QWidget* parent ) : IssueDialog( parent ),
    m_issueId( issueId ),
    m_attributeId( attributeId )
{
    const IssueRow* issue = dataManager->issues()->find( issueId );
    QString issueName = issue ? issue->name() : QString();
    int folderId = issue ? issue->folderId() : 0;

    const AttributeRow* attribute = dataManager->attributes()->find( attributeId );
    QString attrName = attribute ? attribute->name() : QString();
    QString definition = attribute ? attribute->definition() : QString();

    const ValueRow* value = dataManager->values()->find( attributeId, issueId );
    m_oldValue = value ? value->value() : QString();

    const FolderRow* folder = dataManager->folders()->find( folderId );
    int projectId = folder ? folder->projectId() : 0;

    setWindowTitle( tr( "Set Value" ) );
    setPrompt( tr( "Set a new value of attribute <b>%1</b> for issue <b>%2</b>:" ).arg( attrName, issueName ) );
    setPromptPixmap( IconLoader::pixmap( "edit-modify", 22 ) );

    DefinitionInfo info = DefinitionInfo::fromString( definition );

    QVBoxLayout* layout = new QVBoxLayout();

    QRadioButton* button1 = new QRadioButton( tr( "&No Value" ), this );
    layout->addWidget( button1 );

    QHBoxLayout* editorLayout = new QHBoxLayout();
    layout->addLayout( editorLayout );

    QRadioButton* button2 = new QRadioButton( tr( "&Value:" ), this );
    editorLayout->addWidget( button2, 0 );

    m_group = new QButtonGroup( this );
    m_group->addButton( button1, 0 );
    m_group->addButton( button2, 1 );

    m_editor = ValueEditorFactory::createValueEditor( info, projectId, this, this );
    editorLayout->addWidget( m_editor->widget(), 1 );

    if ( m_oldValue.isEmpty() ) {
        m_group->button( 0 )->setChecked( true );
        m_editor->widget()->setEnabled( false );
    } else {
        m_group->button( 1 )->setChecked( true );
        m_editor->setValue( m_oldValue );
    }

    if ( info.required() )
        button1->setEnabled( false );

    connect( button2, SIGNAL( toggled( bool ) ), m_editor->widget(), SLOT( setEnabled( bool ) ) );
    connect( button2, SIGNAL( clicked() ), m_editor->widget(), SLOT( setFocus() ) );

    setContentLayout( layout, true );

    m_editor->widget()->setFocus();
}

SetValueDialog::~SetValueDialog()
{
}

void SetValueDialog::accept()
{
    QString value;

    if ( m_group->checkedId() == 1 ) {
        value = m_editor->value();
        if ( value.isEmpty() ) {
            showWarning( tr( "No value entered" ) );
            return;
        }
    }

    if ( value == m_oldValue ) {
        QDialog::accept();
        return;
    }

    IssueBatch* batch = new IssueBatch( m_issueId );
    batch->setValue( m_attributeId, value );
    batch->setAutoNotify( issueAutoNotify( m_issueId ) );

    executeBatch( batch );
}

AddAttachmentDialog::AddAttachmentDialog( int issueId, const QString& path, const QString& url, QWidget* parent ) :
    IssueDialog( parent ),
    m_issueId( issueId ),
    m_path( path )
{
    QFileInfo info( url );
    m_name = info.fileName();

    const IssueRow* issue = dataManager->issues()->find( issueId );
    QString name = issue ? issue->name() : QString();

    setWindowTitle( tr( "Add Attachment" ) );
    setPrompt( tr( "Add an attachment to issue <b>%1</b>:" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "file-attach", 22 ) );

    QGridLayout* layout = new QGridLayout();

    QLabel* fileLabel = new QLabel( tr( "File:" ), this );
    layout->addWidget( fileLabel, 0, 0 );

    QLineEdit* fileEdit = new QLineEdit( this );
    fileEdit->setReadOnly( true );
    fileEdit->setText( url );
    layout->addWidget( fileEdit, 0, 1 );

    QLabel* descriptionLabel = new QLabel( tr( "&Description:" ), this );
    layout->addWidget( descriptionLabel, 1, 0 );

    m_descriptionEdit = new QLineEdit( QString(), this );
    m_descriptionEdit->setMaxLength( 80 );
    layout->addWidget( m_descriptionEdit, 1, 1 );

    descriptionLabel->setBuddy( m_descriptionEdit );

    setContentLayout( layout, true );

    QFileInfo fileInfo( path );
    m_size = fileInfo.size();

    createProgressPanel( m_size, tr( "Size: %1" ).arg( TableModelsHelper::formatSize( m_size ) ) );

    m_descriptionEdit->setFocus();
}

AddAttachmentDialog::~AddAttachmentDialog()
{
}

void AddAttachmentDialog::accept()
{
    QString description = m_descriptionEdit->text().trimmed();
    m_descriptionEdit->setText( description );

    IssueBatch* batch = new IssueBatch( m_issueId );
    batch->addAttachment( m_name, description, m_path );
    batch->setAutoNotify( issueAutoNotify( m_issueId ) );

    connect( batch, SIGNAL( uploadProgress( int ) ), this, SLOT( uploadProgress( int ) ) );

    uploadProgress( 0 );

    executeBatch( batch );
}

void AddAttachmentDialog::uploadProgress( int done )
{
    QString uploaded = TableModelsHelper::formatSize( done );
    QString total = TableModelsHelper::formatSize( m_size );

    setProgress( done, tr( "Uploaded: %1 of %2" ).arg( uploaded, total ) );
}

bool AddAttachmentDialog::batchSuccessful( AbstractBatch* batch )
{
    int error = ( (IssueBatch*)batch )->fileError();

    if ( error != QFile::NoError ) {
        showWarning( tr( "File could not be read" ) );
        return false;
    }

    return true;
}

GetAttachmentDialog::GetAttachmentDialog( int attachmentId, const QString& path, const QString& url, QWidget* parent ) :
    IssueDialog( parent ),
    m_attachmentId( attachmentId ),
    m_path( path )
{
    const AttachmentRow* attachment = dataManager->attachments()->find( attachmentId );
    QString name = attachment ? attachment->name() : QString();
    QString description = attachment ? attachment->description() : QString();
    m_size = attachment ? attachment->size() : 0;

    setWindowTitle( tr( "Download" ) );
    if ( !url.isEmpty() ) {
        setPrompt( tr( "Download attachment <b>%1</b>:" ).arg( name ) );
        setPromptPixmap( IconLoader::pixmap( "file-save-as", 22 ) );
    } else {
        setPrompt( tr( "Open attachment <b>%1</b>:" ).arg( name ) );
        setPromptPixmap( IconLoader::pixmap( "file-open", 22 ) );
    }

    QGridLayout* layout = new QGridLayout();

    QLabel* fileLabel = new QLabel( tr( "Destination:" ), this );
    layout->addWidget( fileLabel, 0, 0 );

    QLineEdit* fileEdit = new QLineEdit( this );
    fileEdit->setReadOnly( true );
    if ( url.isEmpty() )
        fileEdit->setText( path );
    else
        fileEdit->setText( url );
    layout->addWidget( fileEdit, 0, 1 );

    QLabel* descriptionLabel = new QLabel( tr( "Description:" ), this );
    layout->addWidget( descriptionLabel, 1, 0 );

    QLineEdit* descriptionEdit = new QLineEdit( this );
    descriptionEdit->setReadOnly( true );
    descriptionEdit->setText( description );
    layout->addWidget( descriptionEdit, 1, 1 );

    setContentLayout( layout, true );

    createProgressPanel( m_size, tr( "Size: %1" ).arg( TableModelsHelper::formatSize( m_size ) ) );
}

GetAttachmentDialog::~GetAttachmentDialog()
{
}

void GetAttachmentDialog::download()
{
    accept();
}

void GetAttachmentDialog::accept()
{
    const AttachmentRow* attachment = dataManager->attachments()->find( m_attachmentId );
    int issueId = attachment ? attachment->issueId() : 0;

    IssueBatch* batch = new IssueBatch( issueId );
    batch->getAttachment( m_attachmentId, m_path );

    connect( batch, SIGNAL( downloadProgress( int ) ), this, SLOT( downloadProgress( int ) ) );

    downloadProgress( 0 );

    executeBatch( batch );
}

void GetAttachmentDialog::downloadProgress( int done )
{
    QString downloaded = TableModelsHelper::formatSize( done );
    QString total = TableModelsHelper::formatSize( m_size );

    setProgress( done, tr( "Downloaded: %1 of %2" ).arg( downloaded, total ) );
}

bool GetAttachmentDialog::batchSuccessful( AbstractBatch* batch )
{
    int error = ( (IssueBatch*)batch )->fileError();

    if ( error != QFile::NoError ) {
        showWarning( tr( "File could not be written" ) );
        return false;
    }

    return true;
}
