/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef HTMLREPORTGENERATOR_H
#define HTMLREPORTGENERATOR_H

#include <QObject>

class QDateTime;

class HtmlReportWriter;
class FolderRow;
class IssueRow;

/**
* Class for generating HTML reports.
*
* This class exctracts the information about a folder or an issue, formats it
* into localized text and outputs it to the HtmlReportWriter.
*/
class HtmlReportGenerator : public QObject
{
    Q_OBJECT
public:
    /**
    * Source from which the report is generated.
    */
    enum Source
    {
        /** A folder containing issues. */
        Folder,
        /** A single issue. */
        Issue
    };

    /**
    * Type of the table included in the report.
    */
    enum Table
    {
        /** No table. */
        NoTable,
        /** Table with currently configured columns. */
        CurrentTable,
        /** Table with all available system and user columns. */
        FullTable
    };

public:
    /**
    * Constructor.
    *
    * @param source Type of the source for the report.
    */
    HtmlReportGenerator( Source source );

    /**
    * Destructor.
    */
    ~HtmlReportGenerator();

public:
    /**
    * Configure source in Issue mode.
    *
    * @param issueId Identifier of the issue.
    */
    void setIssue( int issueId );
    /**
    * Configure source in Folder mode.
    *
    * @param folderId Identifier of the folder containing the issues.
    * @param issues Identifiers of issues to be included in the report.
    * @param filterName Name of the custom filter if present.
    */
    void setFolder( int folderId, const QList<int>& issues, const QString& filterName );

    /**
    * Set the type of the table included in the report.
    */
    void setTable( Table table );

    /**
    * Enable or disable summary section for a single issue or multiple issues.
    */
    void setSummaryEnabled( bool enabled );
    /**
    * Enable or disable history section for a single issue.
    */
    void setHistoryEnabled( bool enabled );

    /**
    * Write the report to the given writer.
    */
    void write( HtmlReportWriter* writer );

private:
    void writeFolderInfo( const FolderRow* folder );
    void writeIssueInfo( const IssueRow* issue, const FolderRow* folder );

    void writeIssuesTable( const QList<int>& columns );

    void writeIssueSummary( const IssueRow* issue, const QList<int>& attributes );

    void writeIssueHistory( const IssueRow* issue );

    QString issueHeader( int issueId, const QString& name );
    QString issueCreated( int userId, const QDateTime& date );
    QString issueModified( int userId, const QDateTime& date );

    QString generator();

    QString footer();

private:
    Source m_source;

    int m_folderId;
    QList<int> m_issues;
    QString m_filterName;

    Table m_table;
    bool m_summary;
    bool m_history;

    HtmlReportWriter* m_writer;
};

#endif
