/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This code is partially based on qnetworkcookie.cpp from Qt 4.4,
* licensed unsed the GPL license. 
**************************************************************************/

#include "cookiejar.h"

#include <QList>
#include <QPair>
#include <QDateTime>
#include <QHttp>

class Cookie
{
public:
    Cookie() { }
    ~Cookie() { }

public:
    void setName( const QString& name ) { m_name = name; }
    const QString& name() const { return m_name; }

    void setValue( const QString& value ) { m_value = value; }
    const QString& value() const { return m_value; }

    QString toHeader() const;

    static QList<Cookie> parseHeader( const QString& header );

private:
    QString m_name;
    QString m_value;
};

QString Cookie::toHeader() const
{
    QString header = m_name;
    header += '=';
    if ( m_value.contains( ';' ) || m_value.contains( ',' ) || m_value.contains( ' ' ) || m_value.contains( '"' ) ) {
        header += '"';
        QString escaped = m_value;
        escaped.replace( '"', "\\\"" );
        header += escaped;
        header += '"';
    } else {
        header += m_value;
    }
    return header;
}

static bool isLWS( QChar ch )
{
    return ch == QLatin1Char( ' ' ) || ch == QLatin1Char( '\t' ) || ch == QLatin1Char( '\r' ) || ch == QLatin1Char( '\n' );
}

static int nextNonWhitespace( const QString& text, int position )
{
    while ( position < text.length() ) {
        QChar ch = text.at( position );
        if ( !isLWS( text.at( position ) ) )
            break;
        position++;
    }
    return position;
}

static QPair<QString, QString> nextField( const QString& text, int& position )
{
    int length = text.length();
    position = nextNonWhitespace( text, position );

    int i = position;
    while ( i < length ) {
        QChar ch = text.at( i );
        if ( ch == QLatin1Char( ';' ) || ch == QLatin1Char( ',' ) || ch == QLatin1Char( '=' ) )
            break;
        i++;
    }

    QString first = text.mid( position, i - position ).trimmed();
    position = i;

    if ( first.isEmpty() )
        return qMakePair( QString(), QString() );

    if ( i == length || text.at( i ) != QLatin1Char( '=' ) )
        return qMakePair( first, QString() );

    QString second;
    second.reserve( 32 );

    i = nextNonWhitespace( text, position + 1 );
    if ( i < length && text.at( i ) == QLatin1Char( '"' ) ) {
        i++;
        while ( i < length ) {
            QChar ch = text.at( i );
            if ( ch == QLatin1Char( '"' ) )
                break;
            if ( ch == QLatin1Char( '\\' ) ) {
                i++;
                if ( i >= length )
                    return qMakePair( QString(), QString() );
                ch = text.at( i );
            }
            second += ch;
            i++;
        }
        while ( i < length ) {
            QChar ch = text.at( i );
            if ( ch == QLatin1Char( ',' ) || ch == QLatin1Char( ';' ) )
                break;
            i++;
        }
        position = i;
    } else {
        position = i;
        while ( i < length ) {
            QChar ch = text.at( i );
            if ( ch == QLatin1Char( ',' ) || ch == QLatin1Char( ';' ) || isLWS( ch ) )
                break;
            i++;
        }
        second = text.mid( position, i - position ).trimmed();
        position = i;
    }

    if ( second.isNull() )
        second.resize( 0 );

    return qMakePair( first, second );
}

QList<Cookie> Cookie::parseHeader( const QString& header )
{
    QList<Cookie> result;

    int position = 0;
    int length = header.length();

    while ( position < length ) {
        Cookie cookie;

        QPair<QString, QString> field = nextField( header, position );
        if ( field.first.isEmpty() || field.second.isNull() )
            break;

        cookie.setName( field.first );
        cookie.setValue( field.second );

        position = nextNonWhitespace( header, position );

        while ( position < length ) {
            QChar ch = header.at( position++ );
            if ( ch == QLatin1Char( ',' ) )
                break;
            if ( ch != QLatin1Char( ';' ) )
                continue;

            field = nextField( header, position );

            if ( field.first.toLower() == QLatin1String( "expires" ) ) {
                if ( position < length && header.at( position ) == ',' ) {
                    position++;
                    while ( position < length ) {
                        if ( header.at( position ) == QLatin1Char( ';' ) )
                            break;
                        position++;
                    }
                }
            }

            position = nextNonWhitespace( header, position );
        }

        result.append( cookie );
    }

    return result;
}

CookieJar::CookieJar()
{
}

CookieJar::~CookieJar()
{
}

void CookieJar::extractCookies( const QHttpResponseHeader& response )
{
    QStringList headers = response.allValues( "Set-Cookie" );

    for ( int i = 0; i < headers.count(); i++ ) {
        QList<Cookie> cookies = Cookie::parseHeader( headers.at( i ) );
        for ( int j = 0; j < cookies.count(); j++ )
            m_cookies.insert( cookies.at( j ).name(), cookies.at( j ) );
    }
}

void CookieJar::insertCookies( QHttpRequestHeader& request )
{
    QString header;

    for ( QMap<QString, Cookie>::const_iterator it = m_cookies.constBegin(); it != m_cookies.constEnd(); ++it ) {
        if ( !header.isEmpty() )
            header += QLatin1String( "; " );
        header += it.value().toHeader();
    }

    if ( !header.isEmpty() )
        request.addValue( "Cookie", header );
}

void CookieJar::clear()
{
    m_cookies.clear();
}
