/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QMainWindow>
#include <QSystemTrayIcon>

#include "configdata.h"
#include "xmlui/client.h"
#include "data/bookmark.h"

#if defined( HAVE_OPENSSL )
class QSslError;
#endif

class StartPage;
class View;
class ProjectsView;
class DashboardView;
class FolderView;
class IssueView;
class LoginBatch;
class StatusBar;
class PaneWidget;

/**
* The main window of the application.
*
* When connected to the server, the window embeds the projects view,
* adds its actions to the menu and toolbar and handles its signals.
*
* If the tray icon is disabled, closing this window quits the application.
* If it's enabled, the window is hidden.
*/
class MainWindow : public QMainWindow, public XmlUi::Client
{
    Q_OBJECT
public:
    /**
    * Constructor.
    */
    MainWindow();

    /**
    * Destructor.
    */
    ~MainWindow();

public:
    /**
    * Restore connection to the last server.
    */
    void reconnect();

public: // overrides
    QMenu* createPopupMenu();

    StatusBar* statusBar() { return (StatusBar*)QMainWindow::statusBar(); }

protected: // overrides
    void closeEvent( QCloseEvent* e );

    bool eventFilter( QObject* object, QEvent* e );

private slots:
    void quit();

    void newConnection();
    void closeConnection();

    void importSettings();
    void exportSettings();

    void connectionInfo();

    void showUsers();
    void showTypes();

    void gotoItem();
    void gotoIssue( int issueId, int itemId );
    void gotoItem( int itemId );

    void changePassword();
    void userPreferences();

    void configure();
    void settingsChanged();

    void openConnection( const Bookmark& bookmark );
    void editConnection( const Bookmark& bookmark );

    void loginCompleted( bool successful );
    void abortLogin();

    void trayIconActivated( QSystemTrayIcon::ActivationReason reason );

    void captionChanged( const QString& caption );

    void toolBarCreated( QToolBar* toolBar );
    void builderReset();

#if defined( HAVE_OPENSSL )
    void sslErrors( const QList<QSslError>& errors );
#endif

    void folderEnabledChanged( bool enabled );
    void issueEnabledChanged( bool enabled );

    void selectedFolderChanged( int folderId );
    void selectedWatchChanged( int folderId, const QString& filter );
    void selectedFilterChanged( const QString& filter );
    void selectedIssueChanged( int issueId );

    void updateSelection();

private:
    void showLoginDialog( const Bookmark& bookmark, bool editable );

    void initializeView();
    void closeView( bool startPage );

    void showStartPage();

    void updateActions( bool connected );

    void setActiveView( View* view );

private:
    XmlUi::Builder* m_builder;

    StartPage* m_startPage;

    ProjectsView* m_view;
    DashboardView* m_dashboard;
    FolderView* m_folderView;
    IssueView* m_issueView;

    View* m_activeView;

    PaneWidget* m_folderPane;
    PaneWidget* m_issuePane;

    WindowLayout m_layout;

    QSystemTrayIcon* m_trayIcon;

    LoginBatch* m_batch;

    Bookmark m_bookmark;

    QTimer* m_selectionTimer;

    int m_selectedFolderId;
    QString m_selectedFilter;
    int m_selectedIssueId;

    bool m_supressFilter;
};

#endif
