/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ISSUEHISTORYWRITER_H
#define ISSUEHISTORYWRITER_H

#include <QStringList>

/**
* Class for writing the issue history in HTML format.
*
* This class is responsible for adding HTML markup and writing the issue
* history into an HTML table.
*/
class IssueHistoryWriter
{
public:
    /**
    * Format of the issue history.
    */
    enum Format
    {
        /** History displayed in the issue history view. */
        HistoryView,
        /** History embedded into a report. */
        Report
    };

public:
    /**
    * Default constuctor.
    */
    IssueHistoryWriter( Format format );

    /**
    * Destructor.
    */
    ~IssueHistoryWriter();

public:
    /**
    * Begin the document.
    *
    * This method writes the document header.
    */
    void begin();

    /**
    * End the document.
    *
    * This method writes the document footer.
    */
    void end();

    /**
    * Write a block with a header and a list of changes.
    *
    * @param header Header of the block in HTML format.
    *
    * @param items List of items in HTML format.
    */
    void writeChanges( const QString& header, const QStringList& items );

    /**
    * Write a block with a header, comment text and anchor.
    *
    * The name of the anchor is <tt>id{commentId}</tt>.
    *
    * @param header Header of the block in HTML format.
    *
    * @param text Content of the block in plain text format.
    *
    * @param commentId Identifier of the block's anchor.
    */
    void writeComment( const QString& header, const QString& text, int commentId );

    /**
    * Write a block with a header, attachment item and anchor.
    *
    * The name of the anchor is <tt>id{attachmentId}</tt>.
    *
    * @param header Header of the block in HTML format.
    *
    * @param item The attachment item in HTML format.
    *
    * @param attachmentId Identifier of the block's anchor.
    */
    void writeAttachment( const QString& header, const QString& item, int attachmentId );

    /**
    * Add HTML markup to an attribute name or user name.
    *
    * Names are displayed with bold font.
    */
    QString name( const QString& text );

    /**
    * Add HTML markup to a date.
    *
    * Dates are displayed with bold font.
    */
    QString date( const QString& text );

    /**
    * Add HTML markup to an issue name or attribute value.
    *
    * URLs and issue numbers are converted to links.
    */
    QString value( const QString& text );

    /**
    * Add HTML markup to an attachment description.
    *
    * URLs and issue numbers are converted to links.
    */
    QString description( const QString& text );

    /**
    * Add HTML markup to a comment.
    *
    * URLs and issue numbers are converted to links.
    */
    QString comment( const QString& text );

    /**
    * Create a link to an attachment.
    *
    * The URL of the link is <tt>attachment://{attachmentId}</tt>.
    *
    * @param attachmentId Identifier of the attachmet.
    *
    * @param name The name of the attachment.
    */
    QString attachment( int attachmentId, const QString& name );

    /**
    * Return the generated HTML table.
    */
    QString result() const;

private:
    Format m_format;

    QStringList m_lines;
};

#endif
