/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <time.h>
#include <pwd.h>    /* for getpwuid() */
#include <unistd.h> /* for getuid() */

#include "html.h"
#include "list1.h"
#include "defs.h"

#include "conf.h"
#include "resolv.h" // for service RV@@8
#include "preresolv.h"
#include "timediff.h"

#include "wfipaddr.h"
#include "wfprotocol.h"

#define TIMESIZE 40


wf_outmodule_html::wf_outmodule_html() :
  wf_outmodule(),
  dns(NULL),
  whois(NULL)
{
  conf = new wf_outmodule_html_conf();
}

wf_outmodule_html::~wf_outmodule_html() {
  if (dns != NULL)
    delete dns;
  if (whois != NULL)
    delete whois;
  delete conf;
}


void
wf_outmodule_html::output_header(ostream& os) const {
  char nowstr[TIMESIZE];
  time_t now = time(NULL);
  strftime(nowstr, TIMESIZE, "%a %b %d %H:%M:%S %Z %Y", localtime(&now));

  os << "<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n\
<html>\n<head>\n<title>" << conf->config["title"].tostr() << " - " <<
    nowstr << "</title>\n\
<meta http-equiv=\"pragma\" content=\"no-cache\">\n\
<meta http-equiv=\"expires\" content=\"0\">\n\
</head>\n";

  os << "<body text=\"" << conf->config["textcol"].tostr() <<
    "\" bgcolor=\"" << conf->config["bgcol"].tostr() <<
    "\" link=\"" << conf->config["textcol"].tostr() <<
    "\" alink=\"" << conf->config["textcol"].tostr() <<
    "\" vlink=\"" << conf->config["textcol"].tostr() << "\">\n";
  os << "<font face=\"Arial, Helvetica\"><div align=\"center\">\n<h1>" <<
    conf->config["title"].tostr() << "</h1>\n";

  // if (conf->config["header"].tobool()) { RV@@2
    struct passwd* gen_user;
    gen_user = getpwuid(getuid());
    if (gen_user != NULL)
      ostream_printf(os, _("Generated on %s by %s."), nowstr,
		     gen_user->pw_gecos[0] != '\0' ?
		     gen_user->pw_gecos : gen_user->pw_name);
    else
      ostream_printf(os, _("Generated on %s by an unknown user."), nowstr);
  // } RV@@2
  os << "<br>\n";
}

void
wf_outmodule_html::output_table(ostream& os) const {
  os << ("<br><br>\n<table border=\"0\">\n");
  os << "<tr bgcolor=\"" << conf->config["rowcol1"].tostr() <<
    "\" align=\"center\"><td>#</td>";

  if (conf->config["stimes"].tobool())
    os << "<td>" << _("start") << "</td>";

  if (conf->config["etimes"].tobool())
    os << "<td>" << _("end") << "</td>";

  if (conf->config["duration"].tobool())
    os << "<td>" << _("interval") << "</td>";

  if (conf->config["loghost"].tobool())
    os << "<td>" << _("loghost") << "</td>";

  if (conf->config["chain"].tobool())
    os << "<td>" << _("chain") << "</td>";

  if (conf->config["branch"].tobool())
    os << "<td>" << _("target") << "</td>";

  if (conf->config["interfaces"].tobool())
    os << "<td>" << _("input interface") << "</td>";
  if (conf->config["src_mac"].tobool()) {
    os << "<td>" << _("source MAC address") << "</td>";
    if (conf->config["mac_vendor"].tobool())
      os << "<td>" << _("MAC vendor") << "</td>";
  }

  if (conf->config["interfaces"].tobool())
    os << "<td>" << _("output interface") << "</td>";
  if (conf->config["dst_mac"].tobool()) {
    os << "<td>" << _("destination MAC address") << "</td>";
    if (conf->config["mac_vendor"].tobool())
      os << "<td>" << _("MAC vendor") << "</td>";
  }

  if (conf->config["proto"].tobool())
    os << "<td>" << _("proto") << "</td>";

  if (conf->config["datalen"].tobool())
    os << "<td>" << _("bytes") << "</td>";

  if (conf->config["src_ip"].tobool()) {
    os << "<td>" << _("source") << "</td>";
    if (conf->config["resolve"].toint())
      os << "<td>" << _("hostname") << "</td>";
    if (conf->config["whois_lookup"].toint())
      os << "<td>" << _("source whois info") << "</td>";
  }

  if (conf->config["src_port"].tobool()) {
    os << "<td>" << _("port") << "</td>";
    if (conf->config["sresolve"].tobool())
      os << "<td>" << _("service") << "</td>";
  }

  if (conf->config["dst_ip"].tobool()) {
    os << "<td>" << _("destination") << "</td>";
    if (conf->config["resolve"].toint())
      os << "<td>" << _("hostname") << "</td>";
    if (conf->config["whois_lookup"].toint())
      os << "<td>" << _("dest whois info") << "</td>";
  }

  if (conf->config["dst_port"].tobool()) {
    os << "<td>" << _("port") << "</td>";
    if (conf->config["sresolve"].tobool())
      os << "<td>" << _("service") << "</td>";
  }

  if (conf->config["tcpflags"].tobool())
    os << "<td>" << _("tcpflags") << "</td>";

  os << "</tr>\n";
}

void
wf_outmodule_html::output_html(const wf_logentry* entry, unsigned int color,
			       ostream& os) const {
  char time[TIMESIZE];

  os << "<tr bgcolor=\"";
  if (color)
    os << conf->config["rowcol2"].tostr();
  else
    os << conf->config["rowcol1"].tostr();
  os << "\" align=\"center\"><td>";

  os << entry->count;

  if (conf->config["stimes"].tobool()) {
    os << "</td><td>";
    strftime(time, TIMESIZE, "%b %d %H:%M:%S", localtime(&entry->start_time));
    os << time;
  }

  if (conf->config["etimes"].tobool()) {
    os << "</td><td>";
    if (entry->end_time != 0) {
      strftime(time, TIMESIZE, "%b %d %H:%M:%S", localtime(&entry->end_time));
      os << time;
    }
    else
      os << '-';
  }
  
  if (conf->config["duration"].tobool())
    os << "</td><td>" << timediff_tostr(entry->start_time, entry->end_time);

  if (conf->config["loghost"].tobool())
    os << "</td><td>" << entry->hostname;

  if (conf->config["chain"].tobool())
    os << "</td><td>" << entry->chainlabel;

  if (conf->config["branch"].tobool())
    os << "</td><td>" << entry->branchname;

  if (conf->config["interfaces"].tobool()) {
    os << "</td><td>";
    if (entry->input_iface.empty())
      os << '-';
    else
      os << entry->input_iface;
  }

  if (conf->config["src_mac"].tobool()) {
    os << "</td><td>";
    if (entry->smacaddr.isdefined())
      os << entry->smacaddr;
    else
      os << '-';
    if (conf->config["mac_vendor"].tobool()) {
      os << "</td><td>";
      string vendor = entry->smacaddr.vendor();
      if (vendor.empty())
	os << '-';
      else
	os << vendor;
    }
  }

  if (conf->config["interfaces"].tobool()) {
    os << "</td><td>";
    if (entry->output_iface.empty())
      os << '-';
    else
      os << entry->output_iface;
  }

  if (conf->config["dst_mac"].tobool()) {
    os << "</td><td>";
    if (entry->dmacaddr.isdefined())
      os << entry->dmacaddr;
    else
      os << '-';
    if (conf->config["mac_vendor"].tobool()) {
      os << "</td><td>";
      string vendor = entry->dmacaddr.vendor();
      if (vendor.empty())
	os << '-';
      else
	os << vendor;
    }
  }

  string proto;
  {
    wf_protocol protocol = wf_protocol(entry->protocol);
    proto = protocol.tostr();
  }

  if (conf->config["proto"].tobool())
    os << "</td><td>" << proto;

  if (conf->config["datalen"].tobool())
    os << "</td><td>" << entry->datalen;

  if (conf->config["src_ip"].tobool()) {
    bool src_resolved = false;

    os << "</td><td>" << entry->sipaddr;

    if (conf->config["resolve"].toint() != RESOLV_NO) {
      os << "</td><td>";
#if 0
      os << resolv_hostname(entry->sipaddr);
#else
      wf_dns_entry* dnsentry = dns->resolv(entry->sipaddr);
      if (dnsentry != NULL && dnsentry->name.empty() == false) {
	src_resolved = true;
	os << dnsentry->name;
      }
      else
	os << '-';
#endif
    }

    if (conf->config["whois_lookup"].toint() != WHOIS_NO) {
      os << "</td><td>";
      if (conf->config["whois_lookup"].toint() == WHOIS_YES ||
	  (conf->config["whois_lookup"].toint() == WHOIS_IFNODNS &&
	   src_resolved == false)) {
	wf_whois_entry* we = whois->whois(entry->sipaddr);
	if (we != NULL)
	  os << *we;
	else
	  os << '-';
      }
    }
  }

  if (conf->config["src_port"].tobool()) {
    os << "</td><td>";
    switch (entry->protocol) {
    case IPPROTO_TCP:
    case IPPROTO_UDP:
      os << entry->sport;
      break;
    case IPPROTO_ICMP:
      os << "icmp type " << entry->sport;
      break;
    default:
      os << '-';
    }

    if (conf->config["sresolve"].tobool()) {
      os << "</td><td>";
      switch (entry->protocol) {
      case IPPROTO_TCP:
      case IPPROTO_UDP:
	{
	  string service = resolv_service(entry->sport, proto);
	  if (service.empty())
	    os << '-';
	  else
	    os << service;
	}
	break;
      case IPPROTO_ICMP:
	os << "icmp " <<
	  wf_protocol_icmp_type_tostr(entry->sport, entry->dport);
	break;
      default:
	os << '-';
      }
    }
  }
  
  if (conf->config["dst_ip"].tobool()) {
    bool dst_resolved = false;

    os << "</td><td>" << entry->dipaddr;

    if (conf->config["resolve"].toint() != RESOLV_NO) {
      os << "</td><td>";
#if 0
      os << resolv_hostname(entry->dipaddr);
#else
      wf_dns_entry* dnsentry = dns->resolv(entry->dipaddr);
      if (dnsentry != NULL && dnsentry->name.empty() == false) {
	dst_resolved = true;
	os << dnsentry->name;
      }
      else
	os << '-';
#endif
    }

    if (conf->config["whois_lookup"].toint()) {
      os << "</td><td>";
      if (conf->config["whois_lookup"].toint() == WHOIS_YES ||
	  (conf->config["whois_lookup"].toint() == WHOIS_IFNODNS &&
	   dst_resolved == false)) {
	wf_whois_entry* we = whois->whois(entry->dipaddr);
	if (we != NULL)
	  os << *we;
	else
	  os << '-';
      }
    }
  }

  if (conf->config["dst_port"].tobool()) {
    os << "</td><td>";
    switch (entry->protocol) {
    case IPPROTO_TCP:
    case IPPROTO_UDP:
      os << entry->dport;
      break;
    case IPPROTO_ICMP:
      if (wf_protocol_icmp_type_hascode(entry->sport))
	os << "icmp code " << entry->dport;
      else
	os << '-';
      break;
    default:
      os << '-';
    }
    
    if (conf->config["sresolve"].tobool()) {
      os << "</td><td>";
      switch (entry->protocol) {
      case IPPROTO_TCP:
      case IPPROTO_UDP:
	{
	  string service = resolv_service(entry->dport, proto);
	  if (service.empty())
	    os << '-';
	  else
	    os << service;
	}
	break;
      default:
	os << '-';
      }
    }
  }

  if (conf->config["tcpflags"].tobool())
    os << "</td><td>" << entry->tcpflags_tostr_mini();
  
  os << "</td></tr>" << endl;
}
  
void
wf_outmodule_html::output_footer(ostream& os) const {
  os << "</table></div><br>\n";
  //  if (conf->config["header"].tobool()) { RV@@2
    os << "<small>";
    ostream_printf(os, _("Generated by <a href=\"%s\">wflogs</a> %s"),
		   WALLFIRE_URL, VERSION);
    os << " &copy; Herve Eychenne\n</small>\n";
  //  } RV@@2
  os << "</font></body></html>" << endl;
}

bool
wf_outmodule_html::print(const wf_logentry* entry, ostream& os) {
  return false;
}

bool
wf_outmodule_html::print(const wf_logentries& logentries, ostream& os) {
  unsigned char verbose = conf->config["verbose"].toint();
  int lines = conf->config["lines"].toint();

  wf_logentries* newlogentries;
  if (lines) { /* copy n first logentries */
    newlogentries = new wf_logentries();
    if (newlogentries == NULL)
      return false;
    list1_firstncopy(newlogentries->elems, logentries.elems, lines);
  }
  else /* do not copy logentries */
    newlogentries = (wf_logentries*)&logentries;

  if (conf->config["whois_lookup"].toint() != WHOIS_NO) {
    if (whois == NULL) {
      whois = new wf_whois(verbose);
      if (whois == NULL) {
	if (lines)
	  delete newlogentries;
	return false;
      }
    }
    if (whois->isconnected() == false && whois->connect() == false) {
      if (lines)
	delete newlogentries;
      return false;
    }
  }
  
  if (conf->config["resolve"].toint() == RESOLV_YES) {
    if (dns == NULL) {
      dns = new wf_dns(verbose);
      if (dns == NULL) {
	if (lines)
	  delete newlogentries;
	return false;
      }
    }
    preresolv(*newlogentries, dns,
	      conf->config["src_ip"].tobool(), conf->config["dst_ip"].tobool(),
	      verbose);
  }

  output_header(os);
  output_table(os);

  unsigned char alternate_color = 0;
  list<wf_logentry*>::const_iterator first = newlogentries->elems.begin(),
    last = newlogentries->elems.end();
  for (; first != last; ++first) {
    alternate_color = 1 - alternate_color;
    output_html(*first, alternate_color, os);
  }

  output_footer(os);

  if (conf->config["whois_lookup"].toint() != WHOIS_NO)
    whois->close();
  if (lines)
    delete newlogentries;

  return true;
}

extern "C" wf_outmodule*
wf_outmodule_html_init() {
  return new wf_outmodule_html();
}
