/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h> // for snprintf (suppress) RV@@8
#include <unistd.h>
#include <signal.h>
#include <stdlib.h> // for malloc (suppress) RV@@8
#include <netdb.h>
#include <sys/socket.h>

#include "wfnetwork.h"
#include "whois.h"
#include "defs.h"

#define WHOISCMDLEN 32
#define WHOISDESCLEN 64
#define WHOISROUTELEN 20


wf_whois_entry::wf_whois_entry() :
  ip_route(),
  ip_descr(),
  as_number(0),
  as_descr()
{}


wf_whois::wf_whois() :
  cache(),
  verbose(0),
  _host(WHOIS_SERVER),
  _port(WHOIS_PORT),
  _sock(-1)
{}

wf_whois::wf_whois(unsigned char verb) :
  cache(),
  verbose(verb),
  _host(WHOIS_SERVER),
  _port(WHOIS_PORT),
  _sock(-1)
{}


static int
whois_get_type(int socket, char* type) {
  int cnt = 0;
  char buffer[WHOISCMDLEN];
  signed char c;

  do {
    if (read(socket, &c, 1) == 0)
      break;
    buffer[cnt++] = c;
  } while (c != '\n' && c != EOF && cnt < WHOISCMDLEN);
  buffer[cnt] = '\0';

  switch (buffer[0]) {
  case 'A':
    *type = buffer[0];
    return atoi(&buffer[1]);
  case 'C':
    *type = buffer[0];
    return 0;
  }

  *type = '\0';
  return -1;
}

static void
whois_read_socket(int socket, char* buf, int len) {
  int cnt = 0, retval;

  bzero((void*)buf, len);
  while (cnt < len) {
    retval = read(socket, buf + cnt, len - cnt);
    cnt += retval;
  }
  buf[len] = '\0';
#ifdef WHOIS_DEBUG
  fprintf(stderr, "--- WHOIS_DEBUG ---\n%s--- WHOIS_DEBUG ---\n", buf);
#endif
}

static char*
whois_read_data(int socket) {
  int retval;
  char type, *data = NULL;

  while (1) {
    retval = whois_get_type(socket, &type);
    if (type != 'A')
      break;
    data = (char*)malloc(retval + 1);
    whois_read_socket(socket, data, retval);
  }

  return data;
}

static char*
whois_get_from_as(int socket, int asn) {
  char cmdstr[WHOISCMDLEN];

  snprintf(cmdstr, WHOISCMDLEN, "!man,AS%d\n", asn);
  write(socket, cmdstr, strlen(cmdstr));
  return whois_read_data(socket);
}

static unsigned char
whois_search_desc(int socket, wf_whois_entry* whois_entry) {
  char *obj, *descs, *desce;
  unsigned char ok = 0;

  obj = whois_get_from_as(socket, whois_entry->as_number);
  if (obj != NULL) {
    descs = strstr(obj, "descr:");
    if (descs != NULL) {
      descs += 6;
      while (*descs == ' ' || *descs == '\t')
	descs++;
      desce = strchr(descs, '\n');
      if (desce != NULL)
	*desce = '\0';

      whois_entry->as_descr = descs;
      ok++;
    }
    free(obj);
  }

  return ok;
}

void
wf_whois::whois_from_ip(const wf_ipaddr& ipaddr, wf_whois_entry* whois_entry) {
  //  char cmdstr[WHOISCMDLEN], *data, *descs, *desce;
  char *data, *descs, *desce;

  string cmdstr = "!r" + ipaddr.tostr() + "/32,l\n";
  //  snprintf(cmdstr, WHOISCMDLEN, "!r%s/32,l\n", ipaddr.tostr().c_str());
  //  write(_sock, cmdstr, strlen(cmdstr));
  typedef void (*sighandler_t)(int);
  sighandler_t preserved_sigpipe_handler = signal(SIGPIPE, SIG_IGN);
  /* write() can generate SIGPIPE if the socket is not connected anymore */
  int ret = write(_sock, cmdstr.c_str(), cmdstr.length());
  signal(SIGPIPE, preserved_sigpipe_handler);
  if (ret == -1) { /* we were deconnected */
    _sock = -1;
    if (connect() == false) /* reconnection */
      return; /* connection failed */
    write(_sock, cmdstr.c_str(), cmdstr.length());
  }

  data = whois_read_data(_sock);
  if (data == NULL)
    return;

  descs = desce = data;
  while (*descs != '\0') {
    if (whois_entry->as_number == 0 && strstr(descs, "origin:") == descs) {
      descs += 7;
      while (*descs == ' ' || *descs == '\t')
	descs++;
      descs += 2;
      desce = strchr(descs, '\n');
      if (desce != NULL)
	*desce = '\0';
      whois_entry->as_number = atoi(descs);
      whois_search_desc(_sock, whois_entry);
      descs = desce + 1;
    }
    else if (whois_entry->ip_route.empty() && strstr(descs, "route:") == descs) {
      descs += 6;
      while (*descs == ' ' || *descs == '\t')
	descs++;
      desce = strchr(descs, '\n');
      if (desce != NULL)
	*desce = '\0';
      whois_entry->ip_route = descs;
      descs = desce + 1;
    }
    else if (whois_entry->ip_descr.empty() && strstr(descs, "descr:") == descs) {
      descs += 6;
      while (*descs == ' ' || *descs == '\t')
	descs++;
      desce = strchr(descs, '\n');
      if (desce != NULL)
	*desce = '\0';
      whois_entry->ip_descr = descs;
      descs = desce + 1;
    }
    else
      descs++;
  }
  free(data);

  if (whois_entry->as_number > 0) {
    if (whois_entry->ip_route.empty())
      whois_entry->ip_route = "-";
    if (whois_entry->ip_descr.empty())
      whois_entry->ip_descr = "-";
    if (whois_entry->as_descr.empty())
      whois_entry->as_descr = "-";
  }
}

wf_whois_entry*
wf_whois::whois(const wf_ipaddr& ipaddr) {
  if (ipaddr.isroutable() == false)
    return NULL;

  wf_whois_entry* whois_entry = cache_get(ipaddr);
  if (whois_entry != NULL) { /* found in cache */
    if (verbose > 1)
      fprintf(stderr, _("Found whois info for %s in cache\n"),
	      ipaddr.tostr().c_str());
    return whois_entry;
  }

  /* Not found in cache. */
  if (verbose > 1)
    fprintf(stderr, _("Looking up whois info for %s\n"),
	    ipaddr.tostr().c_str());
  
  if (_sock == -1) { /* if not connected, connect! */
    if (connect() == false)
      return NULL;
  }

  whois_entry = new wf_whois_entry();
  if (whois_entry == NULL)
    return NULL;

  whois_from_ip(ipaddr, whois_entry);
  if (whois_entry->as_number == 0) {
    delete whois_entry;
    return NULL;
  }

  cache_add(whois_entry);  /* Add entry to cache. */
  return whois_entry;
}

bool
wf_whois::isconnected() const {
  return (_sock != -1);
}

bool
wf_whois::connect() {
  return connect(_host, _port);
}

bool
wf_whois::connect(const string& whois_server, uint16_t port) {
  struct hostent* he;
  struct sockaddr_in sin;
  int fd;
  
  if (_sock != -1) { /* close first if already connected */
    ::close(_sock);
    _sock = -1;
  }

  he = gethostbyname(whois_server.c_str());
  if (he == NULL) {
    fprintf(stderr, _("Error: lookup failed for %s.\n"), whois_server.c_str());
    return false;
  }
  
  fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (fd == -1) {
    perror("socket");
    return false;
  }

  sin.sin_family = AF_INET;
  sin.sin_port = htons(port);
  bcopy(he->h_addr, &sin.sin_addr, he->h_length);

  if (verbose > 0)
    ostream_printf(cerr, _("Connecting to whois server %s, port %i... "),
		   whois_server.c_str(), port);

  if (::connect(fd, (struct sockaddr*)&sin, sizeof(sin)) == -1) {
    perror("connect");
    return false;
  }

  if (verbose > 0)
    cerr << _("connected") << endl;

  write(fd, "!!\n", 3);
  _host = whois_server;
  _port = port;
  _sock = fd;
  return true;
}

bool
wf_whois::close() {
  if (_sock == -1)
    return false;
  write(_sock, "q\n", 2);
  if (::close(_sock) == -1) {
    perror("close");
    return false;
  }
  _sock = -1;
  if (verbose > 0)
    cerr << _("Closed connection to whois server.") << endl;
  return true;
}

void
wf_whois::cache_add(const wf_whois_entry* whois_entry) {
  if (whois_entry != NULL)
    cache.push_back((wf_whois_entry*)whois_entry);
}

wf_whois_entry*
wf_whois::cache_get(const wf_ipaddr& ipaddr) const {
  if (ipaddr.isdefined() == false)
    return NULL;

  list<wf_whois_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first) {
    wf_network net((*first)->ip_route);
    if (net.belong(ipaddr) == true)
      return *first;
  }
  return NULL;
}

ostream&
wf_whois::cache_debugprint(ostream& os) const {
  list<wf_whois_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first)
    (*first)->print(os) << endl;
  return os;
}

ostream&
wf_whois_entry::print(ostream& os) const {
  os << ip_route << ' ' << ip_descr << " AS" << as_number << ' ' << as_descr;
  return os;
}

ostream&
operator<<(ostream& os, const wf_whois_entry& whois_entry) {
  return whois_entry.print(os);
}
