/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>

#include <stdio.h>
#include <sys/stat.h>
#include <signal.h>
#include <errno.h>
#include <getopt.h>

#include "wffirewall.h"
#include "wflogs.h"
#include "output_sort.h"
#include "wfinmodule.h"
#include "wfoutmodule.h"
#include "interactive.h"
#include "realtime.h"
#include "defs.h"

// PASSIVE_WAIT is a macro which marks code to add for passive wait (logging
// through a device, for example) RV@@8

#define DEFAULT_SORT_CRITERIAS  "-count,time,dipaddr,protocol,dport"
#define DEFAULT_PARSE_STRICTNESS  WF_LOGENTRIES_PARSING_STRICTNESS_WARNING
#define DEFAULT_PARSE_STRICTNESS_S  "warning"

#define WFLOGS_PROMPT "wflogs> "

char* configfilename = ""; /* define it with a #define RV@@7 */
bool realtime = false, interactive = false;
int verbose = 0;
wflogs_filter* myfilter = NULL;
rvlog wflogs_log;

static wf_interactive* interact = NULL;

#if 1 /* remove this #if */
#include <list>
static list<wf_inmodule*> input_modules;
#endif

static string
guess_fwtool(void) {
  wf_firewall firewall;
  string fwtool;

  firewall.os_guess();
  firewall.fwtool_guess();
  fwtool = firewall.fwtool_get();

  if (fwtool == "iptables")
    return "netfilter";
  return fwtool;
}

static void
input_module_load(const string& name) {
  if (name == "help") {
    cout << _("Available input types:") << endl << _("Built-in modules: ");
    wf_inmodule_available_modules_print(cout);

    string fwtool = guess_fwtool();  /* try to guess the firewalling tool */
    cout << endl << _("Default input type:") << ' ';
    if (fwtool.empty())
      cout << _("could not be guessed.");
    else
      cout << fwtool;
    cout << endl;
    exit(0);
  }

  wf_inmodule* module = wf_inmodule_init(name);
  if (module == NULL) {
    wflogs_log.log(RVLOG_ERROR,
		   _("Error: unable to initialize input module `%s'."),
		   name.c_str());
    exit(1);
  }

  input_modules.push_back(module);
}

static void
input_modules_load(const string& args) {
  if (args == "all") {
    unsigned int i;
    for (i = 0; i < inmodules_size; i++)
      input_module_load(inmodules[i].name);
    return;
  }

  unsigned int pos, oldpos = 0;
  do {
    pos = args.find(',', oldpos);
    string name = args.substr(oldpos, pos - oldpos);
    input_module_load(name);
    oldpos = pos + 1;
  } while (pos != string::npos);
}

static wf_outmodule*
output_module_load(const string& name) {
  if (name == "help") {
    cout << _("Available output types:") << endl;
    wf_outmodule_available_modules_print(cout);
    cout << endl;
    exit(0);
  }

  wf_outmodule* module = wf_outmodule_init(name);
  if (module == NULL) {
    wflogs_log.log(RVLOG_ERROR,
		   _("Error: unable to initialize output module `%s'."),
		   name.c_str());
    exit(1);
  }

  return module;
}

struct display {
  void operator() (const pair<const char*, wf_option>& x) const {
    const wf_option& option = x.second;
    cout << x.first << ": ";
    option.print(cout);
    cout << " (" << option.type_tostr() << ") " << option.long_help() << endl;
  }
};

static void
display_outmodule_options(const wf_outmodule* outmodule) {
  printf(_("Output module `%s' default configuration:"),
	 outmodule->name().c_str());
  cout << endl << endl;
  outmodule->conf->config.for_each(display());
}

static void
init_interactive(int signum) {
  interact = new wf_interactive(WFLOGS_PROMPT);
  if (interact == NULL)
    wflogs_log.log(RVLOG_ERROR, _("Error: could not initialize interactive mode activated by SIGUSR1."));
  else {
    signal(SIGUSR1, SIG_IGN);
    interactive = true;
    interact->display_prompt();
  }
}

static void
print_help_and_exit(void) {
  cerr << _("Type `wflogs --help' for more information.") << endl;
  exit(1);
}

static void
usage(void) {
  cout << "wflogs " VERSION " Herve Eychenne <" WALLFIRE_AUTHOR_EMAIL_ADDR ">\n\n";
  printf(_("Usage: wflogs [options] [logfile]\n\
\n\
wflogs is the log analyser of the WallFire project.\n\
\n\
Options:\n\
-c | --config file       use alternate config file.\n\
                         (defaults to `%s').\n\
-f | --filter expr       set filter expression a la Perl.\n\
-i | --input-format module1[,module2[,...]]\n\
                         set input format(s) that can be parsed.\n\
                         (use name `help' to show available ones and\n\
                         default (dynamically guessed according to the OS)\n\
                         type).  Use `all' to try every available modules.\n\
-I | --interactive       interactive mode.\n\
-o | --output-type name  set output type\n\
                         (use name `help' to show available ones,\n\
                         default is `text').\n\
-O | --obfuscate [field1[,field2[,...]]]\n\
                         obfuscates some logging fields.\n\
                         (available fields are `date', `hostname', `ipaddr',\n\
                         and `macaddr'.  Default is `all': every field).\n\
-P | --proceed           if realtime or interactive modes are set, process\n\
                         current log entries before entering in these modes.\n\
-R | --realtime          real-time mode.\n\
-s | --sort-output[=[-]key1[,[-]key2[,...]]]\n\
                         sort ouput lines according to the given criteria(s)\n\
                         (use key `help' to show available keys).\n\
     --strict-parsing type\n\
                         set the parsing policy\n\
                         (available types are `loose', `nowarning', \n\
                         `warning', and `error'. Default is `%s').\n\
-v | --verbose [level]   set verbosity level (default is 1).\n\
-V | --version           print the current version on stdout.\n\
-h | --help              print this help message.\n\
\n\
See the wflogs(8) manual page for more information.\n\
See %s for comments or bug reports.\n\n"),
	  configfilename, DEFAULT_PARSE_STRICTNESS_S, WALLFIRE_URL);
  exit(0);
}

static void
version(void) {
  puts("wflogs " VERSION);
  exit(0);
}


int
main(int argc, char* argv[]) {
  wf_outmodule* output_module = NULL;
  string filename;
  FILE* file;
  unsigned int lineno = 0;
  wflogs_obfuscator* myobfuscator = NULL;
  bool proceed = false;
  char* sortarg = NULL, *filter_expr = NULL, *obfuscatearg = NULL;
  enum wf_logentries_parsing_strictness strict_parsing = DEFAULT_PARSE_STRICTNESS;
  int ch;
  int long_index = 0;
  static const char* short_options = "c:f:i:Io:O::PRs::v::Vh?";
  static const struct option long_options[] = {
    { "config", required_argument, 0, 'c' },
    { "filter", required_argument, 0, 'f' },
    { "input-format", required_argument, 0, 'i' },
    { "interactive", no_argument, 0, 'I' },
    { "output-type", required_argument, 0, 'o' },
    { "obfuscate", optional_argument, 0, 'O' },
    { "proceed", no_argument, 0, 'P' },
    { "realtime", no_argument, 0, 'R' },
    { "sort-output", optional_argument, 0, 's' },
    { "strict-parsing", required_argument, 0, '%' },
    { "verbose", optional_argument, 0, 'v' },
    { "version", no_argument, 0, 'V' },
    { "help", no_argument, 0, 'h' },
    { NULL, 0, 0, 0 }
  };

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);

  wflogs_log.opts_set(RVLOG_NAME);
  wflogs_log.how_set(RVLOG_STDERR);
  wflogs_log.name_set(argv[0]);

  wf_sort mysort;
  if (mysort.available_set(format_array, format_array_count) == false) {
    wflogs_log.log(RVLOG_ERROR, _("Error: sort criterias are not sorted. Please contact the maintainer."));
    exit(1);
  }

  opterr = 0; /* don't let getopt show error messages: we display our own */
  while ((ch = getopt_long(argc, argv, short_options, long_options,
			   &long_index)) != EOF)
    switch (ch) {
    case 'c':	/* config file */
      configfilename = optarg; break;
    case 'f':	/* filter expression */
      filter_expr = optarg; break;
    case 'i':	/* input type */
      input_modules_load(optarg); break;
    case 'I':	/* interactive mode */
      interactive = true; break;
    case 'o':	/* output type */
      if (output_module != NULL) { /* an output module was already specified */
	wflogs_log.log(RVLOG_ERROR, _("Error: output module already loaded, or output module option specified before module name."));
	exit(1);
      }
      output_module = output_module_load(optarg);
      break;
    case 'O':
      obfuscatearg = ((optarg != NULL) ? optarg : (char*)"" /* default */);
      break;
    case 'P':	/* proceed mode: process current log entries even if realtime
		   or interactive modes are to be activated */
      proceed = true; break;
    case 'R':	/* real-time mode */
      realtime = true; break;
    case 's':
      if (optarg == NULL)
	sortarg = DEFAULT_SORT_CRITERIAS;
      else if (!strcmp(optarg, "help")) {
	cout << _("Available sort keys:") << endl;
	mysort.available_print(cout) << endl;
	cout << _("Default sort key: ") << DEFAULT_SORT_CRITERIAS << endl;
	exit(0);
      }
      else
	sortarg = optarg;
      break;
    case '%':
      if (!strncmp(optarg, "loose", strlen(optarg)))
	strict_parsing = WF_LOGENTRIES_PARSING_STRICTNESS_LOOSE;
      else if (!strncmp(optarg, "nowarning", strlen(optarg)))
	strict_parsing = WF_LOGENTRIES_PARSING_STRICTNESS_NOWARNING;
      else if (!strncmp(optarg, "warning", strlen(optarg)))
	strict_parsing = WF_LOGENTRIES_PARSING_STRICTNESS_WARNING;
      else if (!strncmp(optarg, "error", strlen(optarg)))
	strict_parsing = WF_LOGENTRIES_PARSING_STRICTNESS_ERROR;
      else {
	wflogs_log.log(RVLOG_ERROR,
		       _("Error: unknown argument in option `%s'."),
		       argv[optind - 1]);
	exit(1);
      }
      break;
    case 'v':	/* verbose mode */
      if (optarg == NULL)
	++verbose;
      else
	verbose = atoi(optarg);
      break;
    case 'V':	/* print version and exit */
      version();
    case 'h':	/* print help and exit */
      usage();
    case ':':	/* missing parameter */
      wflogs_log.log(RVLOG_ERROR, _("Error: this shouldn't happen. Please contact the maintainer."));
      exit(1);
      break;
    case '?':
      if (optopt) {
	if (strchr(short_options, optopt) != NULL) {
	  wflogs_log.log(RVLOG_ERROR,
			 _("Error: option requires an argument -- %c."),
			 optopt);
	  exit(1);
	}
	wflogs_log.log(RVLOG_ERROR, _("Error: invalid option -- %c."), optopt);
	exit(1);
      }

      /* unknown global option: we look for output module option */
      --optind;
      if (output_module == NULL) /* no output module was specified */
	output_module = output_module_load("text"); /* text is default mode */
      /* output_module cannot be NULL, we'd already have exited otherwise */
      
      if (!strcmp(argv[optind], "--options")) {
	display_outmodule_options(output_module);
	exit(0);
      }

      int ret = output_module->conf->config.parse(argc, argv);
      if (ret == WF_CONFIG_PARSE_OK)
	break;
      else if (ret == WF_CONFIG_PARSE_UNKNOWN) {
	wflogs_log.log(RVLOG_ERROR, _("Error: unrecognized option `%s'."),
		       argv[optind - 1]);
	print_help_and_exit(); /* exits */
      }
      else if (ret == WF_CONFIG_PARSE_ARG) {
	wflogs_log.log(RVLOG_ERROR,
		       _("Error: problem with argument in option `%s'."),
		       argv[optind - 1]);
	exit(1);
      }
      else {
	wflogs_log.log(RVLOG_ERROR,
		       _("Error: problem during parsing of option `%s'."),
		       argv[optind - 1]);
	exit(1);
      }
    }

  if (output_module == NULL) { /* no output module was specified */
    output_module = output_module_load("text"); /* text is the default mode */
    /* output_module cannot be NULL, we would already have exited otherwise */
  }

  output_module->conf->config["verbose"] = verbose;

  if (argc > optind + 1) {
    wflogs_log.log(RVLOG_ERROR, _("Error: only one filename allowed."));
    print_help_and_exit(); /* exits */
  }

  /* Parse sort criterias. */
  if (sortarg != NULL) {
    const char* err = mysort.parse(sortarg);
    if (err != NULL) {
      wflogs_log.log(RVLOG_ERROR,
		     _("Error: sort criterias parsing failed: %s"), err);
      exit(1);
    }
  }

  /* Check input module(s). */
  if (input_modules.empty()) { /* no input module was forced via -i option */
    string fwtool = guess_fwtool();  /* try to guess the firewalling tool */
    if (fwtool.empty()) { /* default module type could not be guessed either */
      wflogs_log.log(RVLOG_ERROR, _("Error: missing input type: local firewalling tool could not be guessed."));
      wflogs_log.log(RVLOG_ERROR, _("Try to force it via `-i' option."));
      print_help_and_exit(); /* exits */
    }

    input_module_load(fwtool); /* if load fails, we exit here */
  }

  /* Open input file or stream. */
  if (argc == optind + 1 && !strcmp(argv[optind], "-")) {
    filename = "stdin";
    file = stdin;
  }
  else {
    if (argc == optind + 1)
      filename = argv[optind];
    else if (argc == optind)
      filename = WF_DEFAULT_LOGFILE;
    
    file = fopen(filename.c_str(), "r");
    if (file == NULL) {
      wflogs_log.log(RVLOG_ERROR, _("Error: %s: %m"), filename.c_str());
      exit(1);
    }

    /* Check permissions. */
    struct stat sts;
    if (fstat(fileno(file), &sts)) {
      wflogs_log.log(RVLOG_ERROR, _("Error: %s: %m"), filename.c_str());
      exit(1);
    }
    if (S_ISDIR(sts.st_mode)) {
      wflogs_log.log(RVLOG_ERROR, _("Error: %s: %s"), filename.c_str(),
		     strerror(EISDIR));
      exit(1);
    }
  }
  
  /* Parse filter expression. */
  if (filter_expr != NULL) {
    myfilter = new wflogs_filter;
    if (myfilter == NULL || myfilter->set(filter_expr) == false) {
      wflogs_log.log(RVLOG_ERROR, _("Error: wrong filter expression."));
      exit(1);
    }
    if (verbose) {
      cerr << _("Filter expression: ");
      myfilter->print(cerr) << endl;
    }
  }

  /* Parse obfuscation criterias. */
  if (obfuscatearg != NULL) {
    myobfuscator = new wflogs_obfuscator;
    if (myobfuscator->set(obfuscatearg) == false) {
      wflogs_log.log(RVLOG_ERROR, _("Error: wrong obfuscator argument."));
      exit(1);
    }
  }

  if (proceed || (realtime == false && interactive == false)) {
    /* Parse input file. */
    wf_logentries* logentries = wf_inmodule_parse(file, filename, &lineno,
						  input_modules,
						  strict_parsing);
    if (logentries == NULL) {
      wflogs_log.log(RVLOG_ERROR, _("Error: parsing failed."));
      exit(1);
    }
    
    unsigned int deleted_entries;
    
    /* Filter logs. */
    if (myfilter != NULL) {
      if (logentries->filter(*myfilter, &deleted_entries) == false) {
	wflogs_log.log(RVLOG_ERROR, _("Error: filtering failed."));
	exit(1);
      }
      if (verbose > 1)
	fprintf(stderr, _("Entries deleted by filter: %d\n"), deleted_entries);
    }
    
    /* Summarize logs. */
    if (output_module->conf->config["summary"].tobool()) {
      if (logentries->summary(&deleted_entries) == false) {
	wflogs_log.log(RVLOG_ERROR, _("Error: summary failed."));
	exit(1);
      }
      if (verbose > 1)
	fprintf(stderr, _("Entries deleted by summary: %d\n"),
		deleted_entries);
    }
    
    /* Sort logs. */
    if (sortarg != NULL)
      logentries->elems.sort(mysort);
    
    /* Obfuscate logs. */
    if (obfuscatearg != NULL) {
      if (logentries->obfuscate(*myobfuscator) == false) {
	wflogs_log.log(RVLOG_ERROR, _("Error: obfuscator failed."));
	exit(1);
      }
      if (myobfuscator->ipaddr) { /* turn off reverse DNS and whois lookups */
	output_module->conf->config.option_set("resolve", 0);
	output_module->conf->config.option_set("whois_lookup", 0);
      }
      if (myobfuscator->macaddr) /* turn off mac vendor resolution */
	output_module->conf->config.option_set("mac_vendor", 0);
    }
    
    /* Output logs on stdout. */
    if (output_module->print(*logentries, cout) == false) {
      wflogs_log.log(RVLOG_ERROR, _("Error: output failed."));
      exit(1);  // really exit here all the way? ALL@8
    }
  }

  if (interactive || realtime) { /* never returns */
    if (realtime && strict_parsing == WF_LOGENTRIES_PARSING_STRICTNESS_ERROR) {
      cout << _("Warning: strictness parsing is set to error and we are in realtime mode.") << '\n';
      cout << _("Program will exit on parsing error.") << '\n';
    }
    if (interactive) {
      interact = new wf_interactive(WFLOGS_PROMPT);
      if (interact == NULL) {
	wflogs_log.log(RVLOG_ERROR, _("Error: could not initialize interactive mode."));
	exit(1);
      }
    }
    else if (realtime) /* if realtime and non interactive */
      signal(SIGUSR1, init_interactive);

    /* If no processing of the log file has been done yet, we still have to
       count lines in order to be able to provide accurate line numbers later. */
    if (proceed == false) {
      
      if (verbose > 1)
	cerr << _("Counting lines...");
#if 0 /* old method, inefficent on big files */
      int c;
      while ((c = getc(file)) != EOF) { /* getc() is faster than fgetc() */
	if (c == '\n')
	  ++lineno;
      }
#else
#define BUFFER_SIZE (16 * 1024)
      char buf[BUFFER_SIZE + 1];
      size_t bytes_read;
      while ((bytes_read = fread(buf, 1, BUFFER_SIZE, file)) > 0) {
	if (ferror(file)) {
	  cerr << '\n';
	  wflogs_log.log(RVLOG_ERROR, _("Error: %s: %m"), filename.c_str());
	  exit(1);
	}
	const char* p = buf;
	while ((p = (char*)memchr(p, '\n', (buf + bytes_read) - p)) != NULL) {
	  ++p;
	  ++lineno;
	}
      }
#endif
      if (verbose > 1) {
	cerr << _(" done") << '\n';
	if (interactive)
	  interact->display_update();
      }
    }
    
    if (interactive)
      interact->display_prompt();

    /* Entering loop for interactive and/or real-time mode */
    while (1) {
      fd_set fdset;

      /* Handle active wait from log file */
      char line[WF_LOGLINE_MAXLEN];
      bool displayed = false;
      /* We read the line in any case (even if we're not in realtime mode
	 at this time), to keep track of the right line number in case we would
	 set realtime mode later (even if currently turned off). */
      u_int16_t line_block = 0;
      while (fgets(line, sizeof(line), file)) {
	lineno++; /* there is a new line */
	if (realtime) {
	  if (wf_realtime_parse(line, filename, lineno, input_modules,
			    strict_parsing, myfilter, output_module))
	    displayed = true;

	  /* If realtime and interactive mode are both enabled, read only
	     a few lines to still allow a little interactivity even when on
	     log flood. */
	  if (interactive && line_block++ > 10)
	      break;
	}
      }
      /* no more new line */
      
      if (displayed && interactive)
	interact->display_update();

      FD_ZERO(&fdset);
      if (interactive)
	FD_SET(0, &fdset);
      int fds = 0; // only stdin for the moment RV@@9
#ifdef PASSIVE_WAIT
      if (realtime)
	FD_SET(wflogs_realtime_sockd, &fdset);
#endif

      struct timeval tv = { 1, 0 }; /* 1 second timeout */
    intr:
      int ret = select(fds + 1, &fdset, NULL, NULL, &tv);
      if (ret == 0) /* timeout */
	continue;
      else if (ret < 0) {
	if (errno == EINTR) /* SIGWINCH, for example */
	  goto intr;
	printf(_("select error: %s\n"), strerror(errno));
	continue;
      }
    
      /* Handle tty activity */
      if (interactive && FD_ISSET(0, &fdset))
	interact->callback(); /* every char if readline, every line otherw. */

#ifdef PASSIVE_WAIT
      /* Handle passive wait from log stream */
      if (realtime && FD_ISSET(wflogs_realtime_sockd, &fdset)) {
	if (interactive)
	  interact->display_update();
      }
#endif
    }
  }

  exit(0);
}
