/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _WFTK_JOYSTICK_H
#define _WFTK_JOYSTICK_H

#include <vector>
#include <sigc++/object.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#endif

#include <SDL/SDL_joystick.h>
#include <SDL/SDL_events.h>

#include <wftk/marshal.h>
#include <wftk/point.h>

namespace wftk {

/// A class to handle joystick information
class Joystick : virtual public SigC::Object
{
 public:
  /// joystick constructor, used internally
  Joystick(unsigned);
  ~Joystick();

  /// returns the number of joysticks on the system
  static unsigned number() {if(!init_) init(); return joysticks_.size();}
  /// returns a particular Joystick, or 0 if n >= number()
  static Joystick* instance(unsigned n = 0);

  /// Get the system-dependent name of a joystick.
  const char* name() {return SDL_JoystickName((int) num_);}

  /// Get the number of axes of a joystick
  Uint8 numAxes() {return nAxes_;}
  /// Get the number of trackballs of a joystick
  Uint8 numBalls() {return nBalls_;}
  /// Get the number of hats for a joystick
  Uint8 numHats() {return nHats_;}
  /// Get the number of buttons on a joystick
  Uint8 numButtons() {return nButtons_;}

  /// Flags used in the hat status mask
  enum Hat {
    CENTERED = SDL_HAT_CENTERED, // == 0
    LEFT = SDL_HAT_LEFT,
    RIGHT = SDL_HAT_RIGHT,
    UP = SDL_HAT_UP,
    DOWN = SDL_HAT_DOWN,
    RIGHTUP = RIGHT | UP, // == SDL_HAT_RIGHTUP
    RIGHTDOWN = RIGHT | DOWN, // == SDL_HAT_RIGHTDOWN
    LEFTUP = LEFT | UP, // == SDL_HAT_LEFTUP
    LEFTDOWN = LEFT | DOWN // == SDL_HAT_LEFTDOWN
  };

  /// returns the position of the joystick on a particular axis
  Sint16 getAxis(Uint8 axis) {return (axis < nAxes_) ?
	SDL_JoystickGetAxis(stick_, (int) axis) : 0;}
  /// returns a mask based on the Hat enum
  unsigned getHat(Uint8 hat) {return (hat < nHats_) ?
	SDL_JoystickGetHat(stick_, (int) hat) : 0;}
  /// return motion delta since last call to getBall()
  Point getBall(Uint8 ball) {Point p; if(ball < nBalls_)
	SDL_JoystickGetBall(stick_, (int) ball, &p.x, &p.y); return p;}
  /// returns true if pressed, false if not
  bool getButton(Uint8 button) {return (button < nButtons_) ?
	(SDL_JoystickGetButton(stick_, (int) button) != 0) : false;}

  /// emitted if the joystick axis changes
  SigC::Signal2<bool,Uint8,Sint16,BoolMarshal> axisEvent;
  /// emitted if the joystick hat changes
  SigC::Signal2<bool,Uint8,unsigned,BoolMarshal> hatEvent;
  /// emitted if the joystick trackball changes
  SigC::Signal2<bool,Uint8,const Point&,BoolMarshal> ballEvent;
  /// emitted if the joystick button changes
  SigC::Signal2<bool,Uint8,bool,BoolMarshal> buttonEvent;

  /// called by the wftk mainloop
  static bool handleEvent(const SDL_Event*);

 private:
  Joystick(const Joystick&);
  Joystick& operator=(const Joystick&);

  unsigned num_;
  SDL_Joystick* stick_;
  Uint8 nAxes_, nHats_, nBalls_, nButtons_;

  static void init();
  static void shutdown();
  static bool init_;
  static std::vector<Joystick*> joysticks_;
};

} // namespace wftk

#endif // _WFTK_JOYSTICK_H
