/*
 * PreferencesApp.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include "swt.h"
#include "PreferencesApp.h"
#include "DuneApp.h"

#define DEFAULT_NEAR_CLIPPING_PLANE_DIST 0.05f 
#define DEFAULT_FAR_CLIPPING_PLANE_DIST 7000.0f
#define DEFAULT_MAX_INLINES_TO_LOAD 1024
#define DEFAULT_MAX_KEYS_IN_CHANNELVIEW 128

PreferencesApp::PreferencesApp()
{
    _prefs = swLoadPreferences("DuneSoft", "dune");
    _showAllFields = GetBoolPreference("ShowAllFields", false);
    _rotationOrder = GetIntPreference("RotationOrder", EulOrdXYZs);
    _rotationTitle = "XYZs";
    _mouseMode = (MouseMode) GetIntPreference("MouseMode", MOUSE_EXAMINE);
    _handleMode = (HandleMode) GetIntPreference("HandleMode", HM_SELECTED);
    _handleMeshAlways = GetBoolPreference("HandleMeshAlways", false);
    const char* handleSizeBuf        = GetPreference("HandleSize", "2.0");
    _handleSize = atof(handleSizeBuf);
    const char* pointSetSizeBuf      = GetPreference("PointSetSize", "0.0");
    _pointSetSize = atof(pointSetSizeBuf);
    const char* epsilonBuf           = GetPreference("Epsilon", "0.005");
    _epsilon = atof(epsilonBuf);
    char buf[128];
    const char *buf2;
    mysnprintf(buf, 127, "%f", DEFAULT_NEAR_CLIPPING_PLANE_DIST);
    buf2 = GetPreference("NearClippingPlaneDistance", buf); 
    if (atof(buf2) > 0)
        _nearClippingPlaneDist = atof(buf2);
    else {
        _nearClippingPlaneDist = DEFAULT_NEAR_CLIPPING_PLANE_DIST;
        fprintf(stderr,"NearClippingPlaneDistance preference");
        fprintf(stderr," must be greater zero\n");
        fprintf(stderr,"NearClippingPlaneDistance preference set to %f\n",
               _nearClippingPlaneDist);
    }
    mysnprintf(buf, 127, "%f", DEFAULT_FAR_CLIPPING_PLANE_DIST);
    buf2 = GetPreference("FarClippingPlaneDistance", buf); 
    if (atof(buf2) > _nearClippingPlaneDist)
        _farClippingPlaneDist = atof(buf2);
    else {
        _farClippingPlaneDist = DEFAULT_FAR_CLIPPING_PLANE_DIST;
        fprintf(stderr,"FarClippingPlaneDistance preference");
        fprintf(stderr," must be greater than NearClippingPlaneDistance preference\n");
        fprintf(stderr,"FarClippingPlaneDistance preference set to %f\n",
                _farClippingPlaneDist);
    }
    _xSymetricMode = GetBoolPreference("XSymetric", false);    
    _maxInlinesToLoad = GetIntPreference("MaxInlinesToLoad", 
                                         DEFAULT_MAX_INLINES_TO_LOAD);
    _maxKeysInChannelView = GetIntPreference("MaxKeysInChannelView", 
                                             DEFAULT_MAX_KEYS_IN_CHANNELVIEW);
    _renderFasterWorse = GetBoolPreference("RenderFasterWorse", false);    
}

void
PreferencesApp::PreferencesDefaults()
{
    _showAllFields = false;
    _rotationOrder = EulOrdXYZs;
    _rotationTitle = "XYZs";
    _mouseMode = (MouseMode) MOUSE_EXAMINE;
    _handleMode = (HandleMode) HM_SELECTED;
    _handleMeshAlways = false;
    _handleSize = 2.0;
    _pointSetSize = 0.0;
    _epsilon = 0.01;
    _nearClippingPlaneDist = DEFAULT_NEAR_CLIPPING_PLANE_DIST;
    _farClippingPlaneDist = DEFAULT_FAR_CLIPPING_PLANE_DIST;
    _xSymetricMode = false;    
    _maxInlinesToLoad = DEFAULT_MAX_INLINES_TO_LOAD;
    _renderFasterWorse = false;
    _maxKeysInChannelView = DEFAULT_MAX_KEYS_IN_CHANNELVIEW;
}


void PreferencesApp::SavePreferences()
{
    char buf[128];
    SetBoolPreference("ShowAllFields", _showAllFields);
    SetIntPreference("RotationOrder", _rotationOrder);
    SetIntPreference("MouseMode", _mouseMode);
    SetIntPreference("HandleMode", _handleMode);
    SetBoolPreference("HandleMeshAlways", _handleMeshAlways);
    snprintf(buf,127,"%f",_handleSize);
    SetPreference("HandleSize", buf);
    snprintf(buf,127,"%f",_pointSetSize);
    SetPreference("PointSetSize", buf);
    snprintf(buf,127,"%f",_epsilon);
    SetPreference("Epsilon", buf);
    snprintf(buf,127,"%f",_nearClippingPlaneDist);
    SetPreference("NearClippingPlaneDistance", buf);
    snprintf(buf,127,"%f",_farClippingPlaneDist);
    SetPreference("FarClippingPlaneDistance", buf);
    SetBoolPreference("XSymetric", _xSymetricMode);
    SetIntPreference("MaxInlinesToLoad", _maxInlinesToLoad); 
    SetIntPreference("MaxKeysInChannelView", _maxKeysInChannelView);
    SetBoolPreference("RenderFasterWorse", _renderFasterWorse);

    TheApp->EcmaScriptSavePreferences();
    TheApp->StereoViewSavePreferences();
    TheApp->OutputSavePreferences();
    swSavePreferences(_prefs);
}

void PreferencesApp::SetShowAllFields(bool showAllFields)
{
    if (_showAllFields != showAllFields) {
	_showAllFields = showAllFields;
	TheApp->UpdateAllWindows();
    }
}

void PreferencesApp::SetRotationOrder(int rotationOrder)
{
    if (_rotationOrder != rotationOrder) {
	_rotationOrder = rotationOrder;
	TheApp->UpdateAllWindows();
    }
}

float PreferencesApp::GetHandleSize(void)
{
    if (TheApp->useStereo())
        return TheApp->GetStereoHandleSizeMult() * _handleSize;
    else
        return _handleSize;         
}

void PreferencesApp::SetHandleSize(float size)
{
    _handleSize = size;         
    TheApp->UpdateAllWindows();
}

void PreferencesApp::SetPointSetSize(float size)
{
    _pointSetSize = size;         
    TheApp->UpdateAllWindows();
}

void PreferencesApp::SetHandleMode(HandleMode handleMode)
{
    if (_handleMode != handleMode) {
	_handleMode = handleMode;
	TheApp->UpdateAllWindows();
    }
}

void PreferencesApp::SetMouseMode(MouseMode mouseMode)
{
    if (_mouseMode != mouseMode) {
	_mouseMode = mouseMode;
	TheApp->UpdateAllWindows();
    }
}

bool
PreferencesApp::GetBoolPreference(const char *key, bool defaultValue)
{
    return swGetIntPreference(_prefs, key, defaultValue ? 1 : 0) != 0;
}

int
PreferencesApp::GetIntPreference(const char *key, int defaultValue)
{
    return swGetIntPreference(_prefs, key, defaultValue);
}

const char *
PreferencesApp::GetPreference(const char *key, const char *defaultValue)
{
    return swGetPreference(_prefs, key, defaultValue);
}

void
PreferencesApp::SetBoolPreference(const char *key, bool value)
{
    swSetIntPreference(_prefs, key, value ? 1 : 0);
}

void
PreferencesApp::SetIntPreference(const char *key, int value)
{
    swSetIntPreference(_prefs, key, value);
}

void
PreferencesApp::SetPreference(const char *key, const char *value)
{
    swSetPreference(_prefs, key, value);
}



