(**************************************************************************)
(*                                                                        *)
(*  The Why platform for program certification                            *)
(*                                                                        *)
(*  Copyright (C) 2002-2017                                               *)
(*                                                                        *)
(*    Jean-Christophe FILLIATRE, CNRS & Univ. Paris-sud                   *)
(*    Claude MARCHE, INRIA & Univ. Paris-sud                              *)
(*    Yannick MOY, Univ. Paris-sud                                        *)
(*    Romain BARDOU, Univ. Paris-sud                                      *)
(*                                                                        *)
(*  Secondary contributors:                                               *)
(*                                                                        *)
(*    Thierry HUBERT, Univ. Paris-sud  (former Caduceus front-end)        *)
(*    Nicolas ROUSSET, Univ. Paris-sud (on Jessie & Krakatoa)             *)
(*    Ali AYAD, CNRS & CEA Saclay      (floating-point support)           *)
(*    Sylvie BOLDO, INRIA              (floating-point support)           *)
(*    Jean-Francois COUCHOT, INRIA     (sort encodings, hyps pruning)     *)
(*    Mehdi DOGGUY, Univ. Paris-sud    (Why GUI)                          *)
(*                                                                        *)
(*  This software is free software; you can redistribute it and/or        *)
(*  modify it under the terms of the GNU Lesser General Public            *)
(*  License version 2.1, with the special exception on linking            *)
(*  described in file LICENSE.                                            *)
(*                                                                        *)
(*  This software is distributed in the hope that it will be useful,      *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *)
(**************************************************************************)



module Sset = Set.Make(String)

(* small library common to Why and Caduceus *)

let mkdir_p dir =
  if Sys.file_exists dir then begin
    if (Unix.stat dir).Unix.st_kind <> Unix.S_DIR then
      failwith ("failed to create directory " ^ dir)
  end else
    Unix.mkdir dir 0o777

let file ~dir ~file =
  mkdir_p dir;
  Filename.concat dir (Filename.basename file)

let file_subdir ~dir ~file =
  let d = Filename.dirname file in
  let d = Filename.concat d dir in
  mkdir_p d;
  Filename.concat d (Filename.basename file)

let file_copy src dest =
  let cin = open_in src
  and cout = open_out dest
  and buff = Bytes.make 1024 ' '
  and n = ref 0
  in
  while n := input cin buff 0 1024; !n <> 0 do
    output cout buff 0 !n
  done;
  close_in cin; close_out cout

let file_copy_if_different src dst =
  if not (Sys.file_exists dst) || Digest.file dst <> Digest.file src then
    file_copy src dst

let channel_contents_buf cin =
  let buf = Buffer.create 1024
  and buff = Bytes.make 1024 ' ' in
  let n = ref 0 in
  while n := input cin buff 0 1024; !n <> 0 do
    Buffer.add_subbytes buf buff 0 !n
  done;
  buf

let channel_contents cin = Buffer.contents (channel_contents_buf cin)

let file_contents f =
  try
    let cin = open_in f in
    let s = channel_contents cin in
    close_in cin;
    s
  with _ ->
    invalid_arg (Printf.sprintf "(cannot open %s)" f)

let file_contents_buf f =
  try
    let cin = open_in f in
    let buf = channel_contents_buf cin in
    close_in cin;
    buf
  with _ ->
    invalid_arg (Printf.sprintf "(cannot open %s)" f)


let remove_file ~debug f =
  if debug then
    Format.eprintf "Debug: not removing temporary file '%s'@." f
  else
    try
      Sys.remove f;
    with _ ->
      Format.eprintf "Warning: could not remove temporary file '%s'@." f
