/*
 * crypt32 certificate chain functions tests
 *
 * Copyright 2006 Juan Lang
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdio.h>
#include <stdarg.h>

#include <windef.h>
#include <winbase.h>
#include <winerror.h>
#include <wincrypt.h>
#include <wininet.h>

#include "wine/test.h"

static const BYTE selfSignedCert[] = {
 0x30, 0x82, 0x01, 0x1f, 0x30, 0x81, 0xce, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
 0x10, 0xeb, 0x0d, 0x57, 0x2a, 0x9c, 0x09, 0xba, 0xa4, 0x4a, 0xb7, 0x25, 0x49,
 0xd9, 0x3e, 0xb5, 0x73, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1d,
 0x05, 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03,
 0x13, 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30,
 0x1e, 0x17, 0x0d, 0x30, 0x36, 0x30, 0x36, 0x32, 0x39, 0x30, 0x35, 0x30, 0x30,
 0x34, 0x36, 0x5a, 0x17, 0x0d, 0x30, 0x37, 0x30, 0x36, 0x32, 0x39, 0x31, 0x31,
 0x30, 0x30, 0x34, 0x36, 0x5a, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
 0x55, 0x04, 0x03, 0x13, 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e,
 0x67, 0x00, 0x30, 0x5c, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x4b, 0x00, 0x30, 0x48, 0x02, 0x41,
 0x00, 0xe2, 0x54, 0x3a, 0xa7, 0x83, 0xb1, 0x27, 0x14, 0x3e, 0x59, 0xbb, 0xb4,
 0x53, 0xe6, 0x1f, 0xe7, 0x5d, 0xf1, 0x21, 0x68, 0xad, 0x85, 0x53, 0xdb, 0x6b,
 0x1e, 0xeb, 0x65, 0x97, 0x03, 0x86, 0x60, 0xde, 0xf3, 0x6c, 0x38, 0x75, 0xe0,
 0x4c, 0x61, 0xbb, 0xbc, 0x62, 0x17, 0xa9, 0xcd, 0x79, 0x3f, 0x21, 0x4e, 0x96,
 0xcb, 0x0e, 0xdc, 0x61, 0x94, 0x30, 0x18, 0x10, 0x6b, 0xd0, 0x1c, 0x10, 0x79,
 0x02, 0x03, 0x01, 0x00, 0x01, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02,
 0x1d, 0x05, 0x00, 0x03, 0x41, 0x00, 0x25, 0x90, 0x53, 0x34, 0xd9, 0x56, 0x41,
 0x5e, 0xdb, 0x7e, 0x01, 0x36, 0xec, 0x27, 0x61, 0x5e, 0xb7, 0x4d, 0x90, 0x66,
 0xa2, 0xe1, 0x9d, 0x58, 0x76, 0xd4, 0x9c, 0xba, 0x2c, 0x84, 0xc6, 0x83, 0x7a,
 0x22, 0x0d, 0x03, 0x69, 0x32, 0x1a, 0x6d, 0xcb, 0x0c, 0x15, 0xb3, 0x6b, 0xc7,
 0x0a, 0x8c, 0xb4, 0x5c, 0x34, 0x78, 0xe0, 0x3c, 0x9c, 0xe9, 0xf3, 0x30, 0x9f,
 0xa8, 0x76, 0x57, 0x92, 0x36 };

static BOOL (WINAPI *pCertCreateCertificateChainEngine)(PCERT_CHAIN_ENGINE_CONFIG,HCERTCHAINENGINE*);
static BOOL (WINAPI *pCertGetCertificateChain)(HCERTCHAINENGINE,PCCERT_CONTEXT,LPFILETIME,HCERTSTORE,PCERT_CHAIN_PARA,DWORD,LPVOID,PCCERT_CHAIN_CONTEXT*);
static VOID (WINAPI *pCertFreeCertificateChain)(PCCERT_CHAIN_CONTEXT);
static VOID (WINAPI *pCertFreeCertificateChainEngine)(HCERTCHAINENGINE);
static BOOL (WINAPI *pCertVerifyCertificateChainPolicy)(LPCSTR,PCCERT_CHAIN_CONTEXT,PCERT_CHAIN_POLICY_PARA,PCERT_CHAIN_POLICY_STATUS);

#define IS_INTOID(x)    (((ULONG_PTR)(x) >> 16) == 0)

typedef struct _CERT_CHAIN_ENGINE_CONFIG_NO_EXCLUSIVE_ROOT
{
    DWORD       cbSize;
    HCERTSTORE  hRestrictedRoot;
    HCERTSTORE  hRestrictedTrust;
    HCERTSTORE  hRestrictedOther;
    DWORD       cAdditionalStore;
    HCERTSTORE *rghAdditionalStore;
    DWORD       dwFlags;
    DWORD       dwUrlRetrievalTimeout;
    DWORD       MaximumCachedCertificates;
    DWORD       CycleDetectionModulus;
} CERT_CHAIN_ENGINE_CONFIG_NO_EXCLUSIVE_ROOT;

static void testCreateCertChainEngine(void)
{
    BOOL ret;
    CERT_CHAIN_ENGINE_CONFIG_NO_EXCLUSIVE_ROOT config = { 0 };
    CERT_CHAIN_ENGINE_CONFIG *pConfig = (CERT_CHAIN_ENGINE_CONFIG *)&config;
    HCERTCHAINENGINE engine;
    HCERTSTORE store;

    if (!pCertCreateCertificateChainEngine || !pCertFreeCertificateChainEngine)
    {
        win_skip("Cert*CertificateChainEngine() functions are not available\n");
        return;
    }

    /* Crash
    ret = pCertCreateCertificateChainEngine(NULL, NULL);
    ret = pCertCreateCertificateChainEngine(NULL, &engine);
    ret = pCertCreateCertificateChainEngine(pConfig, NULL);
     */
    ret = pCertCreateCertificateChainEngine(pConfig, &engine);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08lx\n", GetLastError());
    /* Crashes
    config.cbSize = sizeof(config);
    ret = pCertCreateCertificateChainEngine(pConfig, NULL);
     */
    config.cbSize = sizeof(config);
    ret = pCertCreateCertificateChainEngine(pConfig, &engine);
    ok(ret, "CertCreateCertificateChainEngine failed: %08lx\n", GetLastError());
    pCertFreeCertificateChainEngine(engine);
    config.dwFlags = 0xff000000;
    ret = pCertCreateCertificateChainEngine(pConfig, &engine);
    ok(ret, "CertCreateCertificateChainEngine failed: %08lx\n", GetLastError());
    pCertFreeCertificateChainEngine(engine);

    /* Creating a cert with no root certs at all is allowed.. */
    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    config.hRestrictedRoot = store;
    ret = pCertCreateCertificateChainEngine(pConfig, &engine);
    ok(ret, "CertCreateCertificateChainEngine failed: %08lx\n", GetLastError());
    pCertFreeCertificateChainEngine(engine);

    /* but creating one with a restricted root with a cert that isn't a member
     * of the Root store isn't allowed.
     */
    CertAddEncodedCertificateToStore(store, X509_ASN_ENCODING, selfSignedCert,
     sizeof(selfSignedCert), CERT_STORE_ADD_ALWAYS, NULL);
    ret = pCertCreateCertificateChainEngine(pConfig, &engine);
    /* ERROR_FILE_NOT_FOUND used in Windows 10 */
    ok(!ret && ((GetLastError() == CRYPT_E_NOT_FOUND) ||
                (GetLastError() == ERROR_FILE_NOT_FOUND)),
        "Expected CRYPT_E_NOT_FOUND or ERROR_FILE_NOT_FOUND, got %08lx\n", GetLastError());

    CertCloseStore(store, 0);
}

static const BYTE bigCert[] = { 0x30, 0x7a, 0x02, 0x01, 0x01, 0x30, 0x02, 0x06,
 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x22,
 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30,
 0x30, 0x30, 0x30, 0x5a, 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30,
 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x15, 0x31, 0x13, 0x30,
 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20,
 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x07, 0x30, 0x02, 0x06, 0x00, 0x03, 0x01,
 0x00, 0xa3, 0x16, 0x30, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01 };
static const BYTE geotrust_global_ca[]={
0x30,0x82,0x03,0x54,0x30,0x82,0x02,0x3c,0xa0,0x03,0x02,0x01,0x02,0x02,0x03,
0x02,0x34,0x56,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
0x05,0x05,0x00,0x30,0x42,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,
0x02,0x55,0x53,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x0a,0x13,0x0d,0x47,
0x65,0x6f,0x54,0x72,0x75,0x73,0x74,0x20,0x49,0x6e,0x63,0x2e,0x31,0x1b,0x30,
0x19,0x06,0x03,0x55,0x04,0x03,0x13,0x12,0x47,0x65,0x6f,0x54,0x72,0x75,0x73,
0x74,0x20,0x47,0x6c,0x6f,0x62,0x61,0x6c,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,
0x30,0x32,0x30,0x35,0x32,0x31,0x30,0x34,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,
0x32,0x32,0x30,0x35,0x32,0x31,0x30,0x34,0x30,0x30,0x30,0x30,0x5a,0x30,0x42,
0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x16,
0x30,0x14,0x06,0x03,0x55,0x04,0x0a,0x13,0x0d,0x47,0x65,0x6f,0x54,0x72,0x75,
0x73,0x74,0x20,0x49,0x6e,0x63,0x2e,0x31,0x1b,0x30,0x19,0x06,0x03,0x55,0x04,
0x03,0x13,0x12,0x47,0x65,0x6f,0x54,0x72,0x75,0x73,0x74,0x20,0x47,0x6c,0x6f,
0x62,0x61,0x6c,0x20,0x43,0x41,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,
0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xda,0xcc,0x18,0x63,0x30,0xfd,
0xf4,0x17,0x23,0x1a,0x56,0x7e,0x5b,0xdf,0x3c,0x6c,0x38,0xe4,0x71,0xb7,0x78,
0x91,0xd4,0xbc,0xa1,0xd8,0x4c,0xf8,0xa8,0x43,0xb6,0x03,0xe9,0x4d,0x21,0x07,
0x08,0x88,0xda,0x58,0x2f,0x66,0x39,0x29,0xbd,0x05,0x78,0x8b,0x9d,0x38,0xe8,
0x05,0xb7,0x6a,0x7e,0x71,0xa4,0xe6,0xc4,0x60,0xa6,0xb0,0xef,0x80,0xe4,0x89,
0x28,0x0f,0x9e,0x25,0xd6,0xed,0x83,0xf3,0xad,0xa6,0x91,0xc7,0x98,0xc9,0x42,
0x18,0x35,0x14,0x9d,0xad,0x98,0x46,0x92,0x2e,0x4f,0xca,0xf1,0x87,0x43,0xc1,
0x16,0x95,0x57,0x2d,0x50,0xef,0x89,0x2d,0x80,0x7a,0x57,0xad,0xf2,0xee,0x5f,
0x6b,0xd2,0x00,0x8d,0xb9,0x14,0xf8,0x14,0x15,0x35,0xd9,0xc0,0x46,0xa3,0x7b,
0x72,0xc8,0x91,0xbf,0xc9,0x55,0x2b,0xcd,0xd0,0x97,0x3e,0x9c,0x26,0x64,0xcc,
0xdf,0xce,0x83,0x19,0x71,0xca,0x4e,0xe6,0xd4,0xd5,0x7b,0xa9,0x19,0xcd,0x55,
0xde,0xc8,0xec,0xd2,0x5e,0x38,0x53,0xe5,0x5c,0x4f,0x8c,0x2d,0xfe,0x50,0x23,
0x36,0xfc,0x66,0xe6,0xcb,0x8e,0xa4,0x39,0x19,0x00,0xb7,0x95,0x02,0x39,0x91,
0x0b,0x0e,0xfe,0x38,0x2e,0xd1,0x1d,0x05,0x9a,0xf6,0x4d,0x3e,0x6f,0x0f,0x07,
0x1d,0xaf,0x2c,0x1e,0x8f,0x60,0x39,0xe2,0xfa,0x36,0x53,0x13,0x39,0xd4,0x5e,
0x26,0x2b,0xdb,0x3d,0xa8,0x14,0xbd,0x32,0xeb,0x18,0x03,0x28,0x52,0x04,0x71,
0xe5,0xab,0x33,0x3d,0xe1,0x38,0xbb,0x07,0x36,0x84,0x62,0x9c,0x79,0xea,0x16,
0x30,0xf4,0x5f,0xc0,0x2b,0xe8,0x71,0x6b,0xe4,0xf9,0x02,0x03,0x01,0x00,0x01,
0xa3,0x53,0x30,0x51,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,
0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,
0x04,0x14,0xc0,0x7a,0x98,0x68,0x8d,0x89,0xfb,0xab,0x05,0x64,0x0c,0x11,0x7d,
0xaa,0x7d,0x65,0xb8,0xca,0xcc,0x4e,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,
0x18,0x30,0x16,0x80,0x14,0xc0,0x7a,0x98,0x68,0x8d,0x89,0xfb,0xab,0x05,0x64,
0x0c,0x11,0x7d,0xaa,0x7d,0x65,0xb8,0xca,0xcc,0x4e,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,
0x35,0xe3,0x29,0x6a,0xe5,0x2f,0x5d,0x54,0x8e,0x29,0x50,0x94,0x9f,0x99,0x1a,
0x14,0xe4,0x8f,0x78,0x2a,0x62,0x94,0xa2,0x27,0x67,0x9e,0xd0,0xcf,0x1a,0x5e,
0x47,0xe9,0xc1,0xb2,0xa4,0xcf,0xdd,0x41,0x1a,0x05,0x4e,0x9b,0x4b,0xee,0x4a,
0x6f,0x55,0x52,0xb3,0x24,0xa1,0x37,0x0a,0xeb,0x64,0x76,0x2a,0x2e,0x2c,0xf3,
0xfd,0x3b,0x75,0x90,0xbf,0xfa,0x71,0xd8,0xc7,0x3d,0x37,0xd2,0xb5,0x05,0x95,
0x62,0xb9,0xa6,0xde,0x89,0x3d,0x36,0x7b,0x38,0x77,0x48,0x97,0xac,0xa6,0x20,
0x8f,0x2e,0xa6,0xc9,0x0c,0xc2,0xb2,0x99,0x45,0x00,0xc7,0xce,0x11,0x51,0x22,
0x22,0xe0,0xa5,0xea,0xb6,0x15,0x48,0x09,0x64,0xea,0x5e,0x4f,0x74,0xf7,0x05,
0x3e,0xc7,0x8a,0x52,0x0c,0xdb,0x15,0xb4,0xbd,0x6d,0x9b,0xe5,0xc6,0xb1,0x54,
0x68,0xa9,0xe3,0x69,0x90,0xb6,0x9a,0xa5,0x0f,0xb8,0xb9,0x3f,0x20,0x7d,0xae,
0x4a,0xb5,0xb8,0x9c,0xe4,0x1d,0xb6,0xab,0xe6,0x94,0xa5,0xc1,0xc7,0x83,0xad,
0xdb,0xf5,0x27,0x87,0x0e,0x04,0x6c,0xd5,0xff,0xdd,0xa0,0x5d,0xed,0x87,0x52,
0xb7,0x2b,0x15,0x02,0xae,0x39,0xa6,0x6a,0x74,0xe9,0xda,0xc4,0xe7,0xbc,0x4d,
0x34,0x1e,0xa9,0x5c,0x4d,0x33,0x5f,0x92,0x09,0x2f,0x88,0x66,0x5d,0x77,0x97,
0xc7,0x1d,0x76,0x13,0xa9,0xd5,0xe5,0xf1,0x16,0x09,0x11,0x35,0xd5,0xac,0xdb,
0x24,0x71,0x70,0x2c,0x98,0x56,0x0b,0xd9,0x17,0xb4,0xd1,0xe3,0x51,0x2b,0x5e,
0x75,0xe8,0xd5,0xd0,0xdc,0x4f,0x34,0xed,0xc2,0x05,0x66,0x80,0xa1,0xcb,0xe6,
0x33
};
/* Google Internet Authority G2 */
static const BYTE google_internet_authority[]={
0x30,0x82,0x03,0xf0,0x30,0x82,0x02,0xd8,0xa0,0x03,0x02,0x01,0x02,0x02,0x03,
0x02,0x3a,0x92,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
0x0b,0x05,0x00,0x30,0x42,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,
0x02,0x55,0x53,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x0a,0x13,0x0d,0x47,
0x65,0x6f,0x54,0x72,0x75,0x73,0x74,0x20,0x49,0x6e,0x63,0x2e,0x31,0x1b,0x30,
0x19,0x06,0x03,0x55,0x04,0x03,0x13,0x12,0x47,0x65,0x6f,0x54,0x72,0x75,0x73,
0x74,0x20,0x47,0x6c,0x6f,0x62,0x61,0x6c,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,
0x31,0x35,0x30,0x34,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,
0x31,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x49,
0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,
0x30,0x11,0x06,0x03,0x55,0x04,0x0a,0x13,0x0a,0x47,0x6f,0x6f,0x67,0x6c,0x65,
0x20,0x49,0x6e,0x63,0x31,0x25,0x30,0x23,0x06,0x03,0x55,0x04,0x03,0x13,0x1c,
0x47,0x6f,0x6f,0x67,0x6c,0x65,0x20,0x49,0x6e,0x74,0x65,0x72,0x6e,0x65,0x74,
0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x20,0x47,0x32,0x30,0x82,
0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,
0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,
0x00,0x9c,0x2a,0x04,0x77,0x5c,0xd8,0x50,0x91,0x3a,0x06,0xa3,0x82,0xe0,0xd8,
0x50,0x48,0xbc,0x89,0x3f,0xf1,0x19,0x70,0x1a,0x88,0x46,0x7e,0xe0,0x8f,0xc5,
0xf1,0x89,0xce,0x21,0xee,0x5a,0xfe,0x61,0x0d,0xb7,0x32,0x44,0x89,0xa0,0x74,
0x0b,0x53,0x4f,0x55,0xa4,0xce,0x82,0x62,0x95,0xee,0xeb,0x59,0x5f,0xc6,0xe1,
0x05,0x80,0x12,0xc4,0x5e,0x94,0x3f,0xbc,0x5b,0x48,0x38,0xf4,0x53,0xf7,0x24,
0xe6,0xfb,0x91,0xe9,0x15,0xc4,0xcf,0xf4,0x53,0x0d,0xf4,0x4a,0xfc,0x9f,0x54,
0xde,0x7d,0xbe,0xa0,0x6b,0x6f,0x87,0xc0,0xd0,0x50,0x1f,0x28,0x30,0x03,0x40,
0xda,0x08,0x73,0x51,0x6c,0x7f,0xff,0x3a,0x3c,0xa7,0x37,0x06,0x8e,0xbd,0x4b,
0x11,0x04,0xeb,0x7d,0x24,0xde,0xe6,0xf9,0xfc,0x31,0x71,0xfb,0x94,0xd5,0x60,
0xf3,0x2e,0x4a,0xaf,0x42,0xd2,0xcb,0xea,0xc4,0x6a,0x1a,0xb2,0xcc,0x53,0xdd,
0x15,0x4b,0x8b,0x1f,0xc8,0x19,0x61,0x1f,0xcd,0x9d,0xa8,0x3e,0x63,0x2b,0x84,
0x35,0x69,0x65,0x84,0xc8,0x19,0xc5,0x46,0x22,0xf8,0x53,0x95,0xbe,0xe3,0x80,
0x4a,0x10,0xc6,0x2a,0xec,0xba,0x97,0x20,0x11,0xc7,0x39,0x99,0x10,0x04,0xa0,
0xf0,0x61,0x7a,0x95,0x25,0x8c,0x4e,0x52,0x75,0xe2,0xb6,0xed,0x08,0xca,0x14,
0xfc,0xce,0x22,0x6a,0xb3,0x4e,0xcf,0x46,0x03,0x97,0x97,0x03,0x7e,0xc0,0xb1,
0xde,0x7b,0xaf,0x45,0x33,0xcf,0xba,0x3e,0x71,0xb7,0xde,0xf4,0x25,0x25,0xc2,
0x0d,0x35,0x89,0x9d,0x9d,0xfb,0x0e,0x11,0x79,0x89,0x1e,0x37,0xc5,0xaf,0x8e,
0x72,0x69,0x02,0x03,0x01,0x00,0x01,0xa3,0x81,0xe7,0x30,0x81,0xe4,0x30,0x1f,
0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0xc0,0x7a,0x98,0x68,
0x8d,0x89,0xfb,0xab,0x05,0x64,0x0c,0x11,0x7d,0xaa,0x7d,0x65,0xb8,0xca,0xcc,
0x4e,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x4a,0xdd,0x06,
0x16,0x1b,0xbc,0xf6,0x68,0xb5,0x76,0xf5,0x81,0xb6,0xbb,0x62,0x1a,0xba,0x5a,
0x81,0x2f,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,
0x02,0x01,0x06,0x30,0x2e,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,
0x04,0x22,0x30,0x20,0x30,0x1e,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,
0x01,0x86,0x12,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x67,0x2e,0x73,0x79,0x6d,
0x63,0x64,0x2e,0x63,0x6f,0x6d,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,
0xff,0x04,0x08,0x30,0x06,0x01,0x01,0xff,0x02,0x01,0x00,0x30,0x35,0x06,0x03,
0x55,0x1d,0x1f,0x04,0x2e,0x30,0x2c,0x30,0x2a,0xa0,0x28,0xa0,0x26,0x86,0x24,
0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x67,0x2e,0x73,0x79,0x6d,0x63,0x62,0x2e,
0x63,0x6f,0x6d,0x2f,0x63,0x72,0x6c,0x73,0x2f,0x67,0x74,0x67,0x6c,0x6f,0x62,
0x61,0x6c,0x2e,0x63,0x72,0x6c,0x30,0x17,0x06,0x03,0x55,0x1d,0x20,0x04,0x10,
0x30,0x0e,0x30,0x0c,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0xd6,0x79,0x02,0x05,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,
0x00,0x03,0x82,0x01,0x01,0x00,0x08,0x4e,0x04,0xa7,0x80,0x7f,0x10,0x16,0x43,
0x5e,0x02,0xad,0xd7,0x42,0x80,0xf4,0xb0,0x8e,0xd2,0xae,0xb3,0xeb,0x11,0x7d,
0x90,0x84,0x18,0x7d,0xe7,0x90,0x15,0xfb,0x49,0x7f,0xa8,0x99,0x05,0x91,0xbb,
0x7a,0xc9,0xd6,0x3c,0x37,0x18,0x09,0x9a,0xb6,0xc7,0x92,0x20,0x07,0x35,0x33,
0x09,0xe4,0x28,0x63,0x72,0x0d,0xb4,0xe0,0x32,0x9c,0x87,0x98,0xc4,0x1b,0x76,
0x89,0x67,0xc1,0x50,0x58,0xb0,0x13,0xaa,0x13,0x1a,0x1b,0x32,0xa5,0xbe,0xea,
0x11,0x95,0x4c,0x48,0x63,0x49,0xe9,0x99,0x5d,0x20,0x37,0xcc,0xfe,0x2a,0x69,
0x51,0x16,0x95,0x4b,0xa9,0xde,0x49,0x82,0xc0,0x10,0x70,0xf4,0x2c,0xf3,0xec,
0xbc,0x24,0x24,0xd0,0x4e,0xac,0xa5,0xd9,0x5e,0x1e,0x6d,0x92,0xc1,0xa7,0xac,
0x48,0x35,0x81,0xf9,0xe5,0xe4,0x9c,0x65,0x69,0xcd,0x87,0xa4,0x41,0x50,0x3f,
0x2e,0x57,0xa5,0x91,0x51,0x12,0x58,0x0e,0x8c,0x09,0xa1,0xac,0x7a,0xa4,0x12,
0xa5,0x27,0xf3,0x9a,0x10,0x97,0x7d,0x55,0x03,0x06,0xf7,0x66,0x58,0x5f,0x5f,
0x64,0xe1,0xab,0x5d,0x6d,0xa5,0x39,0x48,0x75,0x98,0x4c,0x29,0x5a,0x3a,0x8d,
0xd3,0x2b,0xca,0x9c,0x55,0x04,0xbf,0xf4,0xe6,0x14,0xd5,0x80,0xac,0x26,0xed,
0x17,0x89,0xa6,0x93,0x6c,0x5c,0xa4,0xcc,0xb8,0xf0,0x66,0x8e,0x64,0xe3,0x7d,
0x9a,0xe2,0x00,0xb3,0x49,0xc7,0xe4,0x0a,0xaa,0xdd,0x5b,0x83,0xc7,0x70,0x90,
0x46,0x4e,0xbe,0xd0,0xdb,0x59,0x96,0x6c,0x2e,0xf5,0x16,0x36,0xde,0x71,0xcc,
0x01,0xc2,0x12,0xc1,0x21,0xc6,0x16
};
/* *.google.com cert */
static const BYTE google_com[]={
0x30,0x82,0x07,0x2f,0x30,0x82,0x06,0x17,0xa0,0x03,0x02,0x01,0x02,0x02,0x08,
0x73,0x73,0x50,0xab,0x51,0x66,0x4b,0x73,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x49,0x31,0x0b,0x30,0x09,0x06,
0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,
0x04,0x0a,0x13,0x0a,0x47,0x6f,0x6f,0x67,0x6c,0x65,0x20,0x49,0x6e,0x63,0x31,
0x25,0x30,0x23,0x06,0x03,0x55,0x04,0x03,0x13,0x1c,0x47,0x6f,0x6f,0x67,0x6c,
0x65,0x20,0x49,0x6e,0x74,0x65,0x72,0x6e,0x65,0x74,0x20,0x41,0x75,0x74,0x68,
0x6f,0x72,0x69,0x74,0x79,0x20,0x47,0x32,0x30,0x1e,0x17,0x0d,0x31,0x36,0x31,
0x31,0x31,0x30,0x31,0x35,0x34,0x35,0x35,0x33,0x5a,0x17,0x0d,0x31,0x37,0x30,
0x32,0x30,0x32,0x31,0x35,0x33,0x31,0x30,0x30,0x5a,0x30,0x66,0x31,0x0b,0x30,
0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,
0x03,0x55,0x04,0x08,0x0c,0x0a,0x43,0x61,0x6c,0x69,0x66,0x6f,0x72,0x6e,0x69,
0x61,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x07,0x0c,0x0d,0x4d,0x6f,0x75,
0x6e,0x74,0x61,0x69,0x6e,0x20,0x56,0x69,0x65,0x77,0x31,0x13,0x30,0x11,0x06,
0x03,0x55,0x04,0x0a,0x0c,0x0a,0x47,0x6f,0x6f,0x67,0x6c,0x65,0x20,0x49,0x6e,
0x63,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x0c,0x0c,0x2a,0x2e,0x67,
0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x30,0x59,0x30,0x13,0x06,0x07,
0x2a,0x86,0x48,0xce,0x3d,0x02,0x01,0x06,0x08,0x2a,0x86,0x48,0xce,0x3d,0x03,
0x01,0x07,0x03,0x42,0x00,0x04,0xe6,0xec,0xdc,0xce,0x7e,0x73,0xe8,0x34,0x4f,
0x1a,0xe1,0x29,0xf6,0x6d,0xe1,0xd4,0x63,0xc6,0x5f,0xcd,0xa7,0x0e,0x91,0xe3,
0x6a,0x41,0xf6,0x64,0x30,0xfb,0x01,0xec,0x98,0xa2,0xb9,0xfe,0x63,0xef,0x07,
0xec,0x09,0x50,0x99,0x0e,0x91,0xf9,0xca,0x56,0x16,0xdb,0x1c,0x10,0xe8,0x51,
0x84,0xc7,0xb0,0x62,0x05,0x52,0x09,0xfc,0xce,0x1c,0xa3,0x82,0x04,0xc7,0x30,
0x82,0x04,0xc3,0x30,0x1d,0x06,0x03,0x55,0x1d,0x25,0x04,0x16,0x30,0x14,0x06,
0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x01,0x06,0x08,0x2b,0x06,0x01,0x05,
0x05,0x07,0x03,0x02,0x30,0x82,0x03,0x86,0x06,0x03,0x55,0x1d,0x11,0x04,0x82,
0x03,0x7d,0x30,0x82,0x03,0x79,0x82,0x0c,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,
0x65,0x2e,0x63,0x6f,0x6d,0x82,0x0d,0x2a,0x2e,0x61,0x6e,0x64,0x72,0x6f,0x69,
0x64,0x2e,0x63,0x6f,0x6d,0x82,0x16,0x2a,0x2e,0x61,0x70,0x70,0x65,0x6e,0x67,
0x69,0x6e,0x65,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x82,
0x12,0x2a,0x2e,0x63,0x6c,0x6f,0x75,0x64,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,
0x2e,0x63,0x6f,0x6d,0x82,0x16,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2d,
0x61,0x6e,0x61,0x6c,0x79,0x74,0x69,0x63,0x73,0x2e,0x63,0x6f,0x6d,0x82,0x0b,
0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x61,0x82,0x0b,0x2a,0x2e,
0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6c,0x82,0x0e,0x2a,0x2e,0x67,0x6f,
0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x2e,0x69,0x6e,0x82,0x0e,0x2a,0x2e,0x67,
0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x2e,0x6a,0x70,0x82,0x0e,0x2a,0x2e,
0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x2e,0x75,0x6b,0x82,0x0f,0x2a,
0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x2e,0x61,0x72,0x82,
0x0f,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x2e,0x61,
0x75,0x82,0x0f,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,
0x2e,0x62,0x72,0x82,0x0f,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,
0x6f,0x6d,0x2e,0x63,0x6f,0x82,0x0f,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,
0x2e,0x63,0x6f,0x6d,0x2e,0x6d,0x78,0x82,0x0f,0x2a,0x2e,0x67,0x6f,0x6f,0x67,
0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x2e,0x74,0x72,0x82,0x0f,0x2a,0x2e,0x67,0x6f,
0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x2e,0x76,0x6e,0x82,0x0b,0x2a,0x2e,
0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x64,0x65,0x82,0x0b,0x2a,0x2e,0x67,0x6f,
0x6f,0x67,0x6c,0x65,0x2e,0x65,0x73,0x82,0x0b,0x2a,0x2e,0x67,0x6f,0x6f,0x67,
0x6c,0x65,0x2e,0x66,0x72,0x82,0x0b,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,
0x2e,0x68,0x75,0x82,0x0b,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x69,
0x74,0x82,0x0b,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x6e,0x6c,0x82,
0x0b,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x70,0x6c,0x82,0x0b,0x2a,
0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x70,0x74,0x82,0x12,0x2a,0x2e,0x67,
0x6f,0x6f,0x67,0x6c,0x65,0x61,0x64,0x61,0x70,0x69,0x73,0x2e,0x63,0x6f,0x6d,
0x82,0x0f,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x61,0x70,0x69,0x73,0x2e,
0x63,0x6e,0x82,0x14,0x2a,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x63,0x6f,0x6d,
0x6d,0x65,0x72,0x63,0x65,0x2e,0x63,0x6f,0x6d,0x82,0x11,0x2a,0x2e,0x67,0x6f,
0x6f,0x67,0x6c,0x65,0x76,0x69,0x64,0x65,0x6f,0x2e,0x63,0x6f,0x6d,0x82,0x0c,
0x2a,0x2e,0x67,0x73,0x74,0x61,0x74,0x69,0x63,0x2e,0x63,0x6e,0x82,0x0d,0x2a,
0x2e,0x67,0x73,0x74,0x61,0x74,0x69,0x63,0x2e,0x63,0x6f,0x6d,0x82,0x0a,0x2a,
0x2e,0x67,0x76,0x74,0x31,0x2e,0x63,0x6f,0x6d,0x82,0x0a,0x2a,0x2e,0x67,0x76,
0x74,0x32,0x2e,0x63,0x6f,0x6d,0x82,0x14,0x2a,0x2e,0x6d,0x65,0x74,0x72,0x69,
0x63,0x2e,0x67,0x73,0x74,0x61,0x74,0x69,0x63,0x2e,0x63,0x6f,0x6d,0x82,0x0c,
0x2a,0x2e,0x75,0x72,0x63,0x68,0x69,0x6e,0x2e,0x63,0x6f,0x6d,0x82,0x10,0x2a,
0x2e,0x75,0x72,0x6c,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,
0x82,0x16,0x2a,0x2e,0x79,0x6f,0x75,0x74,0x75,0x62,0x65,0x2d,0x6e,0x6f,0x63,
0x6f,0x6f,0x6b,0x69,0x65,0x2e,0x63,0x6f,0x6d,0x82,0x0d,0x2a,0x2e,0x79,0x6f,
0x75,0x74,0x75,0x62,0x65,0x2e,0x63,0x6f,0x6d,0x82,0x16,0x2a,0x2e,0x79,0x6f,
0x75,0x74,0x75,0x62,0x65,0x65,0x64,0x75,0x63,0x61,0x74,0x69,0x6f,0x6e,0x2e,
0x63,0x6f,0x6d,0x82,0x0b,0x2a,0x2e,0x79,0x74,0x69,0x6d,0x67,0x2e,0x63,0x6f,
0x6d,0x82,0x1a,0x61,0x6e,0x64,0x72,0x6f,0x69,0x64,0x2e,0x63,0x6c,0x69,0x65,
0x6e,0x74,0x73,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x82,
0x0b,0x61,0x6e,0x64,0x72,0x6f,0x69,0x64,0x2e,0x63,0x6f,0x6d,0x82,0x1b,0x64,
0x65,0x76,0x65,0x6c,0x6f,0x70,0x65,0x72,0x2e,0x61,0x6e,0x64,0x72,0x6f,0x69,
0x64,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6e,0x82,0x04,0x67,0x2e,
0x63,0x6f,0x82,0x06,0x67,0x6f,0x6f,0x2e,0x67,0x6c,0x82,0x14,0x67,0x6f,0x6f,
0x67,0x6c,0x65,0x2d,0x61,0x6e,0x61,0x6c,0x79,0x74,0x69,0x63,0x73,0x2e,0x63,
0x6f,0x6d,0x82,0x0a,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x82,
0x12,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x63,0x6f,0x6d,0x6d,0x65,0x72,0x63,0x65,
0x2e,0x63,0x6f,0x6d,0x82,0x19,0x70,0x6f,0x6c,0x69,0x63,0x79,0x2e,0x6d,0x74,
0x61,0x2d,0x73,0x74,0x73,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,
0x6d,0x82,0x0a,0x75,0x72,0x63,0x68,0x69,0x6e,0x2e,0x63,0x6f,0x6d,0x82,0x0a,
0x77,0x77,0x77,0x2e,0x67,0x6f,0x6f,0x2e,0x67,0x6c,0x82,0x08,0x79,0x6f,0x75,
0x74,0x75,0x2e,0x62,0x65,0x82,0x0b,0x79,0x6f,0x75,0x74,0x75,0x62,0x65,0x2e,
0x63,0x6f,0x6d,0x82,0x14,0x79,0x6f,0x75,0x74,0x75,0x62,0x65,0x65,0x64,0x75,
0x63,0x61,0x74,0x69,0x6f,0x6e,0x2e,0x63,0x6f,0x6d,0x30,0x0b,0x06,0x03,0x55,
0x1d,0x0f,0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x68,0x06,0x08,0x2b,0x06,0x01,
0x05,0x05,0x07,0x01,0x01,0x04,0x5c,0x30,0x5a,0x30,0x2b,0x06,0x08,0x2b,0x06,
0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x1f,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,
0x70,0x6b,0x69,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x2f,
0x47,0x49,0x41,0x47,0x32,0x2e,0x63,0x72,0x74,0x30,0x2b,0x06,0x08,0x2b,0x06,
0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x1f,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,
0x63,0x6c,0x69,0x65,0x6e,0x74,0x73,0x31,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,
0x2e,0x63,0x6f,0x6d,0x2f,0x6f,0x63,0x73,0x70,0x30,0x1d,0x06,0x03,0x55,0x1d,
0x0e,0x04,0x16,0x04,0x14,0xe8,0xf5,0x4a,0xed,0x91,0xb9,0xc8,0x2d,0x5d,0xfb,
0x47,0x65,0x53,0x1b,0x47,0x6f,0xe6,0xe9,0x63,0xa3,0x30,0x0c,0x06,0x03,0x55,
0x1d,0x13,0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x1f,0x06,0x03,0x55,0x1d,
0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x4a,0xdd,0x06,0x16,0x1b,0xbc,0xf6,0x68,
0xb5,0x76,0xf5,0x81,0xb6,0xbb,0x62,0x1a,0xba,0x5a,0x81,0x2f,0x30,0x21,0x06,
0x03,0x55,0x1d,0x20,0x04,0x1a,0x30,0x18,0x30,0x0c,0x06,0x0a,0x2b,0x06,0x01,
0x04,0x01,0xd6,0x79,0x02,0x05,0x01,0x30,0x08,0x06,0x06,0x67,0x81,0x0c,0x01,
0x02,0x02,0x30,0x30,0x06,0x03,0x55,0x1d,0x1f,0x04,0x29,0x30,0x27,0x30,0x25,
0xa0,0x23,0xa0,0x21,0x86,0x1f,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x70,0x6b,
0x69,0x2e,0x67,0x6f,0x6f,0x67,0x6c,0x65,0x2e,0x63,0x6f,0x6d,0x2f,0x47,0x49,
0x41,0x47,0x32,0x2e,0x63,0x72,0x6c,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x0c,0x58,0x93,
0x06,0x54,0x42,0x57,0x48,0xe4,0xfa,0x57,0xf0,0x5b,0x96,0x94,0xfc,0xf3,0x3d,
0xe3,0xba,0x73,0x71,0xe9,0xe7,0x56,0x3d,0x21,0x70,0x82,0x4d,0x59,0x7e,0x85,
0xbd,0x87,0x36,0xb6,0x43,0x4a,0xe7,0x6f,0x0a,0x95,0xbf,0xde,0x35,0x1c,0x60,
0xbf,0xd1,0x60,0x41,0x98,0x6d,0xe9,0xea,0xcc,0xa1,0x85,0x91,0x94,0x7e,0x71,
0xa3,0x40,0x40,0x5a,0xbb,0x7d,0x49,0xfd,0x39,0xf9,0xf1,0xe5,0x2f,0xee,0xcf,
0xcf,0x65,0xca,0xc1,0xb4,0xf1,0xb5,0xae,0xf3,0x6b,0x53,0x7d,0x8d,0x2b,0x60,
0x1c,0x2a,0xb3,0xa5,0xe0,0x55,0x11,0x8d,0x13,0xd8,0x91,0x5a,0xb0,0xcc,0x91,
0xa8,0xbc,0x2b,0x6b,0x63,0xff,0x14,0xfe,0x00,0x16,0x07,0xfb,0xef,0xae,0xd5,
0xa8,0x13,0x0e,0x59,0xdc,0x68,0x96,0xd5,0x63,0xea,0xf7,0x56,0x0e,0x38,0x5a,
0x57,0xc8,0x5e,0x52,0x95,0xe3,0xab,0x0a,0x33,0xc8,0x61,0xb2,0xdf,0xdb,0x77,
0xea,0x49,0x90,0x3d,0xa8,0xd1,0x56,0x40,0x16,0xa5,0x14,0x5d,0xc3,0xc6,0x09,
0xc2,0xd1,0x03,0xac,0x2e,0x02,0xab,0x88,0x59,0xb2,0x18,0x2b,0xf9,0xfb,0xf5,
0xfc,0x64,0x7c,0x73,0x1c,0xa0,0x05,0xaf,0xc8,0x2f,0x4d,0xb0,0x1b,0x20,0x40,
0xe2,0x8b,0x17,0xb1,0x5b,0x71,0xd0,0x4b,0x5d,0xa4,0x07,0x7b,0xd9,0xbf,0x39,
0x72,0x4e,0x35,0x8f,0xf2,0x13,0x6d,0xb9,0x6b,0x5d,0x02,0x55,0x85,0x4a,0x78,
0xfc,0x76,0x0e,0x65,0xf4,0xd6,0xe0,0x29,0x6e,0x04,0x57,0x22,0x03,0x8e,0x6b,
0x59,0x4f,0x43,0x6a,0x5b,0x36,0xaa,0xa5,0x68,0xb4,0x1a,0x60,0x88
};
/* *.winehq.org cert */
static const BYTE winehq_org[] = {
  0x30, 0x82, 0x05, 0xb5, 0x30, 0x82, 0x04, 0x9d, 0xa0, 0x03, 0x02, 0x01,
  0x02, 0x02, 0x11, 0x00, 0xc9, 0xce, 0x10, 0xbc, 0x5e, 0xcf, 0xf2, 0xfd,
  0xa6, 0x37, 0xb8, 0x51, 0xf1, 0xe4, 0xb9, 0x30, 0x30, 0x0d, 0x06, 0x09,
  0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30,
  0x7a, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
  0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
  0x02, 0x56, 0x41, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07,
  0x13, 0x07, 0x48, 0x65, 0x72, 0x6e, 0x64, 0x6f, 0x6e, 0x31, 0x21, 0x30,
  0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x18, 0x4e, 0x65, 0x74, 0x77,
  0x6f, 0x72, 0x6b, 0x20, 0x53, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e,
  0x73, 0x20, 0x4c, 0x2e, 0x4c, 0x2e, 0x43, 0x2e, 0x31, 0x29, 0x30, 0x27,
  0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x20, 0x4e, 0x65, 0x74, 0x77, 0x6f,
  0x72, 0x6b, 0x20, 0x53, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73,
  0x20, 0x4f, 0x56, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x43,
  0x41, 0x20, 0x32, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x33, 0x30,
  0x36, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x32, 0x30,
  0x30, 0x36, 0x30, 0x37, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30,
  0x81, 0xc8, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
  0x02, 0x55, 0x53, 0x31, 0x0e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x04, 0x11,
  0x13, 0x05, 0x35, 0x35, 0x31, 0x31, 0x34, 0x31, 0x0b, 0x30, 0x09, 0x06,
  0x03, 0x55, 0x04, 0x08, 0x13, 0x02, 0x4d, 0x4e, 0x31, 0x13, 0x30, 0x11,
  0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x0a, 0x53, 0x61, 0x69, 0x6e, 0x74,
  0x20, 0x50, 0x61, 0x75, 0x6c, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
  0x04, 0x09, 0x13, 0x07, 0x53, 0x74, 0x65, 0x20, 0x31, 0x32, 0x30, 0x31,
  0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x09, 0x13, 0x0f, 0x37, 0x30,
  0x30, 0x20, 0x52, 0x61, 0x79, 0x6d, 0x6f, 0x6e, 0x64, 0x20, 0x41, 0x76,
  0x65, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0b,
  0x43, 0x6f, 0x64, 0x65, 0x57, 0x65, 0x61, 0x76, 0x65, 0x72, 0x73, 0x31,
  0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x02, 0x49, 0x54,
  0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x18, 0x53,
  0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 0x4c, 0x69, 0x6e, 0x6b, 0x20, 0x53,
  0x53, 0x4c, 0x20, 0x57, 0x69, 0x6c, 0x64, 0x63, 0x61, 0x72, 0x64, 0x31,
  0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0c, 0x2a, 0x2e,
  0x77, 0x69, 0x6e, 0x65, 0x68, 0x71, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x82,
  0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
  0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82,
  0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xc2, 0x91, 0x35, 0xc7, 0x67,
  0x32, 0xf1, 0x66, 0x12, 0x38, 0x79, 0xdb, 0x02, 0x11, 0x95, 0x89, 0x1b,
  0x4a, 0x6a, 0x63, 0x45, 0x35, 0x58, 0x1b, 0x57, 0x8d, 0xec, 0x19, 0x88,
  0xf5, 0xa3, 0x9e, 0x6b, 0xdb, 0x07, 0xfb, 0x6e, 0x29, 0xe9, 0xa9, 0x14,
  0x07, 0x8c, 0x3a, 0xc1, 0xd7, 0x88, 0xcc, 0xe4, 0x55, 0x79, 0x6e, 0x97,
  0xe9, 0x93, 0x88, 0x9a, 0x89, 0xf9, 0x31, 0x0f, 0x91, 0x61, 0xc2, 0x9c,
  0xae, 0x59, 0x25, 0x93, 0x9a, 0xc8, 0xf9, 0x28, 0x7a, 0x0f, 0x0f, 0x89,
  0x9a, 0xda, 0x46, 0xbe, 0xb2, 0xdc, 0x90, 0x5b, 0xd8, 0x87, 0xda, 0xda,
  0xce, 0x70, 0x70, 0x95, 0x43, 0x51, 0xa0, 0x21, 0x22, 0x54, 0xab, 0xa7,
  0x09, 0xe1, 0x81, 0x09, 0x8c, 0x00, 0x15, 0x30, 0x5a, 0xa7, 0x61, 0x77,
  0xba, 0x48, 0x91, 0xc6, 0x94, 0x90, 0xdd, 0x1a, 0xf4, 0x3f, 0xa6, 0x8a,
  0xb4, 0x2f, 0x98, 0x16, 0x0d, 0x96, 0x9f, 0x0c, 0x96, 0x01, 0x15, 0xbc,
  0x74, 0x7e, 0x5c, 0x4e, 0xe3, 0x95, 0xae, 0x00, 0xf5, 0x5b, 0x88, 0x0f,
  0xef, 0x7c, 0x1f, 0xab, 0x22, 0xfe, 0x95, 0x42, 0x1a, 0xbd, 0xee, 0x23,
  0xca, 0x5b, 0x7f, 0x4a, 0xb0, 0xbe, 0x15, 0x65, 0xe2, 0xce, 0x02, 0xc4,
  0xf6, 0xb8, 0xf3, 0xd6, 0x7e, 0x75, 0x5e, 0x63, 0x47, 0x5d, 0xc1, 0xca,
  0xc3, 0xc4, 0x4a, 0xf5, 0x4d, 0x66, 0x34, 0x03, 0xf8, 0xf4, 0x54, 0x7d,
  0x6b, 0x54, 0x2f, 0x17, 0x7c, 0x39, 0x13, 0x78, 0x09, 0x9e, 0xd0, 0x9c,
  0x51, 0x02, 0x11, 0x9d, 0xa7, 0x53, 0x60, 0xdd, 0x62, 0x2b, 0xee, 0xb0,
  0xd1, 0xf2, 0x1a, 0x5c, 0xdc, 0xa7, 0x58, 0xc2, 0xfa, 0x29, 0xcf, 0xf3,
  0xc9, 0xe9, 0xae, 0xe8, 0x70, 0x60, 0x99, 0xc7, 0x30, 0xdd, 0x1c, 0xda,
  0x32, 0xc0, 0x45, 0xa5, 0x48, 0xc5, 0x22, 0x29, 0x0c, 0x74, 0x25, 0x02,
  0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x01, 0xe5, 0x30, 0x82, 0x01, 0xe1,
  0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
  0x14, 0x20, 0x33, 0xcd, 0xb7, 0x61, 0xf6, 0xa5, 0x86, 0x4f, 0xdc, 0xc9,
  0xd7, 0x73, 0x6a, 0xbc, 0x0a, 0x51, 0x65, 0x98, 0xec, 0x30, 0x1d, 0x06,
  0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xff, 0x4c, 0x0f, 0x46,
  0xb8, 0x3c, 0x5b, 0x72, 0xe4, 0x10, 0x96, 0xbb, 0xa7, 0xbc, 0x91, 0xa2,
  0x63, 0xf6, 0x81, 0x0a, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01,
  0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x05, 0xa0, 0x30, 0x0c, 0x06, 0x03,
  0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x1d,
  0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b,
  0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01,
  0x05, 0x05, 0x07, 0x03, 0x02, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x20,
  0x04, 0x6e, 0x30, 0x6c, 0x30, 0x60, 0x06, 0x0c, 0x2b, 0x06, 0x01, 0x04,
  0x01, 0x86, 0x0e, 0x01, 0x02, 0x01, 0x03, 0x01, 0x30, 0x50, 0x30, 0x4e,
  0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x42,
  0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x6e,
  0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x69,
  0x6f, 0x6e, 0x73, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6c, 0x65, 0x67, 0x61,
  0x6c, 0x2f, 0x53, 0x53, 0x4c, 0x2d, 0x6c, 0x65, 0x67, 0x61, 0x6c, 0x2d,
  0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2d, 0x63,
  0x70, 0x73, 0x2e, 0x6a, 0x73, 0x70, 0x30, 0x08, 0x06, 0x06, 0x67, 0x81,
  0x0c, 0x01, 0x02, 0x02, 0x30, 0x49, 0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04,
  0x42, 0x30, 0x40, 0x30, 0x3e, 0xa0, 0x3c, 0xa0, 0x3a, 0x86, 0x38, 0x68,
  0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x6e, 0x65,
  0x74, 0x73, 0x6f, 0x6c, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
  0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x53, 0x6f, 0x6c, 0x75, 0x74,
  0x69, 0x6f, 0x6e, 0x73, 0x4f, 0x56, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72,
  0x43, 0x41, 0x32, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x7b, 0x06, 0x08, 0x2b,
  0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x6f, 0x30, 0x6d, 0x30,
  0x44, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86,
  0x38, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x74, 0x2e,
  0x6e, 0x65, 0x74, 0x73, 0x6f, 0x6c, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f,
  0x6d, 0x2f, 0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x53, 0x6f, 0x6c,
  0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x4f, 0x56, 0x53, 0x65, 0x72, 0x76,
  0x65, 0x72, 0x43, 0x41, 0x32, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x25, 0x06,
  0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x19, 0x68,
  0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6f, 0x63, 0x73, 0x70, 0x2e, 0x6e,
  0x65, 0x74, 0x73, 0x6f, 0x6c, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d,
  0x30, 0x23, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x1c, 0x30, 0x1a, 0x82,
  0x0c, 0x2a, 0x2e, 0x77, 0x69, 0x6e, 0x65, 0x68, 0x71, 0x2e, 0x6f, 0x72,
  0x67, 0x82, 0x0a, 0x77, 0x69, 0x6e, 0x65, 0x68, 0x71, 0x2e, 0x6f, 0x72,
  0x67, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
  0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x58, 0x90, 0xa4,
  0x5b, 0x56, 0x5f, 0xf6, 0xab, 0x28, 0x6b, 0xc8, 0x17, 0xe6, 0x48, 0x51,
  0x38, 0x51, 0xea, 0xc2, 0x16, 0x07, 0x88, 0x32, 0xa0, 0x75, 0x17, 0xef,
  0xc3, 0xc3, 0x19, 0xce, 0xfb, 0xc9, 0x98, 0x2c, 0x1f, 0x16, 0x94, 0x41,
  0xa0, 0x46, 0xc5, 0x0d, 0x7e, 0x90, 0x40, 0x24, 0xd2, 0xb1, 0xf4, 0xc8,
  0x51, 0x43, 0x21, 0x54, 0xdb, 0xed, 0x47, 0x21, 0x97, 0x37, 0x11, 0xc9,
  0x22, 0xe5, 0xcc, 0xe1, 0x64, 0x5e, 0x3a, 0xd0, 0xbf, 0x95, 0x4a, 0xe9,
  0x3a, 0x32, 0x05, 0xca, 0x9d, 0xb4, 0x8c, 0x7e, 0xce, 0x31, 0x54, 0xb0,
  0x5f, 0xf5, 0xcf, 0xef, 0x59, 0x71, 0x7b, 0x5d, 0xaf, 0xc0, 0x76, 0xe2,
  0xdb, 0x89, 0x03, 0x25, 0x94, 0xc4, 0x55, 0xd4, 0xf8, 0xac, 0xe1, 0xbb,
  0x12, 0xc1, 0x24, 0x97, 0xd0, 0xfb, 0x17, 0x64, 0xae, 0x80, 0xc9, 0x3b,
  0xff, 0x16, 0x20, 0xfd, 0x65, 0x0c, 0x34, 0xdf, 0x7b, 0x11, 0xf6, 0x37,
  0xcb, 0x5c, 0xec, 0x98, 0xef, 0xab, 0xd3, 0x60, 0xaf, 0x6e, 0x1b, 0x0d,
  0x1b, 0xfd, 0xc1, 0x43, 0xf0, 0x8d, 0xb5, 0x44, 0xdf, 0xcc, 0xc3, 0xe7,
  0x6e, 0xb6, 0x51, 0x9e, 0x8b, 0x77, 0x5c, 0x1a, 0x8b, 0xbd, 0xb4, 0xa4,
  0x28, 0xf8, 0x01, 0x41, 0xc5, 0xa1, 0xc0, 0x64, 0x0a, 0xe0, 0x6f, 0x32,
  0x30, 0x2c, 0x06, 0x2a, 0xc4, 0x76, 0x0b, 0x95, 0x8b, 0xbf, 0x14, 0x59,
  0x16, 0xb3, 0xd1, 0xee, 0xc9, 0xef, 0x71, 0xd5, 0x5a, 0x26, 0xb4, 0xb0,
  0xba, 0x86, 0x1f, 0x41, 0x13, 0xd8, 0xff, 0x96, 0x51, 0x7b, 0x2e, 0x8f,
  0x85, 0x21, 0x60, 0xcd, 0x8f, 0x47, 0xe2, 0x48, 0x44, 0x07, 0x38, 0x29,
  0xdd, 0x0a, 0x3a, 0x01, 0xd8, 0x2c, 0x37, 0x8d, 0xc3, 0x0e, 0x68, 0xc5,
  0xd4, 0x24, 0x86, 0x2b, 0x05, 0x8f, 0xf5, 0x08, 0xd7, 0x46, 0x73, 0x59,
  0xb2
};
/* Battle.Net's cert */
static const BYTE battlenet[] = {
0x30,0x82,0x03,0xd8,0x30,0x82,0x02,0xc0,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x1e,0x4c,0xc1,0xf1,0xac,0xbd,0xf3,0xf5,0x96,0x05,0xbd,0x5f,0xbb,0x3f,0x75,
0x6b,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x3c,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,
0x53,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x0a,0x13,0x0c,0x54,0x68,0x61,
0x77,0x74,0x65,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x16,0x30,0x14,0x06,0x03,
0x55,0x04,0x03,0x13,0x0d,0x54,0x68,0x61,0x77,0x74,0x65,0x20,0x53,0x53,0x4c,
0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x31,0x30,0x30,0x38,0x32,0x36,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x31,0x32,0x30,0x39,0x32,0x34,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x71,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,
0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x0c,
0x0a,0x43,0x61,0x6c,0x69,0x66,0x6f,0x72,0x6e,0x69,0x61,0x31,0x0f,0x30,0x0d,
0x06,0x03,0x55,0x04,0x07,0x0c,0x06,0x49,0x72,0x76,0x69,0x6e,0x65,0x31,0x25,
0x30,0x23,0x06,0x03,0x55,0x04,0x0a,0x0c,0x1c,0x42,0x6c,0x69,0x7a,0x7a,0x61,
0x72,0x64,0x20,0x45,0x6e,0x74,0x65,0x72,0x74,0x61,0x69,0x6e,0x6d,0x65,0x6e,
0x74,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,
0x03,0x0c,0x0c,0x2a,0x2e,0x62,0x61,0x74,0x74,0x6c,0x65,0x2e,0x6e,0x65,0x74,
0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,
0x01,0x01,0x00,0xa8,0x27,0x24,0x42,0x24,0xc8,0xe4,0x4e,0xfa,0x12,0x53,0x78,
0x14,0xa9,0xec,0x20,0x2d,0x79,0x07,0x55,0x36,0xad,0x04,0x8b,0xbc,0xd9,0x3b,
0xcc,0x3e,0xae,0xa0,0x3b,0xa1,0x79,0xf9,0x03,0x20,0x3e,0xa6,0x6a,0xeb,0x8c,
0xb1,0x45,0xcb,0x00,0x43,0x76,0x35,0x1b,0x3d,0xc9,0x4b,0xa9,0xc0,0xb6,0x32,
0x88,0xaa,0x4c,0x2c,0x53,0xf8,0xc4,0xcf,0xee,0xee,0xef,0x28,0xdf,0x44,0xfa,
0xa9,0x26,0xf6,0x99,0x7b,0xa4,0x7f,0xe8,0x5c,0x7f,0x59,0x51,0xe1,0x2c,0x57,
0x2f,0x8f,0xb3,0xad,0x7b,0x88,0x50,0xbc,0x76,0xfe,0x03,0xbd,0xfd,0x11,0x5d,
0x6e,0xbc,0x13,0x5b,0xd9,0x2e,0x38,0xc7,0x56,0x89,0x93,0x08,0xa2,0x24,0xbd,
0x1d,0x48,0xd9,0x48,0xce,0x6e,0x12,0x4f,0x10,0x60,0x94,0x54,0xb7,0x6b,0x51,
0xd0,0xdf,0x04,0xa8,0x16,0x39,0xcb,0xa5,0xd9,0xe7,0xb5,0xa9,0x02,0xfa,0xd3,
0xca,0x52,0xe7,0xc8,0x45,0xf8,0x4d,0xbb,0x70,0x1a,0xfd,0xb3,0x7d,0x9c,0x77,
0x8b,0x34,0xbe,0xd2,0xad,0xe7,0x17,0xb0,0x55,0xfa,0x1b,0x3e,0x51,0xcf,0x37,
0xbd,0x29,0x94,0x9f,0x56,0x28,0xd1,0x9d,0xe5,0x56,0xce,0x78,0x61,0x6e,0x8b,
0xae,0x95,0x44,0x3c,0xc8,0x54,0x48,0x78,0x1e,0x4c,0x72,0xff,0x0e,0xb9,0x14,
0x78,0xdc,0x7c,0x2e,0x50,0x05,0xd9,0xd0,0xa5,0x97,0xf7,0xb9,0x45,0x3f,0x7e,
0xdc,0xc6,0x4e,0x64,0x93,0x82,0xb7,0x97,0xcf,0xb1,0x7a,0x04,0xc1,0x1d,0x70,
0xf8,0x6b,0x43,0xc4,0xd0,0xa7,0x03,0x4e,0xc9,0x14,0x90,0x4f,0x05,0xb5,0x11,
0x36,0xc0,0xc2,0xbb,0x02,0x03,0x01,0x00,0x01,0xa3,0x81,0xa0,0x30,0x81,0x9d,
0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,
0x3a,0x06,0x03,0x55,0x1d,0x1f,0x04,0x33,0x30,0x31,0x30,0x2f,0xa0,0x2d,0xa0,
0x2b,0x86,0x29,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x73,0x76,0x72,0x2d,0x6f,
0x76,0x2d,0x63,0x72,0x6c,0x2e,0x74,0x68,0x61,0x77,0x74,0x65,0x2e,0x63,0x6f,
0x6d,0x2f,0x54,0x68,0x61,0x77,0x74,0x65,0x4f,0x56,0x2e,0x63,0x72,0x6c,0x30,
0x1d,0x06,0x03,0x55,0x1d,0x25,0x04,0x16,0x30,0x14,0x06,0x08,0x2b,0x06,0x01,
0x05,0x05,0x07,0x03,0x01,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x02,
0x30,0x32,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x26,0x30,
0x24,0x30,0x22,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x16,
0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,0x74,0x68,0x61,
0x77,0x74,0x65,0x2e,0x63,0x6f,0x6d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x5c,0x44,0xe2,
0x2f,0x50,0x41,0xc8,0x54,0x0a,0xdd,0x2a,0xa2,0xa7,0x62,0x2d,0xc9,0xe8,0xa8,
0xf7,0x53,0x14,0xe1,0x88,0x89,0x81,0x22,0x1b,0x82,0xb5,0xa1,0x75,0xcc,0x91,
0x76,0x30,0x71,0xae,0x56,0x68,0xa9,0x75,0x65,0x7f,0xd8,0xc7,0xae,0x3b,0x68,
0x66,0xd6,0x2d,0x92,0xb7,0x9b,0x28,0x59,0x98,0x89,0x29,0xf9,0x69,0xff,0xff,
0xfa,0x4f,0x04,0x6b,0x96,0x78,0x1d,0xfe,0x67,0x78,0x33,0xd3,0xd5,0x91,0xee,
0xa7,0x36,0xcd,0x86,0x4c,0xc6,0x08,0xf4,0x12,0x4b,0x2b,0xd2,0x95,0x6a,0x87,
0xcd,0xe6,0x2d,0xf3,0xe0,0x8d,0x0c,0x77,0x9d,0xa0,0x2e,0xdc,0xf2,0xc1,0x06,
0xc5,0xb0,0xd5,0xa0,0x00,0xe5,0x0e,0x53,0xad,0x04,0xc4,0xf6,0x6e,0x6b,0x7e,
0x04,0xc2,0xea,0xaa,0xdf,0xe1,0x26,0x4a,0x14,0x33,0x03,0x77,0x15,0x5b,0x3e,
0x41,0x22,0x5d,0xb7,0xaf,0x65,0x2f,0x46,0xbc,0x24,0xd7,0x30,0xe6,0x82,0x7d,
0x2a,0x3b,0x81,0x04,0xa7,0xd5,0x0b,0x61,0x57,0xe0,0x91,0x04,0x6c,0xc6,0x08,
0xbc,0xc0,0x1b,0x26,0x7f,0x69,0x22,0x69,0xd3,0x41,0x4c,0x9d,0x61,0xe0,0xfe,
0x2b,0xd8,0x2e,0xe9,0x2d,0x72,0x30,0x68,0x81,0xa1,0x37,0x06,0xb5,0xdc,0xd3,
0x48,0x65,0x16,0x74,0xfb,0x3c,0xb2,0x70,0xef,0x3d,0xee,0x63,0xea,0x62,0xf5,
0xd2,0xc7,0x48,0x6a,0xb8,0x53,0xcb,0xbe,0x9a,0xeb,0xc1,0x77,0xfb,0x9b,0xec,
0xb8,0x06,0x04,0xaa,0x23,0x2c,0x6d,0x17,0x9e,0xb9,0x6e,0xc9,0xa4,0xde,0x7e,
0x61,0xc4,0xa7,0x45,0x68,0xf6,0x2a,0x57,0xaa,0xad,0xca,0x84,0x03 };
static const BYTE thawte_primary_ca[] = {
0x30,0x82,0x04,0x20,0x30,0x82,0x03,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x34,0x4e,0xd5,0x57,0x20,0xd5,0xed,0xec,0x49,0xf4,0x2f,0xce,0x37,0xdb,0x2b,
0x6d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x81,0xa9,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
0x55,0x53,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x0a,0x13,0x0c,0x74,0x68,
0x61,0x77,0x74,0x65,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x28,0x30,0x26,0x06,
0x03,0x55,0x04,0x0b,0x13,0x1f,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,
0x74,0x69,0x6f,0x6e,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x20,0x44,
0x69,0x76,0x69,0x73,0x69,0x6f,0x6e,0x31,0x38,0x30,0x36,0x06,0x03,0x55,0x04,
0x0b,0x13,0x2f,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x36,0x20,0x74,0x68,0x61,
0x77,0x74,0x65,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x20,0x2d,0x20,0x46,0x6f,0x72,
0x20,0x61,0x75,0x74,0x68,0x6f,0x72,0x69,0x7a,0x65,0x64,0x20,0x75,0x73,0x65,
0x20,0x6f,0x6e,0x6c,0x79,0x31,0x1f,0x30,0x1d,0x06,0x03,0x55,0x04,0x03,0x13,
0x16,0x74,0x68,0x61,0x77,0x74,0x65,0x20,0x50,0x72,0x69,0x6d,0x61,0x72,0x79,
0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x36,0x31,
0x31,0x31,0x37,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x33,0x36,0x30,
0x37,0x31,0x36,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x81,0xa9,0x31,0x0b,
0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x15,0x30,0x13,
0x06,0x03,0x55,0x04,0x0a,0x13,0x0c,0x74,0x68,0x61,0x77,0x74,0x65,0x2c,0x20,
0x49,0x6e,0x63,0x2e,0x31,0x28,0x30,0x26,0x06,0x03,0x55,0x04,0x0b,0x13,0x1f,
0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x20,0x53,
0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x20,0x44,0x69,0x76,0x69,0x73,0x69,0x6f,
0x6e,0x31,0x38,0x30,0x36,0x06,0x03,0x55,0x04,0x0b,0x13,0x2f,0x28,0x63,0x29,
0x20,0x32,0x30,0x30,0x36,0x20,0x74,0x68,0x61,0x77,0x74,0x65,0x2c,0x20,0x49,
0x6e,0x63,0x2e,0x20,0x2d,0x20,0x46,0x6f,0x72,0x20,0x61,0x75,0x74,0x68,0x6f,
0x72,0x69,0x7a,0x65,0x64,0x20,0x75,0x73,0x65,0x20,0x6f,0x6e,0x6c,0x79,0x31,
0x1f,0x30,0x1d,0x06,0x03,0x55,0x04,0x03,0x13,0x16,0x74,0x68,0x61,0x77,0x74,
0x65,0x20,0x50,0x72,0x69,0x6d,0x61,0x72,0x79,0x20,0x52,0x6f,0x6f,0x74,0x20,
0x43,0x41,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,
0x02,0x82,0x01,0x01,0x00,0xac,0xa0,0xf0,0xfb,0x80,0x59,0xd4,0x9c,0xc7,0xa4,
0xcf,0x9d,0xa1,0x59,0x73,0x09,0x10,0x45,0x0c,0x0d,0x2c,0x6e,0x68,0xf1,0x6c,
0x5b,0x48,0x68,0x49,0x59,0x37,0xfc,0x0b,0x33,0x19,0xc2,0x77,0x7f,0xcc,0x10,
0x2d,0x95,0x34,0x1c,0xe6,0xeb,0x4d,0x09,0xa7,0x1c,0xd2,0xb8,0xc9,0x97,0x36,
0x02,0xb7,0x89,0xd4,0x24,0x5f,0x06,0xc0,0xcc,0x44,0x94,0x94,0x8d,0x02,0x62,
0x6f,0xeb,0x5a,0xdd,0x11,0x8d,0x28,0x9a,0x5c,0x84,0x90,0x10,0x7a,0x0d,0xbd,
0x74,0x66,0x2f,0x6a,0x38,0xa0,0xe2,0xd5,0x54,0x44,0xeb,0x1d,0x07,0x9f,0x07,
0xba,0x6f,0xee,0xe9,0xfd,0x4e,0x0b,0x29,0xf5,0x3e,0x84,0xa0,0x01,0xf1,0x9c,
0xab,0xf8,0x1c,0x7e,0x89,0xa4,0xe8,0xa1,0xd8,0x71,0x65,0x0d,0xa3,0x51,0x7b,
0xee,0xbc,0xd2,0x22,0x60,0x0d,0xb9,0x5b,0x9d,0xdf,0xba,0xfc,0x51,0x5b,0x0b,
0xaf,0x98,0xb2,0xe9,0x2e,0xe9,0x04,0xe8,0x62,0x87,0xde,0x2b,0xc8,0xd7,0x4e,
0xc1,0x4c,0x64,0x1e,0xdd,0xcf,0x87,0x58,0xba,0x4a,0x4f,0xca,0x68,0x07,0x1d,
0x1c,0x9d,0x4a,0xc6,0xd5,0x2f,0x91,0xcc,0x7c,0x71,0x72,0x1c,0xc5,0xc0,0x67,
0xeb,0x32,0xfd,0xc9,0x92,0x5c,0x94,0xda,0x85,0xc0,0x9b,0xbf,0x53,0x7d,0x2b,
0x09,0xf4,0x8c,0x9d,0x91,0x1f,0x97,0x6a,0x52,0xcb,0xde,0x09,0x36,0xa4,0x77,
0xd8,0x7b,0x87,0x50,0x44,0xd5,0x3e,0x6e,0x29,0x69,0xfb,0x39,0x49,0x26,0x1e,
0x09,0xa5,0x80,0x7b,0x40,0x2d,0xeb,0xe8,0x27,0x85,0xc9,0xfe,0x61,0xfd,0x7e,
0xe6,0x7c,0x97,0x1d,0xd5,0x9d,0x02,0x03,0x01,0x00,0x01,0xa3,0x42,0x30,0x40,
0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,
0x01,0xff,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,
0x02,0x01,0x06,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x7b,
0x5b,0x45,0xcf,0xaf,0xce,0xcb,0x7a,0xfd,0x31,0x92,0x1a,0x6a,0xb6,0xf3,0x46,
0xeb,0x57,0x48,0x50,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x79,0x11,0xc0,0x4b,0xb3,0x91,
0xb6,0xfc,0xf0,0xe9,0x67,0xd4,0x0d,0x6e,0x45,0xbe,0x55,0xe8,0x93,0xd2,0xce,
0x03,0x3f,0xed,0xda,0x25,0xb0,0x1d,0x57,0xcb,0x1e,0x3a,0x76,0xa0,0x4c,0xec,
0x50,0x76,0xe8,0x64,0x72,0x0c,0xa4,0xa9,0xf1,0xb8,0x8b,0xd6,0xd6,0x87,0x84,
0xbb,0x32,0xe5,0x41,0x11,0xc0,0x77,0xd9,0xb3,0x60,0x9d,0xeb,0x1b,0xd5,0xd1,
0x6e,0x44,0x44,0xa9,0xa6,0x01,0xec,0x55,0x62,0x1d,0x77,0xb8,0x5c,0x8e,0x48,
0x49,0x7c,0x9c,0x3b,0x57,0x11,0xac,0xad,0x73,0x37,0x8e,0x2f,0x78,0x5c,0x90,
0x68,0x47,0xd9,0x60,0x60,0xe6,0xfc,0x07,0x3d,0x22,0x20,0x17,0xc4,0xf7,0x16,
0xe9,0xc4,0xd8,0x72,0xf9,0xc8,0x73,0x7c,0xdf,0x16,0x2f,0x15,0xa9,0x3e,0xfd,
0x6a,0x27,0xb6,0xa1,0xeb,0x5a,0xba,0x98,0x1f,0xd5,0xe3,0x4d,0x64,0x0a,0x9d,
0x13,0xc8,0x61,0xba,0xf5,0x39,0x1c,0x87,0xba,0xb8,0xbd,0x7b,0x22,0x7f,0xf6,
0xfe,0xac,0x40,0x79,0xe5,0xac,0x10,0x6f,0x3d,0x8f,0x1b,0x79,0x76,0x8b,0xc4,
0x37,0xb3,0x21,0x18,0x84,0xe5,0x36,0x00,0xeb,0x63,0x20,0x99,0xb9,0xe9,0xfe,
0x33,0x04,0xbb,0x41,0xc8,0xc1,0x02,0xf9,0x44,0x63,0x20,0x9e,0x81,0xce,0x42,
0xd3,0xd6,0x3f,0x2c,0x76,0xd3,0x63,0x9c,0x59,0xdd,0x8f,0xa6,0xe1,0x0e,0xa0,
0x2e,0x41,0xf7,0x2e,0x95,0x47,0xcf,0xbc,0xfd,0x33,0xf3,0xf6,0x0b,0x61,0x7e,
0x7e,0x91,0x2b,0x81,0x47,0xc2,0x27,0x30,0xee,0xa7,0x10,0x5d,0x37,0x8f,0x5c,
0x39,0x2b,0xe4,0x04,0xf0,0x7b,0x8d,0x56,0x8c,0x68 };
static const BYTE thawte_ssl_ca[] = {
0x30,0x82,0x04,0x6c,0x30,0x82,0x03,0x54,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x4d,0x5f,0x2c,0x34,0x08,0xb2,0x4c,0x20,0xcd,0x6d,0x50,0x7e,0x24,0x4d,0xc9,
0xec,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x81,0xa9,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
0x55,0x53,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x0a,0x13,0x0c,0x74,0x68,
0x61,0x77,0x74,0x65,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x28,0x30,0x26,0x06,
0x03,0x55,0x04,0x0b,0x13,0x1f,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,
0x74,0x69,0x6f,0x6e,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x20,0x44,
0x69,0x76,0x69,0x73,0x69,0x6f,0x6e,0x31,0x38,0x30,0x36,0x06,0x03,0x55,0x04,
0x0b,0x13,0x2f,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x36,0x20,0x74,0x68,0x61,
0x77,0x74,0x65,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x20,0x2d,0x20,0x46,0x6f,0x72,
0x20,0x61,0x75,0x74,0x68,0x6f,0x72,0x69,0x7a,0x65,0x64,0x20,0x75,0x73,0x65,
0x20,0x6f,0x6e,0x6c,0x79,0x31,0x1f,0x30,0x1d,0x06,0x03,0x55,0x04,0x03,0x13,
0x16,0x74,0x68,0x61,0x77,0x74,0x65,0x20,0x50,0x72,0x69,0x6d,0x61,0x72,0x79,
0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x31,0x30,0x30,
0x32,0x30,0x38,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x32,0x30,0x30,
0x32,0x30,0x37,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x3c,0x31,0x0b,0x30,
0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x15,0x30,0x13,0x06,
0x03,0x55,0x04,0x0a,0x13,0x0c,0x54,0x68,0x61,0x77,0x74,0x65,0x2c,0x20,0x49,
0x6e,0x63,0x2e,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x03,0x13,0x0d,0x54,
0x68,0x61,0x77,0x74,0x65,0x20,0x53,0x53,0x4c,0x20,0x43,0x41,0x30,0x82,0x01,
0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,
0x99,0xe4,0x85,0x5b,0x76,0x49,0x7d,0x2f,0x05,0xd8,0xc5,0xac,0xc8,0xc8,0xa9,
0xd3,0xdc,0x98,0xe6,0xd7,0x34,0xa6,0x2f,0x0c,0xf2,0x22,0x26,0xd8,0xa3,0xc9,
0x14,0x4c,0x8f,0x05,0xa4,0x45,0xe8,0x14,0x0c,0x58,0x90,0x05,0x1a,0xb7,0xc5,
0xc1,0x06,0xa5,0x80,0xaf,0xbb,0x1d,0x49,0x6b,0x52,0x34,0x88,0xc3,0x59,0xe7,
0xef,0x6b,0xc4,0x27,0x41,0x8c,0x2b,0x66,0x1d,0xd0,0xe0,0xa3,0x97,0x98,0x19,
0x34,0x4b,0x41,0xd5,0x98,0xd5,0xc7,0x05,0xad,0xa2,0xe4,0xd7,0xed,0x0c,0xad,
0x4f,0xc1,0xb5,0xb0,0x21,0xfd,0x3e,0x50,0x53,0xb2,0xc4,0x90,0xd0,0xd4,0x30,
0x67,0x6c,0x9a,0xf1,0x0e,0x74,0xc4,0xc2,0xdc,0x8a,0xe8,0x97,0xff,0xc9,0x92,
0xae,0x01,0x8a,0x56,0x0a,0x98,0x32,0xb0,0x00,0x23,0xec,0x90,0x1a,0x60,0xc3,
0xed,0xbb,0x3a,0xcb,0x0f,0x63,0x9f,0x0d,0x44,0xc9,0x52,0xe1,0x25,0x96,0xbf,
0xed,0x50,0x95,0x89,0x7f,0x56,0x14,0xb1,0xb7,0x61,0x1d,0x1c,0x07,0x8c,0x3a,
0x2c,0xf7,0xff,0x80,0xde,0x39,0x45,0xd5,0xaf,0x1a,0xd1,0x78,0xd8,0xc7,0x71,
0x6a,0xa3,0x19,0xa7,0x32,0x50,0x21,0xe9,0xf2,0x0e,0xa1,0xc6,0x13,0x03,0x44,
0x48,0xd1,0x66,0xa8,0x52,0x57,0xd7,0x11,0xb4,0x93,0x8b,0xe5,0x99,0x9f,0x5d,
0xe7,0x78,0x51,0xe5,0x4d,0xf6,0xb7,0x59,0xb4,0x76,0xb5,0x09,0x37,0x4d,0x06,
0x38,0x13,0x7a,0x1c,0x08,0x98,0x5c,0xc4,0x48,0x4a,0xcb,0x52,0xa0,0xa9,0xf8,
0xb1,0x9d,0x8e,0x7b,0x79,0xb0,0x20,0x2f,0x3c,0x96,0xa8,0x11,0x62,0x47,0xbb,
0x11,0x02,0x03,0x01,0x00,0x01,0xa3,0x81,0xfb,0x30,0x81,0xf8,0x30,0x32,0x06,
0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x26,0x30,0x24,0x30,0x22,
0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x16,0x68,0x74,0x74,
0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,0x74,0x68,0x61,0x77,0x74,0x65,
0x2e,0x63,0x6f,0x6d,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,
0x08,0x30,0x06,0x01,0x01,0xff,0x02,0x01,0x00,0x30,0x34,0x06,0x03,0x55,0x1d,
0x1f,0x04,0x2d,0x30,0x2b,0x30,0x29,0xa0,0x27,0xa0,0x25,0x86,0x23,0x68,0x74,
0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x74,0x68,0x61,0x77,0x74,0x65,
0x2e,0x63,0x6f,0x6d,0x2f,0x54,0x68,0x61,0x77,0x74,0x65,0x50,0x43,0x41,0x2e,
0x63,0x72,0x6c,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,
0x03,0x02,0x01,0x06,0x30,0x28,0x06,0x03,0x55,0x1d,0x11,0x04,0x21,0x30,0x1f,
0xa4,0x1d,0x30,0x1b,0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,0x03,0x13,0x10,
0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x4d,0x50,0x4b,0x49,0x2d,0x32,0x2d,
0x39,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0xa7,0xa2,0x83,
0xbb,0x34,0x45,0x40,0x3d,0xfc,0xd5,0x30,0x4f,0x12,0xb9,0x3e,0xa1,0x01,0x9f,
0xf6,0xdb,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,
0x7b,0x5b,0x45,0xcf,0xaf,0xce,0xcb,0x7a,0xfd,0x31,0x92,0x1a,0x6a,0xb6,0xf3,
0x46,0xeb,0x57,0x48,0x50,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x80,0x22,0x80,0xe0,0x6c,
0xc8,0x95,0x16,0xd7,0x57,0x26,0x87,0xf3,0x72,0x34,0xdb,0xc6,0x72,0x56,0x27,
0x3e,0xd3,0x96,0xf6,0x2e,0x25,0x91,0xa5,0x3e,0x33,0x97,0xa7,0x4b,0xe5,0x2f,
0xfb,0x25,0x7d,0x2f,0x07,0x61,0xfa,0x6f,0x83,0x74,0x4c,0x4c,0x53,0x72,0x20,
0xa4,0x7a,0xcf,0x51,0x51,0x56,0x81,0x88,0xb0,0x6d,0x1f,0x36,0x2c,0xc8,0x2b,
0xb1,0x88,0x99,0xc1,0xfe,0x44,0xab,0x48,0x51,0x7c,0xd8,0xf2,0x44,0x64,0x2a,
0xd8,0x71,0xa7,0xfb,0x1a,0x2f,0xf9,0x19,0x8d,0x34,0xb2,0x23,0xbf,0xc4,0x4c,
0x55,0x1d,0x8e,0x44,0xe8,0xaa,0x5d,0x9a,0xdd,0x9f,0xfd,0x03,0xc7,0xba,0x24,
0x43,0x8d,0x2d,0x47,0x44,0xdb,0xf6,0xd8,0x98,0xc8,0xb2,0xf9,0xda,0xef,0xed,
0x29,0x5c,0x69,0x12,0xfa,0xd1,0x23,0x96,0x0f,0xbf,0x9c,0x0d,0xf2,0x79,0x45,
0x53,0x37,0x9a,0x56,0x2f,0xe8,0x57,0x10,0x70,0xf6,0xee,0x89,0x0c,0x49,0x89,
0x9a,0xc1,0x23,0xf5,0xc2,0x2a,0xcc,0x41,0xcf,0x22,0xab,0x65,0x6e,0xb7,0x94,
0x82,0x6d,0x2f,0x40,0x5f,0x58,0xde,0xeb,0x95,0x2b,0xa6,0x72,0x68,0x52,0x19,
0x91,0x2a,0xae,0x75,0x9d,0x4e,0x92,0xe6,0xca,0xde,0x54,0xea,0x18,0xab,0x25,
0x3c,0xe6,0x64,0xa6,0x79,0x1f,0x26,0x7d,0x61,0xed,0x7d,0xd2,0xe5,0x71,0x55,
0xd8,0x93,0x17,0x7c,0x14,0x38,0x30,0x3c,0xdf,0x86,0xe3,0x4c,0xad,0x49,0xe3,
0x97,0x59,0xce,0x1b,0x9b,0x2b,0xce,0xdc,0x65,0xd4,0x0b,0x28,0x6b,0x4e,0x84,
0x46,0x51,0x44,0xf7,0x33,0x08,0x2d,0x58,0x97,0x21,0xae };
/* global_sign_root -> global_sign_ca -> openssl_org:
 * "*.openssl.org"'s certificate
 */
static const BYTE global_sign_root[] = {
0x30,0x82,0x04,0x6e,0x30,0x82,0x03,0x56,0xa0,0x03,0x02,0x01,0x02,0x02,0x0b,0x04,
0x00,0x00,0x00,0x00,0x01,0x12,0x56,0xad,0x62,0x04,0x30,0x0d,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x57,0x31,0x0b,0x30,0x09,0x06,
0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,
0x0a,0x13,0x10,0x47,0x6c,0x6f,0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,0x20,0x6e,0x76,
0x2d,0x73,0x61,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x0b,0x13,0x07,0x52,0x6f,
0x6f,0x74,0x20,0x43,0x41,0x31,0x1b,0x30,0x19,0x06,0x03,0x55,0x04,0x03,0x13,0x12,
0x47,0x6c,0x6f,0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,0x20,0x52,0x6f,0x6f,0x74,0x20,
0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x34,0x31,0x30,0x30,0x30,
0x30,0x30,0x5a,0x17,0x0d,0x31,0x34,0x30,0x31,0x32,0x37,0x31,0x31,0x30,0x30,0x30,
0x30,0x5a,0x30,0x71,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,
0x45,0x31,0x1d,0x30,0x1b,0x06,0x03,0x55,0x04,0x0b,0x13,0x14,0x44,0x6f,0x6d,0x61,
0x69,0x6e,0x20,0x56,0x61,0x6c,0x69,0x64,0x61,0x74,0x69,0x6f,0x6e,0x20,0x43,0x41,
0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,0x0a,0x13,0x10,0x47,0x6c,0x6f,0x62,0x61,
0x6c,0x53,0x69,0x67,0x6e,0x20,0x6e,0x76,0x2d,0x73,0x61,0x31,0x28,0x30,0x26,0x06,
0x03,0x55,0x04,0x03,0x13,0x1f,0x47,0x6c,0x6f,0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,
0x20,0x44,0x6f,0x6d,0x61,0x69,0x6e,0x20,0x56,0x61,0x6c,0x69,0x64,0x61,0x74,0x69,
0x6f,0x6e,0x20,0x43,0x41,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,
0x0a,0x02,0x82,0x01,0x01,0x00,0xb4,0x9f,0x49,0xe6,0xb3,0x9f,0x01,0x55,0x36,0xd0,
0x36,0x95,0x6c,0xcb,0x16,0x5a,0x6c,0x72,0xb5,0xe2,0x9d,0xe2,0xfa,0x39,0xe9,0x0a,
0xdc,0x73,0x5c,0xa0,0x3b,0x76,0x6d,0x05,0x28,0x93,0x0d,0xa5,0x42,0xf3,0xe8,0xcb,
0xee,0xd3,0x33,0x1a,0x13,0x27,0x0a,0xdf,0x5c,0xd1,0x54,0x62,0x6a,0x9a,0x17,0x36,
0xd6,0x3c,0x58,0x87,0x1e,0xb2,0x66,0x1f,0x55,0x69,0x2b,0x89,0x56,0x47,0xb4,0xc2,
0x50,0x0c,0xe5,0x5d,0xc1,0xfd,0x29,0x1b,0x66,0x73,0x43,0xf4,0x6f,0xd1,0x14,0x4f,
0x2a,0x6d,0x48,0x52,0x3a,0xf5,0xe4,0x90,0x3f,0xde,0xb2,0xf4,0x10,0xd5,0xf9,0xfa,
0xac,0x96,0x0d,0x10,0x32,0x23,0xd2,0xd3,0x51,0xa7,0x95,0x85,0x50,0xfa,0x0f,0x29,
0x9d,0xa7,0x88,0x4e,0x0d,0x31,0x9b,0x44,0x41,0x8e,0x41,0x2f,0x6e,0x9a,0x1a,0xb4,
0xb6,0xa5,0x59,0xf1,0x49,0x59,0x09,0x7e,0x7e,0x49,0xd3,0x69,0x33,0x4b,0x00,0x92,
0xc5,0x1b,0x42,0x20,0x69,0xf8,0x64,0xdf,0xd8,0xa7,0x44,0xd1,0x90,0x5e,0xfd,0x39,
0xad,0x6b,0x7d,0xfe,0xd5,0x8b,0xa4,0x01,0x64,0xf6,0xa7,0xb4,0x20,0x84,0xcc,0x05,
0x2f,0x27,0xee,0x1e,0xd5,0x24,0x96,0xae,0x84,0x43,0xa4,0x3e,0x43,0xf3,0x3b,0x99,
0xd7,0xf9,0xf2,0xde,0x3d,0x19,0xed,0xd0,0x65,0xf1,0x49,0xca,0x48,0xd2,0x38,0x7d,
0xc6,0x48,0xd7,0x72,0x61,0xb8,0xc6,0xc2,0x67,0x0a,0xd8,0x65,0x5a,0xfd,0x5b,0xf3,
0xc5,0xfd,0xbe,0x8c,0x69,0xcf,0x99,0x20,0xb6,0xea,0xec,0x39,0x80,0xe9,0x66,0xa8,
0xee,0x3d,0x9d,0x46,0x7b,0x87,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x1f,0x30,
0x82,0x01,0x1b,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,
0x02,0x01,0x06,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,
0x06,0x01,0x01,0xff,0x02,0x01,0x00,0x30,0x4b,0x06,0x03,0x55,0x1d,0x20,0x04,0x44,
0x30,0x42,0x30,0x40,0x06,0x09,0x2b,0x06,0x01,0x04,0x01,0xa0,0x32,0x01,0x0a,0x30,
0x33,0x30,0x31,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x25,0x68,
0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x67,0x6c,0x6f,0x62,0x61,0x6c,
0x73,0x69,0x67,0x6e,0x2e,0x6e,0x65,0x74,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,
0x6f,0x72,0x79,0x2f,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x36,
0x12,0x4e,0x9e,0x71,0xc4,0x26,0x41,0xf1,0xfa,0xf1,0x29,0x4c,0xbf,0x17,0xa4,0x53,
0x28,0xb6,0xeb,0x30,0x33,0x06,0x03,0x55,0x1d,0x1f,0x04,0x2c,0x30,0x2a,0x30,0x28,
0xa0,0x26,0xa0,0x24,0x86,0x22,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,
0x2e,0x67,0x6c,0x6f,0x62,0x61,0x6c,0x73,0x69,0x67,0x6e,0x2e,0x6e,0x65,0x74,0x2f,
0x72,0x6f,0x6f,0x74,0x2e,0x63,0x72,0x6c,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,
0x86,0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x02,0x04,0x30,0x20,0x06,0x03,0x55,
0x1d,0x25,0x04,0x19,0x30,0x17,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x0a,
0x03,0x03,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xf8,0x42,0x04,0x01,0x30,0x1f,0x06,
0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x60,0x7b,0x66,0x1a,0x45,0x0d,
0x97,0xca,0x89,0x50,0x2f,0x7d,0x04,0xcd,0x34,0xa8,0xff,0xfc,0xfd,0x4b,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,
0x01,0x00,0xc3,0x25,0x79,0x4a,0xfa,0xcd,0x26,0x6f,0x40,0x6a,0xb8,0x94,0x1b,0xe6,
0xa0,0xe1,0x6b,0xd2,0x31,0x9a,0x3f,0x89,0xfb,0xc5,0x0a,0x22,0x75,0x7a,0xfd,0xbb,
0x40,0x34,0x9c,0xbc,0x9e,0x85,0x49,0x59,0x7a,0x95,0xcc,0x0d,0x2d,0x44,0x0b,0xc7,
0xcb,0x15,0x10,0x1e,0xd2,0xdc,0xe3,0x78,0xea,0x5f,0xf5,0x35,0x51,0xa6,0xc8,0x3f,
0x39,0x90,0x15,0x9e,0x5f,0x15,0xc9,0xb7,0x8f,0x5d,0x6b,0x63,0x99,0x68,0xe1,0x6e,
0x5c,0xfc,0x9d,0x37,0x33,0x0a,0xae,0x87,0x70,0x4a,0xa2,0xe3,0xab,0x26,0x7d,0x73,
0x07,0xe4,0xf7,0x0a,0xaa,0x75,0x8f,0xa6,0xee,0x9b,0x04,0xd1,0x05,0x43,0x74,0x1a,
0xba,0xd4,0xc2,0x9d,0x7f,0xb7,0xa0,0x6c,0xed,0x16,0x9a,0x67,0x40,0x56,0xab,0x83,
0x9f,0x4c,0x9c,0xeb,0x28,0x34,0x1f,0x8d,0xe6,0x5a,0x0f,0x69,0x40,0xd9,0xa9,0x9e,
0x6a,0xf9,0x89,0x7f,0xd0,0xf8,0x8c,0xb3,0x48,0x2e,0x0b,0x4f,0xf1,0xf9,0xcf,0xf5,
0x0e,0x11,0x02,0x0c,0x89,0x16,0x73,0x53,0x5d,0x2c,0xd3,0xef,0xde,0x5f,0x0b,0x25,
0xba,0xd5,0x4e,0x8f,0x85,0x90,0x3f,0x88,0x93,0x79,0xa5,0x20,0xac,0xff,0xb0,0x8a,
0xcb,0xcb,0xa5,0x55,0x78,0x83,0x39,0x41,0x32,0xaa,0x5d,0x25,0x93,0xa7,0xea,0xee,
0x18,0x72,0xb5,0xe2,0x1d,0x7f,0x08,0x74,0x11,0xb6,0x01,0x27,0x4e,0x96,0xcf,0xa8,
0xf0,0x2a,0xf9,0x52,0xac,0x24,0x1a,0x19,0x9e,0x9f,0x6c,0x52,0x7c,0x8b,0x3a,0xa6,
0x54,0xde,0xc1,0xc7,0x21,0x7d,0x3a,0xd3,0xbc,0x1a,0xa8,0x6d,0xf2,0xbf,0xdf,0x76,
0xca,0x03
};
static const BYTE global_sign_ca[] = {
0x30,0x82,0x04,0x6e,0x30,0x82,0x03,0x56,0xa0,0x03,0x02,0x01,0x02,0x02,0x0b,0x04,
0x00,0x00,0x00,0x00,0x01,0x12,0x56,0xad,0x62,0x04,0x30,0x0d,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x57,0x31,0x0b,0x30,0x09,0x06,
0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,
0x0a,0x13,0x10,0x47,0x6c,0x6f,0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,0x20,0x6e,0x76,
0x2d,0x73,0x61,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x0b,0x13,0x07,0x52,0x6f,
0x6f,0x74,0x20,0x43,0x41,0x31,0x1b,0x30,0x19,0x06,0x03,0x55,0x04,0x03,0x13,0x12,
0x47,0x6c,0x6f,0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,0x20,0x52,0x6f,0x6f,0x74,0x20,
0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x34,0x31,0x30,0x30,0x30,
0x30,0x30,0x5a,0x17,0x0d,0x31,0x34,0x30,0x31,0x32,0x37,0x31,0x31,0x30,0x30,0x30,
0x30,0x5a,0x30,0x71,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,
0x45,0x31,0x1d,0x30,0x1b,0x06,0x03,0x55,0x04,0x0b,0x13,0x14,0x44,0x6f,0x6d,0x61,
0x69,0x6e,0x20,0x56,0x61,0x6c,0x69,0x64,0x61,0x74,0x69,0x6f,0x6e,0x20,0x43,0x41,
0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,0x0a,0x13,0x10,0x47,0x6c,0x6f,0x62,0x61,
0x6c,0x53,0x69,0x67,0x6e,0x20,0x6e,0x76,0x2d,0x73,0x61,0x31,0x28,0x30,0x26,0x06,
0x03,0x55,0x04,0x03,0x13,0x1f,0x47,0x6c,0x6f,0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,
0x20,0x44,0x6f,0x6d,0x61,0x69,0x6e,0x20,0x56,0x61,0x6c,0x69,0x64,0x61,0x74,0x69,
0x6f,0x6e,0x20,0x43,0x41,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,
0x0a,0x02,0x82,0x01,0x01,0x00,0xb4,0x9f,0x49,0xe6,0xb3,0x9f,0x01,0x55,0x36,0xd0,
0x36,0x95,0x6c,0xcb,0x16,0x5a,0x6c,0x72,0xb5,0xe2,0x9d,0xe2,0xfa,0x39,0xe9,0x0a,
0xdc,0x73,0x5c,0xa0,0x3b,0x76,0x6d,0x05,0x28,0x93,0x0d,0xa5,0x42,0xf3,0xe8,0xcb,
0xee,0xd3,0x33,0x1a,0x13,0x27,0x0a,0xdf,0x5c,0xd1,0x54,0x62,0x6a,0x9a,0x17,0x36,
0xd6,0x3c,0x58,0x87,0x1e,0xb2,0x66,0x1f,0x55,0x69,0x2b,0x89,0x56,0x47,0xb4,0xc2,
0x50,0x0c,0xe5,0x5d,0xc1,0xfd,0x29,0x1b,0x66,0x73,0x43,0xf4,0x6f,0xd1,0x14,0x4f,
0x2a,0x6d,0x48,0x52,0x3a,0xf5,0xe4,0x90,0x3f,0xde,0xb2,0xf4,0x10,0xd5,0xf9,0xfa,
0xac,0x96,0x0d,0x10,0x32,0x23,0xd2,0xd3,0x51,0xa7,0x95,0x85,0x50,0xfa,0x0f,0x29,
0x9d,0xa7,0x88,0x4e,0x0d,0x31,0x9b,0x44,0x41,0x8e,0x41,0x2f,0x6e,0x9a,0x1a,0xb4,
0xb6,0xa5,0x59,0xf1,0x49,0x59,0x09,0x7e,0x7e,0x49,0xd3,0x69,0x33,0x4b,0x00,0x92,
0xc5,0x1b,0x42,0x20,0x69,0xf8,0x64,0xdf,0xd8,0xa7,0x44,0xd1,0x90,0x5e,0xfd,0x39,
0xad,0x6b,0x7d,0xfe,0xd5,0x8b,0xa4,0x01,0x64,0xf6,0xa7,0xb4,0x20,0x84,0xcc,0x05,
0x2f,0x27,0xee,0x1e,0xd5,0x24,0x96,0xae,0x84,0x43,0xa4,0x3e,0x43,0xf3,0x3b,0x99,
0xd7,0xf9,0xf2,0xde,0x3d,0x19,0xed,0xd0,0x65,0xf1,0x49,0xca,0x48,0xd2,0x38,0x7d,
0xc6,0x48,0xd7,0x72,0x61,0xb8,0xc6,0xc2,0x67,0x0a,0xd8,0x65,0x5a,0xfd,0x5b,0xf3,
0xc5,0xfd,0xbe,0x8c,0x69,0xcf,0x99,0x20,0xb6,0xea,0xec,0x39,0x80,0xe9,0x66,0xa8,
0xee,0x3d,0x9d,0x46,0x7b,0x87,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x1f,0x30,
0x82,0x01,0x1b,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,
0x02,0x01,0x06,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,
0x06,0x01,0x01,0xff,0x02,0x01,0x00,0x30,0x4b,0x06,0x03,0x55,0x1d,0x20,0x04,0x44,
0x30,0x42,0x30,0x40,0x06,0x09,0x2b,0x06,0x01,0x04,0x01,0xa0,0x32,0x01,0x0a,0x30,
0x33,0x30,0x31,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x25,0x68,
0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x67,0x6c,0x6f,0x62,0x61,0x6c,
0x73,0x69,0x67,0x6e,0x2e,0x6e,0x65,0x74,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,
0x6f,0x72,0x79,0x2f,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x36,
0x12,0x4e,0x9e,0x71,0xc4,0x26,0x41,0xf1,0xfa,0xf1,0x29,0x4c,0xbf,0x17,0xa4,0x53,
0x28,0xb6,0xeb,0x30,0x33,0x06,0x03,0x55,0x1d,0x1f,0x04,0x2c,0x30,0x2a,0x30,0x28,
0xa0,0x26,0xa0,0x24,0x86,0x22,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,
0x2e,0x67,0x6c,0x6f,0x62,0x61,0x6c,0x73,0x69,0x67,0x6e,0x2e,0x6e,0x65,0x74,0x2f,
0x72,0x6f,0x6f,0x74,0x2e,0x63,0x72,0x6c,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,
0x86,0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x02,0x04,0x30,0x20,0x06,0x03,0x55,
0x1d,0x25,0x04,0x19,0x30,0x17,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x0a,
0x03,0x03,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xf8,0x42,0x04,0x01,0x30,0x1f,0x06,
0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x60,0x7b,0x66,0x1a,0x45,0x0d,
0x97,0xca,0x89,0x50,0x2f,0x7d,0x04,0xcd,0x34,0xa8,0xff,0xfc,0xfd,0x4b,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,
0x01,0x00,0xc3,0x25,0x79,0x4a,0xfa,0xcd,0x26,0x6f,0x40,0x6a,0xb8,0x94,0x1b,0xe6,
0xa0,0xe1,0x6b,0xd2,0x31,0x9a,0x3f,0x89,0xfb,0xc5,0x0a,0x22,0x75,0x7a,0xfd,0xbb,
0x40,0x34,0x9c,0xbc,0x9e,0x85,0x49,0x59,0x7a,0x95,0xcc,0x0d,0x2d,0x44,0x0b,0xc7,
0xcb,0x15,0x10,0x1e,0xd2,0xdc,0xe3,0x78,0xea,0x5f,0xf5,0x35,0x51,0xa6,0xc8,0x3f,
0x39,0x90,0x15,0x9e,0x5f,0x15,0xc9,0xb7,0x8f,0x5d,0x6b,0x63,0x99,0x68,0xe1,0x6e,
0x5c,0xfc,0x9d,0x37,0x33,0x0a,0xae,0x87,0x70,0x4a,0xa2,0xe3,0xab,0x26,0x7d,0x73,
0x07,0xe4,0xf7,0x0a,0xaa,0x75,0x8f,0xa6,0xee,0x9b,0x04,0xd1,0x05,0x43,0x74,0x1a,
0xba,0xd4,0xc2,0x9d,0x7f,0xb7,0xa0,0x6c,0xed,0x16,0x9a,0x67,0x40,0x56,0xab,0x83,
0x9f,0x4c,0x9c,0xeb,0x28,0x34,0x1f,0x8d,0xe6,0x5a,0x0f,0x69,0x40,0xd9,0xa9,0x9e,
0x6a,0xf9,0x89,0x7f,0xd0,0xf8,0x8c,0xb3,0x48,0x2e,0x0b,0x4f,0xf1,0xf9,0xcf,0xf5,
0x0e,0x11,0x02,0x0c,0x89,0x16,0x73,0x53,0x5d,0x2c,0xd3,0xef,0xde,0x5f,0x0b,0x25,
0xba,0xd5,0x4e,0x8f,0x85,0x90,0x3f,0x88,0x93,0x79,0xa5,0x20,0xac,0xff,0xb0,0x8a,
0xcb,0xcb,0xa5,0x55,0x78,0x83,0x39,0x41,0x32,0xaa,0x5d,0x25,0x93,0xa7,0xea,0xee,
0x18,0x72,0xb5,0xe2,0x1d,0x7f,0x08,0x74,0x11,0xb6,0x01,0x27,0x4e,0x96,0xcf,0xa8,
0xf0,0x2a,0xf9,0x52,0xac,0x24,0x1a,0x19,0x9e,0x9f,0x6c,0x52,0x7c,0x8b,0x3a,0xa6,
0x54,0xde,0xc1,0xc7,0x21,0x7d,0x3a,0xd3,0xbc,0x1a,0xa8,0x6d,0xf2,0xbf,0xdf,0x76,
0xca,0x03
};
static const BYTE openssl_org[] = {
0x30,0x82,0x04,0xc8,0x30,0x82,0x03,0xb0,0xa0,0x03,0x02,0x01,0x02,0x02,0x0b,0x01,
0x00,0x00,0x00,0x00,0x01,0x1c,0x57,0x8e,0x3a,0xeb,0x30,0x0d,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x71,0x31,0x0b,0x30,0x09,0x06,
0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x1d,0x30,0x1b,0x06,0x03,0x55,0x04,
0x0b,0x13,0x14,0x44,0x6f,0x6d,0x61,0x69,0x6e,0x20,0x56,0x61,0x6c,0x69,0x64,0x61,
0x74,0x69,0x6f,0x6e,0x20,0x43,0x41,0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,0x0a,
0x13,0x10,0x47,0x6c,0x6f,0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,0x20,0x6e,0x76,0x2d,
0x73,0x61,0x31,0x28,0x30,0x26,0x06,0x03,0x55,0x04,0x03,0x13,0x1f,0x47,0x6c,0x6f,
0x62,0x61,0x6c,0x53,0x69,0x67,0x6e,0x20,0x44,0x6f,0x6d,0x61,0x69,0x6e,0x20,0x56,
0x61,0x6c,0x69,0x64,0x61,0x74,0x69,0x6f,0x6e,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,
0x30,0x38,0x30,0x39,0x31,0x32,0x31,0x37,0x31,0x34,0x31,0x31,0x5a,0x17,0x0d,0x31,
0x31,0x30,0x39,0x31,0x33,0x31,0x37,0x31,0x34,0x30,0x36,0x5a,0x30,0x60,0x31,0x0b,
0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x47,0x42,0x31,0x21,0x30,0x1f,0x06,
0x03,0x55,0x04,0x0b,0x13,0x18,0x44,0x6f,0x6d,0x61,0x69,0x6e,0x20,0x43,0x6f,0x6e,
0x74,0x72,0x6f,0x6c,0x20,0x56,0x61,0x6c,0x69,0x64,0x61,0x74,0x65,0x64,0x31,0x16,
0x30,0x14,0x06,0x03,0x55,0x04,0x0a,0x14,0x0d,0x2a,0x2e,0x6f,0x70,0x65,0x6e,0x73,
0x73,0x6c,0x2e,0x6f,0x72,0x67,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x03,0x14,
0x0d,0x2a,0x2e,0x6f,0x70,0x65,0x6e,0x73,0x73,0x6c,0x2e,0x6f,0x72,0x67,0x30,0x82,
0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xb9,
0x23,0x4e,0x1d,0xb5,0xef,0x87,0xb2,0xce,0x43,0x75,0x6a,0xc3,0x11,0x1c,0xd2,0xf7,
0x32,0xf3,0xf7,0x92,0x9b,0xed,0xe6,0x86,0xc3,0xb7,0x93,0x93,0x5e,0x7d,0x32,0x6a,
0x2b,0x03,0xb7,0x2a,0xc8,0x80,0x98,0x91,0x41,0x40,0x08,0xac,0xde,0xa6,0x7a,0x7c,
0x4f,0x44,0x8b,0x9c,0xd8,0xd3,0x87,0x71,0xf2,0x4e,0x3e,0x54,0xd7,0xa4,0x11,0x81,
0xb7,0x71,0x68,0x87,0xc1,0xc9,0xaf,0x2a,0xfa,0xb2,0x74,0xa3,0x70,0xa1,0x4d,0x19,
0x06,0x8d,0x1c,0x95,0x04,0x14,0x9c,0x8f,0x1f,0xf0,0x98,0xc1,0xc0,0x70,0xfa,0xb1,
0x10,0x1d,0x77,0xda,0x80,0x14,0x53,0xa5,0x5c,0xbe,0x05,0x9a,0x43,0xf4,0x54,0xf3,
0x70,0xf4,0x71,0x39,0xea,0x50,0x9c,0xe6,0xa8,0xd0,0x78,0xa6,0x30,0x1e,0x6d,0x4d,
0x22,0x10,0xdd,0x15,0x2d,0x50,0x00,0x5d,0xd9,0xc9,0xd2,0xa6,0xb9,0xdb,0x7f,0xad,
0xd8,0xeb,0xef,0x65,0x1f,0xce,0xbb,0x8d,0x4e,0x51,0x95,0x84,0x6e,0x95,0x16,0xef,
0x0b,0xf0,0x7b,0xff,0x2d,0x6c,0x1b,0x81,0x6c,0xca,0xe0,0x25,0x3d,0x2a,0x5c,0x6f,
0xad,0xce,0x4b,0x5c,0x46,0x7c,0xb1,0xa9,0xf4,0xea,0x72,0x02,0xa1,0xcc,0x87,0x63,
0x85,0x19,0x52,0x01,0x23,0x04,0x8e,0x2a,0xb4,0x94,0xde,0x6d,0x69,0x13,0x65,0x0b,
0x70,0x5f,0x92,0x1a,0x44,0x68,0x41,0xcd,0x80,0xef,0xc9,0xe9,0x83,0xd8,0x1e,0x95,
0x86,0xc7,0x1a,0x51,0xaf,0x3d,0xaa,0xca,0x97,0x19,0xe0,0xb4,0x69,0x87,0x75,0xe1,
0x09,0x9a,0x53,0xcd,0xf5,0x43,0xcf,0x2b,0x1d,0xe7,0x81,0x20,0x1e,0x97,0x25,0x02,
0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x70,0x30,0x82,0x01,0x6c,0x30,0x1f,0x06,0x03,
0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x36,0x12,0x4e,0x9e,0x71,0xc4,0x26,
0x41,0xf1,0xfa,0xf1,0x29,0x4c,0xbf,0x17,0xa4,0x53,0x28,0xb6,0xeb,0x30,0x49,0x06,
0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x3d,0x30,0x3b,0x30,0x39,0x06,
0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x2d,0x68,0x74,0x74,0x70,0x3a,
0x2f,0x2f,0x73,0x65,0x63,0x75,0x72,0x65,0x2e,0x67,0x6c,0x6f,0x62,0x61,0x6c,0x73,
0x69,0x67,0x6e,0x2e,0x6e,0x65,0x74,0x2f,0x63,0x61,0x63,0x65,0x72,0x74,0x2f,0x64,
0x76,0x68,0x65,0x31,0x2e,0x63,0x72,0x74,0x30,0x39,0x06,0x03,0x55,0x1d,0x1f,0x04,
0x32,0x30,0x30,0x30,0x2e,0xa0,0x2c,0xa0,0x2a,0x86,0x28,0x68,0x74,0x74,0x70,0x3a,
0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x67,0x6c,0x6f,0x62,0x61,0x6c,0x73,0x69,0x67,0x6e,
0x2e,0x6e,0x65,0x74,0x2f,0x44,0x6f,0x6d,0x61,0x69,0x6e,0x56,0x61,0x6c,0x31,0x2e,
0x63,0x72,0x6c,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0xec,0x81,
0x7d,0x47,0xa8,0xc0,0x19,0xa7,0xbc,0x6b,0x52,0x5e,0xb6,0x90,0xfb,0x17,0xae,0x7f,
0x41,0x8f,0x30,0x09,0x06,0x03,0x55,0x1d,0x13,0x04,0x02,0x30,0x00,0x30,0x0e,0x06,
0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x04,0xf0,0x30,0x29,0x06,
0x03,0x55,0x1d,0x25,0x04,0x22,0x30,0x20,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,
0x03,0x01,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x02,0x06,0x0a,0x2b,0x06,
0x01,0x04,0x01,0x82,0x37,0x0a,0x03,0x03,0x30,0x4b,0x06,0x03,0x55,0x1d,0x20,0x04,
0x44,0x30,0x42,0x30,0x40,0x06,0x09,0x2b,0x06,0x01,0x04,0x01,0xa0,0x32,0x01,0x0a,
0x30,0x33,0x30,0x31,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x25,
0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x67,0x6c,0x6f,0x62,0x61,
0x6c,0x73,0x69,0x67,0x6e,0x2e,0x6e,0x65,0x74,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,
0x74,0x6f,0x72,0x79,0x2f,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xf8,0x42,
0x01,0x01,0x04,0x04,0x03,0x02,0x06,0xc0,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x2b,0x22,0xdd,0xa3,
0x1a,0xf9,0x12,0xee,0x77,0xbf,0x34,0xdd,0xbf,0x57,0x98,0x72,0xb3,0x05,0x8d,0x49,
0xa4,0x5d,0x55,0xec,0x08,0xf4,0x70,0xb9,0x83,0xaf,0x57,0xb2,0x0f,0x54,0x12,0xc5,
0xf4,0x0e,0x6e,0xd9,0xe4,0xd4,0x7c,0x6a,0x11,0x4a,0xd7,0xc3,0x46,0x42,0x7e,0x13,
0x4c,0x39,0x1b,0xe3,0x53,0xb1,0x8b,0x8a,0xd7,0xa2,0x74,0xcb,0x18,0x0a,0x40,0x50,
0xbe,0xbe,0x86,0x81,0xa9,0x8e,0x23,0x0b,0xa0,0x38,0x02,0x1c,0x55,0x57,0xfd,0xf7,
0x98,0x43,0x7f,0x6e,0xe9,0x26,0xc5,0x64,0x7d,0x8a,0x42,0xfb,0x67,0xbd,0x93,0x39,
0x75,0x49,0xcd,0x26,0x20,0x71,0xb4,0x65,0xca,0x04,0x15,0xfb,0x1a,0xcb,0x80,0x2e,
0x20,0x43,0x81,0x3e,0x58,0x7f,0x3f,0x2c,0x93,0x92,0x06,0x96,0x69,0xb0,0x49,0x4c,
0xd1,0xa8,0x53,0x9b,0x41,0x55,0x13,0x75,0xc3,0x51,0xe8,0x16,0x3d,0x3e,0x6f,0xbd,
0xb7,0x0c,0x49,0x48,0x64,0x97,0xb4,0x14,0x89,0x57,0x62,0x75,0x32,0x31,0xda,0x94,
0x82,0xf8,0xb3,0xc6,0x02,0x20,0x5e,0x0e,0x00,0x87,0x9c,0x0f,0x34,0x65,0xda,0xf2,
0x47,0x78,0x18,0xf5,0xd5,0xea,0xf3,0x93,0x4e,0x60,0xd6,0x6b,0x7a,0xe8,0x88,0xd7,
0x69,0x41,0xaa,0x10,0x56,0xd7,0x16,0x14,0xde,0xc9,0x1c,0xb6,0xb6,0x10,0xab,0x6b,
0x97,0xbd,0x6a,0xa9,0xb7,0x67,0xd7,0x96,0xeb,0x52,0x12,0x9e,0x9a,0x69,0xaf,0x38,
0x53,0x4d,0x99,0xd8,0x84,0xea,0x54,0xba,0x89,0x67,0xff,0x40,0x66,0x0d,0x53,0x99,
0xb6,0x65,0xba,0xf3,0x8a,0x01,0x3d,0xd0,0xf1,0x21,0x5f,0xdd
};

/* USERTrust -> InCommon RSA CA -> cs.stanford.edu
 * cs.stanford.edu's chain, also valid for www.cs.stanford.edu
 */
static const BYTE usertrust_ca[] = {
0x30,0x82,0x05,0xde,0x30,0x82,0x03,0xc6,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x01,
0xfd,0x6d,0x30,0xfc,0xa3,0xca,0x51,0xa8,0x1b,0xbc,0x64,0x0e,0x35,0x03,0x2d,0x30,
0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0c,0x05,0x00,0x30,0x81,
0x88,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,
0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x4e,0x65,0x77,0x20,0x4a,0x65,0x72,
0x73,0x65,0x79,0x31,0x14,0x30,0x12,0x06,0x03,0x55,0x04,0x07,0x13,0x0b,0x4a,0x65,
0x72,0x73,0x65,0x79,0x20,0x43,0x69,0x74,0x79,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,
0x04,0x0a,0x13,0x15,0x54,0x68,0x65,0x20,0x55,0x53,0x45,0x52,0x54,0x52,0x55,0x53,
0x54,0x20,0x4e,0x65,0x74,0x77,0x6f,0x72,0x6b,0x31,0x2e,0x30,0x2c,0x06,0x03,0x55,
0x04,0x03,0x13,0x25,0x55,0x53,0x45,0x52,0x54,0x72,0x75,0x73,0x74,0x20,0x52,0x53,
0x41,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x20,
0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x30,0x1e,0x17,0x0d,0x31,0x30,0x30,
0x32,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x33,0x38,0x30,0x31,
0x31,0x38,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x81,0x88,0x31,0x0b,0x30,0x09,
0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,
0x04,0x08,0x13,0x0a,0x4e,0x65,0x77,0x20,0x4a,0x65,0x72,0x73,0x65,0x79,0x31,0x14,
0x30,0x12,0x06,0x03,0x55,0x04,0x07,0x13,0x0b,0x4a,0x65,0x72,0x73,0x65,0x79,0x20,
0x43,0x69,0x74,0x79,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x54,
0x68,0x65,0x20,0x55,0x53,0x45,0x52,0x54,0x52,0x55,0x53,0x54,0x20,0x4e,0x65,0x74,
0x77,0x6f,0x72,0x6b,0x31,0x2e,0x30,0x2c,0x06,0x03,0x55,0x04,0x03,0x13,0x25,0x55,
0x53,0x45,0x52,0x54,0x72,0x75,0x73,0x74,0x20,0x52,0x53,0x41,0x20,0x43,0x65,0x72,
0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x20,0x41,0x75,0x74,0x68,0x6f,
0x72,0x69,0x74,0x79,0x30,0x82,0x02,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,
0x02,0x82,0x02,0x01,0x00,0x80,0x12,0x65,0x17,0x36,0x0e,0xc3,0xdb,0x08,0xb3,0xd0,
0xac,0x57,0x0d,0x76,0xed,0xcd,0x27,0xd3,0x4c,0xad,0x50,0x83,0x61,0xe2,0xaa,0x20,
0x4d,0x09,0x2d,0x64,0x09,0xdc,0xce,0x89,0x9f,0xcc,0x3d,0xa9,0xec,0xf6,0xcf,0xc1,
0xdc,0xf1,0xd3,0xb1,0xd6,0x7b,0x37,0x28,0x11,0x2b,0x47,0xda,0x39,0xc6,0xbc,0x3a,
0x19,0xb4,0x5f,0xa6,0xbd,0x7d,0x9d,0xa3,0x63,0x42,0xb6,0x76,0xf2,0xa9,0x3b,0x2b,
0x91,0xf8,0xe2,0x6f,0xd0,0xec,0x16,0x20,0x90,0x09,0x3e,0xe2,0xe8,0x74,0xc9,0x18,
0xb4,0x91,0xd4,0x62,0x64,0xdb,0x7f,0xa3,0x06,0xf1,0x88,0x18,0x6a,0x90,0x22,0x3c,
0xbc,0xfe,0x13,0xf0,0x87,0x14,0x7b,0xf6,0xe4,0x1f,0x8e,0xd4,0xe4,0x51,0xc6,0x11,
0x67,0x46,0x08,0x51,0xcb,0x86,0x14,0x54,0x3f,0xbc,0x33,0xfe,0x7e,0x6c,0x9c,0xff,
0x16,0x9d,0x18,0xbd,0x51,0x8e,0x35,0xa6,0xa7,0x66,0xc8,0x72,0x67,0xdb,0x21,0x66,
0xb1,0xd4,0x9b,0x78,0x03,0xc0,0x50,0x3a,0xe8,0xcc,0xf0,0xdc,0xbc,0x9e,0x4c,0xfe,
0xaf,0x05,0x96,0x35,0x1f,0x57,0x5a,0xb7,0xff,0xce,0xf9,0x3d,0xb7,0x2c,0xb6,0xf6,
0x54,0xdd,0xc8,0xe7,0x12,0x3a,0x4d,0xae,0x4c,0x8a,0xb7,0x5c,0x9a,0xb4,0xb7,0x20,
0x3d,0xca,0x7f,0x22,0x34,0xae,0x7e,0x3b,0x68,0x66,0x01,0x44,0xe7,0x01,0x4e,0x46,
0x53,0x9b,0x33,0x60,0xf7,0x94,0xbe,0x53,0x37,0x90,0x73,0x43,0xf3,0x32,0xc3,0x53,
0xef,0xdb,0xaa,0xfe,0x74,0x4e,0x69,0xc7,0x6b,0x8c,0x60,0x93,0xde,0xc4,0xc7,0x0c,
0xdf,0xe1,0x32,0xae,0xcc,0x93,0x3b,0x51,0x78,0x95,0x67,0x8b,0xee,0x3d,0x56,0xfe,
0x0c,0xd0,0x69,0x0f,0x1b,0x0f,0xf3,0x25,0x26,0x6b,0x33,0x6d,0xf7,0x6e,0x47,0xfa,
0x73,0x43,0xe5,0x7e,0x0e,0xa5,0x66,0xb1,0x29,0x7c,0x32,0x84,0x63,0x55,0x89,0xc4,
0x0d,0xc1,0x93,0x54,0x30,0x19,0x13,0xac,0xd3,0x7d,0x37,0xa7,0xeb,0x5d,0x3a,0x6c,
0x35,0x5c,0xdb,0x41,0xd7,0x12,0xda,0xa9,0x49,0x0b,0xdf,0xd8,0x80,0x8a,0x09,0x93,
0x62,0x8e,0xb5,0x66,0xcf,0x25,0x88,0xcd,0x84,0xb8,0xb1,0x3f,0xa4,0x39,0x0f,0xd9,
0x02,0x9e,0xeb,0x12,0x4c,0x95,0x7c,0xf3,0x6b,0x05,0xa9,0x5e,0x16,0x83,0xcc,0xb8,
0x67,0xe2,0xe8,0x13,0x9d,0xcc,0x5b,0x82,0xd3,0x4c,0xb3,0xed,0x5b,0xff,0xde,0xe5,
0x73,0xac,0x23,0x3b,0x2d,0x00,0xbf,0x35,0x55,0x74,0x09,0x49,0xd8,0x49,0x58,0x1a,
0x7f,0x92,0x36,0xe6,0x51,0x92,0x0e,0xf3,0x26,0x7d,0x1c,0x4d,0x17,0xbc,0xc9,0xec,
0x43,0x26,0xd0,0xbf,0x41,0x5f,0x40,0xa9,0x44,0x44,0xf4,0x99,0xe7,0x57,0x87,0x9e,
0x50,0x1f,0x57,0x54,0xa8,0x3e,0xfd,0x74,0x63,0x2f,0xb1,0x50,0x65,0x09,0xe6,0x58,
0x42,0x2e,0x43,0x1a,0x4c,0xb4,0xf0,0x25,0x47,0x59,0xfa,0x04,0x1e,0x93,0xd4,0x26,
0x46,0x4a,0x50,0x81,0xb2,0xde,0xbe,0x78,0xb7,0xfc,0x67,0x15,0xe1,0xc9,0x57,0x84,
0x1e,0x0f,0x63,0xd6,0xe9,0x62,0xba,0xd6,0x5f,0x55,0x2e,0xea,0x5c,0xc6,0x28,0x08,
0x04,0x25,0x39,0xb8,0x0e,0x2b,0xa9,0xf2,0x4c,0x97,0x1c,0x07,0x3f,0x0d,0x52,0xf5,
0xed,0xef,0x2f,0x82,0x0f,0x02,0x03,0x01,0x00,0x01,0xa3,0x42,0x30,0x40,0x30,0x1d,
0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x53,0x79,0xbf,0x5a,0xaa,0x2b,0x4a,
0xcf,0x54,0x80,0xe1,0xd8,0x9b,0xc0,0x9d,0xf2,0xb2,0x03,0x66,0xcb,0x30,0x0e,0x06,
0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x01,0x06,0x30,0x0f,0x06,
0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0c,0x05,0x00,0x03,0x82,0x02,
0x01,0x00,0x5c,0xd4,0x7c,0x0d,0xcf,0xf7,0x01,0x7d,0x41,0x99,0x65,0x0c,0x73,0xc5,
0x52,0x9f,0xcb,0xf8,0xcf,0x99,0x06,0x7f,0x1b,0xda,0x43,0x15,0x9f,0x9e,0x02,0x55,
0x57,0x96,0x14,0xf1,0x52,0x3c,0x27,0x87,0x94,0x28,0xed,0x1f,0x3a,0x01,0x37,0xa2,
0x76,0xfc,0x53,0x50,0xc0,0x84,0x9b,0xc6,0x6b,0x4e,0xba,0x8c,0x21,0x4f,0xa2,0x8e,
0x55,0x62,0x91,0xf3,0x69,0x15,0xd8,0xbc,0x88,0xe3,0xc4,0xaa,0x0b,0xfd,0xef,0xa8,
0xe9,0x4b,0x55,0x2a,0x06,0x20,0x6d,0x55,0x78,0x29,0x19,0xee,0x5f,0x30,0x5c,0x4b,
0x24,0x11,0x55,0xff,0x24,0x9a,0x6e,0x5e,0x2a,0x2b,0xee,0x0b,0x4d,0x9f,0x7f,0xf7,
0x01,0x38,0x94,0x14,0x95,0x43,0x07,0x09,0xfb,0x60,0xa9,0xee,0x1c,0xab,0x12,0x8c,
0xa0,0x9a,0x5e,0xa7,0x98,0x6a,0x59,0x6d,0x8b,0x3f,0x08,0xfb,0xc8,0xd1,0x45,0xaf,
0x18,0x15,0x64,0x90,0x12,0x0f,0x73,0x28,0x2e,0xc5,0xe2,0x24,0x4e,0xfc,0x58,0xec,
0xf0,0xf4,0x45,0xfe,0x22,0xb3,0xeb,0x2f,0x8e,0xd2,0xd9,0x45,0x61,0x05,0xc1,0x97,
0x6f,0xa8,0x76,0x72,0x8f,0x8b,0x8c,0x36,0xaf,0xbf,0x0d,0x05,0xce,0x71,0x8d,0xe6,
0xa6,0x6f,0x1f,0x6c,0xa6,0x71,0x62,0xc5,0xd8,0xd0,0x83,0x72,0x0c,0xf1,0x67,0x11,
0x89,0x0c,0x9c,0x13,0x4c,0x72,0x34,0xdf,0xbc,0xd5,0x71,0xdf,0xaa,0x71,0xdd,0xe1,
0xb9,0x6c,0x8c,0x3c,0x12,0x5d,0x65,0xda,0xbd,0x57,0x12,0xb6,0x43,0x6b,0xff,0xe5,
0xde,0x4d,0x66,0x11,0x51,0xcf,0x99,0xae,0xec,0x17,0xb6,0xe8,0x71,0x91,0x8c,0xde,
0x49,0xfe,0xdd,0x35,0x71,0xa2,0x15,0x27,0x94,0x1c,0xcf,0x61,0xe3,0x26,0xbb,0x6f,
0xa3,0x67,0x25,0x21,0x5d,0xe6,0xdd,0x1d,0x0b,0x2e,0x68,0x1b,0x3b,0x82,0xaf,0xec,
0x83,0x67,0x85,0xd4,0x98,0x51,0x74,0xb1,0xb9,0x99,0x80,0x89,0xff,0x7f,0x78,0x19,
0x5c,0x79,0x4a,0x60,0x2e,0x92,0x40,0xae,0x4c,0x37,0x2a,0x2c,0xc9,0xc7,0x62,0xc8,
0x0e,0x5d,0xf7,0x36,0x5b,0xca,0xe0,0x25,0x25,0x01,0xb4,0xdd,0x1a,0x07,0x9c,0x77,
0x00,0x3f,0xd0,0xdc,0xd5,0xec,0x3d,0xd4,0xfa,0xbb,0x3f,0xcc,0x85,0xd6,0x6f,0x7f,
0xa9,0x2d,0xdf,0xb9,0x02,0xf7,0xf5,0x97,0x9a,0xb5,0x35,0xda,0xc3,0x67,0xb0,0x87,
0x4a,0xa9,0x28,0x9e,0x23,0x8e,0xff,0x5c,0x27,0x6b,0xe1,0xb0,0x4f,0xf3,0x07,0xee,
0x00,0x2e,0xd4,0x59,0x87,0xcb,0x52,0x41,0x95,0xea,0xf4,0x47,0xd7,0xee,0x64,0x41,
0x55,0x7c,0x8d,0x59,0x02,0x95,0xdd,0x62,0x9d,0xc2,0xb9,0xee,0x5a,0x28,0x74,0x84,
0xa5,0x9b,0xb7,0x90,0xc7,0x0c,0x07,0xdf,0xf5,0x89,0x36,0x74,0x32,0xd6,0x28,0xc1,
0xb0,0xb0,0x0b,0xe0,0x9c,0x4c,0xc3,0x1c,0xd6,0xfc,0xe3,0x69,0xb5,0x47,0x46,0x81,
0x2f,0xa2,0x82,0xab,0xd3,0x63,0x44,0x70,0xc4,0x8d,0xff,0x2d,0x33,0xba,0xad,0x8f,
0x7b,0xb5,0x70,0x88,0xae,0x3e,0x19,0xcf,0x40,0x28,0xd8,0xfc,0xc8,0x90,0xbb,0x5d,
0x99,0x22,0xf5,0x52,0xe6,0x58,0xc5,0x1f,0x88,0x31,0x43,0xee,0x88,0x1d,0xd7,0xc6,
0x8e,0x3c,0x43,0x6a,0x1d,0xa7,0x18,0xde,0x7d,0x3d,0x16,0xf1,0x62,0xf9,0xca,0x90,
0xa8,0xfd
};
static const BYTE incommon_rsa_ca[] = {
0x30,0x82,0x05,0xf9,0x30,0x82,0x03,0xe1,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x47,
0x20,0xd0,0xfa,0x85,0x46,0x1a,0x7e,0x17,0xa1,0x64,0x02,0x91,0x84,0x63,0x74,0x30,
0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0c,0x05,0x00,0x30,0x81,
0x88,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,
0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x4e,0x65,0x77,0x20,0x4a,0x65,0x72,
0x73,0x65,0x79,0x31,0x14,0x30,0x12,0x06,0x03,0x55,0x04,0x07,0x13,0x0b,0x4a,0x65,
0x72,0x73,0x65,0x79,0x20,0x43,0x69,0x74,0x79,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,
0x04,0x0a,0x13,0x15,0x54,0x68,0x65,0x20,0x55,0x53,0x45,0x52,0x54,0x52,0x55,0x53,
0x54,0x20,0x4e,0x65,0x74,0x77,0x6f,0x72,0x6b,0x31,0x2e,0x30,0x2c,0x06,0x03,0x55,
0x04,0x03,0x13,0x25,0x55,0x53,0x45,0x52,0x54,0x72,0x75,0x73,0x74,0x20,0x52,0x53,
0x41,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x20,
0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x30,0x1e,0x17,0x0d,0x31,0x34,0x31,
0x30,0x30,0x36,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x32,0x34,0x31,0x30,
0x30,0x35,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x76,0x31,0x0b,0x30,0x09,0x06,
0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,
0x08,0x13,0x02,0x4d,0x49,0x31,0x12,0x30,0x10,0x06,0x03,0x55,0x04,0x07,0x13,0x09,
0x41,0x6e,0x6e,0x20,0x41,0x72,0x62,0x6f,0x72,0x31,0x12,0x30,0x10,0x06,0x03,0x55,
0x04,0x0a,0x13,0x09,0x49,0x6e,0x74,0x65,0x72,0x6e,0x65,0x74,0x32,0x31,0x11,0x30,
0x0f,0x06,0x03,0x55,0x04,0x0b,0x13,0x08,0x49,0x6e,0x43,0x6f,0x6d,0x6d,0x6f,0x6e,
0x31,0x1f,0x30,0x1d,0x06,0x03,0x55,0x04,0x03,0x13,0x16,0x49,0x6e,0x43,0x6f,0x6d,
0x6d,0x6f,0x6e,0x20,0x52,0x53,0x41,0x20,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,
0x41,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,
0x01,0x00,0x9c,0x1b,0xf1,0xbb,0x2f,0x7f,0x63,0x18,0x15,0x51,0x51,0x54,0x0f,0x9e,
0xc5,0x4e,0x4d,0x10,0x58,0xfa,0x30,0x9b,0x17,0x29,0x90,0xe6,0x33,0x0c,0xac,0x13,
0x53,0x7c,0x54,0x91,0xb4,0xea,0xd8,0x6e,0x9b,0x89,0x6d,0xbb,0x33,0x3e,0x8f,0xd2,
0x0d,0xa6,0xe9,0xf9,0xba,0xe9,0x0d,0x0c,0x1a,0x9e,0xb2,0x8e,0xc9,0x70,0x2e,0xef,
0x1e,0x05,0x7d,0x95,0xeb,0x2d,0x8d,0xa2,0xa9,0x4d,0xb3,0x9c,0xe7,0xf3,0x19,0x36,
0xbb,0xa7,0xf1,0x7c,0xe6,0x08,0x1e,0x61,0x27,0x44,0x7a,0x96,0xf4,0xa8,0x34,0xdb,
0xe2,0x42,0xc8,0xa5,0xdb,0x37,0xd5,0xb5,0xe7,0xe4,0x42,0x72,0x3f,0xb4,0x13,0xcf,
0x8b,0x07,0x24,0x45,0x1e,0x8c,0x91,0x83,0x46,0xb9,0x09,0xa6,0xfc,0x18,0xa3,0x06,
0x02,0xec,0x34,0x8d,0x32,0x66,0x95,0x27,0xea,0xe1,0x97,0xe8,0xdb,0x35,0xa3,0x2b,
0x56,0xeb,0x57,0xe8,0xf0,0x10,0x59,0xdf,0x6d,0x70,0x0c,0x66,0x6a,0xd0,0x64,0xe5,
0xa8,0xa3,0x98,0x31,0xad,0x1d,0x62,0xd5,0xfa,0x92,0xe3,0x9a,0x43,0xcd,0x2d,0x35,
0xfb,0xd9,0x9e,0x33,0x5b,0x45,0x7d,0xc4,0x86,0x28,0x2c,0x66,0x12,0xc8,0xdb,0x0f,
0x19,0x30,0x0d,0x3f,0xe9,0xf0,0xea,0x4a,0x5e,0x40,0x07,0xc7,0xf6,0x20,0x7a,0x53,
0x78,0x81,0x64,0x7a,0x7e,0x45,0x6a,0x16,0x6f,0xf4,0x93,0x58,0xc9,0x62,0xfb,0x29,
0x27,0x7d,0xa1,0x7f,0x21,0xce,0xe7,0x4f,0x47,0xd6,0x8a,0x56,0xe0,0xe3,0x66,0xf8,
0xec,0xdd,0x89,0xdc,0x26,0x8c,0x19,0x68,0x3b,0x8d,0x8b,0xe2,0xfb,0x47,0x23,0x0b,
0x7f,0x37,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x6e,0x30,0x82,0x01,0x6a,0x30,
0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x53,0x79,0xbf,0x5a,
0xaa,0x2b,0x4a,0xcf,0x54,0x80,0xe1,0xd8,0x9b,0xc0,0x9d,0xf2,0xb2,0x03,0x66,0xcb,
0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x1e,0x05,0xa3,0x77,0x8f,
0x6c,0x96,0xe2,0x5b,0x87,0x4b,0xa6,0xb4,0x86,0xac,0x71,0x00,0x0c,0xe7,0x38,0x30,
0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x01,0x86,0x30,
0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,0x06,0x01,0x01,0xff,
0x02,0x01,0x00,0x30,0x1d,0x06,0x03,0x55,0x1d,0x25,0x04,0x16,0x30,0x14,0x06,0x08,
0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x01,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,
0x03,0x02,0x30,0x1b,0x06,0x03,0x55,0x1d,0x20,0x04,0x14,0x30,0x12,0x30,0x06,0x06,
0x04,0x55,0x1d,0x20,0x00,0x30,0x08,0x06,0x06,0x67,0x81,0x0c,0x01,0x02,0x02,0x30,
0x50,0x06,0x03,0x55,0x1d,0x1f,0x04,0x49,0x30,0x47,0x30,0x45,0xa0,0x43,0xa0,0x41,
0x86,0x3f,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x75,0x73,0x65,
0x72,0x74,0x72,0x75,0x73,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x55,0x53,0x45,0x52,0x54,
0x72,0x75,0x73,0x74,0x52,0x53,0x41,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,
0x74,0x69,0x6f,0x6e,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x2e,0x63,0x72,
0x6c,0x30,0x76,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x6a,0x30,
0x68,0x30,0x3f,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x33,0x68,
0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x74,0x2e,0x75,0x73,0x65,0x72,0x74,0x72,
0x75,0x73,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x55,0x53,0x45,0x52,0x54,0x72,0x75,0x73,
0x74,0x52,0x53,0x41,0x41,0x64,0x64,0x54,0x72,0x75,0x73,0x74,0x43,0x41,0x2e,0x63,
0x72,0x74,0x30,0x25,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x19,
0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,0x75,0x73,0x65,0x72,
0x74,0x72,0x75,0x73,0x74,0x2e,0x63,0x6f,0x6d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x0c,0x05,0x00,0x03,0x82,0x02,0x01,0x00,0x2d,0x11,0x06,
0x38,0xd6,0xdb,0xd7,0x58,0x68,0xaf,0xaa,0x38,0x67,0x17,0x8d,0xe2,0x13,0xd7,0xa3,
0x14,0x24,0xd9,0x06,0x13,0xeb,0xeb,0x91,0x2f,0xdf,0x4f,0x67,0x2d,0xc8,0xd3,0x14,
0xd7,0x56,0x65,0x52,0x9e,0x6e,0x1f,0x98,0x08,0x8e,0x9a,0x48,0x1b,0xc1,0x8b,0x59,
0x9a,0xa3,0x57,0x9b,0xdb,0x86,0xf8,0x59,0x40,0xfc,0x19,0xb0,0x75,0x11,0x2a,0xc2,
0x12,0x36,0xba,0x8e,0x72,0x8a,0x06,0x4e,0x27,0xb7,0x8d,0x58,0x14,0xd1,0x6f,0xb4,
0xf9,0x68,0xfc,0x98,0xdd,0xa4,0x9c,0x25,0x40,0x36,0xde,0xbd,0x17,0x66,0x2b,0x03,
0x7f,0x78,0x81,0xb1,0x80,0x74,0x9e,0x5f,0x3a,0xb4,0x26,0x2f,0x6a,0x48,0x84,0x36,
0x34,0x8e,0xa7,0x28,0xef,0x87,0xf3,0x61,0xe7,0xdb,0x67,0xf5,0x52,0xdb,0xd7,0xd1,
0xe6,0x30,0x71,0xbb,0x8b,0xa3,0xd4,0xff,0xb9,0x64,0x89,0x9e,0x9b,0x81,0x9b,0x8f,
0x57,0xb8,0x64,0x4c,0xd5,0x06,0x19,0x8e,0xe7,0x91,0x85,0x7c,0x18,0xd1,0x89,0xd8,
0xf6,0xea,0x1d,0x68,0x14,0x11,0xd9,0xee,0x17,0x83,0x1f,0x50,0x63,0xcf,0x0e,0xf6,
0x86,0x2a,0x6e,0xe3,0xb1,0xa4,0xc9,0xfa,0xf6,0x34,0x4c,0x77,0x2a,0x80,0x86,0x30,
0xb0,0xa3,0xdc,0x1b,0x71,0xec,0x04,0xa7,0xe4,0x98,0xbc,0x16,0x85,0x3e,0x84,0x26,
0xb3,0xc0,0xe5,0x35,0x55,0x7e,0x79,0x98,0xa3,0xd4,0xd4,0x8d,0xb6,0xe7,0x42,0xe8,
0x44,0x20,0x12,0x37,0x5f,0x09,0xc9,0xfb,0x03,0xe4,0xf5,0x65,0x74,0x96,0xed,0xca,
0xb9,0xb3,0xf6,0x09,0xff,0x4c,0xa6,0xd1,0x5d,0x3a,0xfc,0xd1,0x4d,0xaa,0xe4,0x98,
0x72,0xbe,0x38,0x4b,0x7f,0x89,0x4e,0x26,0x8f,0xd4,0xcc,0xbe,0x56,0x09,0x71,0x03,
0x4a,0x6c,0xa3,0xe2,0x35,0x86,0xdd,0x1e,0xd9,0xf1,0x31,0x03,0xf7,0x13,0x4d,0x0b,
0x11,0x81,0x31,0x79,0xcc,0x7a,0xd7,0xbe,0xdc,0xfb,0xf3,0x76,0x1b,0x2c,0xbd,0xb3,
0x91,0x0f,0x00,0x59,0x07,0x2a,0x20,0x43,0xdc,0x4b,0xd8,0xb5,0x19,0x14,0x8f,0xe2,
0x7a,0x84,0x29,0xd1,0x43,0x3f,0x2f,0xcc,0xdf,0x3f,0x9d,0xbb,0xbd,0x68,0xc4,0xce,
0xe0,0xcd,0xe7,0x1c,0x31,0x32,0x78,0x62,0xfa,0xf0,0x93,0xa2,0x1e,0xc9,0xd7,0x9f,
0x68,0xe5,0xa8,0x76,0xf6,0x63,0xfe,0x68,0x99,0xef,0xba,0x36,0xd7,0x12,0x71,0x9a,
0x9e,0xb3,0x71,0x1f,0x3b,0xbe,0x00,0x63,0x9e,0x3d,0x5f,0x21,0xc2,0xb1,0x86,0x1b,
0xb8,0x4e,0x21,0xc3,0xc3,0x43,0x09,0x2e,0x63,0x0c,0xcd,0xff,0x14,0xf6,0xf6,0x22,
0xe9,0xfd,0xca,0x9f,0xf5,0x98,0x44,0xb6,0x41,0x9c,0x41,0xc2,0x08,0x98,0x7d,0xdb,
0xa0,0x9f,0x22,0x7e,0xc0,0xa7,0x49,0xbb,0xb4,0x18,0x1f,0x4b,0xd3,0xa6,0x2a,0x87,
0xb9,0x5c,0xca,0xf2,0x83,0x4c,0x40,0x03,0xb2,0x52,0x1a,0x79,0x21,0x08,0x37,0x18,
0x4e,0xd9,0x8d,0x5f,0x99,0xc6,0x05,0x5f,0xf1,0x6a,0xae,0xba,0x75,0x5a,0x78,0x47,
0x3a,0x3a,0x65,0x5e,0xe5,0xc4,0xd0,0xe3,0xda,0xd2,0xeb,0x5a,0x28,0x2d,0xb9,0x02,
0x99,0x60,0xa2,0x6f,0x3c,0x2f,0x66,0x7c,0x98,0x45,0x9c,0xc9,0xfa,0x01,0xef,0x32,
0x8e,0x7c,0x3e,0xf9,0xf4,0x03,0x7b,0x24,0xa6,0x56,0x09,0x8c,0x24
};
static const BYTE cs_stanford_edu[] = {
0x30,0x82,0x05,0xa5,0x30,0x82,0x04,0x8d,0xa0,0x03,0x02,0x01,0x02,0x02,0x11,0x00,
0xf1,0x06,0x35,0xc6,0xee,0x01,0x95,0x98,0x11,0xab,0x60,0x00,0x75,0x72,0x30,0xb3,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,
0x76,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0b,
0x30,0x09,0x06,0x03,0x55,0x04,0x08,0x13,0x02,0x4d,0x49,0x31,0x12,0x30,0x10,0x06,
0x03,0x55,0x04,0x07,0x13,0x09,0x41,0x6e,0x6e,0x20,0x41,0x72,0x62,0x6f,0x72,0x31,
0x12,0x30,0x10,0x06,0x03,0x55,0x04,0x0a,0x13,0x09,0x49,0x6e,0x74,0x65,0x72,0x6e,
0x65,0x74,0x32,0x31,0x11,0x30,0x0f,0x06,0x03,0x55,0x04,0x0b,0x13,0x08,0x49,0x6e,
0x43,0x6f,0x6d,0x6d,0x6f,0x6e,0x31,0x1f,0x30,0x1d,0x06,0x03,0x55,0x04,0x03,0x13,
0x16,0x49,0x6e,0x43,0x6f,0x6d,0x6d,0x6f,0x6e,0x20,0x52,0x53,0x41,0x20,0x53,0x65,
0x72,0x76,0x65,0x72,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x31,0x36,0x31,0x30,0x32,
0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x31,0x39,0x31,0x30,0x32,0x31,
0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x81,0xb4,0x31,0x0b,0x30,0x09,0x06,0x03,
0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x11,
0x13,0x05,0x39,0x34,0x33,0x30,0x35,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x08,
0x13,0x02,0x43,0x41,0x31,0x11,0x30,0x0f,0x06,0x03,0x55,0x04,0x07,0x13,0x08,0x53,
0x74,0x61,0x6e,0x66,0x6f,0x72,0x64,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x09,
0x13,0x0e,0x34,0x35,0x30,0x20,0x53,0x65,0x72,0x72,0x61,0x20,0x4d,0x61,0x6c,0x6c,
0x31,0x1c,0x30,0x1a,0x06,0x03,0x55,0x04,0x0a,0x13,0x13,0x53,0x74,0x61,0x6e,0x66,
0x6f,0x72,0x64,0x20,0x55,0x6e,0x69,0x76,0x65,0x72,0x73,0x69,0x74,0x79,0x31,0x24,
0x30,0x22,0x06,0x03,0x55,0x04,0x0b,0x13,0x1b,0x43,0x6f,0x6d,0x70,0x75,0x74,0x65,
0x72,0x20,0x53,0x63,0x69,0x65,0x6e,0x63,0x65,0x20,0x44,0x65,0x70,0x61,0x72,0x74,
0x6d,0x65,0x6e,0x74,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x03,0x13,0x0f,0x63,
0x73,0x2e,0x73,0x74,0x61,0x6e,0x66,0x6f,0x72,0x64,0x2e,0x65,0x64,0x75,0x30,0x82,
0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xb5,
0x8c,0x69,0x22,0xf1,0xd4,0x39,0x0a,0x45,0x7b,0x5f,0x1e,0x15,0x00,0xae,0x9a,0x0f,
0x79,0x5c,0x08,0xea,0x52,0x0e,0x74,0x6d,0xcf,0x0b,0x1a,0x92,0x88,0x57,0xe4,0x8c,
0x9e,0xbc,0xc6,0xd9,0x29,0x8f,0xe1,0x5f,0x52,0x11,0x59,0xf8,0x12,0x78,0xfa,0xdf,
0xac,0xe6,0xdf,0xfb,0x3f,0x10,0xf3,0x93,0x23,0xdd,0xea,0x6c,0x0b,0xf6,0xda,0x99,
0xbc,0x15,0x51,0xf9,0x8d,0x03,0x9e,0x79,0x28,0xf9,0x44,0x5d,0x06,0xac,0x95,0x61,
0xc6,0x81,0xe4,0x6e,0x49,0x1a,0x8f,0xe9,0xea,0x1c,0x7d,0x05,0x53,0x20,0xfc,0xea,
0x14,0xc4,0x2f,0x11,0x57,0x86,0x3a,0x53,0xc1,0x17,0x6b,0xb1,0xe6,0x77,0x4c,0x21,
0x0d,0xe0,0x68,0x6c,0xac,0xc9,0xcb,0xe9,0xc2,0x5a,0x91,0xcb,0x51,0x1a,0xb8,0x53,
0x13,0x75,0x13,0xef,0xfd,0x50,0x0b,0xd7,0x84,0xb0,0xb8,0x29,0x2c,0x64,0x00,0xd9,
0x11,0xe5,0xef,0x9d,0x9d,0xb3,0xae,0x5f,0xa2,0x78,0x24,0x18,0xc3,0x68,0x13,0x86,
0x0d,0x86,0xf8,0x2d,0xfe,0x47,0xba,0xae,0xe8,0xc2,0x6c,0xf3,0x80,0xc9,0x96,0x33,
0xc2,0x08,0xa3,0x01,0xc9,0x70,0x04,0x25,0x2a,0x32,0x0e,0xd5,0x84,0xe3,0x91,0x57,
0xb3,0x9c,0xbf,0x9f,0x37,0x71,0xbb,0x0a,0x21,0x22,0x09,0x01,0xfc,0x10,0x75,0x14,
0x0c,0x53,0x3f,0xbe,0x6d,0x1f,0x96,0x4d,0xb5,0x3c,0xcd,0x2f,0xd9,0x9d,0xe7,0xb8,
0x52,0x8b,0xac,0x24,0x8a,0x95,0xf4,0xd3,0x44,0xc3,0xe0,0x9f,0x85,0x86,0x3f,0x2e,
0xf0,0xbe,0x73,0x8e,0x82,0x01,0x46,0x59,0x8d,0xd7,0x90,0x64,0x4a,0x2d,0x4d,0x02,
0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0xed,0x30,0x82,0x01,0xe9,0x30,0x1f,0x06,0x03,
0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x1e,0x05,0xa3,0x77,0x8f,0x6c,0x96,
0xe2,0x5b,0x87,0x4b,0xa6,0xb4,0x86,0xac,0x71,0x00,0x0c,0xe7,0x38,0x30,0x1d,0x06,
0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x48,0xf2,0x68,0xbb,0xd9,0xab,0xe3,0x04,
0xba,0x2f,0x9d,0x7d,0x94,0xca,0xcf,0x77,0xd7,0xb1,0x0b,0x23,0x30,0x0e,0x06,0x03,
0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x05,0xa0,0x30,0x0c,0x06,0x03,
0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x1d,0x06,0x03,0x55,0x1d,
0x25,0x04,0x16,0x30,0x14,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x01,0x06,
0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x02,0x30,0x67,0x06,0x03,0x55,0x1d,0x20,
0x04,0x60,0x30,0x5e,0x30,0x52,0x06,0x0c,0x2b,0x06,0x01,0x04,0x01,0xae,0x23,0x01,
0x04,0x03,0x01,0x01,0x30,0x42,0x30,0x40,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,
0x02,0x01,0x16,0x34,0x68,0x74,0x74,0x70,0x73,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,
0x69,0x6e,0x63,0x6f,0x6d,0x6d,0x6f,0x6e,0x2e,0x6f,0x72,0x67,0x2f,0x63,0x65,0x72,
0x74,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,0x72,0x79,0x2f,0x63,0x70,0x73,
0x5f,0x73,0x73,0x6c,0x2e,0x70,0x64,0x66,0x30,0x08,0x06,0x06,0x67,0x81,0x0c,0x01,
0x02,0x02,0x30,0x44,0x06,0x03,0x55,0x1d,0x1f,0x04,0x3d,0x30,0x3b,0x30,0x39,0xa0,
0x37,0xa0,0x35,0x86,0x33,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,
0x69,0x6e,0x63,0x6f,0x6d,0x6d,0x6f,0x6e,0x2d,0x72,0x73,0x61,0x2e,0x6f,0x72,0x67,
0x2f,0x49,0x6e,0x43,0x6f,0x6d,0x6d,0x6f,0x6e,0x52,0x53,0x41,0x53,0x65,0x72,0x76,
0x65,0x72,0x43,0x41,0x2e,0x63,0x72,0x6c,0x30,0x75,0x06,0x08,0x2b,0x06,0x01,0x05,
0x05,0x07,0x01,0x01,0x04,0x69,0x30,0x67,0x30,0x3e,0x06,0x08,0x2b,0x06,0x01,0x05,
0x05,0x07,0x30,0x02,0x86,0x32,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x74,
0x2e,0x75,0x73,0x65,0x72,0x74,0x72,0x75,0x73,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x49,
0x6e,0x43,0x6f,0x6d,0x6d,0x6f,0x6e,0x52,0x53,0x41,0x53,0x65,0x72,0x76,0x65,0x72,
0x43,0x41,0x5f,0x32,0x2e,0x63,0x72,0x74,0x30,0x25,0x06,0x08,0x2b,0x06,0x01,0x05,
0x05,0x07,0x30,0x01,0x86,0x19,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,
0x70,0x2e,0x75,0x73,0x65,0x72,0x74,0x72,0x75,0x73,0x74,0x2e,0x63,0x6f,0x6d,0x30,
0x44,0x06,0x03,0x55,0x1d,0x11,0x04,0x3d,0x30,0x3b,0x82,0x0f,0x63,0x73,0x2e,0x73,
0x74,0x61,0x6e,0x66,0x6f,0x72,0x64,0x2e,0x65,0x64,0x75,0x82,0x13,0x77,0x77,0x77,
0x2d,0x63,0x73,0x2e,0x73,0x74,0x61,0x6e,0x66,0x6f,0x72,0x64,0x2e,0x65,0x64,0x75,
0x82,0x13,0x77,0x77,0x77,0x2e,0x63,0x73,0x2e,0x73,0x74,0x61,0x6e,0x66,0x6f,0x72,
0x64,0x2e,0x65,0x64,0x75,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x0b,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x7e,0xad,0x31,0x04,0x9a,0xe0,0xc2,
0x42,0x8a,0x3b,0x5a,0x8a,0x5f,0xe9,0x50,0x33,0xd7,0xf9,0x18,0xd3,0x96,0x05,0x15,
0xe5,0xcf,0x61,0x99,0x0d,0x97,0xda,0xf4,0xdd,0x6e,0x45,0xfb,0xe4,0x0e,0x4d,0x23,
0xcd,0xe1,0xe0,0x46,0x8f,0x96,0x35,0xa7,0x04,0xe5,0x78,0xfe,0x78,0xed,0xf9,0x33,
0xeb,0xde,0x0b,0x9f,0xf4,0x79,0xba,0x2a,0x04,0xea,0xdb,0x63,0xd4,0x8c,0x82,0x12,
0x0e,0xdd,0xf1,0xa5,0xa4,0x36,0x86,0x43,0x75,0x1f,0x2e,0x84,0xcd,0xbf,0x7a,0x80,
0x9f,0x96,0x89,0x58,0x69,0xe1,0x31,0x17,0xb1,0x16,0x34,0x53,0xb6,0xf0,0x31,0xfd,
0xc1,0x1e,0x9e,0x40,0x9f,0xa7,0x56,0x26,0xb0,0xa4,0x15,0x19,0x70,0x1c,0xb3,0x47,
0x2b,0x8e,0x90,0x43,0x37,0xd7,0xb5,0xba,0x54,0xe5,0xa4,0xda,0x69,0xae,0xb8,0x73,
0x35,0x47,0x01,0xac,0xd9,0x21,0xff,0x18,0x88,0x0b,0xda,0x4a,0xff,0x26,0x2f,0xef,
0x15,0xcd,0x6f,0x79,0xf3,0xcb,0x0a,0xaf,0x11,0x3b,0x7e,0xc0,0x11,0xbb,0xeb,0x13,
0xe7,0xb2,0x28,0x62,0x92,0x90,0x63,0x50,0xf2,0x6f,0x93,0x92,0x67,0xc0,0xde,0x22,
0x3f,0x6e,0x25,0x00,0x60,0x42,0xfe,0x29,0x8d,0x7b,0x5c,0xc1,0xa9,0x09,0xb4,0xc6,
0x18,0x56,0x5b,0xb6,0xd0,0xd2,0xdd,0x1b,0x51,0xcb,0x97,0xe7,0x20,0x94,0x97,0xb1,
0x40,0xff,0x05,0xec,0x10,0xd6,0xa3,0x4d,0xc8,0x32,0x99,0xf0,0xab,0x51,0xda,0x5b,
0x12,0x70,0x63,0x8a,0x5b,0x63,0x6d,0xc8,0x31,0x76,0x70,0xe3,0x02,0x18,0x4e,0x0e,
0x9c,0xea,0xab,0x78,0xfa,0x07,0x48,0x59,0xc6
};

/* chain0_0 -> chain0_1:
 * A simple, valid chain
 */
static const BYTE chain0_0[] = {
0x30,0x82,0x01,0xc2,0x30,0x82,0x01,0x2d,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x73,0x87,0xdb,0x32,0x3a,0x6c,0x89,0x74,0x0f,0xda,0x07,0xd4,0x6a,0x2c,0x81,
0x59,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x82,0xcd,0x5c,0x0a,
0x9d,0x0d,0x3f,0xd1,0xd8,0x74,0xfc,0x52,0x65,0x11,0x76,0x10,0x62,0xb0,0x9a,
0xc5,0x54,0x66,0xae,0xfa,0xd5,0x66,0x36,0xcb,0x1e,0xe7,0xad,0xb9,0xfa,0x2e,
0xbb,0xb9,0x0b,0x4b,0xbf,0xe7,0x26,0x75,0x2c,0xae,0x2d,0x14,0x84,0x92,0xae,
0x56,0xdf,0x55,0x56,0xf3,0xa6,0xc5,0x21,0xa8,0x5e,0xce,0x7a,0x9d,0xff,0x7f,
0x10,0xd9,0xc3,0x05,0x52,0x86,0xdc,0x9b,0x46,0xe7,0x69,0xfb,0x1e,0xf0,0x68,
0x7d,0x34,0xd7,0xf3,0x56,0xe3,0x0a,0xf3,0xe4,0xdc,0x0f,0xd6,0x4d,0xa8,0xc5,
0xb5,0x5b,0xbf,0x0b,0x5d,0xe7,0x76,0xa1,0x56,0xc1,0xa7,0x55,0xa0,0x88,0x7f,
0x57,0x8e,0x64,0xda,0x6f,0xaa,0x9c,0xca,0x47,0xf3,0x37,0x08,0x4b,0x9a,0xdc,
0xbc,0x65,0x4d,0x9a };
static const BYTE chain0_1[] = {
0x30,0x82,0x01,0x98,0x30,0x82,0x01,0x01,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x31,0x31,0x32,
0x32,0x31,0x38,0x35,0x32,0x30,0x37,0x5a,0x18,0x0f,0x31,0x36,0x32,0x38,0x30,
0x38,0x32,0x36,0x30,0x35,0x32,0x30,0x35,0x37,0x5a,0x30,0x10,0x31,0x0e,0x30,
0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,
0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,
0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,
0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,
0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,
0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,
0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,
0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,
0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,
0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,
0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x3c,
0x90,0x45,0x5b,0x81,0x25,0x9d,0x7a,0x10,0x9f,0x15,0x25,0x03,0x52,0x78,0x09,
0x6f,0x62,0x66,0x78,0x01,0x69,0x6a,0x7f,0x19,0x02,0x18,0xd8,0xc0,0x89,0x2f,
0x0f,0x47,0xa7,0x17,0xc2,0xd7,0xe2,0x81,0x7e,0xea,0x54,0xd4,0xe6,0xcc,0x8f,
0x88,0x1a,0xee,0x28,0x42,0x6c,0x77,0xf5,0xec,0x8c,0x05,0xa5,0x7f,0xee,0x2e,
0x0b,0xbc,0xde,0xb0,0x71,0x2c,0x83,0x32,0xcb,0x49,0x00,0x46,0x03,0x95,0x58,
0x6c,0x57,0x0c,0x5a,0x69,0x3c,0xfa,0xbf,0x46,0x7a,0x4d,0xd0,0x23,0x77,0x23,
0xac,0x1d,0x33,0xb7,0x87,0xb4,0xc6,0x6b,0x52,0xa1,0xed,0x9a,0xb8,0x8e,0x1a,
0xcb,0xc3,0xd3,0x50,0xe0,0x0d,0x12,0x5d,0x0e,0x84,0x52,0xa5,0x12,0x76,0x6b,
0xbf,0xb8,0xbe,0x51,0xc0,0xa6,0xae };
/* chain0_0 -> chain1_1:
 * A chain whose signature is bad
 */
static const BYTE chain1_1[] = {
0x30,0x82,0x01,0x98,0x30,0x82,0x01,0x01,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x31,0x31,0x32,
0x32,0x31,0x38,0x35,0x32,0x30,0x37,0x5a,0x18,0x0f,0x31,0x36,0x32,0x38,0x30,
0x38,0x32,0x36,0x30,0x35,0x32,0x30,0x35,0x37,0x5a,0x30,0x10,0x31,0x0e,0x30,
0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,
0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xad,0x7e,0xca,
0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,
0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,
0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,
0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,
0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,
0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,
0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,
0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,
0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0xad,
0x16,0xaa,0x77,0xab,0xd5,0x1a,0x6c,0x90,0x5d,0x4e,0x61,0x49,0x89,0x6e,0x9f,
0x03,0x7b,0x4a,0x49,0xb1,0x46,0x75,0xf3,0x69,0xeb,0x96,0x13,0x3c,0x35,0xb5,
0x37,0x3b,0xc2,0x84,0x5c,0xe5,0x7c,0x46,0xf6,0x6b,0xc7,0x4e,0x72,0x91,0xf4,
0xde,0xc5,0x66,0x0f,0x2d,0x39,0xc6,0x9e,0x8e,0x25,0x4d,0x3f,0x7b,0x45,0xe1,
0xc6,0x1a,0x7e,0x28,0x83,0xf8,0x87,0x30,0x4e,0xa5,0xfd,0x32,0x7a,0xed,0x53,
0x10,0x3b,0x14,0xe5,0xf1,0x32,0x77,0xf1,0x29,0x72,0x98,0x2b,0xa3,0x17,0xd6,
0x8c,0x65,0xaa,0x3b,0x6d,0xab,0xb3,0xbe,0xfa,0x72,0x38,0xf7,0xd0,0xb6,0x0f,
0x5c,0x0d,0xec,0x17,0x43,0x2a,0xfd,0xc1,0x01,0x43,0x98,0xd3,0x1b,0x84,0x3e,
0x32,0xcf,0xda,0x1d,0xc2,0xc9,0x61 };
/* chain2_0 -> chain0_1:
 * A chain whose time nesting is invalid.
 */
static const BYTE chain2_0[] = {
0x30,0x82,0x01,0xc2,0x30,0x82,0x01,0x2d,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x11,0x58,0x73,0x34,0x62,0x2b,0xa5,0xa5,0x54,0x4a,0x14,0x48,0x3c,0x90,0x4e,
0x86,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x36,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x36,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x88,0x44,0x83,0xfc,
0x38,0xa4,0x6e,0x53,0x24,0x8f,0xac,0xcf,0xc0,0xb9,0xaa,0xdc,0x5f,0x30,0xf3,
0xc3,0x5c,0xd5,0x10,0x4e,0xb7,0x30,0x24,0x80,0xb3,0x5e,0xcb,0x6a,0xec,0x88,
0xa6,0x02,0xbd,0x37,0x97,0xcc,0x60,0x04,0x03,0xdd,0xfa,0xb0,0x4e,0xd6,0x00,
0x22,0x41,0xaa,0x2b,0x3c,0x3e,0x1f,0xcd,0xe3,0xf0,0x88,0x84,0x1c,0x00,0xc4,
0x97,0x69,0x3d,0x40,0x6d,0x33,0x8c,0xb7,0xb4,0x41,0xf3,0x28,0x00,0xa8,0xa1,
0x81,0x76,0x50,0x69,0xb2,0x2c,0xc4,0x56,0xab,0x2b,0x4a,0x77,0x41,0x3e,0x6c,
0x78,0x0b,0xf9,0x6c,0x86,0x84,0x1a,0x5a,0x0e,0x8c,0x12,0x59,0xbd,0x74,0xf3,
0xb9,0x91,0xa1,0x70,0x91,0xf3,0xe8,0x8d,0x78,0x42,0xc3,0x4f,0xfa,0xef,0xe7,
0xca,0x09,0x5c,0x18 };
/* chain3_0 -> chain0_1:
 * A chain whose root cannot be a CA.
 */
static const BYTE chain3_0[] = {
0x30,0x82,0x01,0xad,0x30,0x82,0x01,0x1a,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x63,0x73,0xee,0x45,0x59,0x76,0x1a,0x9e,0x47,0xf1,0xfb,0xf0,0x42,0x44,0x26,
0xeb,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x10,0x30,0x0e,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,
0xff,0x04,0x02,0x30,0x00,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,
0x00,0x03,0x81,0x81,0x00,0x53,0xb7,0xa9,0x90,0xe0,0x88,0x36,0xc0,0x2d,0xb1,
0xa3,0x97,0xff,0x16,0x62,0x29,0xa2,0xfd,0x92,0x90,0xb7,0x8d,0xd9,0xdf,0x57,
0x94,0x5a,0xce,0x36,0x5c,0x95,0x24,0xab,0x4f,0x63,0x55,0xa7,0xea,0x8b,0xaf,
0xd9,0x41,0x6a,0x2f,0xd8,0x9e,0x1c,0x2d,0xc0,0x11,0x52,0x09,0x4b,0x31,0x1f,
0x5e,0x8f,0x4b,0xbf,0x20,0x73,0x2d,0x04,0x11,0x56,0x6d,0xb1,0xbe,0xb4,0x5d,
0x70,0x6b,0xb2,0xd0,0xd3,0xf5,0x5d,0x60,0x08,0x65,0x2c,0xe3,0x78,0x5a,0x05,
0x36,0xe1,0xf1,0x67,0x3a,0x25,0xf3,0x47,0x83,0xce,0x4f,0xb0,0x74,0x7a,0x35,
0xa0,0x7f,0x70,0x64,0x8a,0x14,0x66,0x03,0xf6,0xae,0xdf,0x1b,0xf5,0x80,0x87,
0x06,0x39,0x76,0x70,0x7b,0xd2,0x83,0xe9,0xce,0x1f,0xf8,0x66,0xc1 };
/* chain4_0 -> chain4_1 -> chain4_2:
 * A chain whose path length constraint is violated.
 */
static const BYTE chain4_0[] = {
0x30,0x82,0x01,0xc5,0x30,0x82,0x01,0x30,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x01,0xcd,0x67,0x9e,0xec,0xae,0x1e,0x69,0x16,0x3f,0x92,0x8a,0xed,0x6d,0x57,
0xac,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x26,0x30,0x24,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x08,0x30,0x06,0x01,0x01,0xff,0x02,0x01,0x00,0x30,0x0b,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x70,
0x60,0x8d,0x5f,0x29,0x8e,0x6b,0x48,0x79,0xaa,0xd1,0x7a,0xbb,0x6c,0x7c,0x54,
0x11,0x55,0x08,0xb9,0x2e,0x5e,0x53,0xd6,0x9b,0xb9,0xe4,0xc8,0x0e,0x48,0xe7,
0x20,0x9e,0xbd,0x7c,0x55,0xb8,0xf1,0x69,0x0c,0x08,0xd6,0x32,0x32,0xd9,0x05,
0x81,0x8a,0x33,0x4c,0x57,0x20,0xae,0xe8,0xde,0x61,0x63,0x85,0xc8,0xe1,0x1d,
0xef,0x4b,0xa4,0x42,0x4e,0x0a,0x25,0x48,0x2b,0xc2,0x06,0x79,0x90,0x45,0x90,
0x94,0x3d,0xb0,0x36,0xe6,0x60,0xe6,0xd2,0x1a,0x11,0x01,0x4b,0xaf,0x23,0x4d,
0x62,0x6b,0xdc,0x3a,0xae,0x61,0x93,0xce,0x7b,0xae,0x21,0xfe,0x42,0xd8,0x86,
0x36,0x19,0x00,0x18,0x13,0x4d,0xf9,0xd1,0x94,0xa2,0xb7,0xbd,0xb0,0x5c,0x9b,
0x5c,0x03,0xf5,0x86,0x85,0x07,0x1c };
static const BYTE chain4_1[] = {
0x30,0x82,0x01,0xb3,0x30,0x82,0x01,0x1e,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,
0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,
0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,
0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,
0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,
0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,
0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,
0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,
0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x26,0x56,0x31,0x53,
0x65,0xe3,0xd8,0x59,0xc8,0x6f,0xa3,0x5b,0x8a,0x11,0x53,0xde,0x8d,0x6f,0x6d,
0x8c,0xe1,0x79,0x25,0xb2,0xbf,0xc6,0x3b,0xa1,0x8b,0x6b,0xcd,0x99,0xcf,0x0c,
0x6f,0xaa,0xbc,0xd0,0x3d,0xb2,0xf8,0x1c,0x6f,0xb0,0x84,0x7b,0xb1,0x88,0xec,
0xd6,0x77,0xbe,0xb1,0x0b,0x8c,0x18,0x85,0xb7,0x38,0x81,0x12,0xc9,0xc7,0x4a,
0xf6,0x83,0xc4,0xd9,0xae,0xb6,0x1d,0xbb,0xd2,0x08,0x01,0x4a,0xe1,0xdc,0x8f,
0x67,0xa9,0xed,0xe2,0x3f,0x96,0xf6,0x72,0x7e,0x4a,0x63,0x46,0xc4,0x82,0x89,
0x0a,0x81,0xc8,0x3a,0x9c,0xe7,0x23,0x5f,0x8d,0xd9,0x6c,0xb2,0x5d,0x9a,0x40,
0x04,0x04,0xd5,0x4a,0x29,0x57,0xbd,0x18,0xd8,0xca,0x1f,0xf7,0xf8,0x23,0x07,
0x4b,0x03,0xa3,0xd9 };
static const BYTE chain4_2[] = {
0x30,0x82,0x01,0x98,0x30,0x82,0x01,0x01,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x32,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x31,0x31,0x32,
0x32,0x31,0x38,0x35,0x32,0x30,0x37,0x5a,0x18,0x0f,0x31,0x36,0x32,0x38,0x30,
0x38,0x32,0x36,0x30,0x35,0x32,0x30,0x35,0x37,0x5a,0x30,0x10,0x31,0x0e,0x30,
0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x33,0x30,0x81,
0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xc3,0x31,0x35,
0xc3,0x9d,0x8a,0x87,0x20,0xc6,0x32,0xe9,0xb1,0xf6,0x8f,0xf9,0x05,0x73,0x1d,
0xa7,0xde,0xab,0x15,0x8a,0x9c,0x7f,0x11,0x7e,0x77,0xa0,0x42,0x80,0xf4,0x79,
0xda,0x98,0x7b,0x00,0xfa,0x8f,0x0c,0xd0,0xeb,0x8a,0x80,0xed,0x07,0xfc,0x64,
0x71,0x03,0xc1,0xd6,0x2f,0x9b,0xde,0x42,0x63,0xd9,0x79,0xea,0xdd,0x10,0xe8,
0x68,0xc8,0x69,0x4f,0x4a,0x39,0x23,0x87,0xca,0xd1,0xc9,0x77,0x14,0x30,0x85,
0x9e,0xf7,0x79,0xf9,0x07,0xb7,0x7c,0x55,0xcb,0xa7,0xd5,0xb8,0x44,0xb5,0x20,
0xb5,0x01,0x5c,0xa2,0xd1,0xd5,0xad,0x0f,0x87,0xaf,0x37,0xd1,0x39,0x0c,0x0d,
0xd5,0xde,0x26,0x7a,0xed,0xf9,0x2a,0xb1,0x60,0x65,0x2d,0x08,0x24,0x51,0x1d,
0xb0,0x0a,0xb5,0x13,0xc7,0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0xaa,
0x0e,0x17,0x8d,0x5b,0xca,0x11,0xda,0x5a,0xb5,0xd4,0xe8,0x63,0xce,0xa5,0x67,
0x11,0x2f,0xf8,0xb8,0x9f,0x89,0x46,0x7d,0x20,0x94,0x5f,0x0d,0x02,0x7e,0x5e,
0x8c,0x33,0xfc,0x24,0x56,0x08,0xe6,0xf1,0x4d,0x29,0xd5,0xef,0x03,0xe3,0x9a,
0xc6,0xcd,0x6e,0xe3,0xe6,0x45,0xaa,0xce,0x5a,0x2f,0x37,0x6c,0x07,0xd2,0x28,
0x63,0x8e,0x55,0x8a,0xad,0xe3,0x8a,0x58,0x3d,0x24,0x09,0x1e,0x9a,0x7d,0x5f,
0x05,0xc4,0x51,0x18,0x17,0x46,0x33,0x4d,0x72,0x05,0xc2,0x02,0xb6,0x05,0x22,
0x23,0xcc,0xf9,0x1f,0x87,0x6c,0x4a,0x3d,0x6f,0xdb,0x59,0xf6,0x79,0x81,0x02,
0x45,0x9f,0xc8,0xc3,0xf0,0x90,0x82,0xaf,0x58,0x80,0x18,0xea,0x73,0x50,0x29,
0x71,0x8d,0xc1,0x9e,0x53,0x19,0x5a };
/* chain5_0 -> chain5_1:
 * A chain whose name constraints aren't met.
 */
static const BYTE chain5_0[] = {
0x30,0x82,0x01,0xe0,0x30,0x82,0x01,0x4b,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0xed,0xb8,0x3b,0xf2,0x95,0xe6,0x8e,0xdf,0x75,0x6a,0x22,0xbc,0xca,0x51,0xb6,
0x83,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x41,0x30,0x3f,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1c,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x12,0x30,0x10,0xa0,0x0e,0x30,0x0c,0x82,0x0a,0x77,
0x69,0x6e,0x65,0x68,0x71,0x2e,0x6f,0x72,0x67,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x7f,0x98,0xd1,0x58,
0x70,0x36,0x2c,0x08,0x63,0x2c,0xf9,0xe0,0x16,0xda,0x59,0xd9,0x1b,0xa5,0xba,
0xe0,0xec,0xd9,0x0a,0x1b,0x78,0xd0,0xea,0x49,0xc7,0x01,0x06,0x78,0xe5,0xb3,
0x81,0x5a,0x53,0x3d,0x9b,0x0d,0xaf,0xc8,0x04,0x57,0x15,0xee,0xfd,0x05,0x80,
0x72,0x8a,0x54,0x51,0x6a,0x15,0xdc,0xbf,0xfc,0xe3,0x04,0xe4,0x6f,0x20,0xb2,
0xfd,0xa6,0x32,0x4c,0xbc,0x1b,0x60,0xf7,0xc3,0x9c,0x5f,0x17,0x94,0xc7,0xdf,
0x9f,0xae,0xdb,0x72,0x8a,0x98,0x28,0x9d,0x26,0x1f,0x14,0x76,0xb1,0x8e,0x26,
0xf6,0x97,0x96,0x57,0x14,0x9e,0x44,0x13,0xef,0x85,0xc1,0x6f,0x38,0xda,0xf2,
0x3a,0xf6,0xad,0x0b,0x35,0xcf,0xea,0xb0,0x5a,0x9c,0x36,0x80,0x45,0x1f,0x88,
0x2f,0xff,0xb4,0xb5 };
static const BYTE chain5_1[] = {
0x30,0x82,0x01,0xb2,0x30,0x82,0x01,0x1b,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,
0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x1c,0x30,0x1a,0x30,0x18,0x06,0x03,0x55,
0x1d,0x07,0x01,0x01,0xff,0x04,0x0e,0x30,0x0c,0x82,0x0a,0x77,0x69,0x6e,0x65,
0x68,0x71,0x2e,0x63,0x6f,0x6d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x08,0x08,0xf5,0x1c,0xca,
0x3b,0x5a,0xd7,0x45,0xac,0x29,0x5c,0x10,0x9d,0xb0,0x4a,0xdb,0x36,0x0b,0x39,
0x14,0x87,0x3d,0x06,0xe8,0x0d,0x54,0xa7,0xb9,0x50,0x14,0x29,0x05,0xd2,0x75,
0xe4,0x88,0xd8,0x8e,0x7e,0xab,0x61,0xba,0x3c,0xb2,0x4f,0x8f,0xfd,0x00,0x6b,
0x94,0x69,0xf0,0x13,0x40,0x7d,0x4e,0x7c,0xac,0xf0,0x1b,0x07,0x2c,0x96,0x46,
0xf6,0x38,0x38,0xc6,0x65,0x2c,0x02,0x28,0x62,0x6f,0x7d,0xf9,0x4d,0xff,0xb1,
0x38,0x0e,0xce,0x89,0x0e,0xdd,0x22,0x51,0xd2,0x6b,0x0e,0x80,0x3c,0xa6,0x0c,
0xe7,0x8b,0x3b,0x2a,0x2f,0x46,0xed,0xda,0x96,0x1a,0xc1,0x5c,0xfe,0xe0,0xef,
0xdb,0xe0,0x11,0xc4,0x56,0x12,0xe2,0x4b,0x84,0xde,0xaf,0x7f,0x3c,0x9f,0xcb,
0xb6,0xac,0x92 };
/* chain0_0 -> chain4_1:
 * A chain whose end certificate is a CA.
 */
/* chain0_0 -> chain 7_1:
 * A chain whose end cert has a bad critical extension.
 */
static const BYTE chain7_1[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,
0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,
0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,
0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,
0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,
0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,
0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,
0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,
0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x9f,0x69,0xfd,0x26,0xd5,0x4b,
0xe0,0xab,0x12,0x21,0xb9,0xfc,0xf7,0xe0,0x0c,0x09,0x94,0xad,0x27,0xd7,0x9d,
0xa3,0xcc,0x46,0x2a,0x25,0x9a,0x24,0xa7,0x31,0x58,0x78,0xf5,0xfc,0x30,0xe1,
0x6d,0xfd,0x59,0xab,0xbe,0x69,0xa0,0xea,0xe3,0x7d,0x7a,0x7b,0xe5,0x85,0xeb,
0x86,0x6a,0x84,0x3c,0x96,0x01,0x1a,0x70,0xa7,0xb8,0xcb,0xf2,0x11,0xe7,0x52,
0x9c,0x58,0x2d,0xac,0x63,0xce,0x72,0x4b,0xad,0x62,0xa8,0x1d,0x75,0x96,0xe2,
0x27,0xf5,0x6f,0xba,0x91,0xf8,0xf1,0xb0,0xbf,0x90,0x24,0x6d,0xba,0x5d,0xd7,
0x39,0x63,0x3b,0x7c,0x04,0x5d,0x89,0x9d,0x1c,0xf2,0xf7,0xcc,0xdf,0x6e,0x8a,
0x43,0xa9,0xdd,0x86,0x05,0xa2,0xf3,0x22,0x2d,0x1e,0x70,0xa1,0x59,0xd7,0xa5,
0x94,0x7d };
/* chain8_0 -> chain8_1 -> chain8_2:
 * A chain whose root cert validity time doesn't overlap the end cert's
 * validity time.
 */
static const BYTE chain8_0[] = {
0x30,0x82,0x01,0xc2,0x30,0x82,0x01,0x2d,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x76,0x09,0xeb,0xc1,0x06,0x61,0x69,0x50,0x0e,0x0c,0xaf,0xc9,0x0a,0x6b,0x93,
0x50,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x30,0x36,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x87,0x74,0x4a,0xb7,
0xbe,0xd4,0x21,0xa5,0x5f,0x51,0xd6,0xea,0x89,0x4c,0x68,0x16,0x38,0x09,0x49,
0xf5,0xef,0xac,0x0c,0x59,0x27,0xa3,0x0f,0x8f,0xb9,0x21,0x35,0x91,0xef,0x15,
0x91,0xe2,0xac,0xf2,0xad,0xfe,0x05,0x9a,0x9f,0x24,0x81,0x75,0x69,0x57,0x51,
0xe3,0x2f,0x29,0xad,0x55,0x8b,0xc7,0x25,0xbe,0xcd,0x38,0xa7,0xff,0xdb,0xb0,
0x85,0x4e,0x8d,0x56,0x5a,0x79,0xf7,0x7a,0xfb,0x64,0x3d,0x4e,0x04,0x34,0xf7,
0x2f,0x7f,0xe3,0x21,0xee,0x8d,0xd3,0xb0,0x26,0x7d,0xb8,0xc3,0xb6,0x0f,0xe2,
0x3f,0xb6,0x8a,0x53,0x27,0xc0,0xe1,0xfb,0xd8,0xea,0xfe,0xed,0x80,0x51,0xb6,
0x46,0x59,0x65,0x6c,0x34,0x01,0xcc,0x2e,0x50,0xc7,0x4e,0x37,0x83,0x54,0x3f,
0x7b,0xa2,0x1d,0x3a };
static const BYTE chain8_1[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,
0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,
0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,
0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,
0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,
0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,
0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,
0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,
0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x22,0xf1,0x66,0x00,0x79,0xd2,
0xe6,0xb2,0xb2,0xf7,0x2f,0x98,0x92,0x7d,0x73,0xc3,0x6c,0x5c,0x77,0x20,0xe3,
0xbf,0x3e,0xe0,0xb3,0x5c,0x68,0xb4,0x9b,0x3a,0x41,0xae,0x94,0xa0,0x80,0x3a,
0xfe,0x5d,0x7a,0x56,0x87,0x85,0x44,0x45,0xcf,0xa6,0xd3,0x10,0xe7,0x73,0x41,
0xf2,0x7f,0x88,0x85,0x91,0x8e,0xe6,0xec,0xe2,0xce,0x08,0xbc,0xa5,0x76,0xe5,
0x4d,0x1d,0xb7,0x70,0x31,0xdd,0xc9,0x9a,0x15,0x32,0x11,0x5a,0x4e,0x62,0xc8,
0xd1,0xf8,0xec,0x46,0x39,0x5b,0xe7,0x67,0x1f,0x58,0xe8,0xa1,0xa0,0x5b,0xf7,
0x8a,0x6d,0x5f,0x91,0x18,0xd4,0x90,0x85,0xff,0x30,0xc7,0xca,0x9c,0xc6,0x92,
0xb0,0xca,0x16,0xc4,0xa4,0xc0,0xd6,0xe8,0xff,0x15,0x19,0xd1,0x30,0x61,0xf3,
0xef,0x9f };
static const BYTE chain8_2[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x32,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x39,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x33,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xc3,0x31,0x35,0xc3,0x9d,0x8a,0x87,0x20,
0xc6,0x32,0xe9,0xb1,0xf6,0x8f,0xf9,0x05,0x73,0x1d,0xa7,0xde,0xab,0x15,0x8a,
0x9c,0x7f,0x11,0x7e,0x77,0xa0,0x42,0x80,0xf4,0x79,0xda,0x98,0x7b,0x00,0xfa,
0x8f,0x0c,0xd0,0xeb,0x8a,0x80,0xed,0x07,0xfc,0x64,0x71,0x03,0xc1,0xd6,0x2f,
0x9b,0xde,0x42,0x63,0xd9,0x79,0xea,0xdd,0x10,0xe8,0x68,0xc8,0x69,0x4f,0x4a,
0x39,0x23,0x87,0xca,0xd1,0xc9,0x77,0x14,0x30,0x85,0x9e,0xf7,0x79,0xf9,0x07,
0xb7,0x7c,0x55,0xcb,0xa7,0xd5,0xb8,0x44,0xb5,0x20,0xb5,0x01,0x5c,0xa2,0xd1,
0xd5,0xad,0x0f,0x87,0xaf,0x37,0xd1,0x39,0x0c,0x0d,0xd5,0xde,0x26,0x7a,0xed,
0xf9,0x2a,0xb1,0x60,0x65,0x2d,0x08,0x24,0x51,0x1d,0xb0,0x0a,0xb5,0x13,0xc7,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x33,0x95,0x73,0x7c,0x7c,0xb0,
0xbb,0x2b,0xe0,0xc4,0x12,0xf0,0x26,0x85,0x97,0x0c,0x81,0xde,0xc5,0x52,0x04,
0xcb,0xee,0x11,0x9a,0x2e,0xa2,0xc4,0x76,0xd5,0x57,0xd1,0x2f,0x2d,0x6f,0x42,
0xbf,0x23,0x85,0x59,0x5c,0x96,0x8d,0xef,0xbc,0xb6,0xf5,0xd3,0x66,0x99,0x54,
0x95,0x09,0x28,0x0b,0x4f,0x05,0x5f,0x61,0x20,0xa8,0x0d,0x46,0x9a,0xab,0x29,
0x97,0x6b,0xa3,0xcd,0xd7,0x5a,0xda,0xaa,0x8e,0xad,0x5a,0x66,0x58,0xf6,0x63,
0x76,0x10,0xdc,0xb0,0xee,0x3f,0x82,0x6d,0x29,0xb9,0xcb,0x99,0x15,0xbb,0xb9,
0x74,0x73,0x00,0x14,0x74,0x79,0xad,0x89,0x73,0xaa,0xa2,0x0e,0x52,0xe8,0x28,
0x69,0xe6,0x8c,0x1e,0xce,0x3a,0x6d,0x85,0x33,0x1e,0x21,0xa3,0x42,0x61,0x3a,
0xeb,0x25 };
/* chain9_0 -> chain7_1:
 * A cyclic chain.
 */
static const BYTE chain9_0[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x32,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,
0xca,0x50,0x82,0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,
0xc5,0xc3,0x34,0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,
0x03,0x7a,0xe7,0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,
0xc1,0x33,0x91,0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,
0x1b,0x99,0xc5,0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,
0x49,0xba,0xcd,0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,
0x72,0xb6,0x1c,0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,
0x1f,0x2f,0x35,0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x2c,0xf5,0x25,0xaa,0xb4,0x63,
0x5a,0x5d,0xae,0x7b,0x42,0x79,0xbc,0x41,0x2f,0x8a,0x2a,0x70,0x9f,0x8e,0xb2,
0x14,0x1b,0x6f,0xb2,0xf3,0x3e,0xab,0x06,0x9c,0xa8,0x92,0x43,0xc4,0x57,0x63,
0xa4,0xee,0x83,0x04,0xab,0xa5,0x2c,0xd9,0x90,0xed,0xc9,0x4d,0xa8,0x7b,0xca,
0x3b,0x5f,0xdf,0x46,0xe1,0x3b,0x7f,0x72,0xb7,0x92,0x82,0xe6,0x05,0xc3,0x64,
0x38,0xfb,0x86,0x13,0xeb,0x19,0xb6,0x16,0x82,0x63,0x84,0x13,0x04,0x38,0x51,
0x4c,0x89,0x12,0xf1,0x13,0x46,0x75,0x4e,0x99,0x46,0xcb,0xc6,0x76,0x9f,0x97,
0x9c,0xda,0x63,0x01,0x6a,0x71,0x68,0xc4,0x21,0x67,0xa6,0x51,0x19,0x5e,0xc7,
0x8c,0x8d,0x63,0x5c,0xbe,0x04,0x8f,0xad,0xea,0xa5,0xcd,0xc3,0xd6,0x6b,0xc7,
0xb5,0x56 };
/* chain0_0 -+
 *            +-> chain7_1
 * chain10_1 -+
 * A chain with two issuers, only one of whose dates is valid.
 */
static const BYTE chain10_1[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0xbf,0x99,0x4f,0x14,0x03,0x77,0x44,0xb8,0x49,0x02,0x70,0xa1,0xb8,0x9c,0xa7,
0x24,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0xa8,0xec,0x8c,0x34,0xe7,0x2c,0xdf,0x75,0x87,0xc4,0xf7,0xda,0x71,
0x72,0x29,0xb2,0x48,0xa8,0x2a,0xec,0x7b,0x7d,0x19,0xb9,0x5f,0x1d,0xd9,0x91,
0x2b,0xc4,0x28,0x7e,0xd6,0xb5,0x91,0x69,0xa5,0x8a,0x1a,0x1f,0x97,0x98,0x46,
0x9d,0xdf,0x12,0xf6,0x45,0x62,0xad,0x60,0xb6,0xba,0xb0,0xfd,0xf5,0x9f,0xc6,
0x98,0x05,0x4f,0x4d,0x48,0xdc,0xee,0x69,0xbe,0xb8,0xc4,0xc4,0xd7,0x1b,0xb1,
0x1f,0x64,0xd6,0x45,0xa7,0xdb,0xb3,0x87,0x63,0x0f,0x54,0xe1,0x3a,0x6b,0x57,
0x36,0xd7,0x68,0x65,0xcf,0xda,0x57,0x8d,0xcd,0x84,0x75,0x47,0x26,0x2c,0xef,
0x1e,0x8f,0xc7,0x3b,0xee,0x5d,0x03,0xa6,0xdf,0x3a,0x20,0xb2,0xcc,0xc9,0x09,
0x2c,0xfe,0x2b,0x79,0xb0,0xca,0x2c,0x9a,0x81,0x6b };
/* chain0_0 -+
 *            +-> chain7_1
 * chain11_1 -+
 * A chain with two issuers, only one of whose public keys matches the
 * signature.
 */
static const BYTE chain11_1[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x72,0x41,0xa2,0x53,0xd2,0x33,0x0b,0xac,0x4a,0x30,0xfe,0x0a,0xe0,0x2e,0xd4,
0x03,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xc3,0x31,0x35,0xc3,0x9d,0x8a,0x87,0x20,0xc6,0x32,0xe9,
0xb1,0xf6,0x8f,0xf9,0x05,0x73,0x1d,0xa7,0xde,0xab,0x15,0x8a,0x9c,0x7f,0x11,
0x7e,0x77,0xa0,0x42,0x80,0xf4,0x79,0xda,0x98,0x7b,0x00,0xfa,0x8f,0x0c,0xd0,
0xeb,0x8a,0x80,0xed,0x07,0xfc,0x64,0x71,0x03,0xc1,0xd6,0x2f,0x9b,0xde,0x42,
0x63,0xd9,0x79,0xea,0xdd,0x10,0xe8,0x68,0xc8,0x69,0x4f,0x4a,0x39,0x23,0x87,
0xca,0xd1,0xc9,0x77,0x14,0x30,0x85,0x9e,0xf7,0x79,0xf9,0x07,0xb7,0x7c,0x55,
0xcb,0xa7,0xd5,0xb8,0x44,0xb5,0x20,0xb5,0x01,0x5c,0xa2,0xd1,0xd5,0xad,0x0f,
0x87,0xaf,0x37,0xd1,0x39,0x0c,0x0d,0xd5,0xde,0x26,0x7a,0xed,0xf9,0x2a,0xb1,
0x60,0x65,0x2d,0x08,0x24,0x51,0x1d,0xb0,0x0a,0xb5,0x13,0xc7,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0x51,0x17,0xfb,0x2a,0x69,0x31,0x36,0xa7,0x37,0x2c,0x1f,0x38,0xee,
0xe3,0xf1,0xae,0x63,0x61,0xac,0x63,0x13,0xc2,0x3b,0x29,0xca,0xb1,0x8e,0x27,
0x41,0xfa,0xb3,0x7d,0x42,0xbe,0xf0,0x9c,0xe0,0xc7,0x69,0xba,0x80,0xf4,0x87,
0x7c,0x6f,0x77,0x70,0x6b,0x77,0x7d,0x5c,0x58,0x95,0x07,0xbd,0xac,0xe2,0x5d,
0x65,0x12,0xb0,0xae,0xf4,0x01,0x14,0x01,0xba,0xf9,0xa0,0x32,0x10,0xea,0x09,
0x12,0xcf,0xf1,0xbc,0x37,0xc9,0x42,0xc3,0x8c,0x9e,0xa2,0xc3,0x5e,0x2e,0x56,
0xe9,0xa7,0xd8,0x1c,0x1c,0x8c,0x1c,0x2f,0xd8,0xfb,0x67,0x44,0xd4,0xa3,0x79,
0x6d,0x53,0x17,0x56,0x3c,0x3e,0xd0,0x67,0x36,0x0c,0x8b,0x77,0xab,0x40,0xc7,
0x27,0xd9,0x44,0x69,0x13,0xeb,0x09,0xec,0x06,0xe2 };
/* chain12_0 -> chain7_1:
 * A chain whose root has an invalid signature.
 */
static const BYTE chain12_0[] = {
0x30,0x82,0x01,0xc2,0x30,0x82,0x01,0x2d,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x73,0x87,0xdb,0x32,0x3a,0x6c,0x89,0x74,0x0f,0xda,0x07,0xd4,0x6a,0x2c,0x81,
0x59,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x82,0xcd,0x5c,0x0a,
0x9d,0x0d,0x3f,0xd1,0xd8,0x74,0xfc,0x52,0x65,0x11,0x76,0x10,0x62,0xb0,0x9a,
0xc5,0x54,0x66,0xae,0xfa,0xd5,0x66,0x36,0xcb,0x1e,0xe7,0xad,0xb9,0xfa,0x2e,
0xbb,0xb9,0x0b,0x4b,0xbf,0xe7,0x26,0x75,0x2c,0xae,0x2d,0x14,0x84,0x92,0xae,
0x56,0xdf,0x55,0x56,0xf3,0xa6,0xc5,0x21,0xa8,0x5e,0xce,0x7a,0x9d,0xff,0x7f,
0x10,0xd9,0xc3,0x05,0x52,0x86,0xdc,0x9b,0x46,0xe7,0x69,0xfb,0x1e,0xf0,0x68,
0x7d,0x34,0xd7,0xf3,0x56,0xe3,0x0a,0xf3,0xe4,0xdc,0x0f,0xd6,0x4d,0xa8,0xc5,
0xb5,0x5b,0xbf,0x0b,0x5d,0xe7,0x76,0xa1,0x56,0xc1,0xa7,0x55,0xa0,0x88,0x7f,
0x57,0x8e,0x64,0xda,0x6f,0xaa,0x9c,0xca,0x47,0xf3,0x37,0x08,0x4b,0x9a,0xdc,
0xbc,0x65,0x4d,0xff };
/* chain0_0 -> chain13_1:
 * A chain whose time validity nesting is invalid.
 */
static const BYTE chain13_1[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x36,0x30,0x31,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,
0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,
0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,
0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,
0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,
0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,
0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,
0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,
0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x85,0xc9,0xc5,0x08,0x0b,0xb1,
0xff,0x58,0x8f,0xee,0x66,0x5f,0xa3,0x77,0x63,0x9f,0xd1,0x53,0x03,0xdc,0xf6,
0x59,0x08,0x02,0x1f,0xdc,0x52,0x5a,0x52,0x38,0xa4,0x0b,0x83,0x12,0x19,0xd6,
0xb5,0xb8,0x81,0x17,0xcd,0x10,0x5d,0x96,0x16,0x9d,0x51,0x5b,0xf9,0x31,0x3e,
0x36,0xac,0x30,0xd5,0xd6,0xa0,0x30,0x5c,0x9c,0xcb,0xe1,0x02,0xfd,0x73,0xc3,
0xa1,0x58,0xfe,0x12,0xb0,0xa6,0xbd,0x11,0x86,0xdb,0x20,0x17,0x29,0x1e,0x09,
0x63,0x4d,0x28,0xe4,0x55,0x2d,0xa1,0x78,0xde,0x28,0x0b,0x1b,0x35,0x40,0x4c,
0xf1,0xb0,0x6e,0x82,0x34,0x2b,0x7f,0x3b,0x2e,0x27,0x6b,0xf7,0xde,0x73,0x9a,
0x98,0xac,0x79,0x33,0x1a,0xdd,0x03,0x95,0x60,0x8a,0x08,0xba,0xcb,0x8a,0xbf,
0xea,0xe3 };
/* chain14_0 -> chain14_1:
 * A chain whose root is allowed to sign certificates.
 */
static const BYTE chain14_0[] = {
0x30,0x82,0x01,0xc2,0x30,0x82,0x01,0x2d,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0xe0,0x3a,0xae,0xb4,0x23,0x65,0x59,0x5c,0x05,0x85,0x76,0x8b,0x8e,0x7f,0xa4,
0x12,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x10,0xb9,0x5c,0x13,
0x7f,0xe9,0xf3,0x45,0xfb,0x40,0xa2,0x2b,0xe6,0x99,0x5e,0x74,0x9a,0x0b,0x15,
0xa0,0x16,0xf2,0xff,0x6e,0xfa,0x92,0x78,0x4f,0x05,0x02,0xbd,0x70,0x4f,0x28,
0x6f,0x43,0xc7,0xaa,0xaf,0xb6,0xd1,0x60,0x77,0xc6,0x8a,0x58,0x85,0xf0,0xac,
0x04,0x90,0x4a,0x06,0xf0,0x17,0x88,0x12,0x90,0x22,0x65,0x98,0xb1,0x67,0x93,
0x8a,0xe6,0x2d,0x54,0x79,0x99,0xd3,0x8c,0xe7,0x86,0xf4,0xec,0x08,0xae,0x18,
0xa3,0xa9,0x23,0x7b,0x4f,0x8d,0xde,0xe6,0xf9,0x11,0x92,0xcf,0xe0,0x45,0xc3,
0x42,0x72,0x43,0x64,0x07,0x8b,0x7f,0x28,0x11,0x0a,0xea,0xf2,0xbd,0x24,0x1b,
0x93,0x5c,0x6f,0xed,0x67,0xae,0xee,0x7e,0x40,0x88,0x8b,0x28,0x2c,0xe7,0xee,
0xec,0x34,0xe0,0x7b };
static const BYTE chain14_1[] = {
0x30,0x82,0x01,0x8d,0x30,0x81,0xf9,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,0x10,
0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,
0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,
0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,
0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,
0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,
0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,0x48,
0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,0x20,
0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,0xb8,
0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,0x3c,
0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,0x8e,
0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,0x28,
0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,0x98,
0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,0x4a,
0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,0x00,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,
0x81,0x81,0x00,0x63,0x67,0x7f,0x82,0x91,0xe2,0x42,0x45,0x98,0xbc,0xe5,0x03,
0x6c,0x7d,0xc9,0xab,0xdb,0x95,0xc4,0x96,0x86,0xfd,0xcf,0xe6,0x32,0x4a,0x0b,
0x7c,0x5b,0x76,0x39,0x71,0x26,0x54,0x60,0x1d,0xce,0x4f,0x29,0x22,0x9a,0xb7,
0x7d,0x8d,0xfa,0x2c,0x99,0x0d,0xa6,0xa4,0x2e,0xd3,0xf6,0x3e,0xaf,0xea,0x89,
0xa0,0x16,0x40,0x0f,0xf1,0x50,0xaa,0x98,0xdc,0xc2,0x69,0x90,0x6f,0xc6,0x4a,
0xee,0x26,0x9e,0x5e,0x7d,0x4b,0x06,0x65,0x54,0x15,0x67,0x9e,0xe5,0x1d,0x1c,
0x6f,0x0c,0x6e,0x84,0x29,0xf9,0x12,0xe9,0x71,0xdd,0xaf,0x9a,0x3d,0x2a,0xc1,
0x60,0xeb,0x27,0xed,0x7c,0xbf,0xc4,0x84,0x17,0xa5,0xa6,0x41,0x29,0x13,0x8d,
0x6c,0xdd,0x7c,0x3d,0x01,0x03,0xf8,0x70,0x97,0x3c,0x45 };
/* chain15_0 -> chain14_1:
 * A chain whose root is not allowed to sign certificates.
 */
static const BYTE chain15_0[] = {
0x30,0x82,0x01,0xc2,0x30,0x82,0x01,0x2d,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x10,0xb3,0x62,0xed,0xed,0x4a,0x92,0xda,0x74,0x62,0x74,0x48,0x99,0xa5,0xe9,
0x0d,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x00,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x51,0x48,0x43,0xc9,
0xe3,0xb4,0xbc,0xc1,0x1d,0x98,0x14,0x69,0xd9,0xec,0x83,0x4d,0x07,0x70,0x58,
0xe9,0x45,0x88,0xe5,0x11,0x4a,0xf0,0x58,0x2d,0x8d,0x08,0x35,0x76,0x0e,0x42,
0x9f,0x62,0xda,0xf9,0x71,0x51,0x58,0xc7,0xd3,0x7e,0x88,0x6a,0x92,0xd2,0xfe,
0x86,0x4f,0xa2,0x17,0xe3,0xdd,0x9b,0x55,0xe0,0xdc,0xa7,0xe7,0x08,0xe0,0x0a,
0xa6,0xf3,0x09,0x17,0x63,0x10,0x89,0xe6,0x07,0x00,0x8e,0x2c,0x44,0x79,0x3e,
0x6c,0x51,0x9a,0xa3,0x59,0x17,0x8b,0x90,0xa8,0x82,0x64,0x09,0x77,0x95,0x55,
0xd6,0x38,0x43,0x8c,0x66,0xe6,0x7b,0xb9,0xf4,0xe1,0x7b,0x86,0x9b,0x8e,0xe3,
0x56,0xc9,0x62,0x62,0x6e,0x3d,0x4d,0x67,0x8d,0x5e,0x20,0xe5,0xd0,0xfa,0xaa,
0xbe,0x91,0xa8,0xf1 };
/* chain0_0 -> chain16_1:
 * A chain whose end cert has a name constraints extension saying it can't
 * be a CA, and which also has a key usage extension allowing it to sign
 * certs.
 */
static const BYTE chain16_1[] = {
0x30,0x82,0x01,0xb0,0x30,0x82,0x01,0x1b,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,
0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,
0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,
0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,
0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,
0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,
0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,
0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,
0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,
0x00,0x01,0xa3,0x20,0x30,0x1e,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x47,0xdb,0xb4,0x95,0xb1,0xe2,0x5c,
0x03,0x3b,0x28,0x16,0x6b,0xed,0xe9,0x38,0xeb,0x74,0xe6,0xde,0xc5,0xaf,0x93,
0xad,0x2e,0xaa,0x29,0x14,0x21,0xae,0xe9,0x7d,0x46,0x18,0x05,0x67,0xad,0x2d,
0xaa,0xfa,0xf8,0x41,0xc8,0x8c,0xde,0xe0,0xd7,0x3e,0x9a,0x89,0x58,0x71,0x73,
0x57,0xc1,0xd9,0x42,0x51,0xfb,0xaa,0xb8,0x50,0x98,0x9b,0x87,0xb1,0x14,0xa2,
0x23,0xa7,0xfc,0x51,0x23,0x97,0x9f,0xef,0x06,0x83,0x63,0x91,0x14,0xc3,0xbe,
0xaa,0xc8,0x7c,0x03,0x0e,0x41,0x5a,0x23,0x9c,0x1d,0x32,0x3a,0xa5,0xf1,0x6a,
0x56,0x6c,0x36,0xe3,0x3d,0xa1,0xac,0x8e,0x1a,0x75,0x30,0x1c,0xd8,0xbc,0xae,
0x3e,0x4a,0x1a,0x05,0xf4,0x09,0x24,0xb2,0x89,0x75,0xa4,0x28,0x78,0x6b,0xfa,
0xcc };
/* chain0_0 -> chain17_1 -> chain17_2:
 * A root cert, an intermediate cert that's a CA and allowed to sign a cert,
 * and an end cert.
 */
static const BYTE chain17_1[] = {
0x30,0x82,0x01,0xb3,0x30,0x82,0x01,0x1e,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,
0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,
0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,
0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,
0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,
0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,
0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,
0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,
0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x26,0x56,0x31,0x53,
0x65,0xe3,0xd8,0x59,0xc8,0x6f,0xa3,0x5b,0x8a,0x11,0x53,0xde,0x8d,0x6f,0x6d,
0x8c,0xe1,0x79,0x25,0xb2,0xbf,0xc6,0x3b,0xa1,0x8b,0x6b,0xcd,0x99,0xcf,0x0c,
0x6f,0xaa,0xbc,0xd0,0x3d,0xb2,0xf8,0x1c,0x6f,0xb0,0x84,0x7b,0xb1,0x88,0xec,
0xd6,0x77,0xbe,0xb1,0x0b,0x8c,0x18,0x85,0xb7,0x38,0x81,0x12,0xc9,0xc7,0x4a,
0xf6,0x83,0xc4,0xd9,0xae,0xb6,0x1d,0xbb,0xd2,0x08,0x01,0x4a,0xe1,0xdc,0x8f,
0x67,0xa9,0xed,0xe2,0x3f,0x96,0xf6,0x72,0x7e,0x4a,0x63,0x46,0xc4,0x82,0x89,
0x0a,0x81,0xc8,0x3a,0x9c,0xe7,0x23,0x5f,0x8d,0xd9,0x6c,0xb2,0x5d,0x9a,0x40,
0x04,0x04,0xd5,0x4a,0x29,0x57,0xbd,0x18,0xd8,0xca,0x1f,0xf7,0xf8,0x23,0x07,
0x4b,0x03,0xa3,0xd9 };
static const BYTE chain17_2[] = {
0x30,0x82,0x01,0x8d,0x30,0x81,0xf9,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,0x10,
0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,
0x32,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,
0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,
0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,
0x05,0x43,0x65,0x72,0x74,0x33,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,
0x81,0x81,0x00,0xc3,0x31,0x35,0xc3,0x9d,0x8a,0x87,0x20,0xc6,0x32,0xe9,0xb1,
0xf6,0x8f,0xf9,0x05,0x73,0x1d,0xa7,0xde,0xab,0x15,0x8a,0x9c,0x7f,0x11,0x7e,
0x77,0xa0,0x42,0x80,0xf4,0x79,0xda,0x98,0x7b,0x00,0xfa,0x8f,0x0c,0xd0,0xeb,
0x8a,0x80,0xed,0x07,0xfc,0x64,0x71,0x03,0xc1,0xd6,0x2f,0x9b,0xde,0x42,0x63,
0xd9,0x79,0xea,0xdd,0x10,0xe8,0x68,0xc8,0x69,0x4f,0x4a,0x39,0x23,0x87,0xca,
0xd1,0xc9,0x77,0x14,0x30,0x85,0x9e,0xf7,0x79,0xf9,0x07,0xb7,0x7c,0x55,0xcb,
0xa7,0xd5,0xb8,0x44,0xb5,0x20,0xb5,0x01,0x5c,0xa2,0xd1,0xd5,0xad,0x0f,0x87,
0xaf,0x37,0xd1,0x39,0x0c,0x0d,0xd5,0xde,0x26,0x7a,0xed,0xf9,0x2a,0xb1,0x60,
0x65,0x2d,0x08,0x24,0x51,0x1d,0xb0,0x0a,0xb5,0x13,0xc7,0x02,0x03,0x01,0x00,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,
0x81,0x81,0x00,0x5c,0xb5,0x55,0xd7,0x2c,0x70,0xc5,0x6e,0xc0,0x8e,0x30,0x1f,
0xf2,0x8e,0x55,0xd6,0xf6,0x95,0xa4,0x51,0xd8,0xf0,0x39,0x98,0x12,0x57,0xa7,
0x6d,0x0c,0xfd,0xcb,0x25,0xf1,0xc4,0x73,0x85,0x78,0x82,0x15,0x12,0x08,0xb6,
0x05,0x85,0x4f,0x5a,0x59,0x1f,0x03,0xdf,0x04,0xef,0xc2,0xb0,0xee,0xd7,0x73,
0x09,0xca,0xbb,0xfb,0xd8,0xec,0x8b,0x39,0x01,0x14,0xc9,0x03,0xc9,0x08,0x4d,
0x97,0xc7,0xf2,0x0b,0x54,0x67,0xf7,0x7c,0xc6,0xee,0x74,0xd5,0xa2,0x10,0x3e,
0xd9,0x3a,0x7a,0x9d,0xeb,0xeb,0xd8,0x8e,0x43,0x74,0xc3,0xa5,0x3b,0x1a,0x4b,
0x8a,0xbc,0x6a,0x2c,0x15,0x4b,0x72,0x0b,0xac,0xa4,0x1a,0x97,0x80,0xf8,0xb1,
0xd3,0xd7,0x1a,0xf3,0xf5,0x9f,0x7c,0x4c,0x7d,0x7c,0xc7 };
/* chain0_0 -> chain18_1 -> chain18_2:
 * A root, an intermediate CA not allowed to sign certs, and an end cert.
 */
static const BYTE chain18_1[] = {
0x30,0x82,0x01,0xb3,0x30,0x82,0x01,0x1e,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,
0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,
0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,
0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,
0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,
0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,
0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,
0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,
0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,
0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,
0x00,0x01,0xa3,0x23,0x30,0x21,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x00,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x2c,0x49,0x9f,0xc2,
0xce,0xf5,0x0d,0x59,0xaa,0xff,0xa0,0x72,0xa7,0xcf,0x4c,0x35,0xc5,0x5e,0x1b,
0x32,0xe6,0x46,0x53,0x3e,0x70,0xe6,0xbc,0xa2,0x37,0xe0,0x9d,0x41,0x40,0x12,
0x9e,0x81,0x2b,0xd9,0x46,0x3a,0xdc,0xd7,0x37,0xc4,0xbf,0x0f,0x0c,0xf8,0x74,
0x7a,0xf9,0x92,0x2d,0xf8,0xfe,0xec,0x17,0xfb,0x38,0x4f,0x83,0x02,0x6f,0x49,
0xb4,0xad,0xb2,0x84,0x41,0x13,0x71,0x32,0xee,0x34,0x8d,0xea,0x2b,0x64,0x9b,
0x26,0x76,0xe9,0xcc,0xdf,0x73,0xe0,0x84,0xc1,0x8b,0xea,0xfc,0x8c,0x18,0x93,
0xfd,0xa3,0x5f,0x8f,0xde,0x05,0x11,0x88,0x62,0x0b,0x03,0x35,0x1a,0xef,0x65,
0x82,0x06,0xc8,0xa5,0xf6,0xd2,0x1e,0x46,0xc2,0x86,0xe7,0x26,0x8e,0x9e,0x10,
0x33,0xe2,0x89,0x93 };
static const BYTE chain18_2[] = {
0x30,0x82,0x01,0x8d,0x30,0x81,0xf9,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,0x10,
0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,
0x32,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,
0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,
0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,
0x05,0x43,0x65,0x72,0x74,0x33,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,
0x81,0x81,0x00,0xc3,0x31,0x35,0xc3,0x9d,0x8a,0x87,0x20,0xc6,0x32,0xe9,0xb1,
0xf6,0x8f,0xf9,0x05,0x73,0x1d,0xa7,0xde,0xab,0x15,0x8a,0x9c,0x7f,0x11,0x7e,
0x77,0xa0,0x42,0x80,0xf4,0x79,0xda,0x98,0x7b,0x00,0xfa,0x8f,0x0c,0xd0,0xeb,
0x8a,0x80,0xed,0x07,0xfc,0x64,0x71,0x03,0xc1,0xd6,0x2f,0x9b,0xde,0x42,0x63,
0xd9,0x79,0xea,0xdd,0x10,0xe8,0x68,0xc8,0x69,0x4f,0x4a,0x39,0x23,0x87,0xca,
0xd1,0xc9,0x77,0x14,0x30,0x85,0x9e,0xf7,0x79,0xf9,0x07,0xb7,0x7c,0x55,0xcb,
0xa7,0xd5,0xb8,0x44,0xb5,0x20,0xb5,0x01,0x5c,0xa2,0xd1,0xd5,0xad,0x0f,0x87,
0xaf,0x37,0xd1,0x39,0x0c,0x0d,0xd5,0xde,0x26,0x7a,0xed,0xf9,0x2a,0xb1,0x60,
0x65,0x2d,0x08,0x24,0x51,0x1d,0xb0,0x0a,0xb5,0x13,0xc7,0x02,0x03,0x01,0x00,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,
0x81,0x81,0x00,0x5c,0xb5,0x55,0xd7,0x2c,0x70,0xc5,0x6e,0xc0,0x8e,0x30,0x1f,
0xf2,0x8e,0x55,0xd6,0xf6,0x95,0xa4,0x51,0xd8,0xf0,0x39,0x98,0x12,0x57,0xa7,
0x6d,0x0c,0xfd,0xcb,0x25,0xf1,0xc4,0x73,0x85,0x78,0x82,0x15,0x12,0x08,0xb6,
0x05,0x85,0x4f,0x5a,0x59,0x1f,0x03,0xdf,0x04,0xef,0xc2,0xb0,0xee,0xd7,0x73,
0x09,0xca,0xbb,0xfb,0xd8,0xec,0x8b,0x39,0x01,0x14,0xc9,0x03,0xc9,0x08,0x4d,
0x97,0xc7,0xf2,0x0b,0x54,0x67,0xf7,0x7c,0xc6,0xee,0x74,0xd5,0xa2,0x10,0x3e,
0xd9,0x3a,0x7a,0x9d,0xeb,0xeb,0xd8,0x8e,0x43,0x74,0xc3,0xa5,0x3b,0x1a,0x4b,
0x8a,0xbc,0x6a,0x2c,0x15,0x4b,0x72,0x0b,0xac,0xa4,0x1a,0x97,0x80,0xf8,0xb1,
0xd3,0xd7,0x1a,0xf3,0xf5,0x9f,0x7c,0x4c,0x7d,0x7c,0xc7 };
/* A chain which allows a.com, where the end cert is issued to a.com */
static const BYTE chain19_0[] = {
0x30,0x82,0x01,0xd9,0x30,0x82,0x01,0x46,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x22,0xf4,0xa5,0xbe,0xe7,0x5f,0x91,0x89,0x4b,0x10,0x5c,0x86,0x89,0x94,0x36,
0xab,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3c,0x30,0x3a,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x17,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0d,0x30,0x0b,0xa0,0x09,0x30,0x07,0x82,0x05,0x61,
0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,
0x03,0x81,0x81,0x00,0x19,0x84,0xa8,0xed,0x39,0xb3,0xc0,0x7e,0xc1,0x50,0x50,
0xb7,0x4c,0xf2,0xf4,0x16,0xc8,0x2a,0x00,0xc9,0x9e,0x0a,0x77,0x7f,0x65,0xb0,
0xe9,0x46,0x88,0x39,0x03,0xd9,0xcc,0x83,0x27,0x22,0xd3,0xec,0x98,0xb3,0x0d,
0xf6,0x85,0xe2,0x3f,0x82,0x02,0x96,0xd2,0x9f,0x17,0xcb,0x7c,0x06,0xbb,0xe5,
0x58,0x0d,0x9e,0xc3,0x01,0xcd,0xba,0xee,0x26,0x7d,0xc6,0x2b,0x50,0xa7,0xf3,
0x9e,0x28,0x87,0xa8,0x9a,0x8b,0x1c,0x73,0x79,0x3e,0xed,0xa0,0xcf,0x70,0xf0,
0x53,0x89,0x1f,0xbf,0x0d,0x6b,0x30,0xb1,0x5e,0x16,0x6d,0x57,0x81,0xe9,0xad,
0x63,0xe5,0x03,0x50,0x99,0x3d,0xb3,0xad,0x1d,0x9c,0x6b,0x32,0xea,0xc3,0x7f,
0x54,0x22,0x89,0x69,0xbf,0x73,0xdc,0x87,0xff,0x4b,0xf1,0xf6 };
static const BYTE chain19_1[] = {
0x30,0x82,0x01,0xaa,0x30,0x82,0x01,0x13,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x14,0x30,0x12,0x30,0x10,0x06,0x03,0x55,
0x1d,0x07,0x04,0x09,0x30,0x07,0x82,0x05,0x61,0x2e,0x63,0x6f,0x6d,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,
0x81,0x00,0x8e,0x1d,0xf5,0xac,0x59,0x8e,0x39,0x7a,0x4d,0x79,0x79,0x68,0xa9,
0x13,0x25,0xbf,0x71,0x62,0x73,0x55,0xb0,0x22,0xcb,0x8a,0x22,0xa2,0x4f,0xa3,
0xfc,0x22,0xa3,0x2c,0x4c,0xe3,0x95,0xc6,0x97,0xe1,0x97,0xf1,0xee,0xb0,0xb4,
0xac,0x7a,0x80,0xe3,0xdd,0x02,0x5a,0x42,0x30,0xe7,0x27,0x0e,0x69,0xff,0x6b,
0xb2,0x83,0x7a,0x4f,0xc7,0xaa,0x05,0x95,0xb5,0x18,0x5f,0xa8,0x29,0x97,0x3b,
0xf2,0xb8,0x35,0x06,0x1b,0xfe,0xfa,0xa8,0x91,0xc7,0xb7,0x1b,0x8b,0xaa,0xd5,
0x23,0x88,0x7f,0xd9,0x88,0x45,0xee,0xd4,0xb9,0xcd,0x36,0xd4,0x5a,0x28,0x86,
0x0d,0x92,0x5d,0xcb,0x8c,0xf1,0xd9,0xfb,0xcc,0xa4,0x65,0xda,0x8d,0x6e,0xfd,
0x99,0xed,0xf1,0xc6,0x45,0xdf,0x5b,0x1f,0xf5,0xa0 };
/* A chain which allows a.com, where the end cert is issued to b.com */
static const BYTE chain20_0[] = {
0x30,0x82,0x01,0xd9,0x30,0x82,0x01,0x46,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x5e,0xf5,0xbf,0x5a,0x8d,0x92,0xd9,0x8e,0x46,0x09,0xbf,0x50,0x27,0x2f,0xc1,
0x4b,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3c,0x30,0x3a,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x17,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0d,0x30,0x0b,0xa0,0x09,0x30,0x07,0x82,0x05,0x61,
0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,
0x03,0x81,0x81,0x00,0x4e,0xd8,0x55,0xa5,0x18,0xa8,0xc3,0xc4,0xa2,0xed,0x11,
0xff,0x12,0x78,0xbe,0xa5,0xaf,0x6c,0x9f,0xa5,0x94,0x46,0xd3,0x74,0x8e,0x44,
0x45,0x9f,0xf2,0xf9,0x0b,0xbe,0x28,0x08,0xf2,0x28,0x4a,0xd5,0x00,0x71,0xb2,
0xa5,0x01,0x3b,0x86,0xd6,0x8f,0xb8,0x9a,0xbc,0x8e,0xe4,0x33,0x9e,0xc2,0x5d,
0xcc,0x5c,0x24,0x90,0x52,0xbc,0xe6,0x83,0x38,0x77,0xd6,0x25,0x1b,0xfa,0x37,
0xc1,0xf6,0x22,0xa7,0xff,0xc6,0x4b,0x16,0x73,0xf5,0xca,0x04,0xad,0xa5,0x71,
0xff,0x15,0x5b,0x34,0x84,0xd4,0x45,0xce,0xfb,0xd1,0x76,0x13,0x2f,0x53,0x3e,
0x82,0xba,0x2f,0x53,0x82,0x48,0xcb,0x3a,0x77,0xd4,0xb0,0x0f,0xb5,0x38,0x68,
0x73,0x3f,0x49,0x60,0x0f,0x61,0x58,0xae,0x25,0xd8,0xe6,0x73 };
static const BYTE chain20_1[] = {
0x30,0x82,0x01,0xaa,0x30,0x82,0x01,0x13,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x14,0x30,0x12,0x30,0x10,0x06,0x03,0x55,
0x1d,0x07,0x04,0x09,0x30,0x07,0x82,0x05,0x62,0x2e,0x63,0x6f,0x6d,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,
0x81,0x00,0x17,0xbb,0xec,0x37,0x4c,0xaf,0x0f,0xb0,0x57,0x04,0x01,0x67,0x69,
0x77,0xe4,0x73,0xc4,0x12,0x0a,0xa5,0xfe,0x0c,0x8b,0x89,0x90,0x98,0x0d,0x3b,
0x37,0xbe,0xe7,0x7e,0xba,0x84,0x27,0x5f,0xed,0x40,0xc9,0x57,0x91,0x6b,0x72,
0xe3,0x4c,0xfa,0x9f,0x6d,0xd5,0xe6,0xf8,0x33,0xa1,0x6e,0xbd,0x5f,0x8d,0xfb,
0x8f,0xe6,0xe1,0x6a,0x64,0x51,0x1e,0x3b,0x04,0xe4,0x3b,0xa2,0x49,0x15,0x23,
0x35,0x17,0x61,0x4a,0xa6,0x03,0xee,0xa2,0xa6,0x4f,0x42,0x13,0x75,0x14,0xd3,
0x87,0x3b,0x71,0x6a,0x2f,0x11,0xcd,0xe6,0xd0,0x63,0xce,0x62,0x8c,0xd9,0x64,
0xc0,0xaa,0x53,0x4e,0x5e,0xa2,0x4c,0x6e,0xd0,0x13,0x49,0x2e,0xdf,0x81,0x3a,
0x3c,0x67,0xcf,0x4e,0xde,0x95,0x14,0x74,0xa8,0x31 };
/* A chain which allows A.com, where the end cert is issued to a.COM */
static const BYTE chain21_0[] = {
0x30,0x82,0x01,0xd9,0x30,0x82,0x01,0x46,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x66,0xaf,0x57,0xcc,0x4b,0x41,0xd1,0x96,0x40,0xef,0xa4,0xc9,0xe0,0x96,0xad,
0x5c,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3c,0x30,0x3a,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x17,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0d,0x30,0x0b,0xa0,0x09,0x30,0x07,0x82,0x05,0x41,
0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,
0x03,0x81,0x81,0x00,0x66,0x0e,0x8e,0x3d,0x1b,0xe8,0x51,0x2d,0x5f,0xfa,0xf5,
0x01,0xb1,0xf8,0x8e,0x65,0xf6,0xcf,0x63,0x40,0x27,0x20,0x41,0xfc,0xa2,0x97,
0x14,0x8d,0x46,0x33,0x3b,0x79,0xc7,0x3d,0xc8,0x1d,0xb3,0xf1,0xf3,0x52,0xfb,
0x76,0x55,0x00,0xb9,0x4d,0x89,0x4c,0xfd,0xa9,0xcc,0xa0,0x96,0x86,0x15,0x40,
0xfc,0xd2,0x5b,0x60,0xca,0x9c,0x00,0xf8,0x02,0x6a,0x56,0x1c,0x79,0x94,0x01,
0x8a,0x60,0x44,0x1a,0x2f,0x34,0x94,0xb6,0xfb,0x00,0xa7,0xd1,0x47,0xd8,0x8f,
0xf7,0xf5,0xfc,0xfd,0xeb,0xf7,0xa1,0xab,0xa7,0x38,0x13,0x16,0x50,0x04,0x8e,
0xf2,0x17,0x45,0xac,0x10,0x83,0x08,0x1c,0x0d,0xe9,0xa9,0xea,0x4e,0x0e,0x06,
0x09,0xca,0xb3,0x54,0xd1,0x44,0x10,0x47,0xac,0xaf,0xbc,0xbb };
static const BYTE chain21_1[] = {
0x30,0x82,0x01,0xaa,0x30,0x82,0x01,0x13,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x14,0x30,0x12,0x30,0x10,0x06,0x03,0x55,
0x1d,0x07,0x04,0x09,0x30,0x07,0x82,0x05,0x61,0x2e,0x43,0x4f,0x4d,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,
0x81,0x00,0x65,0xd5,0xaa,0x17,0x1f,0xa8,0xda,0x62,0x04,0x9d,0xf5,0xe0,0x82,
0xe4,0xed,0xc6,0x1b,0x54,0x8f,0xb7,0x46,0xac,0xcd,0xaf,0xaa,0xe3,0xd8,0x4a,
0x8b,0xd9,0xe6,0x87,0x4f,0x8c,0xa6,0x86,0x97,0x75,0x08,0x42,0xf1,0xed,0x3e,
0x70,0x99,0x13,0xf2,0x00,0xf8,0xcc,0xf4,0x55,0x36,0xf5,0x29,0x97,0x47,0x71,
0xb8,0x40,0x8d,0x47,0x03,0xe2,0x66,0x3c,0x00,0x01,0xcc,0xa7,0x39,0xf7,0xf9,
0x76,0xfc,0x14,0x31,0x1f,0x6f,0xf0,0x87,0xe0,0x43,0x8b,0xad,0x98,0xa2,0xe3,
0xfe,0x49,0xa3,0x1e,0x57,0x34,0xe8,0x61,0x97,0x23,0xc8,0x03,0x05,0xa4,0x2f,
0xab,0x49,0x97,0xff,0x3f,0xbf,0x6c,0x08,0x2c,0x6b,0x4d,0x6a,0x54,0x62,0x2e,
0x16,0x83,0xac,0xdd,0x21,0x8f,0xe7,0x71,0x12,0x5f };
/* A chain which allows A.com, where the end cert is issued to b.COM */
static const BYTE chain22_0[] = {
0x30,0x82,0x01,0xd9,0x30,0x82,0x01,0x46,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x75,0xff,0x32,0x4c,0x24,0xbb,0x4e,0xb9,0x45,0x70,0xc3,0x4a,0xe1,0xfd,0xdb,
0xdc,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3c,0x30,0x3a,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x17,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0d,0x30,0x0b,0xa0,0x09,0x30,0x07,0x82,0x05,0x41,
0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,
0x03,0x81,0x81,0x00,0x43,0x9b,0xcf,0x0d,0x31,0x9a,0x33,0x9d,0x7a,0xf8,0x2b,
0x5a,0x6f,0x52,0x68,0x25,0x7e,0x14,0x04,0xb1,0x88,0x90,0xa3,0xa8,0x4e,0xa8,
0x0c,0xc2,0x06,0x35,0x6c,0xb9,0x07,0x9a,0x28,0xd0,0x54,0xad,0x5f,0xc7,0x5e,
0xbf,0x1d,0xde,0x3d,0x76,0x6d,0x32,0x6c,0xc2,0x95,0x26,0x8e,0x6a,0xc5,0xfd,
0x45,0x47,0x3a,0x2e,0x81,0x47,0x37,0xf9,0xe3,0xdb,0x57,0xd5,0x24,0xe9,0x10,
0x91,0x55,0xbf,0x48,0xaa,0x27,0x7a,0x32,0xc8,0x96,0xbc,0xd8,0x9d,0xb5,0x03,
0x6e,0x64,0x50,0xa6,0x9d,0xc2,0xd9,0xdb,0x67,0x8d,0xad,0x76,0x82,0x75,0x42,
0x78,0x38,0x3c,0x31,0x55,0x1e,0x8f,0xbb,0x18,0xc9,0xf3,0xa4,0x49,0xc7,0x24,
0x1a,0x9b,0xef,0x61,0x03,0x75,0xb7,0x2b,0x8b,0xd4,0x50,0x8b };
static const BYTE chain22_1[] = {
0x30,0x82,0x01,0xaa,0x30,0x82,0x01,0x13,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x14,0x30,0x12,0x30,0x10,0x06,0x03,0x55,
0x1d,0x07,0x04,0x09,0x30,0x07,0x82,0x05,0x62,0x2e,0x43,0x4f,0x4d,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,
0x81,0x00,0x4b,0x4c,0x2f,0x68,0xd2,0x64,0x82,0x7d,0x95,0x1d,0xd0,0x2b,0xa4,
0xab,0xc3,0x9d,0xd8,0xc1,0x0f,0xaa,0x3e,0xc6,0xa3,0x11,0x9a,0xc6,0x58,0x40,
0x49,0xcc,0x92,0x8e,0xd4,0x0d,0x20,0x81,0x65,0x59,0xca,0x7d,0x54,0xce,0xd4,
0x43,0x95,0xb5,0xc3,0x8c,0x79,0x01,0xc6,0x17,0xf8,0x04,0x13,0xf7,0x7c,0xf7,
0x7f,0xd7,0xc6,0x62,0x2c,0x21,0x0d,0xab,0x79,0x82,0xd7,0x70,0x9f,0x38,0xba,
0x5f,0x2f,0xb4,0xfd,0xe8,0x62,0x40,0xe3,0x69,0xe7,0xc4,0xe5,0x5a,0xc1,0x71,
0xc0,0x9a,0xdc,0xcc,0xa6,0x0e,0x5e,0x84,0x27,0x70,0xc3,0x47,0x98,0x72,0x89,
0xf9,0x7c,0xe4,0x15,0x4b,0x4c,0xec,0x56,0x1b,0xbf,0x6c,0xf1,0xf7,0x5a,0x76,
0x80,0x1f,0x43,0x7c,0xae,0xdb,0xb3,0x49,0xbc,0xcb };
/* A chain which allows *.com, where the end cert is issued to foo.com */
static const BYTE chain23_0[] = {
0x30,0x82,0x01,0xd9,0x30,0x82,0x01,0x46,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x70,0x29,0x25,0xb2,0x21,0xd9,0x44,0x8a,0x4b,0x9d,0xf3,0x4e,0x6b,0xdb,0x7c,
0xa7,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3c,0x30,0x3a,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x17,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0d,0x30,0x0b,0xa0,0x09,0x30,0x07,0x82,0x05,0x2a,
0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,
0x03,0x81,0x81,0x00,0x39,0x5b,0xf7,0xc9,0x4e,0xac,0x65,0x8b,0x56,0xf4,0xaf,
0x23,0xaa,0x1b,0x95,0x0a,0x6e,0x62,0xa8,0x39,0xa0,0xc7,0x86,0x6f,0xb2,0xac,
0x45,0xb6,0x99,0x21,0x66,0x48,0x1a,0x7d,0xe9,0xa9,0xfe,0x5c,0xc6,0xa7,0x25,
0x89,0x66,0x84,0xc4,0x8e,0x05,0x3b,0x53,0xcf,0xee,0x78,0x88,0x85,0xf1,0x8f,
0xce,0x6e,0x72,0xc6,0x0c,0x06,0x3d,0xa6,0x12,0xad,0x0a,0x07,0xbb,0x16,0x2f,
0x0c,0x96,0x0c,0x78,0x31,0xfa,0x93,0x5a,0x2d,0x8d,0x02,0xf5,0xb0,0x1c,0x83,
0x53,0x14,0xc6,0xa6,0x29,0xa6,0xfe,0xb1,0xc0,0x65,0x72,0x47,0xe9,0xf6,0x01,
0x09,0xa4,0xff,0xd2,0x60,0x87,0x81,0xc1,0xb4,0xe7,0xe8,0x3e,0x9a,0xba,0x25,
0xb2,0x36,0x50,0xc4,0xf8,0xb1,0x94,0x07,0xba,0xdd,0x6a,0x95 };
static const BYTE chain23_1[] = {
0x30,0x82,0x01,0xac,0x30,0x82,0x01,0x15,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x16,0x30,0x14,0x30,0x12,0x06,0x03,0x55,
0x1d,0x07,0x04,0x0b,0x30,0x09,0x82,0x07,0x66,0x6f,0x6f,0x2e,0x63,0x6f,0x6d,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x03,0x81,0x81,0x00,0xa5,0xc0,0xf6,0xb1,0x63,0x3a,0x23,0xc5,0xb0,0x07,0xb4,
0x83,0xcf,0x36,0xb4,0xfe,0x3c,0xcb,0xa6,0x6b,0xed,0xf7,0x4c,0x25,0x9f,0x2d,
0x88,0xcd,0x8a,0xe4,0x88,0xe3,0x33,0xbb,0x50,0x54,0xee,0xc9,0xad,0xed,0x8b,
0xec,0x1e,0x64,0x36,0xe8,0xb5,0x51,0xca,0xd3,0xa5,0xb9,0x7a,0x2d,0x4b,0xe7,
0xe8,0xc9,0xa5,0x1d,0x49,0xa1,0x31,0x0f,0x50,0x46,0xb0,0x0f,0xd0,0xf6,0xad,
0x51,0x18,0x93,0x25,0x8f,0x4f,0x57,0x12,0x72,0x1e,0xa7,0x44,0x37,0x9f,0xf2,
0xc0,0x28,0xe8,0xbe,0xb8,0x75,0x42,0xef,0x1d,0x7a,0xca,0x87,0x0c,0x11,0x25,
0xeb,0xab,0xcf,0xe0,0xe1,0x12,0x38,0xf6,0x68,0x31,0xfb,0x16,0xcb,0xef,0x67,
0x8c,0x4b,0x6b,0xfd,0x69,0xba,0x0b,0x69,0xee,0x74,0xe0,0x73 };
/* A chain which allows *.com, where the end cert is issued to foo.bar.com */
static const BYTE chain24_0[] = {
0x30,0x82,0x01,0xd9,0x30,0x82,0x01,0x46,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x57,0xa6,0x19,0xe0,0x2a,0x65,0x89,0x8d,0x4c,0xd9,0x58,0x8f,0xd0,0xe5,0x5a,
0x02,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3c,0x30,0x3a,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x17,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0d,0x30,0x0b,0xa0,0x09,0x30,0x07,0x82,0x05,0x2a,
0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,
0x03,0x81,0x81,0x00,0x44,0xec,0x96,0x0c,0x82,0x70,0xd6,0x4f,0xa0,0xa0,0x9e,
0x6f,0xfc,0x93,0xcb,0xbb,0x9b,0xb7,0x34,0xf9,0x9c,0x00,0x00,0x7e,0x11,0xeb,
0x38,0x85,0x8d,0xc9,0xc5,0x69,0x05,0x8e,0xcd,0x82,0x6c,0xe4,0xe5,0xb8,0x63,
0xa6,0x8b,0xe5,0x17,0x9e,0x24,0x67,0x34,0xdd,0x4c,0x62,0xdd,0x08,0xfe,0xdc,
0x90,0xe1,0x8e,0x02,0xfe,0xe9,0xc4,0x19,0xfc,0x7d,0xef,0x24,0xcc,0x99,0x9f,
0x11,0x81,0x6a,0x4d,0x46,0xc8,0x48,0x73,0xc8,0x52,0x89,0xab,0xc7,0x0d,0x9b,
0x77,0xc0,0xbc,0xb5,0x57,0x51,0x2e,0x64,0xad,0x98,0x39,0xd3,0x6d,0xe9,0xe1,
0x58,0x32,0x01,0xfa,0xc7,0xa9,0x89,0x58,0x86,0x3c,0x34,0xc3,0x90,0xd6,0xed,
0x4a,0xdc,0x23,0x66,0xc5,0xab,0x76,0xb9,0x47,0xc8,0x00,0x4c };
static const BYTE chain24_1[] = {
0x30,0x82,0x01,0xb0,0x30,0x82,0x01,0x19,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x1a,0x30,0x18,0x30,0x16,0x06,0x03,0x55,
0x1d,0x07,0x04,0x0f,0x30,0x0d,0x82,0x0b,0x66,0x6f,0x6f,0x2e,0x62,0x61,0x72,
0x2e,0x63,0x6f,0x6d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x28,0x64,0x0b,0xad,0x9f,0x12,0xd4,
0x77,0xbe,0x1d,0x20,0x3b,0x69,0x74,0xce,0x91,0x69,0xbe,0x3f,0x76,0x1d,0xe7,
0x81,0x92,0x2b,0xdc,0xbb,0xf9,0xd9,0xba,0xf0,0xba,0x58,0x4a,0x9b,0x47,0x59,
0x34,0xf7,0x19,0xb5,0x15,0x31,0x35,0x4a,0x6d,0x7b,0x91,0x3d,0xc2,0x6c,0x8f,
0x58,0x27,0x0f,0x79,0x96,0x57,0xee,0x4e,0x66,0x3e,0xfa,0x5c,0x22,0xb9,0x57,
0xbf,0x8e,0x7e,0xce,0x98,0x62,0x28,0xb1,0x39,0x0e,0x2d,0x7b,0x5d,0x23,0x70,
0xf0,0x4a,0x63,0x11,0x66,0x80,0xf9,0x99,0x53,0x80,0xda,0x87,0x87,0xf7,0x3c,
0x44,0x76,0x66,0x26,0xa7,0x05,0x3c,0x68,0x66,0x1c,0x07,0x4d,0xcf,0x54,0xaa,
0x5d,0xba,0x7a,0x8f,0x06,0xa7,0x1e,0x86,0xf1,0x5a,0x4b,0x50,0x16,0xad,0x9f,
0x89 };
/* A chain which allows f*.com, where the end cert is issued to foo.com */
static const BYTE chain25_0[] = {
0x30,0x82,0x01,0xda,0x30,0x82,0x01,0x47,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x58,0x61,0x5e,0xb3,0x52,0x48,0xc2,0xa7,0x4a,0xa4,0x70,0x8a,0x64,0x96,0xac,
0xc6,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3d,0x30,0x3b,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x18,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0e,0x30,0x0c,0xa0,0x0a,0x30,0x08,0x82,0x06,0x66,
0x2a,0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,
0x00,0x03,0x81,0x81,0x00,0x18,0xb1,0xb1,0x84,0x3c,0x55,0x29,0xfc,0xd0,0xa7,
0xcd,0x40,0x88,0x9b,0x35,0x9a,0x14,0xa1,0x34,0xd7,0x20,0xd9,0xfb,0x52,0x2f,
0xba,0x6c,0x19,0x5c,0xdc,0x0b,0x0b,0xa5,0x21,0xf9,0x0f,0x98,0x23,0xb6,0xcd,
0x36,0x3f,0xed,0xeb,0xf3,0x10,0xbf,0x62,0x33,0xbb,0xb1,0x94,0x79,0x45,0xe4,
0x42,0x2b,0x9e,0x32,0x53,0x18,0x19,0x59,0x4d,0xc9,0x2d,0x65,0xa4,0xcb,0xd4,
0x0b,0x90,0x2a,0xfe,0xea,0x7f,0x10,0xa9,0xb7,0x65,0xf8,0xc3,0xad,0x26,0x4c,
0x29,0x45,0x60,0xce,0x6a,0x63,0x3e,0xa4,0x6f,0x47,0xaf,0x96,0xea,0xc4,0x35,
0xb0,0x83,0xd7,0x52,0xa9,0x1a,0xd0,0x9a,0x6d,0xac,0xcf,0x6a,0xa6,0xe8,0x8e,
0x52,0x61,0x34,0x95,0xd2,0x22,0x83,0x1f,0x86,0x15,0xa0,0x91,0x2b };
static const BYTE chain25_1[] = {
0x30,0x82,0x01,0xac,0x30,0x82,0x01,0x15,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x16,0x30,0x14,0x30,0x12,0x06,0x03,0x55,
0x1d,0x07,0x04,0x0b,0x30,0x09,0x82,0x07,0x66,0x6f,0x6f,0x2e,0x63,0x6f,0x6d,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x03,0x81,0x81,0x00,0xa5,0xc0,0xf6,0xb1,0x63,0x3a,0x23,0xc5,0xb0,0x07,0xb4,
0x83,0xcf,0x36,0xb4,0xfe,0x3c,0xcb,0xa6,0x6b,0xed,0xf7,0x4c,0x25,0x9f,0x2d,
0x88,0xcd,0x8a,0xe4,0x88,0xe3,0x33,0xbb,0x50,0x54,0xee,0xc9,0xad,0xed,0x8b,
0xec,0x1e,0x64,0x36,0xe8,0xb5,0x51,0xca,0xd3,0xa5,0xb9,0x7a,0x2d,0x4b,0xe7,
0xe8,0xc9,0xa5,0x1d,0x49,0xa1,0x31,0x0f,0x50,0x46,0xb0,0x0f,0xd0,0xf6,0xad,
0x51,0x18,0x93,0x25,0x8f,0x4f,0x57,0x12,0x72,0x1e,0xa7,0x44,0x37,0x9f,0xf2,
0xc0,0x28,0xe8,0xbe,0xb8,0x75,0x42,0xef,0x1d,0x7a,0xca,0x87,0x0c,0x11,0x25,
0xeb,0xab,0xcf,0xe0,0xe1,0x12,0x38,0xf6,0x68,0x31,0xfb,0x16,0xcb,0xef,0x67,
0x8c,0x4b,0x6b,0xfd,0x69,0xba,0x0b,0x69,0xee,0x74,0xe0,0x73 };
/* A chain which allows f*.com, where the end cert is issued to bar.com */
static const BYTE chain26_0[] = {
0x30,0x82,0x01,0xda,0x30,0x82,0x01,0x47,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x6f,0x9d,0x2b,0x8a,0x6d,0xf2,0xd4,0x9f,0x42,0xea,0x3d,0x16,0xe8,0xa5,0x80,
0x7e,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3d,0x30,0x3b,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x18,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0e,0x30,0x0c,0xa0,0x0a,0x30,0x08,0x82,0x06,0x66,
0x2a,0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,
0x00,0x03,0x81,0x81,0x00,0x6c,0x34,0x75,0x37,0xd9,0x65,0x9e,0x20,0x05,0x67,
0x16,0x2e,0xc9,0x53,0xc1,0x69,0x53,0xd6,0x00,0x2c,0x92,0x53,0x91,0xa2,0x2a,
0x94,0x2c,0xfa,0x6f,0xc1,0x7e,0xee,0x3c,0x22,0x06,0xcc,0xd1,0xdc,0xd7,0x21,
0x6a,0xf2,0xd3,0x1b,0xaf,0x34,0x20,0x97,0x0f,0xa1,0xec,0x7f,0x72,0x22,0xf2,
0x98,0x5d,0xf4,0x0b,0x83,0x23,0xa7,0x73,0xb2,0xde,0x12,0x56,0xc9,0x11,0xa6,
0x5d,0xbc,0x32,0x79,0x41,0x8b,0xe6,0x8c,0x9c,0xf9,0x78,0xd2,0x20,0x96,0x27,
0x85,0x41,0xc1,0xb8,0x8a,0x74,0x1e,0xbb,0x27,0xc1,0xf7,0xaf,0xe8,0x1c,0x21,
0xe4,0x10,0x64,0x6a,0xdf,0x79,0x30,0x29,0x0e,0x5a,0x04,0x48,0x18,0xdd,0xb0,
0xaa,0x43,0x08,0x71,0x2e,0x49,0x9f,0x9b,0xc1,0x1a,0xcc,0x5f,0xee };
static const BYTE chain26_1[] = {
0x30,0x82,0x01,0xac,0x30,0x82,0x01,0x15,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x16,0x30,0x14,0x30,0x12,0x06,0x03,0x55,
0x1d,0x07,0x04,0x0b,0x30,0x09,0x82,0x07,0x62,0x61,0x72,0x2e,0x63,0x6f,0x6d,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x03,0x81,0x81,0x00,0x47,0xa6,0x37,0x73,0x4f,0x19,0xf9,0xaf,0xa9,0xb5,0xa8,
0x41,0x99,0x31,0xa5,0xaa,0x0d,0x6c,0xb8,0x9d,0xe6,0xf6,0x59,0x4d,0xd8,0x12,
0x76,0xcc,0x0b,0xaa,0x2f,0x08,0x1c,0x75,0x54,0x5a,0xae,0x09,0x39,0x25,0xfc,
0x44,0xa1,0x00,0x24,0x5c,0xec,0x20,0x04,0xc7,0xe9,0x14,0xb0,0x71,0x4a,0x33,
0x41,0x7f,0x1d,0x47,0xe5,0x7c,0xe4,0x86,0xb1,0xeb,0xa4,0x9d,0x2a,0x89,0xeb,
0x1d,0x15,0xd6,0x42,0x39,0xe4,0x05,0x26,0xbf,0xc2,0x09,0x6c,0x5c,0xd1,0x56,
0xfd,0x18,0x42,0xd3,0xd5,0x7c,0x52,0xa8,0x4a,0x77,0xe8,0xc0,0x53,0x2b,0xa6,
0x01,0xf9,0x90,0x68,0x60,0x3f,0xcb,0xb8,0x24,0x16,0x0b,0x69,0x4b,0xd9,0xf8,
0x2c,0x92,0xed,0x89,0xa5,0xff,0x71,0xc5,0xed,0xe8,0xbc,0x70 };
/* A chain which allows foo.com, where the end cert is issued to
 * foo.com\0baddie
 */
static const BYTE chain27_0[] = {
0x30,0x82,0x01,0xdb,0x30,0x82,0x01,0x48,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x3d,0xd2,0x79,0xf1,0xa7,0x1a,0xac,0xb2,0x43,0x5e,0x53,0x67,0xc9,0xa1,0x8c,
0x53,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x3e,0x30,0x3c,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x19,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x0f,0x30,0x0d,0xa0,0x0b,0x30,0x09,0x82,0x07,0x66,
0x6f,0x6f,0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,
0x05,0x00,0x03,0x81,0x81,0x00,0x18,0x5d,0xb0,0x42,0x96,0xf7,0x96,0x7b,0x23,
0x46,0xa9,0xf4,0x99,0x57,0x27,0x8e,0x24,0xae,0x29,0xb8,0x04,0xe3,0x1b,0x64,
0x22,0x8e,0x20,0xf1,0x45,0xb8,0x84,0x38,0x41,0x99,0x73,0xb2,0xbc,0xd9,0xcb,
0x9d,0x9f,0xf4,0x86,0xed,0x21,0x34,0x42,0x21,0xf4,0x8f,0xa7,0xcf,0x19,0xe5,
0xd3,0x4f,0x7c,0x4f,0xc7,0x9e,0x91,0x3a,0x16,0x0b,0x61,0x9e,0xfa,0x08,0x1b,
0x01,0xc0,0x19,0xeb,0xe8,0xd4,0x5a,0xe9,0xe0,0x71,0x04,0xaa,0x88,0x93,0x10,
0x3e,0xc5,0x23,0xbe,0x0b,0xab,0xa9,0x07,0x46,0x4d,0x4b,0xf2,0x95,0xeb,0x69,
0xf2,0x70,0x33,0x5c,0xf7,0xd1,0xa9,0x5b,0xd2,0x84,0x10,0xa6,0x00,0xbe,0xb5,
0xd4,0xe7,0xa3,0x54,0x63,0x3c,0xca,0xb0,0xbf,0x96,0xd1,0x6f,0x64,0xaf };
static const BYTE chain27_1[] = {
0x30,0x82,0x01,0xb3,0x30,0x82,0x01,0x1c,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x1d,0x30,0x1b,0x30,0x19,0x06,0x03,0x55,
0x1d,0x07,0x04,0x12,0x30,0x10,0x82,0x0e,0x66,0x6f,0x6f,0x2e,0x63,0x6f,0x6d,
0x00,0x62,0x61,0x64,0x64,0x69,0x65,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x6f,0x06,0xc6,0xd0,
0xa1,0x8c,0xc6,0xed,0x33,0x04,0xf6,0x64,0x81,0xa1,0x0f,0x79,0x4d,0x9a,0xbd,
0xc0,0x8f,0x18,0xc2,0xb8,0x82,0xb9,0x70,0x6f,0xbe,0xaa,0xbb,0x43,0x95,0x79,
0x58,0xec,0xdf,0x94,0x8b,0x4d,0xe1,0xee,0x56,0xc4,0x57,0x7c,0xff,0xde,0x5f,
0x52,0xf9,0x3a,0x9e,0x33,0x99,0x6e,0xdf,0xf2,0xce,0x40,0xc7,0xd8,0xb5,0x86,
0xae,0xb7,0x52,0x8b,0xc5,0x62,0x17,0xce,0x03,0x06,0x29,0x5a,0xf6,0x5a,0x9a,
0x8e,0xb7,0xe3,0xb2,0xcd,0x3f,0xa7,0x61,0x2b,0xba,0xe6,0x68,0xa8,0x92,0x8a,
0xe5,0x32,0xe0,0x49,0xd4,0x42,0xc0,0x1a,0x92,0x72,0xb0,0x66,0x48,0x84,0x74,
0xd3,0x25,0xe3,0x66,0x2f,0x30,0xea,0x40,0x72,0x82,0xc4,0xc5,0x4a,0x4e,0xcc,
0x32,0xbc,0x36,0xa4 };
/* A chain which allows foo.*.com, where the end cert is issued to foo.bar.com.
 */
static const BYTE chain28_0[] = {
0x30,0x82,0x01,0xdd,0x30,0x82,0x01,0x4a,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x72,0x7f,0x98,0xcd,0x99,0xa6,0x5f,0x89,0x4a,0x5c,0xd2,0x15,0xd0,0x68,0x25,
0xc8,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x40,0x30,0x3e,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1b,0x06,0x03,0x55,0x1d,
0x1e,0x01,0x01,0xff,0x04,0x11,0x30,0x0f,0xa0,0x0d,0x30,0x0b,0x82,0x09,0x66,
0x6f,0x6f,0x2e,0x2a,0x2e,0x63,0x6f,0x6d,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,
0x02,0x1d,0x05,0x00,0x03,0x81,0x81,0x00,0x01,0x9a,0x8c,0x75,0x20,0x9f,0x91,
0xe9,0xde,0x28,0x3c,0x7a,0xd9,0xf1,0x2d,0xa3,0x0d,0x66,0x9f,0xd8,0x2a,0xb0,
0xb5,0x19,0x9a,0x29,0xc7,0xd5,0xe5,0xf6,0x75,0xee,0x06,0xe4,0xd2,0x71,0x77,
0x7e,0xbc,0x1a,0x5f,0xbe,0x28,0xba,0x6e,0x50,0x8e,0xe6,0xd9,0xd5,0x5a,0x21,
0xa6,0x29,0xff,0xb0,0xc7,0xfe,0x06,0xb6,0xde,0x51,0x5b,0xab,0xe4,0xbb,0xbf,
0xaa,0x03,0xa9,0xfe,0x71,0xb8,0x3f,0x65,0xe2,0x2c,0xa9,0x31,0x18,0x6d,0xe6,
0x54,0xd0,0x90,0x6b,0x7b,0x3a,0x9a,0x45,0xa9,0x1b,0x05,0x2e,0xca,0xd7,0x5f,
0x79,0x3d,0x1c,0xbf,0xdb,0xce,0x23,0x5d,0x0a,0x9b,0xb4,0x91,0xfe,0xd4,0x94,
0x92,0x76,0x36,0xae,0x03,0xb8,0x30,0x13,0x58,0x69,0x71,0xea,0xc2,0xbc,0x31,
0x83 };
static const BYTE chain28_1[] = {
0x30,0x82,0x01,0xb0,0x30,0x82,0x01,0x19,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x1a,0x30,0x18,0x30,0x16,0x06,0x03,0x55,
0x1d,0x07,0x04,0x0f,0x30,0x0d,0x82,0x0b,0x66,0x6f,0x6f,0x2e,0x62,0x61,0x72,
0x2e,0x63,0x6f,0x6d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x28,0x64,0x0b,0xad,0x9f,0x12,0xd4,
0x77,0xbe,0x1d,0x20,0x3b,0x69,0x74,0xce,0x91,0x69,0xbe,0x3f,0x76,0x1d,0xe7,
0x81,0x92,0x2b,0xdc,0xbb,0xf9,0xd9,0xba,0xf0,0xba,0x58,0x4a,0x9b,0x47,0x59,
0x34,0xf7,0x19,0xb5,0x15,0x31,0x35,0x4a,0x6d,0x7b,0x91,0x3d,0xc2,0x6c,0x8f,
0x58,0x27,0x0f,0x79,0x96,0x57,0xee,0x4e,0x66,0x3e,0xfa,0x5c,0x22,0xb9,0x57,
0xbf,0x8e,0x7e,0xce,0x98,0x62,0x28,0xb1,0x39,0x0e,0x2d,0x7b,0x5d,0x23,0x70,
0xf0,0x4a,0x63,0x11,0x66,0x80,0xf9,0x99,0x53,0x80,0xda,0x87,0x87,0xf7,0x3c,
0x44,0x76,0x66,0x26,0xa7,0x05,0x3c,0x68,0x66,0x1c,0x07,0x4d,0xcf,0x54,0xaa,
0x5d,0xba,0x7a,0x8f,0x06,0xa7,0x1e,0x86,0xf1,0x5a,0x4b,0x50,0x16,0xad,0x9f,
0x89 };
/* A chain whose end certificate is issued to *.winehq.org. */
static const BYTE chain29_1[] = {
0x30,0x82,0x01,0xab,0x30,0x82,0x01,0x16,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,
0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,
0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,
0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,
0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,
0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,
0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,
0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,
0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,
0x00,0x01,0xa3,0x1b,0x30,0x19,0x30,0x17,0x06,0x03,0x55,0x1d,0x07,0x04,0x10,
0x30,0x0e,0x82,0x0c,0x2a,0x2e,0x77,0x69,0x6e,0x65,0x68,0x71,0x2e,0x6f,0x72,
0x67,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,
0x81,0x81,0x00,0x65,0xbf,0xfa,0xf7,0xc3,0x09,0x70,0x25,0x8a,0x46,0x69,0xf6,
0xdc,0x07,0x1e,0x30,0xc9,0xe4,0x58,0x89,0x65,0x3a,0xa8,0xda,0xbd,0x17,0xf8,
0x1d,0x0d,0x7d,0x47,0xb1,0xb2,0xda,0x17,0x9f,0xf6,0x47,0xe0,0xe4,0x4a,0xeb,
0x02,0xc9,0x2e,0x69,0x1c,0x57,0x2a,0x80,0xc9,0x01,0x77,0x7b,0x27,0xff,0x2f,
0xaf,0xdf,0xf3,0x65,0x12,0xd8,0x7d,0xc2,0xbf,0x1b,0x1d,0x18,0x96,0x5c,0xf6,
0xba,0x43,0xc5,0x43,0x57,0xc0,0xdd,0x97,0x95,0xfb,0x1c,0xad,0x64,0x0f,0x61,
0x3a,0xe9,0x27,0xa4,0x57,0x27,0x34,0xa7,0x42,0xde,0x78,0x1a,0x71,0x80,0x23,
0xd6,0xd7,0x22,0xf0,0x24,0x0d,0x71,0xf1,0x2b,0xd0,0xd8,0x76,0x3d,0xef,0x4c,
0xce,0x1c,0x3b,0x83,0x1b,0x63,0x10,0x6c,0x63,0xe5,0x69 };
/* chain0_0 -> chain30_1 -> chain30_2
 * A chain whose intermediate certificate has an unsupported critical
 * extension.
 */
static const BYTE chain30_1[] = {
0x30,0x82,0x01,0xc0,0x30,0x82,0x01,0x2b,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,
0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,
0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,
0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,
0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,
0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,
0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,
0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,
0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,
0x00,0x01,0xa3,0x30,0x30,0x2e,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x04,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0b,0x06,0x02,0x2a,0x03,
0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0x51,0x3e,0x35,0x1b,0x66,0x3c,
0xca,0x5e,0xf3,0xf9,0x1b,0xd5,0x03,0x13,0xf8,0xcf,0x87,0xdf,0xed,0x75,0xa6,
0xcd,0x4b,0x1c,0x15,0xd3,0xd8,0x58,0x85,0x85,0x2c,0x64,0x31,0xbd,0xbb,0xad,
0xff,0x38,0x64,0xc5,0x16,0x43,0x14,0x0e,0x71,0x35,0xf3,0xe9,0xca,0xf9,0xf4,
0x69,0xa7,0x67,0xa8,0x0f,0xc9,0xcf,0x6f,0x22,0xe5,0x39,0xb8,0xfc,0xe7,0x50,
0x82,0xf7,0xa4,0xaa,0x29,0xe1,0xa9,0xb5,0x03,0x5e,0x0b,0x5f,0x9c,0x8e,0x29,
0x64,0xe5,0xb6,0xed,0xde,0x04,0x0e,0xdb,0xad,0xa3,0xc6,0x2a,0xb0,0x12,0x86,
0x60,0xd4,0xff,0xd8,0xea,0x85,0x54,0x34,0xca,0xc1,0x85,0x4e,0xb5,0x15,0x96,
0xb7,0xa5,0x64,0x7b,0xc7,0x76,0xcb,0x04,0x75,0x9e,0x1e,0xbd,0x62,0x79,0xc5,
0x1f,0x32 };
static const BYTE chain30_2[] = {
0x30,0x82,0x01,0x8d,0x30,0x81,0xf9,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,0x10,
0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,
0x32,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x39,0x30,0x31,0x30,0x30,0x30,0x30,
0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,
0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,
0x05,0x43,0x65,0x72,0x74,0x33,0x30,0x81,0x9d,0x30,0x0b,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,
0x81,0x81,0x00,0xc3,0x31,0x35,0xc3,0x9d,0x8a,0x87,0x20,0xc6,0x32,0xe9,0xb1,
0xf6,0x8f,0xf9,0x05,0x73,0x1d,0xa7,0xde,0xab,0x15,0x8a,0x9c,0x7f,0x11,0x7e,
0x77,0xa0,0x42,0x80,0xf4,0x79,0xda,0x98,0x7b,0x00,0xfa,0x8f,0x0c,0xd0,0xeb,
0x8a,0x80,0xed,0x07,0xfc,0x64,0x71,0x03,0xc1,0xd6,0x2f,0x9b,0xde,0x42,0x63,
0xd9,0x79,0xea,0xdd,0x10,0xe8,0x68,0xc8,0x69,0x4f,0x4a,0x39,0x23,0x87,0xca,
0xd1,0xc9,0x77,0x14,0x30,0x85,0x9e,0xf7,0x79,0xf9,0x07,0xb7,0x7c,0x55,0xcb,
0xa7,0xd5,0xb8,0x44,0xb5,0x20,0xb5,0x01,0x5c,0xa2,0xd1,0xd5,0xad,0x0f,0x87,
0xaf,0x37,0xd1,0x39,0x0c,0x0d,0xd5,0xde,0x26,0x7a,0xed,0xf9,0x2a,0xb1,0x60,
0x65,0x2d,0x08,0x24,0x51,0x1d,0xb0,0x0a,0xb5,0x13,0xc7,0x02,0x03,0x01,0x00,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x03,
0x81,0x81,0x00,0x14,0x75,0x85,0xcc,0x68,0xfe,0x98,0x6f,0xf6,0x67,0x00,0x5b,
0x0c,0xfc,0x36,0x18,0xf4,0x56,0x46,0x7c,0xb9,0xfa,0x6c,0xe6,0x37,0xaf,0x69,
0x37,0x93,0x8c,0x35,0x3a,0x1b,0x58,0x2f,0xe2,0x06,0x39,0x85,0x3f,0x73,0xcf,
0xe1,0x3f,0x27,0x19,0x60,0xc3,0x1b,0xf6,0x69,0x3b,0x8e,0x57,0x7b,0xd8,0xb9,
0xc6,0x9f,0x13,0x72,0x22,0x04,0x8f,0x5c,0x54,0x13,0x8c,0x63,0xe3,0x6b,0x70,
0x98,0xec,0xcc,0xe1,0x93,0xb1,0x4b,0x30,0x4c,0xde,0xe8,0x3c,0x68,0x38,0x44,
0x5e,0xe2,0x2b,0xf5,0xa1,0xee,0x02,0x7e,0x09,0x15,0xff,0xc9,0xf6,0xaf,0xf5,
0xcc,0xeb,0xfc,0xe7,0x3c,0x92,0xdb,0x31,0xab,0x1e,0xb8,0x9e,0xf0,0x5e,0xa3,
0x93,0xfe,0xab,0x26,0x7b,0x01,0xa8,0x98,0x88,0xbb,0xee };
/* chain0_0 -> chain31_1: a chain whose end certificate has two CNs, a
 * wildcard name "*.foo.com" and a non-wildcard name "foo.com".
 */
static const BYTE chain31_1[] = {
0x30,0x82,0x01,0xa2,0x30,0x82,0x01,0x0d,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x30,
0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,
0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x30,0x24,0x31,0x22,0x30,0x0e,0x06,0x03,0x55,0x04,0x03,
0x13,0x07,0x66,0x6f,0x6f,0x2e,0x63,0x6f,0x6d,0x30,0x10,0x06,0x03,0x55,0x04,
0x03,0x13,0x09,0x2a,0x2e,0x66,0x6f,0x6f,0x2e,0x63,0x6f,0x6d,0x30,0x81,0x9d,
0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x03,0x81,
0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,
0xe5,0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,
0xc6,0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,
0xa7,0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,
0x8b,0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,
0x6b,0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,
0xfc,0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,
0x85,0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,
0xb2,0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,
0x72,0xa3,0x02,0x03,0x01,0x00,0x01,0x30,0x0b,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x05,0x03,0x81,0x81,0x00,0xa0,0x93,0x52,0x87,0x81,0xe2,
0xff,0x2a,0xc7,0xef,0x5f,0x3c,0xbc,0x88,0x99,0xc0,0x47,0x3e,0x13,0xe9,0x87,
0xfa,0x36,0xd7,0xb5,0xe8,0xdf,0x70,0xcc,0x36,0xe4,0x70,0x3c,0xcd,0xa2,0x0b,
0x31,0x6e,0x0a,0xb9,0x00,0xf0,0x4f,0xb6,0xc2,0xce,0xf4,0x33,0x1e,0xc0,0x29,
0xc0,0x73,0x0c,0xcf,0x28,0xa5,0x26,0x9d,0xc2,0xaf,0x85,0x30,0x81,0xbf,0xd1,
0x70,0x3f,0x69,0x15,0xc5,0x41,0x1d,0x8e,0xd4,0xfa,0x02,0xcd,0xba,0xf1,0xf2,
0x67,0xb5,0x45,0x29,0xad,0xe8,0x54,0x9a,0x0f,0x1a,0x8f,0xdf,0x16,0xf4,0xcb,
0x43,0x08,0xe5,0x78,0x2b,0x95,0xf3,0x75,0xb6,0x88,0xf0,0x6b,0x5c,0x5b,0x50,
0x04,0x91,0x3b,0x89,0x5a,0x60,0x1f,0xfc,0x36,0x53,0x32,0x36,0x0a,0x4d,0x03,
0x2c,0xd7 };
/* Microsoft Root Certificate Authority 2010 */
static const BYTE chain32_1[] = {
0x30,0x82,0x05,0xed,0x30,0x82,0x03,0xd5,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x28,0xcc,0x3a,0x25,0xbf,0xba,0x44,0xac,0x44,0x9a,0x9b,0x58,0x6b,0x43,0x39,
0xaa,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,
0x00,0x30,0x81,0x88,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x57,0x61,
0x73,0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,
0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,0x30,0x1c,
0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,
0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x32,
0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,0x73,
0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,
0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,
0x20,0x32,0x30,0x31,0x30,0x30,0x1e,0x17,0x0d,0x31,0x30,0x30,0x36,0x32,0x33,
0x32,0x31,0x35,0x37,0x32,0x34,0x5a,0x17,0x0d,0x33,0x35,0x30,0x36,0x32,0x33,
0x32,0x32,0x30,0x34,0x30,0x31,0x5a,0x30,0x81,0x88,0x31,0x0b,0x30,0x09,0x06,
0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,
0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,
0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,
0x6e,0x64,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,
0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,
0x74,0x69,0x6f,0x6e,0x31,0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,
0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,
0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,
0x68,0x6f,0x72,0x69,0x74,0x79,0x20,0x32,0x30,0x31,0x30,0x30,0x82,0x02,0x22,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,
0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,0x02,0x82,0x02,0x01,0x00,0xb9,
0x08,0x9e,0x28,0xe4,0xe4,0xec,0x06,0x4e,0x50,0x68,0xb3,0x41,0xc5,0x7b,0xeb,
0xae,0xb6,0x8e,0xaf,0x81,0xba,0x22,0x44,0x1f,0x65,0x34,0x69,0x4c,0xbe,0x70,
0x40,0x17,0xf2,0x16,0x7b,0xe2,0x79,0xfd,0x86,0xed,0x0d,0x39,0xf4,0x1b,0xa8,
0xad,0x92,0x90,0x1e,0xcb,0x3d,0x76,0x8f,0x5a,0xd9,0xb5,0x91,0x10,0x2e,0x3c,
0x05,0x8d,0x8a,0x6d,0x24,0x54,0xe7,0x1f,0xed,0x56,0xad,0x83,0xb4,0x50,0x9c,
0x15,0xa5,0x17,0x74,0x88,0x59,0x20,0xfc,0x08,0xc5,0x84,0x76,0xd3,0x68,0xd4,
0x6f,0x28,0x78,0xce,0x5c,0xb8,0xf3,0x50,0x90,0x44,0xff,0xe3,0x63,0x5f,0xbe,
0xa1,0x9a,0x2c,0x96,0x15,0x04,0xd6,0x07,0xfe,0x1e,0x84,0x21,0xe0,0x42,0x31,
0x11,0xc4,0x28,0x36,0x94,0xcf,0x50,0xa4,0x62,0x9e,0xc9,0xd6,0xab,0x71,0x00,
0xb2,0x5b,0x0c,0xe6,0x96,0xd4,0x0a,0x24,0x96,0xf5,0xff,0xc6,0xd5,0xb7,0x1b,
0xd7,0xcb,0xb7,0x21,0x62,0xaf,0x12,0xdc,0xa1,0x5d,0x37,0xe3,0x1a,0xfb,0x1a,
0x46,0x98,0xc0,0x9b,0xc0,0xe7,0x63,0x1f,0x2a,0x08,0x93,0x02,0x7e,0x1e,0x6a,
0x8e,0xf2,0x9f,0x18,0x89,0xe4,0x22,0x85,0xa2,0xb1,0x84,0x57,0x40,0xff,0xf5,
0x0e,0xd8,0x6f,0x9c,0xed,0xe2,0x45,0x31,0x01,0xcd,0x17,0xe9,0x7f,0xb0,0x81,
0x45,0xe3,0xaa,0x21,0x40,0x26,0xa1,0x72,0xaa,0xa7,0x4f,0x3c,0x01,0x05,0x7e,
0xee,0x83,0x58,0xb1,0x5e,0x06,0x63,0x99,0x62,0x91,0x78,0x82,0xb7,0x0d,0x93,
0x0c,0x24,0x6a,0xb4,0x1b,0xdb,0x27,0xec,0x5f,0x95,0x04,0x3f,0x93,0x4a,0x30,
0xf5,0x97,0x18,0xb3,0xa7,0xf9,0x19,0xa7,0x93,0x33,0x1d,0x01,0xc8,0xdb,0x22,
0x52,0x5c,0xd7,0x25,0xc9,0x46,0xf9,0xa2,0xfb,0x87,0x59,0x43,0xbe,0x9b,0x62,
0xb1,0x8d,0x2d,0x86,0x44,0x1a,0x46,0xac,0x78,0x61,0x7e,0x30,0x09,0xfa,0xae,
0x89,0xc4,0x41,0x2a,0x22,0x66,0x03,0x91,0x39,0x45,0x9c,0xc7,0x8b,0x0c,0xa8,
0xca,0x0d,0x2f,0xfb,0x52,0xea,0x0c,0xf7,0x63,0x33,0x23,0x9d,0xfe,0xb0,0x1f,
0xad,0x67,0xd6,0xa7,0x50,0x03,0xc6,0x04,0x70,0x63,0xb5,0x2c,0xb1,0x86,0x5a,
0x43,0xb7,0xfb,0xae,0xf9,0x6e,0x29,0x6e,0x21,0x21,0x41,0x26,0x06,0x8c,0xc9,
0xc3,0xee,0xb0,0xc2,0x85,0x93,0xa1,0xb9,0x85,0xd9,0xe6,0x32,0x6c,0x4b,0x4c,
0x3f,0xd6,0x5d,0xa3,0xe5,0xb5,0x9d,0x77,0xc3,0x9c,0xc0,0x55,0xb7,0x74,0x00,
0xe3,0xb8,0x38,0xab,0x83,0x97,0x50,0xe1,0x9a,0x42,0x24,0x1d,0xc6,0xc0,0xa3,
0x30,0xd1,0x1a,0x5a,0xc8,0x52,0x34,0xf7,0x73,0xf1,0xc7,0x18,0x1f,0x33,0xad,
0x7a,0xec,0xcb,0x41,0x60,0xf3,0x23,0x94,0x20,0xc2,0x48,0x45,0xac,0x5c,0x51,
0xc6,0x2e,0x80,0xc2,0xe2,0x77,0x15,0xbd,0x85,0x87,0xed,0x36,0x9d,0x96,0x91,
0xee,0x00,0xb5,0xa3,0x70,0xec,0x9f,0xe3,0x8d,0x80,0x68,0x83,0x76,0xba,0xaf,
0x5d,0x70,0x52,0x22,0x16,0xe2,0x66,0xfb,0xba,0xb3,0xc5,0xc2,0xf7,0x3e,0x2f,
0x77,0xa6,0xca,0xde,0xc1,0xa6,0xc6,0x48,0x4c,0xc3,0x37,0x51,0x23,0xd3,0x27,
0xd7,0xb8,0x4e,0x70,0x96,0xf0,0xa1,0x44,0x76,0xaf,0x78,0xcf,0x9a,0xe1,0x66,
0x13,0x02,0x03,0x01,0x00,0x01,0xa3,0x51,0x30,0x4f,0x30,0x0b,0x06,0x03,0x55,
0x1d,0x0f,0x04,0x04,0x03,0x02,0x01,0x86,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,
0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,0x55,
0x1d,0x0e,0x04,0x16,0x04,0x14,0xd5,0xf6,0x56,0xcb,0x8f,0xe8,0xa2,0x5c,0x62,
0x68,0xd1,0x3d,0x94,0x90,0x5b,0xd7,0xce,0x9a,0x18,0xc4,0x30,0x10,0x06,0x09,
0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x15,0x01,0x04,0x03,0x02,0x01,0x00,0x30,
0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,
0x82,0x02,0x01,0x00,0xac,0xa5,0x96,0x8c,0xbf,0xbb,0xae,0xa6,0xf6,0xd7,0x71,
0x87,0x43,0x31,0x56,0x88,0xfd,0x1c,0x32,0x71,0x5b,0x35,0xb7,0xd4,0xf0,0x91,
0xf2,0xaf,0x37,0xe2,0x14,0xf1,0xf3,0x02,0x26,0x05,0x3e,0x16,0x14,0x7f,0x14,
0xba,0xb8,0x4f,0xfb,0x89,0xb2,0xb2,0xe7,0xd4,0x09,0xcc,0x6d,0xb9,0x5b,0x3b,
0x64,0x65,0x70,0x66,0xb7,0xf2,0xb1,0x5a,0xdf,0x1a,0x02,0xf3,0xf5,0x51,0xb8,
0x67,0x6d,0x79,0xf3,0xbf,0x56,0x7b,0xe4,0x84,0xb9,0x2b,0x1e,0x9b,0x40,0x9c,
0x26,0x34,0xf9,0x47,0x18,0x98,0x69,0xd8,0x1c,0xd7,0xb6,0xd1,0xbf,0x8f,0x61,
0xc2,0x67,0xc4,0xb5,0xef,0x60,0x43,0x8e,0x10,0x1b,0x36,0x49,0xe4,0x20,0xca,
0xad,0xa7,0xc1,0xb1,0x27,0x65,0x09,0xf8,0xcd,0xf5,0x5b,0x2a,0xd0,0x84,0x33,
0xf3,0xef,0x1f,0xf2,0xf5,0x9c,0x0b,0x58,0x93,0x37,0xa0,0x75,0xa0,0xde,0x72,
0xde,0x6c,0x75,0x2a,0x66,0x22,0xf5,0x8c,0x06,0x30,0x56,0x9f,0x40,0xb9,0x30,
0xaa,0x40,0x77,0x15,0x82,0xd7,0x8b,0xec,0xc0,0xd3,0xb2,0xbd,0x83,0xc5,0x77,
0x0c,0x1e,0xae,0xaf,0x19,0x53,0xa0,0x4d,0x79,0x71,0x9f,0x0f,0xaf,0x30,0xce,
0x67,0xf9,0xd6,0x2c,0xcc,0x22,0x41,0x7a,0x07,0xf2,0x97,0x42,0x18,0xce,0x59,
0x79,0x10,0x55,0xde,0x6f,0x10,0xe4,0xb8,0xda,0x83,0x66,0x40,0x16,0x09,0x68,
0x23,0x5b,0x97,0x2e,0x26,0x9a,0x02,0xbb,0x57,0x8c,0xc5,0xb8,0xba,0x69,0x62,
0x32,0x80,0x89,0x9e,0xa1,0xfd,0xc0,0x92,0x7c,0x7b,0x2b,0x33,0x19,0x84,0x2a,
0x63,0xc5,0x00,0x68,0x62,0xfa,0x9f,0x47,0x8d,0x99,0x7a,0x45,0x3a,0xa7,0xe9,
0xed,0xee,0x69,0x42,0xb5,0xf3,0x81,0x9b,0x47,0x56,0x10,0x7b,0xfc,0x70,0x36,
0x84,0x18,0x73,0xea,0xef,0xf9,0x97,0x4d,0x9e,0x33,0x23,0xdd,0x26,0x0b,0xba,
0x2a,0xb7,0x3f,0x44,0xdc,0x83,0x27,0xff,0xbd,0x61,0x59,0x2b,0x11,0xb7,0xca,
0x4f,0xdb,0xc5,0x8b,0x0c,0x1c,0x31,0xae,0x32,0xf8,0xf8,0xb9,0x42,0xf7,0x7f,
0xdc,0x61,0x9a,0x76,0xb1,0x5a,0x04,0xe1,0x11,0x3d,0x66,0x45,0xb7,0x18,0x71,
0xbe,0xc9,0x24,0x85,0xd6,0xf3,0xd4,0xba,0x41,0x34,0x5d,0x12,0x2d,0x25,0xb9,
0x8d,0xa6,0x13,0x48,0x6d,0x4b,0xb0,0x07,0x7d,0x99,0x93,0x09,0x61,0x81,0x74,
0x57,0x26,0x8a,0xab,0x69,0xe3,0xe4,0xd9,0xc7,0x88,0xcc,0x24,0xd8,0xec,0x52,
0x24,0x5c,0x1e,0xbc,0x91,0x14,0xe2,0x96,0xde,0xeb,0x0a,0xda,0x9e,0xdd,0x5f,
0xb3,0x5b,0xdb,0xd4,0x82,0xec,0xc6,0x20,0x50,0x87,0x25,0x40,0x3a,0xfb,0xc7,
0xee,0xcd,0xfe,0x33,0xe5,0x6e,0xc3,0x84,0x09,0x55,0x03,0x25,0x39,0xc0,0xe9,
0x35,0x5d,0x65,0x31,0xa8,0xf6,0xbf,0xa0,0x09,0xcd,0x29,0xc7,0xb3,0x36,0x32,
0x2e,0xdc,0x95,0xf3,0x83,0xc1,0x5a,0xcf,0x8b,0x8d,0xf6,0xea,0xb3,0x21,0xf8,
0xa4,0xed,0x1e,0x31,0x0e,0xb6,0x4c,0x11,0xab,0x60,0x0b,0xa4,0x12,0x23,0x22,
0x17,0xa3,0x36,0x64,0x82,0x91,0x04,0x12,0xe0,0xab,0x6f,0x1e,0xcb,0x50,0x05,
0x61,0xb4,0x40,0xff,0x59,0x86,0x71,0xd1,0xd5,0x33,0x69,0x7c,0xa9,0x73,0x8a,
0x38,0xd7,0x64,0x0c,0xf1,0x69 };
/* Microsoft Code Signing PCA 2010 */
static const BYTE chain32_2[] = {
0x30,0x82,0x06,0x70,0x30,0x82,0x04,0x58,0xa0,0x03,0x02,0x01,0x02,0x02,0x0a,
0x61,0x0c,0x52,0x4c,0x00,0x00,0x00,0x00,0x00,0x03,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x81,0x88,0x31,0x0b,
0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,
0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,0x6e,0x67,0x74,
0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,
0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,
0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,
0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x32,0x30,0x30,0x06,0x03,0x55,0x04,
0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,
0x6f,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,
0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x20,0x32,0x30,0x31,0x30,0x30,
0x1e,0x17,0x0d,0x31,0x30,0x30,0x37,0x30,0x36,0x32,0x30,0x34,0x30,0x31,0x37,
0x5a,0x17,0x0d,0x32,0x35,0x30,0x37,0x30,0x36,0x32,0x30,0x35,0x30,0x31,0x37,
0x5a,0x30,0x7e,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,
0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x57,0x61,0x73,
0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,
0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,0x30,0x1c,0x06,
0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,
0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x28,0x30,
0x26,0x06,0x03,0x55,0x04,0x03,0x13,0x1f,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,
0x66,0x74,0x20,0x43,0x6f,0x64,0x65,0x20,0x53,0x69,0x67,0x6e,0x69,0x6e,0x67,
0x20,0x50,0x43,0x41,0x20,0x32,0x30,0x31,0x30,0x30,0x82,0x01,0x22,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,
0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xe9,0x0e,0x64,
0x50,0x79,0x67,0xb5,0xc4,0xe3,0xfd,0x09,0x00,0x4c,0x9e,0x94,0xac,0xf7,0x56,
0x68,0xea,0x44,0xd8,0xcf,0xc5,0x58,0x4f,0xa9,0xa5,0x76,0x7c,0x6d,0x45,0xba,
0xd3,0x39,0x92,0xb4,0xa4,0x1e,0xf9,0xf9,0x65,0x82,0xe4,0x17,0xd2,0x8f,0xfd,
0x44,0x9c,0x08,0xe8,0x65,0x93,0xce,0x2c,0x55,0x84,0xbf,0x7d,0x08,0xe3,0x2e,
0x2b,0xa8,0x41,0x2b,0x18,0xb7,0xa2,0x4b,0x6e,0x49,0x4c,0x6b,0x15,0x07,0xde,
0xd1,0xd2,0xc2,0x89,0x1e,0x71,0x94,0xcd,0xb5,0x7f,0x4b,0xb4,0xaf,0x08,0xd8,
0xcc,0x88,0xd6,0x6b,0x17,0x94,0x3a,0x93,0xce,0x26,0x3f,0xec,0xe6,0xfe,0x34,
0x98,0x57,0xd5,0x1d,0x5d,0x49,0xf6,0xb2,0x2a,0x2e,0xd5,0x85,0xbb,0x59,0x3f,
0xf8,0x90,0xb4,0x2b,0x83,0x74,0xca,0x2b,0xb3,0x3b,0x46,0xe3,0xf0,0x46,0x49,
0xc1,0x17,0x66,0x54,0xc9,0x1c,0xbd,0x1d,0xc4,0x55,0x62,0x57,0x72,0xf8,0x67,
0xb9,0x25,0x20,0x34,0xde,0x5d,0xa6,0xa5,0x95,0x5e,0xab,0x28,0x80,0xcd,0xd5,
0xb2,0x9e,0xe5,0x03,0xb5,0x63,0xd3,0xb2,0x14,0xc8,0xc1,0xc8,0x8a,0x26,0x0a,
0x59,0x7f,0x07,0xec,0xff,0x0e,0xed,0x80,0x12,0x35,0x4c,0x12,0xa6,0xbe,0x52,
0x5b,0xf5,0xa6,0xda,0xe0,0x8b,0x0b,0x48,0x77,0xd6,0x85,0x47,0xd5,0x10,0xb9,
0xc6,0xe8,0xaa,0xee,0x8b,0x6a,0x2d,0x05,0x5c,0x60,0xc6,0xb4,0x2a,0x5b,0x9c,
0x23,0x1c,0x5f,0x45,0xe3,0x1a,0x14,0x1e,0x6f,0x37,0xcb,0x19,0x33,0x80,0x6a,
0x89,0x4d,0xa3,0x6a,0x66,0x63,0x78,0x93,0xd5,0x30,0xcf,0x95,0x1f,0x02,0x03,
0x01,0x00,0x01,0xa3,0x82,0x01,0xe3,0x30,0x82,0x01,0xdf,0x30,0x10,0x06,0x09,
0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x15,0x01,0x04,0x03,0x02,0x01,0x00,0x30,
0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0xe6,0xfc,0x5f,0x7b,0xbb,
0x22,0x00,0x58,0xe4,0x72,0x4e,0xb5,0xf4,0x21,0x74,0x23,0x32,0xe6,0xef,0xac,
0x30,0x19,0x06,0x09,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x14,0x02,0x04,0x0c,
0x1e,0x0a,0x00,0x53,0x00,0x75,0x00,0x62,0x00,0x43,0x00,0x41,0x30,0x0b,0x06,
0x03,0x55,0x1d,0x0f,0x04,0x04,0x03,0x02,0x01,0x86,0x30,0x0f,0x06,0x03,0x55,
0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1f,0x06,
0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0xd5,0xf6,0x56,0xcb,0x8f,
0xe8,0xa2,0x5c,0x62,0x68,0xd1,0x3d,0x94,0x90,0x5b,0xd7,0xce,0x9a,0x18,0xc4,
0x30,0x56,0x06,0x03,0x55,0x1d,0x1f,0x04,0x4f,0x30,0x4d,0x30,0x4b,0xa0,0x49,
0xa0,0x47,0x86,0x45,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,
0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x70,
0x6b,0x69,0x2f,0x63,0x72,0x6c,0x2f,0x70,0x72,0x6f,0x64,0x75,0x63,0x74,0x73,
0x2f,0x4d,0x69,0x63,0x52,0x6f,0x6f,0x43,0x65,0x72,0x41,0x75,0x74,0x5f,0x32,
0x30,0x31,0x30,0x2d,0x30,0x36,0x2d,0x32,0x33,0x2e,0x63,0x72,0x6c,0x30,0x5a,
0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x4e,0x30,0x4c,0x30,
0x4a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x3e,0x68,0x74,
0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x6d,0x69,0x63,0x72,0x6f,0x73,
0x6f,0x66,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x70,0x6b,0x69,0x2f,0x63,0x65,0x72,
0x74,0x73,0x2f,0x4d,0x69,0x63,0x52,0x6f,0x6f,0x43,0x65,0x72,0x41,0x75,0x74,
0x5f,0x32,0x30,0x31,0x30,0x2d,0x30,0x36,0x2d,0x32,0x33,0x2e,0x63,0x72,0x74,
0x30,0x81,0x9d,0x06,0x03,0x55,0x1d,0x20,0x04,0x81,0x95,0x30,0x81,0x92,0x30,
0x81,0x8f,0x06,0x09,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x2e,0x03,0x30,0x81,
0x81,0x30,0x3d,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x31,
0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x6d,0x69,0x63,0x72,
0x6f,0x73,0x6f,0x66,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x50,0x4b,0x49,0x2f,0x64,
0x6f,0x63,0x73,0x2f,0x43,0x50,0x53,0x2f,0x64,0x65,0x66,0x61,0x75,0x6c,0x74,
0x2e,0x68,0x74,0x6d,0x30,0x40,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,
0x02,0x30,0x34,0x1e,0x32,0x20,0x1d,0x00,0x4c,0x00,0x65,0x00,0x67,0x00,0x61,
0x00,0x6c,0x00,0x5f,0x00,0x50,0x00,0x6f,0x00,0x6c,0x00,0x69,0x00,0x63,0x00,
0x79,0x00,0x5f,0x00,0x53,0x00,0x74,0x00,0x61,0x00,0x74,0x00,0x65,0x00,0x6d,
0x00,0x65,0x00,0x6e,0x00,0x74,0x00,0x2e,0x20,0x1d,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x02,0x01,0x00,
0x1a,0x74,0xef,0x57,0x4f,0x29,0x7b,0xc4,0x16,0x85,0x78,0xb8,0x50,0xd3,0x22,
0xfc,0x09,0x9d,0xac,0x82,0x97,0xf8,0x34,0xff,0x2a,0x2c,0x97,0x95,0x12,0xe5,
0xe4,0xbf,0xcf,0xbf,0x93,0xc8,0xe3,0x34,0xa9,0xdb,0x81,0xb8,0xdc,0x1e,0x00,
0xbe,0xd2,0x35,0x6f,0xaf,0xe5,0x7f,0x79,0x95,0x77,0xe5,0x02,0xd4,0xf1,0xeb,
0xd8,0xcd,0x4e,0x1e,0x1b,0x61,0xa2,0xc2,0x5a,0x23,0x1a,0xf0,0x8c,0xa8,0x62,
0x51,0x45,0x67,0x08,0xe3,0x3f,0x3c,0x1e,0x93,0xf8,0x30,0x85,0x17,0xc8,0x39,
0x40,0xa6,0xd7,0x0e,0xb3,0x21,0x29,0xe5,0xa5,0xa1,0x69,0x8c,0x22,0x93,0xcc,
0x74,0x98,0xe7,0xa1,0x47,0x43,0xf2,0x53,0xac,0xc0,0x0f,0x30,0x69,0x7f,0xfe,
0xd2,0x25,0x20,0x6d,0x6f,0x61,0xd3,0xdf,0x07,0xd5,0xd9,0x72,0x00,0x2c,0x69,
0x86,0x76,0x3d,0x51,0xdb,0xa6,0x39,0x48,0xc9,0x37,0x61,0x6d,0x07,0xdd,0x53,
0x19,0xcb,0xa7,0xd6,0x61,0xc2,0xbf,0xe2,0x83,0xab,0x0f,0xe0,0x6b,0x9b,0x95,
0xd6,0x7d,0x28,0x51,0xb0,0x89,0x4a,0x51,0xa4,0x9a,0x6c,0xc8,0xb7,0x1f,0x4a,
0x1a,0x0e,0x69,0xa9,0xd7,0xdc,0xc1,0x7e,0xd1,0x49,0x70,0xaa,0xb6,0xad,0xbb,
0x72,0x47,0x63,0x17,0xfa,0xa6,0xd6,0xa2,0xa6,0x86,0xec,0xa8,0x10,0x44,0x9b,
0x63,0xb6,0xb2,0x69,0x89,0x06,0xc7,0x46,0x86,0x7a,0x18,0x3f,0xe8,0xc5,0x1d,
0x21,0xd5,0x7b,0xf9,0x02,0x23,0x2d,0xc5,0x41,0xcb,0xbf,0x1d,0x4c,0xc8,0x16,
0xef,0xb1,0x9c,0x7f,0xfc,0x22,0x4b,0x49,0x8a,0x6e,0x15,0xe3,0xa6,0x7f,0x76,
0x5b,0xd1,0x53,0x79,0x91,0x85,0x9d,0xd5,0xd2,0xdb,0x3d,0x73,0x35,0xf3,0x3c,
0xae,0x54,0xb2,0x52,0x47,0x6a,0xc0,0xaa,0x13,0x95,0xd2,0x8e,0x11,0xda,0x99,
0x67,0x5e,0x32,0x8c,0xfb,0x37,0x85,0xd1,0xdc,0x75,0x85,0x9c,0x87,0xc6,0x5a,
0x57,0x85,0xc2,0xbf,0xdd,0x0d,0x8f,0x8c,0x9b,0x2d,0xeb,0xb4,0xee,0xcf,0x27,
0xd3,0xb5,0x5e,0x69,0xfa,0xa4,0x16,0x04,0x01,0xa7,0x24,0x67,0x73,0xcf,0x4d,
0x4f,0xb6,0xde,0x05,0x56,0x97,0x7a,0xf7,0xe9,0x52,0x4d,0xf4,0x77,0x05,0x4f,
0x85,0xc6,0xd8,0x0b,0xf1,0x8e,0xed,0x42,0x09,0xd1,0x0d,0x76,0xe3,0x23,0x56,
0x78,0x22,0x26,0x36,0xbe,0xca,0xb1,0x8c,0x6e,0xaa,0x1d,0xe4,0x85,0xda,0x47,
0x33,0x62,0x8f,0xa4,0xc9,0x91,0x33,0x5f,0x71,0x1e,0x40,0xaf,0x98,0x65,0xc9,
0x22,0xe8,0x42,0x21,0x25,0x8a,0x1c,0x2d,0x60,0xd9,0x37,0x89,0x41,0x89,0x2a,
0x16,0x0f,0xd7,0x61,0x3c,0x94,0x68,0x60,0x52,0xef,0xd6,0x47,0x99,0xa0,0x80,
0x40,0xee,0x15,0x81,0x77,0x3e,0x9c,0xe0,0x53,0x18,0x1a,0x50,0x1d,0x38,0x95,
0x9b,0x1e,0x66,0x33,0x13,0x27,0x39,0x17,0x78,0x87,0x36,0xce,0x4e,0xc3,0x5f,
0xb2,0xf5,0x3d,0x47,0x53,0xb6,0xe0,0xe5,0xdb,0x0b,0x61,0x3d,0x2a,0xd7,0x92,
0x2c,0xce,0x37,0x5a,0x3e,0x40,0x42,0x31,0xa4,0x1f,0x10,0x08,0xc2,0x56,0x9c,
0xbf,0x24,0x5d,0x51,0x02,0x9d,0x6a,0x79,0xd2,0x17,0xd3,0xda,0xc1,0x94,0x8e,
0x07,0x7b,0x25,0x71,0x44,0xab,0x06,0x6a,0xe6,0xd4,0xc6,0xdf,0x23,0x9a,0x96,
0x75,0xc5 };
/* Microsoft Corporation (expired) */
static const BYTE chain32_3[] = {
0x30,0x82,0x04,0xfe,0x30,0x82,0x03,0xe6,0xa0,0x03,0x02,0x01,0x02,0x02,0x13,
0x33,0x00,0x00,0x02,0xcf,0x6d,0x2c,0xc5,0x7c,0xaa,0x65,0xa6,0xd8,0x00,0x00,
0x00,0x00,0x02,0xcf,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x0b,0x05,0x00,0x30,0x7e,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,
0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,
0x57,0x61,0x73,0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,
0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,
0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,
0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,
0x31,0x28,0x30,0x26,0x06,0x03,0x55,0x04,0x03,0x13,0x1f,0x4d,0x69,0x63,0x72,
0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x64,0x65,0x20,0x53,0x69,0x67,0x6e,
0x69,0x6e,0x67,0x20,0x50,0x43,0x41,0x20,0x32,0x30,0x31,0x30,0x30,0x1e,0x17,
0x0d,0x31,0x39,0x30,0x35,0x30,0x32,0x32,0x31,0x32,0x35,0x34,0x32,0x5a,0x17,
0x0d,0x32,0x30,0x30,0x35,0x30,0x32,0x32,0x31,0x32,0x35,0x34,0x32,0x5a,0x30,
0x74,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,
0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,
0x6e,0x67,0x74,0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,
0x07,0x52,0x65,0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,
0x04,0x0a,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,
0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x1e,0x30,0x1c,0x06,
0x03,0x55,0x04,0x03,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,
0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x30,0x82,0x01,
0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,
0xb9,0x5a,0xf3,0xb5,0x5d,0x58,0xf9,0x43,0x0f,0x95,0x01,0x79,0xfa,0x74,0xe5,
0xb0,0xca,0xb5,0x38,0xf0,0x92,0xca,0xec,0x9b,0x99,0x42,0x1d,0x3a,0xea,0xb5,
0x8f,0x75,0x75,0x83,0xa9,0x1b,0x36,0xff,0x40,0x96,0x8e,0xd7,0x75,0x0a,0xd1,
0x96,0xae,0xc7,0xb3,0x77,0x12,0x39,0x82,0xd9,0xee,0x48,0x75,0x3d,0x47,0xad,
0x53,0x39,0x3c,0x69,0xd1,0xca,0xd8,0xf0,0x3f,0x21,0x5a,0x4c,0x3a,0xe8,0x28,
0x67,0xc3,0x7d,0x20,0x53,0x4b,0x9c,0x8b,0xbc,0x93,0xf9,0x20,0x46,0xf0,0x50,
0x82,0xca,0x6f,0x6a,0x4e,0x56,0xf1,0xd4,0x43,0xae,0xd8,0xe9,0xdb,0xbe,0x7c,
0x77,0x7d,0x46,0x73,0x03,0x3a,0x64,0xe9,0xf6,0x2d,0xcb,0x25,0x51,0x0d,0x90,
0xc9,0x16,0x59,0x29,0x0c,0x15,0x20,0x63,0xa8,0xb8,0x4f,0x70,0x24,0x9d,0x01,
0xe5,0x69,0x94,0x49,0xc9,0xd6,0x58,0xa8,0x9b,0x89,0xb8,0x2f,0xe0,0x6e,0x1b,
0xa7,0xcc,0x9d,0x75,0x08,0xa6,0xc8,0x31,0x65,0x0f,0xa3,0xed,0xbd,0xd6,0xc6,
0x93,0x8c,0x66,0x22,0xbf,0x48,0x57,0x1d,0x8c,0xa1,0xc1,0xe6,0xad,0x90,0x81,
0x29,0x2b,0x8a,0x62,0xf4,0xc2,0xa1,0xdb,0xc8,0xc1,0x79,0x23,0x74,0x37,0x08,
0x92,0x65,0xe3,0x57,0x01,0xe4,0x4f,0xa1,0x0a,0xf5,0x68,0x96,0xd6,0x09,0x80,
0x4a,0x05,0xdc,0xae,0x30,0xd7,0xb1,0x5a,0xfb,0x1d,0xf9,0x84,0xde,0x92,0xcb,
0xf2,0xc0,0xf8,0xea,0x26,0x6e,0xd3,0x73,0x7e,0xc5,0x66,0xa5,0x05,0xac,0xc4,
0xeb,0x10,0xda,0x0c,0x5c,0x7c,0x36,0x99,0x9c,0x59,0x89,0xc6,0x94,0x47,0xf6,
0xbb,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x7d,0x30,0x82,0x01,0x79,0x30,
0x1f,0x06,0x03,0x55,0x1d,0x25,0x04,0x18,0x30,0x16,0x06,0x0a,0x2b,0x06,0x01,
0x04,0x01,0x82,0x37,0x3d,0x06,0x01,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,
0x03,0x03,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x37,0xf3,
0x4d,0xec,0x4d,0xaf,0x2c,0x04,0x31,0xa9,0x10,0x88,0x0b,0x58,0xc0,0x6f,0x03,
0xbc,0x55,0x08,0x30,0x54,0x06,0x03,0x55,0x1d,0x11,0x04,0x4d,0x30,0x4b,0xa4,
0x49,0x30,0x47,0x31,0x2d,0x30,0x2b,0x06,0x03,0x55,0x04,0x0b,0x13,0x24,0x4d,
0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x49,0x72,0x65,0x6c,0x61,0x6e,
0x64,0x20,0x4f,0x70,0x65,0x72,0x61,0x74,0x69,0x6f,0x6e,0x73,0x20,0x4c,0x69,
0x6d,0x69,0x74,0x65,0x64,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x05,0x13,
0x0d,0x32,0x33,0x30,0x38,0x36,0x35,0x2b,0x34,0x35,0x34,0x32,0x34,0x34,0x30,
0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0xe6,0xfc,0x5f,
0x7b,0xbb,0x22,0x00,0x58,0xe4,0x72,0x4e,0xb5,0xf4,0x21,0x74,0x23,0x32,0xe6,
0xef,0xac,0x30,0x56,0x06,0x03,0x55,0x1d,0x1f,0x04,0x4f,0x30,0x4d,0x30,0x4b,
0xa0,0x49,0xa0,0x47,0x86,0x45,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,
0x6c,0x2e,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x2e,0x63,0x6f,0x6d,
0x2f,0x70,0x6b,0x69,0x2f,0x63,0x72,0x6c,0x2f,0x70,0x72,0x6f,0x64,0x75,0x63,
0x74,0x73,0x2f,0x4d,0x69,0x63,0x43,0x6f,0x64,0x53,0x69,0x67,0x50,0x43,0x41,
0x5f,0x32,0x30,0x31,0x30,0x2d,0x30,0x37,0x2d,0x30,0x36,0x2e,0x63,0x72,0x6c,
0x30,0x5a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x4e,0x30,
0x4c,0x30,0x4a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x3e,
0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x6d,0x69,0x63,0x72,
0x6f,0x73,0x6f,0x66,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x70,0x6b,0x69,0x2f,0x63,
0x65,0x72,0x74,0x73,0x2f,0x4d,0x69,0x63,0x43,0x6f,0x64,0x53,0x69,0x67,0x50,
0x43,0x41,0x5f,0x32,0x30,0x31,0x30,0x2d,0x30,0x37,0x2d,0x30,0x36,0x2e,0x63,
0x72,0x74,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x02,0x30,
0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,
0x00,0x03,0x82,0x01,0x01,0x00,0xaa,0x30,0x8f,0x08,0x74,0xb1,0x28,0x88,0x0a,
0x81,0x0c,0xcc,0x38,0xea,0xb1,0xc0,0x47,0x9a,0x7d,0xf3,0x19,0x35,0xa6,0xe8,
0x8b,0xe9,0x16,0x04,0xe4,0xc3,0x32,0xe1,0x29,0xf7,0xcc,0xb8,0xf0,0xc6,0xc3,
0x02,0xd5,0xb0,0x8d,0x0b,0x41,0xfe,0x5c,0xa9,0x03,0x86,0x25,0xf5,0x24,0x9f,
0xb0,0x04,0xc4,0x69,0x0a,0x27,0x94,0x04,0xb2,0x09,0xdc,0x19,0xaf,0xbb,0x01,
0x8b,0x92,0xe4,0xd8,0x72,0x04,0x07,0x72,0x81,0xd3,0x89,0xca,0xd7,0x79,0x98,
0x41,0xe5,0xa0,0x29,0xfb,0x95,0x91,0xd9,0xf8,0x8c,0x40,0x23,0x70,0x9a,0xb2,
0xf6,0xcb,0x0a,0x75,0x0b,0xbd,0x1c,0x72,0xd7,0xff,0x98,0x20,0x01,0xde,0xc0,
0xcd,0xd0,0xd3,0xa5,0xff,0x6b,0x0c,0x8b,0xf9,0x85,0x69,0x22,0x15,0x09,0xd7,
0x4e,0xf2,0xc1,0xe9,0xd4,0x78,0xa7,0xd1,0xe3,0x46,0xf6,0x3d,0xa9,0x06,0x66,
0x11,0xa3,0x9e,0x2f,0xef,0xe6,0xd3,0xc5,0x80,0x29,0xc8,0xfa,0xd3,0xb7,0x98,
0x0d,0xdc,0xa4,0xce,0x84,0x68,0xeb,0x9b,0xfb,0x14,0x83,0x28,0x6d,0x22,0x57,
0x9a,0x46,0xbe,0x37,0x71,0xc4,0xec,0xed,0x38,0x8f,0x4d,0x6e,0x13,0x54,0x39,
0x87,0xc6,0xf4,0x1e,0x41,0xdf,0xba,0x87,0x28,0xda,0xc2,0xc0,0x02,0xa5,0x80,
0x52,0x3c,0xbd,0x9a,0xe2,0x66,0xc0,0x5e,0xc3,0x56,0xe8,0x65,0x1d,0xcc,0xd4,
0xec,0xf5,0x4d,0x05,0xae,0xf3,0x9b,0x96,0x7b,0xf2,0xa9,0x3b,0xae,0xe0,0xbb,
0x3b,0x42,0x9b,0xec,0x20,0xba,0xea,0x1c,0x6d,0xf4,0x6a,0x24,0x8f,0x31,0x01,
0x3c,0x8a,0x0a,0x58,0x61,0x67,0x58 };
/* Microsoft Root Certificate Authority 2011 */
static const BYTE chain33_1[] = {
0x30,0x82,0x05,0xed,0x30,0x82,0x03,0xd5,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x3f,0x8b,0xc8,0xb5,0xfc,0x9f,0xb2,0x96,0x43,0xb5,0x69,0xd6,0x6c,0x42,0xe1,
0x44,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,
0x00,0x30,0x81,0x88,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x57,0x61,
0x73,0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,
0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,0x30,0x1c,
0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,
0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x32,
0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,0x73,
0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,
0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,
0x20,0x32,0x30,0x31,0x31,0x30,0x1e,0x17,0x0d,0x31,0x31,0x30,0x33,0x32,0x32,
0x32,0x32,0x30,0x35,0x32,0x38,0x5a,0x17,0x0d,0x33,0x36,0x30,0x33,0x32,0x32,
0x32,0x32,0x31,0x33,0x30,0x34,0x5a,0x30,0x81,0x88,0x31,0x0b,0x30,0x09,0x06,
0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,
0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,
0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,
0x6e,0x64,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,
0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,
0x74,0x69,0x6f,0x6e,0x31,0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,
0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,
0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,
0x68,0x6f,0x72,0x69,0x74,0x79,0x20,0x32,0x30,0x31,0x31,0x30,0x82,0x02,0x22,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,
0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,0x02,0x82,0x02,0x01,0x00,0xb2,
0x80,0x41,0xaa,0x35,0x38,0x4d,0x13,0x72,0x32,0x68,0x22,0x4d,0xb8,0xb2,0xf1,
0xff,0xd5,0x52,0xbc,0x6c,0xc7,0xf5,0xd2,0x4a,0x8c,0x36,0xee,0xd1,0xc2,0x5c,
0x7e,0x8c,0x8a,0xae,0xaf,0x13,0x28,0x6f,0xc0,0x73,0xe3,0x3a,0xce,0xd0,0x25,
0xa8,0x5a,0x3a,0x6d,0xef,0xa8,0xb8,0x59,0xab,0x13,0x23,0x68,0xcd,0x0c,0x29,
0x87,0xd1,0x6f,0x80,0x5c,0x8f,0x44,0x7f,0x5d,0x90,0x01,0x52,0x58,0xac,0x51,
0xc5,0x5f,0x2a,0x87,0xdc,0xdc,0xd8,0x0a,0x1d,0xc1,0x03,0xb9,0x7b,0xb0,0x56,
0xe8,0xa3,0xde,0x64,0x61,0xc2,0x9e,0xf8,0xf3,0x7c,0xb9,0xec,0x0d,0xb5,0x54,
0xfe,0x4c,0xb6,0x65,0x4f,0x88,0xf0,0x9c,0x48,0x99,0x0c,0x42,0x0b,0x09,0x7c,
0x31,0x59,0x17,0x79,0x06,0x78,0x28,0x8d,0x89,0x3a,0x4c,0x03,0x25,0xbe,0x71,
0x6a,0x5c,0x0b,0xe7,0x84,0x60,0xa4,0x99,0x22,0xe3,0xd2,0xaf,0x84,0xa4,0xa7,
0xfb,0xd1,0x98,0xed,0x0c,0xa9,0xde,0x94,0x89,0xe1,0x0e,0xa0,0xdc,0xc0,0xce,
0x99,0x3d,0xea,0x08,0x52,0xbb,0x56,0x79,0xe4,0x1f,0x84,0xba,0x1e,0xb8,0xb4,
0xc4,0x49,0x5c,0x4f,0x31,0x4b,0x87,0xdd,0xdd,0x05,0x67,0x26,0x99,0x80,0xe0,
0x71,0x11,0xa3,0xb8,0xa5,0x41,0xe2,0xa4,0x53,0xb9,0xf7,0x32,0x29,0x83,0x0c,
0x13,0xbf,0x36,0x5e,0x04,0xb3,0x4b,0x43,0x47,0x2f,0x6b,0xe2,0x91,0x1e,0xd3,
0x98,0x4f,0xdd,0x42,0x07,0xc8,0xe8,0x1d,0x12,0xfc,0x99,0xa9,0x6b,0x3e,0x92,
0x7e,0xc8,0xd6,0x69,0x3a,0xfc,0x64,0xbd,0xb6,0x09,0x9d,0xca,0xfd,0x0c,0x0b,
0xa2,0x9b,0x77,0x60,0x4b,0x03,0x94,0xa4,0x30,0x69,0x12,0xd6,0x42,0x2d,0xc1,
0x41,0x4c,0xca,0xdc,0xaa,0xfd,0x8f,0x5b,0x83,0x46,0x9a,0xd9,0xfc,0xb1,0xd1,
0xe3,0xb3,0xc9,0x7f,0x48,0x7a,0xcd,0x24,0xf0,0x41,0x8f,0x5c,0x74,0xd0,0xac,
0xb0,0x10,0x20,0x06,0x49,0xb7,0xc7,0x2d,0x21,0xc8,0x57,0xe3,0xd0,0x86,0xf3,
0x03,0x68,0xfb,0xd0,0xce,0x71,0xc1,0x89,0x99,0x4a,0x64,0x01,0x6c,0xfd,0xec,
0x30,0x91,0xcf,0x41,0x3c,0x92,0xc7,0xe5,0xba,0x86,0x1d,0x61,0x84,0xc7,0x5f,
0x83,0x39,0x62,0xae,0xb4,0x92,0x2f,0x47,0xf3,0x0b,0xf8,0x55,0xeb,0xa0,0x1f,
0x59,0xd0,0xbb,0x74,0x9b,0x1e,0xd0,0x76,0xe6,0xf2,0xe9,0x06,0xd7,0x10,0xe8,
0xfa,0x64,0xde,0x69,0xc6,0x35,0x96,0x88,0x02,0xf0,0x46,0xb8,0x3f,0x27,0x99,
0x6f,0xcb,0x71,0x89,0x29,0x35,0xf7,0x48,0x16,0x02,0x35,0x8f,0xd5,0x79,0x7c,
0x4d,0x02,0xcf,0x5f,0xeb,0x8a,0x83,0x4f,0x45,0x71,0x88,0xf9,0xa9,0x0d,0x4e,
0x72,0xe9,0xc2,0x9c,0x07,0xcf,0x49,0x1b,0x4e,0x04,0x0e,0x63,0x51,0x8c,0x5e,
0xd8,0x00,0xc1,0x55,0x2c,0xb6,0xc6,0xe0,0xc2,0x65,0x4e,0xc9,0x34,0x39,0xf5,
0x9c,0xb3,0xc4,0x7e,0xe8,0x61,0x6e,0x13,0x5f,0x15,0xc4,0x5f,0xd9,0x7e,0xed,
0x1d,0xce,0xee,0x44,0xec,0xcb,0x2e,0x86,0xb1,0xec,0x38,0xf6,0x70,0xed,0xab,
0x5c,0x13,0xc1,0xd9,0x0f,0x0d,0xc7,0x80,0xb2,0x55,0xed,0x34,0xf7,0xac,0x9b,
0xe4,0xc3,0xda,0xe7,0x47,0x3c,0xa6,0xb5,0x8f,0x31,0xdf,0xc5,0x4b,0xaf,0xeb,
0xf1,0x02,0x03,0x01,0x00,0x01,0xa3,0x51,0x30,0x4f,0x30,0x0b,0x06,0x03,0x55,
0x1d,0x0f,0x04,0x04,0x03,0x02,0x01,0x86,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,
0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,0x55,
0x1d,0x0e,0x04,0x16,0x04,0x14,0x72,0x2d,0x3a,0x02,0x31,0x90,0x43,0xb9,0x14,
0x05,0x4e,0xe1,0xea,0xa7,0xc7,0x31,0xd1,0x23,0x89,0x34,0x30,0x10,0x06,0x09,
0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x15,0x01,0x04,0x03,0x02,0x01,0x00,0x30,
0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,
0x82,0x02,0x01,0x00,0x7f,0x72,0xcf,0x0f,0xb7,0xc5,0x15,0xdb,0x9b,0xc0,0x49,
0xca,0x26,0x5b,0xfe,0x9e,0x13,0xe6,0xd3,0xf0,0xd2,0xdb,0x97,0x5f,0xf2,0x4b,
0x3f,0x4d,0xb3,0xae,0x19,0xae,0xed,0xd7,0x97,0xa0,0xac,0xef,0xa9,0x3a,0xa3,
0xc2,0x41,0xb0,0xe5,0xb8,0x91,0x9e,0x13,0x81,0x24,0x03,0xe6,0x09,0xfd,0x3f,
0x57,0x40,0x39,0x21,0x24,0x56,0xd1,0x10,0x2f,0x4b,0x40,0xa9,0x36,0x86,0x4b,
0xb4,0x53,0x57,0x9a,0xfb,0xf1,0x7e,0x89,0x8f,0x11,0xfe,0x18,0x6c,0x51,0xaa,
0xe8,0xed,0x09,0x95,0xb5,0xe5,0x71,0xc9,0xa1,0xe9,0x87,0x75,0xa6,0x15,0x7f,
0xc9,0x7e,0x37,0x54,0x5e,0x74,0x93,0xc5,0xc3,0x67,0xcc,0x0d,0x4f,0x6b,0xa8,
0x17,0x0c,0x6d,0x08,0x92,0x7e,0x8b,0xdd,0x81,0xaa,0x2d,0x70,0x21,0xc3,0x3d,
0x06,0x14,0xbb,0xbf,0x24,0x5e,0xa7,0x84,0xd7,0x3f,0x0f,0x21,0x22,0xbd,0x4b,
0x00,0x06,0xdb,0x97,0x1c,0xd8,0x5e,0xd4,0xc5,0x0b,0x5c,0x87,0x6e,0x50,0xa4,
0xe8,0xc3,0x38,0xa4,0xfb,0xcb,0x2c,0xc5,0x92,0x66,0x9b,0x85,0x5e,0xcb,0x7a,
0x6c,0x93,0x7c,0x80,0x29,0x58,0x5b,0x57,0xb5,0x40,0x69,0xba,0x08,0x79,0xa6,
0x64,0x62,0x15,0x9d,0x87,0x96,0x45,0xb5,0x66,0x23,0x20,0x03,0x8b,0x1c,0x73,
0xa0,0xd3,0xa2,0x79,0x33,0xe0,0x50,0x59,0x86,0xdb,0x2f,0xe5,0x02,0x25,0xea,
0x73,0x2a,0x9f,0x00,0x14,0xc8,0x36,0xc7,0x92,0x3b,0xe9,0x4e,0x00,0xec,0xd8,
0x56,0x09,0xb9,0x33,0x49,0x12,0xd2,0x54,0x0b,0x01,0xab,0xac,0x47,0xb6,0x91,
0x29,0x7d,0x4c,0xb4,0x75,0x80,0x52,0x01,0xe8,0xca,0x82,0xf6,0x9f,0xcc,0xac,
0x9c,0x8f,0x17,0xea,0x2f,0x26,0xb0,0xab,0x72,0xac,0x0b,0xfe,0x9e,0x51,0x1e,
0xc7,0x43,0x55,0x67,0x4f,0x51,0xb3,0x57,0xd6,0xb6,0xec,0xee,0x52,0xb7,0x3a,
0xe9,0x4e,0xe1,0xd7,0x81,0x88,0xbc,0x4f,0x8e,0x75,0xbb,0x4b,0xa8,0xf0,0x35,
0xaa,0x26,0xd4,0x67,0x67,0x49,0xb2,0x70,0x4c,0x3b,0x93,0xdc,0x1d,0xdf,0x78,
0x90,0x86,0x72,0xb2,0x38,0xa4,0xd1,0xdc,0x92,0x4d,0xc9,0x58,0xeb,0x2b,0x12,
0x5c,0xd4,0x3b,0xae,0x8c,0x6b,0xb0,0x83,0xe5,0x01,0x3f,0xf8,0x09,0x32,0xf6,
0x93,0x35,0x34,0x22,0xaf,0xdd,0x37,0x0d,0x77,0x09,0x80,0x2b,0xcd,0x48,0x00,
0xf1,0x8c,0x99,0x19,0x47,0x05,0x01,0xe9,0xd1,0xbf,0xd1,0x4e,0xd0,0xe6,0x28,
0x43,0x37,0x99,0xa4,0x0a,0x4a,0x08,0xd9,0x9a,0x71,0x73,0xd2,0xaa,0xcd,0x31,
0x13,0x63,0x76,0xa1,0x37,0x6f,0x92,0x38,0x1e,0x7d,0x12,0x3c,0x66,0x32,0xe7,
0xcb,0x6d,0xe1,0xfc,0x52,0x89,0xdd,0xca,0xd6,0x66,0x05,0x9a,0x96,0x61,0xbe,
0xa2,0x28,0xc7,0x1c,0xa3,0xa7,0x36,0x50,0x3c,0x3a,0xa4,0xdf,0x4a,0x6e,0xe6,
0x87,0x3b,0xce,0xeb,0xf0,0xe0,0x81,0x37,0x9d,0x13,0x3c,0x52,0x8e,0xbd,0xb9,
0x1d,0x34,0xc6,0x1d,0xd5,0x0a,0x6a,0x3d,0x98,0x29,0x70,0x8c,0x89,0x2a,0xd1,
0xab,0x82,0x10,0x48,0x1f,0xdc,0xf4,0xef,0xa5,0xc5,0xbb,0x55,0x1a,0x38,0x63,
0x84,0x4e,0xb7,0x6c,0xad,0x95,0x54,0xec,0x65,0x22,0x10,0x49,0x17,0xb8,0xc0,
0x1e,0xc7,0x0f,0xac,0x54,0x47 };

typedef struct _CONST_DATA_BLOB
{
    DWORD       cbData;
    const BYTE *pbData;
} CONST_DATA_BLOB;

typedef struct _CONST_BLOB_ARRAY
{
    DWORD cBlob;
    CONST_DATA_BLOB *rgBlob;
} CONST_BLOB_ARRAY;

#define TODO_CHAIN    1
#define TODO_ERROR    2
#define TODO_INFO     4
#define TODO_ELEMENTS 8
#define TODO_CHAINS   16
#define TODO_POLICY   32

/* Gets a certificate chain built from a store containing all the certs in
 * certArray, where the last certificate in the chain is expected to be the
 * end certificate (the one from which the chain is built.)
 */
static PCCERT_CHAIN_CONTEXT getChain(HCERTCHAINENGINE engine,
 const CONST_BLOB_ARRAY *certArray, DWORD flags, BOOL includeStore,
 LPSYSTEMTIME checkTime, DWORD todo, DWORD testIndex)
{
    HCERTSTORE store;
    PCCERT_CHAIN_CONTEXT chain = NULL;

    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    if (store)
    {
        BOOL ret;
        PCCERT_CONTEXT endCert;

        if (certArray->cBlob > 1)
        {
            DWORD i;

            for (i = 0, ret = TRUE; ret && i < certArray->cBlob - 1; i++)
            {
                ret = CertAddEncodedCertificateToStore(store,
                 X509_ASN_ENCODING, certArray->rgBlob[i].pbData,
                 certArray->rgBlob[i].cbData, CERT_STORE_ADD_ALWAYS, NULL);
                ok(ret, "Chain %ld: adding cert %ld failed: %08lx\n",
                 testIndex, i, GetLastError());
            }
        }
        ret = CertAddEncodedCertificateToStore(store,
         X509_ASN_ENCODING, certArray->rgBlob[certArray->cBlob - 1].pbData,
         certArray->rgBlob[certArray->cBlob - 1].cbData, CERT_STORE_ADD_ALWAYS,
         &endCert);
        ok(ret, "Chain %ld: adding end cert failed: %08lx\n",
         testIndex, GetLastError());
        if (ret)
        {
            /* FIXME: allow caller to specify usage matches? */
            CERT_CHAIN_PARA chainPara = { sizeof(chainPara), { 0 } };
            FILETIME fileTime;

            ok(SystemTimeToFileTime(checkTime, &fileTime),
             "SystemTimeToFileTime failed for day %d, month %d, year %d\n",
             checkTime->wDay, checkTime->wMonth, checkTime->wYear);
            ret = pCertGetCertificateChain(engine, endCert, &fileTime,
             includeStore ? store : NULL, &chainPara, flags, NULL, &chain);
            todo_wine_if (todo & TODO_CHAIN)
                ok(ret, "Chain %ld: CertGetCertificateChain failed: %08lx\n",
                 testIndex, GetLastError());
            CertFreeCertificateContext(endCert);
        }
        CertCloseStore(store, 0);
    }
    return chain;
}

typedef struct _SimpleChainStatusCheck
{
    DWORD                    cElement;
    const CERT_TRUST_STATUS *rgElementStatus;
} SimpleChainStatusCheck;

static void checkElementStatus(const CERT_TRUST_STATUS *expected,
 const CERT_TRUST_STATUS *got, const CERT_TRUST_STATUS *ignore,
 DWORD todo, LPCSTR testName, DWORD testIndex, DWORD chainIndex,
 DWORD elementIndex)
{
    if (got->dwErrorStatus == expected->dwErrorStatus)
        ok(got->dwErrorStatus == expected->dwErrorStatus,
         "%s[%ld], element [%ld,%ld]: expected error %08lx, got %08lx\n",
         testName, testIndex, chainIndex, elementIndex, expected->dwErrorStatus,
         got->dwErrorStatus);
    else todo_wine_if (todo & TODO_ERROR)
        ok(got->dwErrorStatus == expected->dwErrorStatus ||
         broken((got->dwErrorStatus & ~ignore->dwErrorStatus) ==
         (expected->dwErrorStatus & ~ignore->dwErrorStatus)),
         "%s[%ld], element [%ld,%ld]: expected error %08lx, got %08lx. %08x is "
         "expected if no valid root certificate is available.\n",
         testName, testIndex, chainIndex, elementIndex, expected->dwErrorStatus,
         got->dwErrorStatus, CERT_TRUST_IS_UNTRUSTED_ROOT);
    if (got->dwInfoStatus == expected->dwInfoStatus)
        ok(got->dwInfoStatus == expected->dwInfoStatus,
         "%s[%ld], element [%ld,%ld]: expected info %08lx, got %08lx\n",
         testName, testIndex, chainIndex, elementIndex, expected->dwInfoStatus,
         got->dwInfoStatus);
    else todo_wine_if (todo & TODO_INFO)
        ok(got->dwInfoStatus == expected->dwInfoStatus ||
         broken((got->dwInfoStatus & ~ignore->dwInfoStatus) ==
         (expected->dwInfoStatus & ~ignore->dwInfoStatus)),
         "%s[%ld], element [%ld,%ld]: expected info %08lx, got %08lx\n",
         testName, testIndex, chainIndex, elementIndex, expected->dwInfoStatus,
         got->dwInfoStatus);
}

static void checkSimpleChainStatus(const CERT_SIMPLE_CHAIN *simpleChain,
 const SimpleChainStatusCheck *simpleChainStatus,
 const CERT_TRUST_STATUS *ignore, DWORD todo, LPCSTR testName, DWORD testIndex,
 DWORD chainIndex)
{
    todo_wine_if (todo & TODO_ELEMENTS)
        ok(simpleChain->cElement == simpleChainStatus->cElement,
         "%s[%ld]: expected %ld elements, got %ld\n", testName, testIndex,
         simpleChainStatus->cElement, simpleChain->cElement);
    if (simpleChain->cElement == simpleChainStatus->cElement)
    {
        DWORD i;

        for (i = 0; i < simpleChain->cElement; i++)
            checkElementStatus(&simpleChainStatus->rgElementStatus[i],
             &simpleChain->rgpElement[i]->TrustStatus, ignore, todo, testName,
             testIndex, chainIndex, i);
    }
}

typedef struct _ChainStatusCheck
{
    CERT_TRUST_STATUS             statusToIgnore;
    CERT_TRUST_STATUS             status;
    DWORD                         cChain;
    const SimpleChainStatusCheck *rgChainStatus;
} ChainStatusCheck;

static void checkChainStatus(PCCERT_CHAIN_CONTEXT chain,
 const ChainStatusCheck *chainStatus, DWORD todo, LPCSTR testName,
 DWORD testIndex)
{
    ok(chain->cChain == chainStatus->cChain,
     "%s[%ld]: expected %ld simple chains, got %ld\n", testName, testIndex,
     chainStatus->cChain, chain->cChain);
    todo_wine_if (todo & TODO_ERROR &&
     chain->TrustStatus.dwErrorStatus != chainStatus->status.dwErrorStatus)
        ok(chain->TrustStatus.dwErrorStatus ==
         chainStatus->status.dwErrorStatus ||
         broken((chain->TrustStatus.dwErrorStatus &
         ~chainStatus->statusToIgnore.dwErrorStatus) ==
         (chainStatus->status.dwErrorStatus &
         ~chainStatus->statusToIgnore.dwErrorStatus)),
         "%s[%ld]: expected error %08lx, got %08lx. %08x is expected if no valid "
         "root certificate is available.\n",
         testName, testIndex, chainStatus->status.dwErrorStatus,
         chain->TrustStatus.dwErrorStatus, CERT_TRUST_IS_UNTRUSTED_ROOT);
    todo_wine_if (todo & TODO_INFO &&
     chain->TrustStatus.dwInfoStatus != chainStatus->status.dwInfoStatus)
        ok(chain->TrustStatus.dwInfoStatus ==
         chainStatus->status.dwInfoStatus ||
         broken((chain->TrustStatus.dwInfoStatus &
         ~chainStatus->statusToIgnore.dwInfoStatus) ==
         (chainStatus->status.dwInfoStatus &
         ~chainStatus->statusToIgnore.dwInfoStatus)),
         "%s[%ld]: expected info %08lx, got %08lx\n",
         testName, testIndex, chainStatus->status.dwInfoStatus,
         chain->TrustStatus.dwInfoStatus);
    if (chain->cChain == chainStatus->cChain)
    {
        DWORD i;

        for (i = 0; i < chain->cChain; i++)
            checkSimpleChainStatus(chain->rgpChain[i],
             &chainStatus->rgChainStatus[i], &chainStatus->statusToIgnore,
             todo, testName, testIndex, i);
    }
}

/* Wednesday, Oct 1, 2007 */
static SYSTEMTIME oct2007 = { 2007, 10, 1, 1, 0, 0, 0, 0 };
/* Wednesday, Oct 28, 2009 */
static SYSTEMTIME oct2009 = { 2009, 10, 3, 28, 0, 0, 0, 0 };
/* Wednesday, Oct 28, 2010 */
static SYSTEMTIME oct2010 = { 2010, 10, 3, 28, 0, 0, 0, 0 };
/* Friday, June 6, 2013 */
static SYSTEMTIME jun2013 = { 2013, 6, 5, 6, 0, 0, 0, 0 };
/* Saturday, Oct 1, 2016 */
static SYSTEMTIME oct2016 = { 2016, 10, 6, 1, 0, 0, 0, 0 };
/* Wednesday, Nov 17, 2016 */
static SYSTEMTIME nov2016 = { 2016, 11, 3, 17, 0, 0, 0, 0 };
/* Thursday, May 14, 2020 */
static SYSTEMTIME may2020 = { 2020, 5, 4, 14, 0, 0, 0, 0 };

typedef struct _ChainCheck
{
    CONST_BLOB_ARRAY certs;
    ChainStatusCheck status;
    DWORD            todo;
    PSYSTEMTIME      validfor;
} ChainCheck;

static CONST_DATA_BLOB chain0[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain0_1), chain0_1 },
};
static const CERT_TRUST_STATUS elementStatus0[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus0[] = {
 { ARRAY_SIZE(elementStatus0), elementStatus0 },
};
static CONST_DATA_BLOB chain1[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain1_1), chain1_1 },
};
static const CERT_TRUST_STATUS elementStatus1[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_NOT_SIGNATURE_VALID,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus1[] = {
 { ARRAY_SIZE(elementStatus1), elementStatus1 },
};
static CONST_DATA_BLOB chain2[] = {
 { sizeof(chain2_0), chain2_0 },
 { sizeof(chain0_1), chain0_1 },
};
static const CERT_TRUST_STATUS elementStatus2[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus2[] = {
 { ARRAY_SIZE(elementStatus2), elementStatus2 },
};
static CONST_DATA_BLOB chain3[] = {
 { sizeof(chain3_0), chain3_0 },
 { sizeof(chain0_1), chain0_1 },
};
static const CERT_TRUST_STATUS elementStatus3[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus3[] = {
 { ARRAY_SIZE(elementStatus3), elementStatus3 },
};
static CONST_DATA_BLOB chain4[] = {
 { sizeof(chain4_0), chain4_0 },
 { sizeof(chain4_1), chain4_1 },
 { sizeof(chain4_2), chain4_2 },
};
static const CERT_TRUST_STATUS elementStatus4[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus4[] = {
 { ARRAY_SIZE(elementStatus4), elementStatus4 },
};
static CONST_DATA_BLOB chain5[] = {
 { sizeof(chain5_0), chain5_0 },
 { sizeof(chain5_1), chain5_1 },
};
static const CERT_TRUST_STATUS elementStatus5[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT |
   CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus5[] = {
 { ARRAY_SIZE(elementStatus5), elementStatus5 },
};
static CONST_DATA_BLOB chain6[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain4_1), chain4_1 },
};
static const CERT_TRUST_STATUS elementStatus6[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus6[] = {
 { ARRAY_SIZE(elementStatus6), elementStatus6 },
};
static CONST_DATA_BLOB chain7[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus7[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus7[] = {
 { ARRAY_SIZE(elementStatus7), elementStatus7 },
};
static CONST_DATA_BLOB chain8[] = {
 { sizeof(chain8_0), chain8_0 },
 { sizeof(chain8_1), chain8_1 },
 { sizeof(chain8_2), chain8_2 },
};
static const CERT_TRUST_STATUS elementStatus8[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_TIME_VALID,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus8[] = {
 { ARRAY_SIZE(elementStatus8), elementStatus8 },
};
static CONST_DATA_BLOB chain9[] = {
 { sizeof(chain9_0), chain9_0 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus9[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_CYCLIC,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus9[] = {
 { ARRAY_SIZE(elementStatus9), elementStatus9 },
};
static CONST_DATA_BLOB chain10[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain10_1), chain10_1 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus10[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus10[] = {
 { ARRAY_SIZE(elementStatus10), elementStatus10 },
};
static CONST_DATA_BLOB chain11[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain11_1), chain11_1 },
 { sizeof(chain7_1), chain7_1 },
};
static CONST_DATA_BLOB chain12[] = {
 { sizeof(chain12_0), chain12_0 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus12[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_NOT_SIGNATURE_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus12[] = {
 { ARRAY_SIZE(elementStatus12), elementStatus12 },
};
static CONST_DATA_BLOB chain13[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain13_1), chain13_1 },
};
static const CERT_TRUST_STATUS elementStatus13[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus13[] = {
 { ARRAY_SIZE(elementStatus13), elementStatus13 },
};
static CONST_DATA_BLOB chain14[] = {
 { sizeof(chain14_0), chain14_0 },
 { sizeof(chain14_1), chain14_1 },
};
static const CERT_TRUST_STATUS elementStatus14[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus14[] = {
 { ARRAY_SIZE(elementStatus14), elementStatus14 },
};
static CONST_DATA_BLOB chain15[] = {
 { sizeof(chain15_0), chain15_0 },
 { sizeof(chain14_1), chain14_1 },
};
static const CERT_TRUST_STATUS elementStatus15[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus15[] = {
 { ARRAY_SIZE(elementStatus15), elementStatus15 },
};
static CONST_DATA_BLOB chain16[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain16_1), chain16_1 },
};
static const CERT_TRUST_STATUS elementStatus16[] = {
 { CERT_TRUST_NO_ERROR | CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus16[] = {
 { ARRAY_SIZE(elementStatus16), elementStatus16 },
};
static CONST_DATA_BLOB chain17[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain17_1), chain17_1 },
 { sizeof(chain17_2), chain17_2 },
};
static const CERT_TRUST_STATUS elementStatus17[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus17[] = {
 { ARRAY_SIZE(elementStatus17), elementStatus17 },
};
static CONST_DATA_BLOB chain18[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain18_1), chain18_1 },
 { sizeof(chain18_2), chain18_2 },
};
static const CERT_TRUST_STATUS elementStatus18[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_NOT_VALID_FOR_USAGE, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus18[] = {
 { ARRAY_SIZE(elementStatus18), elementStatus18 },
};
static CONST_DATA_BLOB chain19[] = {
 { sizeof(chain19_0), chain19_0 },
 { sizeof(chain19_1), chain19_1 },
};
static const CERT_TRUST_STATUS elementStatus19[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER |
   CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS },
};
static const SimpleChainStatusCheck simpleStatus19[] = {
 { ARRAY_SIZE(elementStatus19), elementStatus19 },
};
static CONST_DATA_BLOB chain20[] = {
 { sizeof(chain20_0), chain20_0 },
 { sizeof(chain20_1), chain20_1 },
};
static const CERT_TRUST_STATUS elementStatus20[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus20[] = {
 { ARRAY_SIZE(elementStatus20), elementStatus20 },
};
static CONST_DATA_BLOB chain21[] = {
 { sizeof(chain21_0), chain21_0 },
 { sizeof(chain21_1), chain21_1 },
};
static const CERT_TRUST_STATUS elementStatus21[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER |
   CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS },
};
static const SimpleChainStatusCheck simpleStatus21[] = {
 { ARRAY_SIZE(elementStatus21), elementStatus21 },
};
static CONST_DATA_BLOB chain22[] = {
 { sizeof(chain22_0), chain22_0 },
 { sizeof(chain22_1), chain22_1 },
};
static const CERT_TRUST_STATUS elementStatus22[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus22[] = {
 { ARRAY_SIZE(elementStatus22), elementStatus22 },
};
static CONST_DATA_BLOB chain23[] = {
 { sizeof(chain23_0), chain23_0 },
 { sizeof(chain23_1), chain23_1 },
};
static const CERT_TRUST_STATUS elementStatus23[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus23[] = {
 { ARRAY_SIZE(elementStatus23), elementStatus23 },
};
static CONST_DATA_BLOB chain24[] = {
 { sizeof(chain24_0), chain24_0 },
 { sizeof(chain24_1), chain24_1 },
};
static const CERT_TRUST_STATUS elementStatus24[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus24[] = {
 { ARRAY_SIZE(elementStatus24), elementStatus24 },
};
static CONST_DATA_BLOB chain25[] = {
 { sizeof(chain25_0), chain25_0 },
 { sizeof(chain25_1), chain25_1 },
};
static const CERT_TRUST_STATUS elementStatus25[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus25[] = {
 { ARRAY_SIZE(elementStatus25), elementStatus25 },
};
static CONST_DATA_BLOB chain26[] = {
 { sizeof(chain26_0), chain26_0 },
 { sizeof(chain26_1), chain26_1 },
};
static const CERT_TRUST_STATUS elementStatus26[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus26[] = {
 { ARRAY_SIZE(elementStatus26), elementStatus26 },
};
static CONST_DATA_BLOB chain27[] = {
 { sizeof(chain27_0), chain27_0 },
 { sizeof(chain27_1), chain27_1 },
};
static const CERT_TRUST_STATUS elementStatus27[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_INVALID_NAME_CONSTRAINTS |
   CERT_TRUST_INVALID_EXTENSION,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus27[] = {
 { ARRAY_SIZE(elementStatus27), elementStatus27 },
};
static const CERT_TRUST_STATUS elementStatus27Broken[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER |
   CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS },
};
static const SimpleChainStatusCheck simpleStatus27Broken[] = {
 { ARRAY_SIZE(elementStatus27Broken),
   elementStatus27Broken },
};
static CONST_DATA_BLOB chain28[] = {
 { sizeof(chain28_0), chain28_0 },
 { sizeof(chain28_1), chain28_1 },
};
static const CERT_TRUST_STATUS elementStatus28[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus28[] = {
 { ARRAY_SIZE(elementStatus28), elementStatus28 },
};
static CONST_DATA_BLOB chain29[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain29_1), chain29_1 },
};
static CONST_DATA_BLOB chain30[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain30_1), chain30_1 },
 { sizeof(chain30_2), chain30_2 },
};
static const CERT_TRUST_STATUS elementStatus30[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_EXTENSION | CERT_TRUST_HAS_NOT_SUPPORTED_CRITICAL_EXT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus30[] = {
 { ARRAY_SIZE(elementStatus30), elementStatus30 },
};
static CONST_DATA_BLOB chain31[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain31_1), chain31_1 },
};
static CONST_DATA_BLOB chain32[] = {
 { sizeof(chain32_1), chain32_1 },
 { sizeof(chain32_2), chain32_2 },
 { sizeof(chain32_3), chain32_3 },
};
static CONST_DATA_BLOB chain33[] = {
 { sizeof(chain33_1), chain33_1 },
};
static CONST_DATA_BLOB selfSignedChain[] = {
 { sizeof(selfSignedCert), selfSignedCert }
};
static const CERT_TRUST_STATUS selfSignedElementStatus[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck selfSignedSimpleStatus[] = {
 { ARRAY_SIZE(selfSignedElementStatus),
   selfSignedElementStatus },
};
static CONST_DATA_BLOB googleChain[] = {
 { sizeof(geotrust_global_ca), geotrust_global_ca },
 { sizeof(google_internet_authority), google_internet_authority },
 { sizeof(google_com), google_com },
};
/* The Google cert is only valid from 11/10/2016 to 2/2/2017, so with the date
 * tested (October 2016) it's not time valid.
 */
static const CERT_TRUST_STATUS googleElementStatus[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR,
   /* TODO_INFO */
   CERT_TRUST_HAS_KEY_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_PREFERRED_ISSUER},
};
static const SimpleChainStatusCheck googleSimpleStatus[] = {
 { ARRAY_SIZE(googleElementStatus),
   googleElementStatus },
};
static CONST_DATA_BLOB battlenetChain[] = {
 { sizeof(thawte_primary_ca), thawte_primary_ca },
 { sizeof(thawte_ssl_ca), thawte_ssl_ca },
 { sizeof(battlenet), battlenet },
};
/* The openssl cert is only valid from 9/12/2008 to 9/13/2012, so with the date
 * tested (October 2007) it's not time valid.
 */
static CONST_DATA_BLOB opensslChain[] = {
 { sizeof(global_sign_root), global_sign_root },
 { sizeof(global_sign_ca), global_sign_ca },
 { sizeof(openssl_org), openssl_org },
};
static const CERT_TRUST_STATUS opensslElementStatus[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED},
};
static const SimpleChainStatusCheck opensslSimpleStatus[] = {
 { ARRAY_SIZE(opensslElementStatus),
   opensslElementStatus },
};
/* The OpenSSL chain may not have its root trusted, in which case the chain
 * is truncated (on Win98).
 */
static CONST_DATA_BLOB incompleteOpensslChain[] = {
 { sizeof(global_sign_ca), global_sign_ca },
 { sizeof(openssl_org), openssl_org },
};
static const CERT_TRUST_STATUS incompleteOpensslElementStatus[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
};
static const SimpleChainStatusCheck incompleteOpensslSimpleStatus[] = {
 { ARRAY_SIZE(incompleteOpensslElementStatus),
   incompleteOpensslElementStatus },
};
/*
 * USERTrust -> InCommon RSA CA -> cs.stanford.edu
 * cs.stanford.edu's cert is only valid from 10/21/2016 to 10/22/2019, so with
 * the date tested (October 1, 2016) it's not time valid.
 */
static CONST_DATA_BLOB stanfordChain[] = {
 { sizeof(usertrust_ca), usertrust_ca },
 { sizeof(incommon_rsa_ca), incommon_rsa_ca },
 { sizeof(cs_stanford_edu), cs_stanford_edu },
};
static const CERT_TRUST_STATUS stanfordElementStatus[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED},
};
static const SimpleChainStatusCheck stanfordSimpleStatus[] = {
 { ARRAY_SIZE(stanfordElementStatus),
   stanfordElementStatus },
};
static ChainCheck chainCheck[] = {
 /* Windows XP erroneously sets CERT_TRUST_HAS_PREFERRED_ISSUER on all
  * elements, so ignore it.
  * Windows 98/NT4 also set CERT_TRUST_IS_NOT_TIME_NESTED on chains they
  * shouldn't, so ignore those too.
  */
 { { ARRAY_SIZE(chain0), chain0 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus0 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain1), chain1 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_SIGNATURE_VALID |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus1 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain2), chain2 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus2 }, 0, &oct2007 },
 /* Earlier versions of Windows incorrectly don't set
  * CERT_TRUST_INVALID_BASIC_CONSTRAINTS on this chain.
  */
 { { ARRAY_SIZE(chain3), chain3 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_INVALID_BASIC_CONSTRAINTS,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus3 }, 0, &oct2007 },
 /* Earlier versions of Windows incorrectly don't set
  * CERT_TRUST_INVALID_BASIC_CONSTRAINTS on this chain.
  */
 { { ARRAY_SIZE(chain4), chain4 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_INVALID_BASIC_CONSTRAINTS,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus4 }, 0, &oct2007 },
 /* Windows versions prior to Vista/2008 incorrectly set
  * CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT on this chain, so ignore it.
  * Similarly, some older versions of Windows incorrectly set
  * CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, ignore that too.
  */
 { { ARRAY_SIZE(chain5), chain5 },
   { { CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT |
       CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus5 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain6), chain6 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus6 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain7), chain7 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus7 }, 0, &oct2007 },
 /* Earlier versions of Windows incorrectly don't set
  * CERT_TRUST_INVALID_BASIC_CONSTRAINTS on this chain.
  */
 { { ARRAY_SIZE(chain8), chain8 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_INVALID_BASIC_CONSTRAINTS,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus8 }, 0, &oct2007 },
 /* Earlier versions of Windows incorrectly don't set
  * CERT_TRUST_INVALID_BASIC_CONSTRAINTS on this chain.
  */
 { { ARRAY_SIZE(chain9), chain9 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_INVALID_BASIC_CONSTRAINTS,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_PARTIAL_CHAIN |
       CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_CYCLIC, 0 },
     1, simpleStatus9 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain10), chain10 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus10 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain11), chain11 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus10 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain12), chain12 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_SIGNATURE_VALID, 0 },
     1, simpleStatus12 }, 0, &oct2007 },
 { { ARRAY_SIZE(chain13), chain13 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus13 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain14), chain14 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus14 },
   0, &oct2007 },
 /* Earlier versions of crypt32 incorrectly do not complain that the end cert's
  * key usage is invalid, so ignore that error.
  */
 { { ARRAY_SIZE(chain15), chain15 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_VALID_FOR_USAGE, 0 },
     1, simpleStatus15 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain16), chain16 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_VALID_FOR_USAGE, 0 },
     1, simpleStatus16 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain17), chain17 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus17 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain18), chain18 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_VALID_FOR_USAGE, 0 },
     1, simpleStatus18 },
   0, &oct2007 },
 /* Older versions of crypt32 set CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT
  * even though the constraint and alt name match.
  * They also do not set CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS, since they
  * incorrectly find a name constraint error.
  */
 { { ARRAY_SIZE(chain19), chain19 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER | CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS
     },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS },
     1, simpleStatus19 },
   0, &oct2007 },
 /* Older versions of crypt32 do not set
  * CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, though they should.  So
  * ignore it (on Windows) but require it (on Wine.)
  */
 { { ARRAY_SIZE(chain20), chain20 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, 0 },
     1, simpleStatus20 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain21), chain21 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER | CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS
     },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS },
     1, simpleStatus21 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain22), chain22 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, 0 },
     1, simpleStatus22 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain23), chain23 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, 0 },
     1, simpleStatus23 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain24), chain24 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, 0 },
     1, simpleStatus24 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain25), chain25 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, 0 },
     1, simpleStatus25 },
   0, &oct2007 },
 { { ARRAY_SIZE(chain26), chain26 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, 0 },
     1, simpleStatus26 },
   0, &oct2007 },
 /* chain27 is handled separately elsewhere */
 { { ARRAY_SIZE(chain28), chain28 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, 0 },
     1, simpleStatus28 },
   0, &oct2007 },
 /* chain29 is handled separately elsewhere */
 /* Microsoft incorrectly ignores unknown/unsupported critical extensions on
  * older Windows versions, so ignore the error on Windows.
  */
 { { ARRAY_SIZE(chain30), chain30 },
   { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
       CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT |
       CERT_TRUST_INVALID_EXTENSION |
       CERT_TRUST_HAS_NOT_SUPPORTED_CRITICAL_EXT,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_INVALID_EXTENSION |
       CERT_TRUST_HAS_NOT_SUPPORTED_CRITICAL_EXT, 0 },
     1, simpleStatus30 },
   0, &oct2007 },
 { { ARRAY_SIZE(selfSignedChain), selfSignedChain },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT, 0 },
     1, selfSignedSimpleStatus }, 0, &oct2007 },
 /* The google chain may or may not have its root trusted, so ignore the error.
  * The chain is also considered not time nested on Win98, so ignore that
  * error too.
  * TODO_INFO: If the root cert is considered self-signed, InfoStatus is hardcoded in Wine
  * e.g. has always CERT_TRUST_HAS_NAME_MATCH_ISSUER flag
  * @see CRYPT_CheckSimpleChain
  */
 { { ARRAY_SIZE(googleChain), googleChain },
   { { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_TIME_NESTED,
       CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID, 0 },
       1, googleSimpleStatus }, TODO_INFO | TODO_ERROR, &oct2016 },
 /* The stanford chain may or may not have its root trusted, so ignore the error
  */
 { { ARRAY_SIZE(stanfordChain), stanfordChain },
   { { CERT_TRUST_IS_UNTRUSTED_ROOT, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID, 0 },
       1, stanfordSimpleStatus }, 0, &oct2016 },
};

static const CERT_TRUST_STATUS elementStatus8NoStore[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus8NoStore[] = {
 { ARRAY_SIZE(elementStatus8NoStore),
   elementStatus8NoStore },
};
static ChainCheck chainCheckNoStore[] = {
 { { ARRAY_SIZE(selfSignedChain), selfSignedChain },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT, 0 },
     1, selfSignedSimpleStatus },
   0 },
 { { ARRAY_SIZE(chain8), chain8 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_PARTIAL_CHAIN, 0 },
     1, simpleStatus8NoStore },
   0 },
};

 /* The openssl chain may or may not have its root trusted, so ignore the error
  */
static ChainCheck opensslChainCheck =
 { { ARRAY_SIZE(opensslChain), opensslChain },
   { { CERT_TRUST_IS_UNTRUSTED_ROOT, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID, 0 },
       1, opensslSimpleStatus }, 0 };
static ChainCheck incompleteOpensslChainCheck =
 { { ARRAY_SIZE(incompleteOpensslChain),
     incompleteOpensslChain },
   { { CERT_TRUST_IS_UNTRUSTED_ROOT, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_PARTIAL_CHAIN, 0 },
       1, incompleteOpensslSimpleStatus }, 0 };

/* Chain27 checks a certificate with a subject alternate name containing an
 * embedded NULL.  Newer crypt32 versions fail to decode such alternate names,
 * correctly prohibiting them.  Older crypt32 versions do not.  Rather than
 * ignoring the expected error bits, check each version separately depending
 * on the chain's error status.
 */
static ChainCheck chainCheckEmbeddedNull = {
 { ARRAY_SIZE(chain27), chain27 },
 { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
     CERT_TRUST_HAS_PREFERRED_ISSUER },
   { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_INVALID_NAME_CONSTRAINTS |
     CERT_TRUST_INVALID_EXTENSION, 0 },
   1, simpleStatus27 },
 0 };
static ChainCheck chainCheckEmbeddedNullBroken = {
 { ARRAY_SIZE(chain27), chain27 },
 { { CERT_TRUST_IS_NOT_TIME_NESTED | CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
     CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT,
     CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS | CERT_TRUST_HAS_PREFERRED_ISSUER },
   { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT,
     CERT_TRUST_HAS_VALID_NAME_CONSTRAINTS },
   1, simpleStatus27Broken },
 0 };

#define test_name_blob(a,b) _test_name_blob(__LINE__,a,b)
static void _test_name_blob(unsigned line, CERT_NAME_BLOB *blob, const char *exdata)
{
    char buf[1024];

    CertNameToStrA(CRYPT_ASN_ENCODING, blob, CERT_SIMPLE_NAME_STR, buf, sizeof(buf));
    ok_(__FILE__,line)(!strcmp(buf, exdata), "got string %s, expected %s\n", buf, exdata);
}

static void testGetCertChain(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert;
    CERT_CHAIN_PARA para = { 0 };
    PCCERT_CHAIN_CONTEXT chain;
    const CERT_SIMPLE_CHAIN *simple_chain;
    const CERT_CHAIN_ELEMENT *chain_elem;
    FILETIME fileTime;
    DWORD i;
    HCERTSTORE store;
    static char one_two_three[] = "1.2.3";
    static char oid_server_auth[] = szOID_PKIX_KP_SERVER_AUTH;
    LPSTR oids[2];

    /* Basic parameter checks */
    if (0)
    {
        /* Crash on Vista */
        ret = pCertGetCertificateChain(NULL, NULL, NULL, NULL, NULL, 0, NULL, NULL);
        ok(!ret && GetLastError() == E_INVALIDARG,
         "Expected E_INVALIDARG, got %08lx\n", GetLastError());
    }
    ret = pCertGetCertificateChain(NULL, NULL, NULL, NULL, NULL, 0, NULL,
     &chain);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08lx\n", GetLastError());
    /* Crash
    ret = pCertGetCertificateChain(NULL, NULL, NULL, NULL, &para, 0, NULL, NULL);
    ret = pCertGetCertificateChain(NULL, NULL, NULL, NULL, &para, 0, NULL,
     &chain);
     */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert,
     sizeof(bigCert));
    if (0)
    {
        /* Crash on Vista */
        ret = pCertGetCertificateChain(NULL, cert, NULL, NULL, NULL, 0, NULL, NULL);
        ok(!ret && GetLastError() == E_INVALIDARG,
         "Expected E_INVALIDARG, got %08lx\n", GetLastError());
    }
    /* Crash
    ret = pCertGetCertificateChain(NULL, cert, NULL, NULL, &para, 0, NULL, NULL);
     */

    /* Tests with an invalid cert (one whose signature is bad) */
    SetLastError(0xdeadbeef);
    ret = pCertGetCertificateChain(NULL, cert, NULL, NULL, &para, 0, NULL,
     &chain);
    ok(!ret, "Expected failure\n");
    ok(GetLastError() == ERROR_INVALID_DATA ||
     GetLastError() == CRYPT_E_ASN1_BADTAG /* Vista */,
     "Expected ERROR_INVALID_DATA or CRYPT_E_ASN1_BADTAG, got %ld\n", GetLastError());
    para.cbSize = sizeof(para);
    SetLastError(0xdeadbeef);
    ret = pCertGetCertificateChain(NULL, cert, NULL, NULL, &para, 0, NULL,
     &chain);
    ok(!ret, "Expected failure\n");
    ok(GetLastError() == ERROR_INVALID_DATA ||
     GetLastError() == CRYPT_E_ASN1_BADTAG /* Vista */,
     "Expected ERROR_INVALID_DATA or CRYPT_E_ASN1_BADTAG, got %ld\n", GetLastError());

    para.cbSize = 0;
    SetLastError(0xdeadbeef);
    ret = pCertGetCertificateChain(NULL, cert, NULL, NULL, &para, 0, NULL,
     &chain);
    ok(!ret, "Expected failure\n");
    ok(GetLastError() == ERROR_INVALID_DATA ||
     GetLastError() == CRYPT_E_ASN1_BADTAG, /* Vista and higher */
     "Expected ERROR_INVALID_DATA or CRYPT_E_ASN1_BADTAG, got %ld\n", GetLastError());

    CertFreeCertificateContext(cert);

    /* Test usage match with Google's cert */
    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    CertAddEncodedCertificateToStore(store, X509_ASN_ENCODING,
     geotrust_global_ca, sizeof(geotrust_global_ca), CERT_STORE_ADD_ALWAYS, NULL);
    CertAddEncodedCertificateToStore(store, X509_ASN_ENCODING,
     google_internet_authority, sizeof(google_internet_authority), CERT_STORE_ADD_ALWAYS, NULL);
    cert = CertCreateCertificateContext(X509_ASN_ENCODING,
     google_com, sizeof(google_com));
    SystemTimeToFileTime(&oct2009, &fileTime);
    memset(&para, 0, sizeof(para));
    para.cbSize = sizeof(para);
    oids[0] = one_two_three;
    para.RequestedUsage.dwType = USAGE_MATCH_TYPE_AND;
    para.RequestedUsage.Usage.rgpszUsageIdentifier = oids;
    para.RequestedUsage.Usage.cUsageIdentifier = 1;
    ret = pCertGetCertificateChain(NULL, cert, &fileTime, store, &para,
     0, NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %08lx\n", GetLastError());
    if (ret)
    {
        ok(chain->TrustStatus.dwErrorStatus & CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
         "expected CERT_TRUST_IS_NOT_VALID_FOR_USAGE\n");
        pCertFreeCertificateChain(chain);
    }
    oids[0] = oid_server_auth;
    ret = pCertGetCertificateChain(NULL, cert, &fileTime, store, &para,
     0, NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %08lx\n", GetLastError());
    if (ret)
    {
        ok(!(chain->TrustStatus.dwErrorStatus & CERT_TRUST_IS_NOT_VALID_FOR_USAGE),
           "didn't expect CERT_TRUST_IS_NOT_VALID_FOR_USAGE, got %lx\n", chain->TrustStatus.dwErrorStatus);
        pCertFreeCertificateChain(chain);
    }
    oids[1] = one_two_three;
    para.RequestedUsage.Usage.cUsageIdentifier = 2;
    para.RequestedUsage.dwType = USAGE_MATCH_TYPE_AND;
    ret = pCertGetCertificateChain(NULL, cert, &fileTime, store, &para,
     0, NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %08lx\n", GetLastError());
    if (ret)
    {
        ok(chain->TrustStatus.dwErrorStatus & CERT_TRUST_IS_NOT_VALID_FOR_USAGE,
         "expected CERT_TRUST_IS_NOT_VALID_FOR_USAGE\n");
        pCertFreeCertificateChain(chain);
    }
    para.RequestedUsage.dwType = USAGE_MATCH_TYPE_OR;
    ret = pCertGetCertificateChain(NULL, cert, &fileTime, store, &para,
     0, NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %08lx\n", GetLastError());
    if (ret)
    {
        ok(!(chain->TrustStatus.dwErrorStatus &
         CERT_TRUST_IS_NOT_VALID_FOR_USAGE),
         "didn't expect CERT_TRUST_IS_NOT_VALID_FOR_USAGE\n");
        pCertFreeCertificateChain(chain);
    }
    CertCloseStore(store, 0);
    CertFreeCertificateContext(cert);

    for (i = 0; i < ARRAY_SIZE(chainCheck); i++)
    {
        chain = getChain(NULL, &chainCheck[i].certs, 0, TRUE, chainCheck[i].validfor,
         chainCheck[i].todo, i);
        if (chain)
        {
            checkChainStatus(chain, &chainCheck[i].status, chainCheck[i].todo,
             "chainCheck", i);
            pCertFreeCertificateChain(chain);
        }
    }
    chain = getChain(NULL, &opensslChainCheck.certs, 0, TRUE, &oct2007,
     opensslChainCheck.todo, 0);
    if (chain)
    {
        ok(chain->TrustStatus.dwErrorStatus ==
         opensslChainCheck.status.status.dwErrorStatus ||
         broken((chain->TrustStatus.dwErrorStatus &
         ~incompleteOpensslChainCheck.status.statusToIgnore.dwErrorStatus) ==
         (incompleteOpensslChainCheck.status.status.dwErrorStatus &
         ~incompleteOpensslChainCheck.status.statusToIgnore.dwErrorStatus)),
         "unexpected chain error status %08lx\n",
         chain->TrustStatus.dwErrorStatus);
        if (opensslChainCheck.status.status.dwErrorStatus ==
         chain->TrustStatus.dwErrorStatus)
            checkChainStatus(chain, &opensslChainCheck.status,
             opensslChainCheck.todo, "opensslChainCheck", 0);
        else
            checkChainStatus(chain, &incompleteOpensslChainCheck.status,
             incompleteOpensslChainCheck.todo, "incompleteOpensslChainCheck",
             0);
        pCertFreeCertificateChain(chain);
    }
    for (i = 0; i < ARRAY_SIZE(chainCheckNoStore); i++)
    {
        chain = getChain(NULL, &chainCheckNoStore[i].certs, 0, FALSE, &oct2007,
         chainCheckNoStore[i].todo, i);
        if (chain)
        {
            checkChainStatus(chain, &chainCheckNoStore[i].status,
             chainCheckNoStore[i].todo, "chainCheckNoStore", i);
            pCertFreeCertificateChain(chain);
        }
    }
    chain = getChain(NULL, &chainCheckEmbeddedNull.certs, 0, TRUE, &oct2007,
     chainCheckEmbeddedNull.todo, 0);
    if (chain)
    {
        ok(chain->TrustStatus.dwErrorStatus ==
         chainCheckEmbeddedNull.status.status.dwErrorStatus ||
         broken((chain->TrustStatus.dwErrorStatus &
         ~chainCheckEmbeddedNullBroken.status.statusToIgnore.dwErrorStatus) ==
         (chainCheckEmbeddedNullBroken.status.status.dwErrorStatus &
         ~chainCheckEmbeddedNullBroken.status.statusToIgnore.dwErrorStatus)),
         "unexpected chain error status %08lx\n",
         chain->TrustStatus.dwErrorStatus);
        if (chainCheckEmbeddedNull.status.status.dwErrorStatus ==
         chain->TrustStatus.dwErrorStatus)
            checkChainStatus(chain, &chainCheckEmbeddedNull.status,
             chainCheckEmbeddedNull.todo, "chainCheckEmbeddedNull", 0);
        else
            checkChainStatus(chain, &chainCheckEmbeddedNullBroken.status,
             chainCheckEmbeddedNullBroken.todo, "chainCheckEmbeddedNullBroken",
             0);
        pCertFreeCertificateChain(chain);
    }

    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0, CERT_STORE_CREATE_NEW_FLAG, NULL);
    ok(store != NULL, "CertOpenStore failed: %lu\n", GetLastError());

    ret = CertAddEncodedCertificateToStore(store, X509_ASN_ENCODING, winehq_org, sizeof(winehq_org),
            CERT_STORE_ADD_ALWAYS, &cert);
    ok(ret, "CertAddEncodedCertificateToStore failed: %lu\n", GetLastError());

    oids[0] = oid_server_auth;
    memset(&para, 0, sizeof(para));
    para.cbSize = sizeof(para);
    para.RequestedUsage.Usage.cUsageIdentifier = 1;
    para.RequestedUsage.Usage.rgpszUsageIdentifier = oids;
    SystemTimeToFileTime(&jun2013, &fileTime);

    /* Pass store that does not contain all certs in chain. */
    ret = CertGetCertificateChain(NULL, cert, &fileTime, store, &para, 0, NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %lu\n", GetLastError());

    if(chain->TrustStatus.dwErrorStatus == CERT_TRUST_IS_PARTIAL_CHAIN) { /* win2k */
        win_skip("winehq cert reported as partial chain, skipping its tests\n");
        pCertFreeCertificateChain(chain);
        CertCloseStore(store, 0);
        return;
    }

    ok(chain->TrustStatus.dwErrorStatus == CERT_TRUST_HAS_EXACT_MATCH_ISSUER, "chain->TrustStatus.dwErrorStatus = %lx\n",
       chain->TrustStatus.dwErrorStatus);
    todo_wine
    ok(chain->TrustStatus.dwInfoStatus == CERT_TRUST_HAS_PREFERRED_ISSUER, "chain->TrustStatus.dwInfoStatus = %lx\n",
       chain->TrustStatus.dwInfoStatus);

    ok(chain->cChain == 1, "chain->cChain = %ld\n", chain->cChain);
    ok(!chain->cLowerQualityChainContext, "chain->cLowerQualityChainContext = %lx\n", chain->cLowerQualityChainContext);
    ok(!chain->rgpLowerQualityChainContext, "chain->rgpLowerQualityChainContext =  %p\n", chain->rgpLowerQualityChainContext);

    simple_chain = *chain->rgpChain;
    ok(simple_chain->cbSize == sizeof(*simple_chain), "simple_chain->cbSize = %lu\n", simple_chain->cbSize);
    ok(simple_chain->TrustStatus.dwErrorStatus == CERT_TRUST_HAS_EXACT_MATCH_ISSUER, "simple_chain->TrustStatus.dwErrorStatus = %lx\n",
       simple_chain->TrustStatus.dwErrorStatus);
    todo_wine
    ok(simple_chain->TrustStatus.dwInfoStatus == CERT_TRUST_HAS_PREFERRED_ISSUER,
       "simple_chain->TrustStatus.dwInfoStatus = %lx\n", simple_chain->TrustStatus.dwInfoStatus);
    ok(simple_chain->cElement == 3, "simple_chain->cElement = %lu\n", simple_chain->cElement);

    for(i=0; i < simple_chain->cElement; i++) {
        chain_elem = simple_chain->rgpElement[i];
        ok(chain_elem->cbSize == sizeof(*chain_elem), "chain_elem->cbSize = %lu\n", chain_elem->cbSize);
        if (!i)
            ok(chain_elem->TrustStatus.dwErrorStatus == CERT_TRUST_HAS_EXACT_MATCH_ISSUER,
               "chain_elem->TrustStatus.dwErrorStatus = %lx\n", chain_elem->TrustStatus.dwErrorStatus);
        else if (i == 1)
            todo_wine ok(!chain_elem->TrustStatus.dwErrorStatus, "chain_elem->TrustStatus.dwErrorStatus = %lx\n",
                         chain_elem->TrustStatus.dwErrorStatus);
        else
            ok(!chain_elem->TrustStatus.dwErrorStatus, "chain_elem->TrustStatus.dwErrorStatus = %lx\n",
               chain_elem->TrustStatus.dwErrorStatus);
        trace("info[%lu] = %lx\n", i, chain_elem->TrustStatus.dwInfoStatus);
        ok(chain_elem->pCertContext->dwCertEncodingType == CRYPT_ASN_ENCODING,
           "chain_elem->pCertContext->dwCertEncodingType = %lx\n",
           chain_elem->pCertContext->dwCertEncodingType);
    }

    ok(simple_chain->rgpElement[0]->pCertContext == cert, "simple_chain->rgpElement[0]->pCertContext != cert\n");
    test_name_blob(&simple_chain->rgpElement[1]->pCertContext->pCertInfo->Issuer,
                   "US, New Jersey, Jersey City, The USERTRUST Network, USERTrust RSA Certification Authority");
    test_name_blob(&simple_chain->rgpElement[1]->pCertContext->pCertInfo->Subject,
                   "US, VA, Herndon, Network Solutions L.L.C., Network Solutions OV Server CA 2");
    test_name_blob(&simple_chain->rgpElement[2]->pCertContext->pCertInfo->Issuer,
                   "US, New Jersey, Jersey City, The USERTRUST Network, USERTrust RSA Certification Authority");
    test_name_blob(&simple_chain->rgpElement[2]->pCertContext->pCertInfo->Subject,
                   "US, New Jersey, Jersey City, The USERTRUST Network, USERTrust RSA Certification Authority");

    pCertFreeCertificateChain(chain);

    /* Test revocation flags */
    ret = CertGetCertificateChain(NULL, cert, &fileTime, store, &para, CERT_CHAIN_REVOCATION_CHECK_END_CERT, NULL,
                                  &chain);
    ok(ret, "CertGetCertificateChain failed: %lu\n", GetLastError());
    ok(chain->TrustStatus.dwErrorStatus == CERT_TRUST_HAS_EXACT_MATCH_ISSUER, "chain->TrustStatus.dwErrorStatus = %lx\n",
       chain->TrustStatus.dwErrorStatus);
    pCertFreeCertificateChain(chain);

    ret = CertGetCertificateChain(NULL, cert, &fileTime, store, &para, CERT_CHAIN_REVOCATION_CHECK_CHAIN, NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %lu\n", GetLastError());
    ok(chain->TrustStatus.dwErrorStatus == CERT_TRUST_HAS_EXACT_MATCH_ISSUER ||
       broken(chain->TrustStatus.dwErrorStatus == (CERT_TRUST_IS_NOT_TIME_VALID|CERT_TRUST_REVOCATION_STATUS_UNKNOWN)), /* XP */
       "chain->TrustStatus.dwErrorStatus = %lx\n", chain->TrustStatus.dwErrorStatus);
    pCertFreeCertificateChain(chain);

    ret = CertGetCertificateChain(NULL, cert, &fileTime, store, &para, CERT_CHAIN_REVOCATION_CHECK_CHAIN_EXCLUDE_ROOT,
                                  NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %lu\n", GetLastError());
    ok(chain->TrustStatus.dwErrorStatus == CERT_TRUST_HAS_EXACT_MATCH_ISSUER, "chain->TrustStatus.dwErrorStatus = %lx\n",
       chain->TrustStatus.dwErrorStatus);
    pCertFreeCertificateChain(chain);

    /* Test HCCE_LOCAL_MACHINE */
    ret = CertGetCertificateChain(HCCE_LOCAL_MACHINE, cert, &fileTime, store, &para, 0, NULL, &chain);
    ok(ret, "CertGetCertificateChain failed: %lu\n", GetLastError());
    pCertFreeCertificateChain(chain);

    CertFreeCertificateContext(cert);
    CertCloseStore(store, 0);
}

static void test_CERT_CHAIN_PARA_cbSize(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert;
    CERT_CHAIN_PARA para = { 0 };
    PCCERT_CHAIN_CONTEXT chain;
    HCERTSTORE store;
    DWORD i;

    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);

    ret = CertAddEncodedCertificateToStore(store,
     X509_ASN_ENCODING, chain0_0, sizeof(chain0_0),
     CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCertificateToStore failed: %08lx\n", GetLastError());
    ret = CertAddEncodedCertificateToStore(store,
     X509_ASN_ENCODING, chain0_1, sizeof(chain0_1),
     CERT_STORE_ADD_ALWAYS, &cert);
    ok(ret, "CertAddEncodedCertificateToStore failed: %08lx\n", GetLastError());

    for (i = 0; i < sizeof(CERT_CHAIN_PARA) + 2; i++)
    {
        FILETIME fileTime;

        SystemTimeToFileTime(&oct2007, &fileTime);

        para.cbSize = i;
        ret = pCertGetCertificateChain(NULL, cert, &fileTime,
         NULL, &para, 0, NULL, &chain);
        ok(ret, "CertGetCertificateChain failed %lu\n", GetLastError());
        pCertFreeCertificateChain(chain);
    }

    CertFreeCertificateContext(cert);
    CertCloseStore(store, 0);
}

typedef struct _ChainPolicyCheck
{
    CONST_BLOB_ARRAY                certs;
    CERT_CHAIN_POLICY_STATUS        status;
    const CERT_CHAIN_POLICY_STATUS *brokenStatus;
    DWORD                           todo;
} ChainPolicyCheck;

static const ChainPolicyCheck basePolicyCheck[] = {
 { { ARRAY_SIZE(chain0), chain0 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain1), chain1 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 0, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain2), chain2 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain3), chain3 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain4), chain4 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain5), chain5 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain6), chain6 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain7), chain7 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain8), chain8 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain9), chain9 },
   { 0, CERT_E_CHAINING, 0, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain10), chain10 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain11), chain11 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain12), chain12 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain13), chain13 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain14), chain14 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain15), chain15 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain16), chain16 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain17), chain17 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain18), chain18 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(selfSignedChain), selfSignedChain },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL }, NULL, 0 },
};

/* Windows NT 4 has a different error code when the validity period doesn't
 * nest.  (It's arguably more correct than other Windows versions, but since
 * others do not emulate its behavior, we mark its behavior broken.)
 */
static const CERT_CHAIN_POLICY_STATUS badDateNestingStatus =
 { 0, CERT_E_VALIDITYPERIODNESTING, 0, 0, NULL };

static const ChainPolicyCheck ignoredBadDateNestingBasePolicyCheck = {
 { ARRAY_SIZE(chain2), chain2 },
 { 0, 0, -1, -1, NULL}, NULL, 0
};

static const ChainPolicyCheck ignoredInvalidDateBasePolicyCheck = {
 { ARRAY_SIZE(googleChain), googleChain },
 { 0, 0, -1, -1, NULL}, NULL, 0
};

static const ChainPolicyCheck ignoredInvalidUsageBasePolicyCheck = {
 { ARRAY_SIZE(chain15), chain15 },
 { 0, 0, -1, -1, NULL}, NULL, 0
};

static const ChainPolicyCheck invalidUsageBasePolicyCheck = {
 { ARRAY_SIZE(chain15), chain15 },
 { 0, CERT_E_WRONG_USAGE, 0, 1, NULL}, NULL, 0
};

static const ChainPolicyCheck sslPolicyCheck[] = {
 { { ARRAY_SIZE(chain0), chain0 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain1), chain1 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 0, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain2), chain2 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain3), chain3 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain4), chain4 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain5), chain5 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain6), chain6 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain7), chain7 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain8), chain8 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain9), chain9 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain10), chain10 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain11), chain11 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain12), chain12 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain13), chain13 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain14), chain14 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain15), chain15 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain16), chain16 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain17), chain17 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain18), chain18 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(selfSignedChain), selfSignedChain },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL }, NULL, 0 },
};

static const ChainPolicyCheck ignoredUnknownCAPolicyCheck = {
 { ARRAY_SIZE(chain0), chain0 },
 { 0, CERT_E_EXPIRED, 0, 0, NULL }, NULL, 0
};

static const ChainPolicyCheck googlePolicyCheckWithMatchingNameExpired = {
 { ARRAY_SIZE(googleChain), googleChain },
 { 0, CERT_E_EXPIRED, 0, 0, NULL}, NULL, 0
};

static const ChainPolicyCheck googlePolicyCheckWithMatchingName = {
 { ARRAY_SIZE(googleChain), googleChain },
 { 0, 0, -1, -1, NULL}, NULL, 0
};

/* Win98 does not trust the root of the OpenSSL chain or the Stanford chain */
static const CERT_CHAIN_POLICY_STATUS untrustedRootStatus =
 { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL };

static const ChainPolicyCheck opensslPolicyCheckWithMatchingName = {
 { ARRAY_SIZE(opensslChain), opensslChain },
 { 0, 0, -1, -1, NULL}, &untrustedRootStatus, 0
};

static const ChainPolicyCheck opensslPolicyCheckWithoutMatchingName = {
 { ARRAY_SIZE(opensslChain), opensslChain },
 { 0, CERT_E_CN_NO_MATCH, 0, 0, NULL}, &untrustedRootStatus, 0
};

static const ChainPolicyCheck winehqPolicyCheckWithMatchingName = {
 { ARRAY_SIZE(chain29), chain29 },
 { 0, 0, -1, -1, NULL}, NULL, 0
};

static const ChainPolicyCheck winehqPolicyCheckWithoutMatchingName = {
 { ARRAY_SIZE(chain29), chain29 },
 { 0, CERT_E_CN_NO_MATCH, 0, 0, NULL}, NULL, 0
};

static const ChainPolicyCheck stanfordPolicyCheckWithMatchingName = {
 { ARRAY_SIZE(stanfordChain), stanfordChain },
 { 0, 0, -1, -1, NULL}, &untrustedRootStatus, 0
};

static const ChainPolicyCheck stanfordPolicyCheckWithoutMatchingName = {
 { ARRAY_SIZE(stanfordChain), stanfordChain },
 { 0, CERT_E_CN_NO_MATCH, 0, 0, NULL}, &untrustedRootStatus, 0
};

static const ChainPolicyCheck nullTerminatedDomainComponentPolicyCheck = {
 { ARRAY_SIZE(battlenetChain), battlenetChain },
 { 0, 0, -1, -1, NULL}, &untrustedRootStatus, 0
};

static const ChainPolicyCheck invalidExtensionPolicyCheck = {
 { ARRAY_SIZE(chain30), chain30 },
 { 0, CERT_E_CRITICAL, 0, 1, NULL}, &badDateNestingStatus, 0
};

static const ChainPolicyCheck fooPolicyCheckWithMatchingName = {
 { ARRAY_SIZE(chain31), chain31 },
 { 0, 0, -1, -1, NULL}, NULL, 0
};

static const ChainPolicyCheck fooPolicyCheckWithoutMatchingName = {
 { ARRAY_SIZE(chain31), chain31 },
 { 0, CERT_E_CN_NO_MATCH, 0, 0, NULL}, NULL, 0
};

static const ChainPolicyCheck authenticodePolicyCheck[] = {
 { { ARRAY_SIZE(chain0), chain0 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain1), chain1 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 0, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain2), chain2 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain3), chain3 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain4), chain4 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain5), chain5 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain6), chain6 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain7), chain7 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain8), chain8 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain9), chain9 },
   { 0, CERT_E_CHAINING, 0, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain10), chain10 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain11), chain11 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain12), chain12 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain13), chain13 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain14), chain14 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain15), chain15 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain16), chain16 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain17), chain17 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain18), chain18 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, NULL, 0 },
 { { ARRAY_SIZE(selfSignedChain), selfSignedChain },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL }, NULL, 0 },
};

/* On some older systems, the element index is set to 2 rather than 1 for
 * chain 4, because they do not catch the basic constraints error in the
 * chain, which occurs at element 1.
 */
static const CERT_CHAIN_POLICY_STATUS chain4BrokenStatus =
 { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 2, NULL };

static const ChainPolicyCheck basicConstraintsPolicyCheck[] = {
 { { ARRAY_SIZE(chain0), chain0 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain1), chain1 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain2), chain2 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain3), chain3 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain4), chain4 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL }, &chain4BrokenStatus, 0 },
 { { ARRAY_SIZE(chain5), chain5 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain6), chain6 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain7), chain7 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain8), chain8 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain9), chain9 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain10), chain10 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain11), chain11 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain12), chain12 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain13), chain13 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain14), chain14 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain15), chain15 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain16), chain16 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain17), chain17 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain18), chain18 },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
 { { ARRAY_SIZE(selfSignedChain), selfSignedChain },
   { 0, 0, -1, -1, NULL }, NULL, 0 },
};

static const CERT_CHAIN_POLICY_STATUS chain32BrokenStatus =
 { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL };

static const ChainPolicyCheck msRootPolicyCheck[] = {
 { { ARRAY_SIZE(chain0), chain0 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, NULL, TODO_ELEMENTS },
 { { ARRAY_SIZE(selfSignedChain), selfSignedChain },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL }, NULL, 0 },
 { { ARRAY_SIZE(chain32), chain32 },
   { 0, 0, 0, 0, NULL }, &chain32BrokenStatus, 0 },
 { { ARRAY_SIZE(chain33), chain33 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL }, NULL, 0 },
};

static const char *num_to_str(WORD num)
{
    static char buf[6];

    sprintf(buf, "#%04X", num);
    return buf;
}

static void checkChainPolicyStatus(LPCSTR policy, HCERTCHAINENGINE engine,
 const ChainPolicyCheck *check, LPCSTR testName, DWORD testIndex,
 SYSTEMTIME *sysTime, PCERT_CHAIN_POLICY_PARA para)
{
    PCCERT_CHAIN_CONTEXT chain = getChain(engine, &check->certs, 0, TRUE,
     sysTime, check->todo, testIndex);

    if (chain)
    {
        CERT_CHAIN_POLICY_STATUS policyStatus = { 0 };
        BOOL ret = pCertVerifyCertificateChainPolicy(policy, chain, para,
         &policyStatus);

        if (check->todo & TODO_POLICY)
            todo_wine ok(ret,
             "%s[%ld](%s): CertVerifyCertificateChainPolicy failed: %08lx\n",
             testName, testIndex,
             IS_INTOID(policy) ? num_to_str(LOWORD(policy)) : policy,
             GetLastError());
        else
        {
            if (!ret && GetLastError() == ERROR_FILE_NOT_FOUND)
            {
                skip("%s[%ld]: missing policy %s, skipping test\n",
                 testName, testIndex,
                 IS_INTOID(policy) ? num_to_str(LOWORD(policy)) : policy);
                pCertFreeCertificateChain(chain);
                return;
            }
            ok(ret, "%s[%ld](%s): CertVerifyCertificateChainPolicy failed: %08lx\n",
             testName, testIndex,
             IS_INTOID(policy) ? num_to_str(LOWORD(policy)) : policy,
             GetLastError());
        }
        if (ret)
        {
            todo_wine_if (check->todo & TODO_ERROR)
                ok(policyStatus.dwError == check->status.dwError ||
                 broken(policyStatus.dwError == CERT_TRUST_NO_ERROR) ||
                 (check->brokenStatus && broken(policyStatus.dwError ==
                 check->brokenStatus->dwError)),
                 "%s[%ld](%s): expected %08lx, got %08lx\n",
                 testName, testIndex,
                 IS_INTOID(policy) ? num_to_str(LOWORD(policy)) : policy,
                 check->status.dwError, policyStatus.dwError);
            if (policyStatus.dwError != check->status.dwError)
            {
                skip("%s[%ld](%s): error %08lx doesn't match expected %08lx, not checking indexes\n",
                 testName, testIndex,
                 IS_INTOID(policy) ? num_to_str(LOWORD(policy)) : policy,
                 policyStatus.dwError, check->status.dwError);
                pCertFreeCertificateChain(chain);
                return;
            }
            todo_wine_if (check->todo & TODO_CHAINS)
                ok(policyStatus.lChainIndex == check->status.lChainIndex ||
                 (check->brokenStatus && broken(policyStatus.lChainIndex ==
                 check->brokenStatus->lChainIndex)),
                 "%s[%ld](%s): expected %ld, got %ld\n",
                 testName, testIndex,
                 IS_INTOID(policy) ? num_to_str(LOWORD(policy)) : policy,
                 check->status.lChainIndex, policyStatus.lChainIndex);
            todo_wine_if (check->todo & TODO_ELEMENTS)
                ok(policyStatus.lElementIndex == check->status.lElementIndex ||
                 (check->brokenStatus && broken(policyStatus.lElementIndex ==
                 check->brokenStatus->lElementIndex)),
                 "%s[%ld](%s): expected %ld, got %ld\n",
                 testName, testIndex,
                 IS_INTOID(policy) ? num_to_str(LOWORD(policy)) : policy,
                 check->status.lElementIndex, policyStatus.lElementIndex);
        }
        pCertFreeCertificateChain(chain);
    }
}

#define CHECK_CHAIN_POLICY_STATUS_ARRAY(policy, engine, array, date, para) \
    do { \
        DWORD i; \
        for (i = 0; i < ARRAY_SIZE(array); i++) \
            checkChainPolicyStatus((policy), (engine), &(array)[i], \
             #array, i, (date), (para)); \
    } while(0)

#define CHECK_CHAIN_POLICY_STATUS(policy, engine, policyCheck, date, para) \
    checkChainPolicyStatus((policy), (engine), &(policyCheck), \
     #policyCheck, 0, (date), (para))

static void check_base_policy(void)
{
    CERT_CHAIN_POLICY_PARA policyPara = { 0 };

    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_BASE, NULL,
     basePolicyCheck, &oct2007, NULL);
    policyPara.cbSize = sizeof(policyPara);
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_BASE, NULL,
     ignoredUnknownCAPolicyCheck, &oct2007, &policyPara);
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG |
     CERT_CHAIN_POLICY_IGNORE_NOT_TIME_VALID_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_BASE, NULL,
     ignoredBadDateNestingBasePolicyCheck, &oct2007, &policyPara);
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG |
     CERT_CHAIN_POLICY_IGNORE_NOT_TIME_VALID_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_BASE, NULL,
     ignoredInvalidDateBasePolicyCheck, &oct2007, &policyPara);
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG |
     CERT_CHAIN_POLICY_IGNORE_WRONG_USAGE_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_BASE, NULL,
     ignoredInvalidUsageBasePolicyCheck, &oct2007, &policyPara);
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG |
     CERT_CHAIN_POLICY_IGNORE_NOT_TIME_VALID_FLAG |
     CERT_CHAIN_POLICY_IGNORE_WRONG_USAGE_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_BASE, NULL,
     ignoredInvalidUsageBasePolicyCheck, &oct2007, &policyPara);
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG |
     CERT_CHAIN_POLICY_IGNORE_NOT_TIME_VALID_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_BASE, NULL,
     invalidUsageBasePolicyCheck, &oct2007, &policyPara);
    /* Test chain30, which has an invalid critical extension in an intermediate
     * cert, against the base policy.
     */
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_BASE, NULL,
     invalidExtensionPolicyCheck, &oct2007, &policyPara);
}

static void check_authenticode_policy(void)
{
    CERT_CHAIN_POLICY_PARA policyPara = { 0 };
    SYSTEMTIME epochStart = { 0 };

    /* The authenticode policy doesn't seem to check anything beyond the base
     * policy.  It might check for chains signed by the MS test cert, but none
     * of these chains is.
     */
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_AUTHENTICODE, NULL,
     authenticodePolicyCheck, &oct2007, NULL);
    policyPara.cbSize = sizeof(policyPara);
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_AUTHENTICODE, NULL,
     ignoredUnknownCAPolicyCheck, &oct2007, &policyPara);
    epochStart.wDay = epochStart.wMonth = 1;
    epochStart.wYear = 1601;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_AUTHENTICODE, NULL,
     ignoredUnknownCAPolicyCheck, &epochStart, &policyPara);
    policyPara.dwFlags |= CERT_CHAIN_POLICY_IGNORE_NOT_TIME_VALID_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_AUTHENTICODE, NULL,
     ignoredInvalidDateBasePolicyCheck, &oct2007, &policyPara);
}

static void check_ssl_policy(void)
{
    CERT_CHAIN_POLICY_PARA policyPara = { 0 };
    SSL_EXTRA_CERT_CHAIN_POLICY_PARA sslPolicyPara = { { 0 } };
    WCHAR winehq[] = { 'w','i','n','e','h','q','.','o','r','g',0 };
    WCHAR google_dot_com[] = { 'w','w','w','.','g','o','o','g','l','e','.',
     'c','o','m',0 };
    WCHAR battle_dot_net[] = { 'w','w','w','.','b','a','t','t','l','e','.',
     'n','e','t',0 };
    WCHAR a_dot_openssl_dot_org[] = { 'a','.','o','p','e','n','s','s','l','.',
     'o','r','g',0 };
    WCHAR openssl_dot_org[] = { 'o','p','e','n','s','s','l','.','o','r','g',0 };
    WCHAR fopenssl_dot_org[] = { 'f','o','p','e','n','s','s','l','.',
     'o','r','g',0 };
    WCHAR a_dot_b_dot_openssl_dot_org[] = { 'a','.','b','.',
     'o','p','e','n','s','s','l','.','o','r','g',0 };
    WCHAR cs_dot_stanford_dot_edu[] = { 'c','s','.',
     's','t','a','n','f','o','r','d','.','e','d','u',0 };
    WCHAR www_dot_cs_dot_stanford_dot_edu[] = { 'w','w','w','.','c','s','.',
     's','t','a','n','f','o','r','d','.','e','d','u',0 };
    WCHAR a_dot_cs_dot_stanford_dot_edu[] = { 'a','.','c','s','.',
     's','t','a','n','f','o','r','d','.','e','d','u',0 };
    WCHAR test_dot_winehq_dot_org[] = { 't','e','s','t','.',
     'w','i','n','e','h','q','.','o','r','g',0 };
    WCHAR a_dot_b_dot_winehq_dot_org[] = { 'a','.','b','.',
     'w','i','n','e','h','q','.','o','r','g',0 };
    WCHAR foo_dot_com[] = { 'f','o','o','.','c','o','m',0 };
    WCHAR afoo_dot_com[] = { 'a','f','o','o','.','c','o','m',0 };
    WCHAR a_dot_foo_dot_com[] = { 'a','.','f','o','o','.','c','o','m',0 };
    HCERTSTORE testRoot;
    CERT_CHAIN_ENGINE_CONFIG engineConfig = { sizeof(engineConfig), 0 };
    HCERTCHAINENGINE engine;

    /* Check ssl policy with no parameter */
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, NULL);
    /* Check again with a policy parameter that specifies nothing */
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* Check yet again, but specify an empty SSL_EXTRA_CERT_CHAIN_POLICY_PARA
     * argument.
     */
    policyPara.pvExtraPolicyPara = &sslPolicyPara;
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* And again, but specify the auth type as a client */
    sslPolicyPara.dwAuthType = AUTHTYPE_CLIENT;
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* And again, but specify the auth type as a server */
    sslPolicyPara.dwAuthType = AUTHTYPE_SERVER;
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* And again authenticating a client, but specify the size of the policy
     * parameter.
     */
    U(sslPolicyPara).cbSize = sizeof(sslPolicyCheck);
    sslPolicyPara.dwAuthType = AUTHTYPE_CLIENT;
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* One more time authenticating a client, but specify winehq.org as the
     * server name.
     */
    sslPolicyPara.pwszServerName = winehq;
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* And again authenticating a server, still specifying winehq.org as the
     * server name.
     */
    sslPolicyPara.dwAuthType = AUTHTYPE_SERVER;
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* And again authenticating a server, this time specifying the size of the
     * policy param.
     */
    policyPara.cbSize = sizeof(policyPara);
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_SSL, NULL, sslPolicyCheck,
     &oct2007, &policyPara);
    /* And again, specifying a chain with an untrusted root, but ignoring
     * unknown CAs.
     */
    sslPolicyPara.fdwChecks = SECURITY_FLAG_IGNORE_UNKNOWN_CA;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     ignoredUnknownCAPolicyCheck, &oct2007, &policyPara);
    sslPolicyPara.fdwChecks = 0;
    /* And again, but specifying the ignore in dwFlags */
    policyPara.dwFlags = CERT_CHAIN_POLICY_ALLOW_UNKNOWN_CA_FLAG;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     ignoredUnknownCAPolicyCheck, &oct2007, &policyPara);
    /* And again, but checking the Google chain at a bad date */
    sslPolicyPara.pwszServerName = google_dot_com;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     googlePolicyCheckWithMatchingNameExpired, &oct2007, &policyPara);
    policyPara.dwFlags = 0;
    /* Again checking the Google chain at a bad date, but ignoring date
     * errors.
     */
    sslPolicyPara.fdwChecks = SECURITY_FLAG_IGNORE_UNKNOWN_CA |
     SECURITY_FLAG_IGNORE_CERT_DATE_INVALID;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     googlePolicyCheckWithMatchingName, &oct2007, &policyPara);
    /* And again, but checking the Google chain at a good date */
    sslPolicyPara.fdwChecks = SECURITY_FLAG_IGNORE_UNKNOWN_CA;
    sslPolicyPara.pwszServerName = google_dot_com;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     googlePolicyCheckWithMatchingName, &nov2016, &policyPara);
    sslPolicyPara.fdwChecks = 0;

    /* Check again with the openssl cert, which has a wildcard in its name,
     * with various combinations of matching and non-matching names.
     * With "a.openssl.org": match
     */
    sslPolicyPara.pwszServerName = a_dot_openssl_dot_org;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     opensslPolicyCheckWithMatchingName, &oct2009, &policyPara);
    /* With "openssl.org": no match */
    sslPolicyPara.pwszServerName = openssl_dot_org;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     opensslPolicyCheckWithoutMatchingName, &oct2009, &policyPara);
    /* With "fopenssl.org": no match */
    sslPolicyPara.pwszServerName = fopenssl_dot_org;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     opensslPolicyCheckWithoutMatchingName, &oct2009, &policyPara);
    /* with "a.b.openssl.org": no match */
    sslPolicyPara.pwszServerName = a_dot_b_dot_openssl_dot_org;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     opensslPolicyCheckWithoutMatchingName, &oct2009, &policyPara);
    /* Check again with the cs.stanford.edu, which has both cs.stanford.edu
     * and www.cs.stanford.edu in its subject alternative name.
     * With "cs.stanford.edu": match
     */
    sslPolicyPara.pwszServerName = cs_dot_stanford_dot_edu;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     stanfordPolicyCheckWithMatchingName, &nov2016, &policyPara);
    /* With "www.cs.stanford.edu": match */
    sslPolicyPara.pwszServerName = www_dot_cs_dot_stanford_dot_edu;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     stanfordPolicyCheckWithMatchingName, &nov2016, &policyPara);
    /* With "a.cs.stanford.edu": no match */
    sslPolicyPara.pwszServerName = a_dot_cs_dot_stanford_dot_edu;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     stanfordPolicyCheckWithoutMatchingName, &nov2016, &policyPara);
    /* Check chain29, which has a wildcard in its subject alternative name,
     * but not in its distinguished name.
     * Step 1: create a chain engine that trusts chain29's root.
     */
    testRoot = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    CertAddEncodedCertificateToStore(testRoot, X509_ASN_ENCODING, chain0_0,
     sizeof(chain0_0), CERT_STORE_ADD_ALWAYS, NULL);
    engineConfig.hExclusiveRoot = testRoot;
    if (!pCertCreateCertificateChainEngine(&engineConfig, &engine))
    {
        skip("Couldn't create chain engine\n");
        return;
    }
    /* With "winehq.org": no match */
    sslPolicyPara.pwszServerName = winehq;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, engine,
     winehqPolicyCheckWithoutMatchingName, &oct2007, &policyPara);
    /* With "test.winehq.org": match */
    sslPolicyPara.pwszServerName = test_dot_winehq_dot_org;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, engine,
     winehqPolicyCheckWithMatchingName, &oct2007, &policyPara);
    /* With "a.b.winehq.org": no match */
    sslPolicyPara.pwszServerName = a_dot_b_dot_winehq_dot_org;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, engine,
     winehqPolicyCheckWithoutMatchingName, &oct2007, &policyPara);
    /* When specifying to ignore name mismatch: match */
    sslPolicyPara.fdwChecks |= SECURITY_FLAG_IGNORE_CERT_CN_INVALID;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, engine,
     winehqPolicyCheckWithMatchingName, &oct2007, &policyPara);
    pCertFreeCertificateChainEngine(engine);
    CertCloseStore(testRoot, 0);
    /* Test chain30, which has an invalid critical extension in an intermediate
     * cert, against the SSL policy.
     */
    sslPolicyPara.fdwChecks = SECURITY_FLAG_IGNORE_UNKNOWN_CA;
    sslPolicyPara.pwszServerName = NULL;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     invalidExtensionPolicyCheck, &oct2007, &policyPara);
    /* Test chain31, which has two CNs, "*.foo.com" and "foo.com", against
     * some names that match one of the CNs:
     */
    sslPolicyPara.pwszServerName = foo_dot_com;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     fooPolicyCheckWithMatchingName, &oct2007, &policyPara);
    sslPolicyPara.pwszServerName = a_dot_foo_dot_com;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     fooPolicyCheckWithMatchingName, &oct2007, &policyPara);
    /* and against a name that doesn't match either CN: */
    sslPolicyPara.pwszServerName = afoo_dot_com;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     fooPolicyCheckWithoutMatchingName, &oct2007, &policyPara);
    /* The Battle.Net chain checks a certificate with a domain component
     * containing a terminating NULL.
     */
    sslPolicyPara.pwszServerName = battle_dot_net;
    CHECK_CHAIN_POLICY_STATUS(CERT_CHAIN_POLICY_SSL, NULL,
     nullTerminatedDomainComponentPolicyCheck, &oct2010, &policyPara);
}

static void check_msroot_policy(void)
{
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_MICROSOFT_ROOT, NULL,
     msRootPolicyCheck, &may2020, NULL);
}

static void testVerifyCertChainPolicy(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert;
    CERT_CHAIN_PARA chainPara = { sizeof(CERT_CHAIN_PARA), { 0 } };
    PCCERT_CHAIN_CONTEXT chain;
    CERT_CHAIN_POLICY_STATUS policyStatus = { 0 };
    CERT_CHAIN_POLICY_PARA policyPara = { 0 };

    if (!pCertVerifyCertificateChainPolicy)
    {
        win_skip("CertVerifyCertificateChainPolicy() is not available\n");
        return;
    }

    /* Crash
    ret = pCertVerifyCertificateChainPolicy(NULL, NULL, NULL, NULL);
    ret = pCertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, NULL, NULL,
     NULL);
    ret = pCertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, NULL,
     &chainPara, NULL);
     */
    SetLastError(0xdeadbeef);
    ret = pCertVerifyCertificateChainPolicy(NULL, NULL, NULL, &policyStatus);
    ok(!ret && GetLastError() == ERROR_FILE_NOT_FOUND,
     "Expected ERROR_FILE_NOT_FOUND, got %08lx\n", GetLastError());
    /* Crashes
    ret = pCertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, NULL, NULL,
     &policyStatus);
     */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, selfSignedCert,
     sizeof(selfSignedCert));
    pCertGetCertificateChain(NULL, cert, NULL, NULL, &chainPara, 0, NULL,
     &chain);
    /* Crash
    ret = pCertVerifyCertificateChainPolicy(NULL, chain, NULL, NULL);
    ret = pCertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain, NULL,
     NULL);
    ret = pCertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain,
     &chainPara, NULL);
     */
    /* Size of policy status is apparently ignored, as is pChainPolicyPara */
    ret = pCertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain, NULL,
     &policyStatus);
    ok(ret, "CertVerifyCertificateChainPolicy failed: %08lx\n", GetLastError());
    ok(policyStatus.dwError == CERT_E_UNTRUSTEDROOT ||
        policyStatus.dwError == TRUST_E_CERT_SIGNATURE, /* win7 + win8 */
        "Expected CERT_E_UNTRUSTEDROOT or TRUST_E_CERT_SIGNATURE, got %08lx\n", policyStatus.dwError);
    ok(policyStatus.lChainIndex == 0 && policyStatus.lElementIndex == 0,
     "Expected both indexes 0, got %ld, %ld\n", policyStatus.lChainIndex,
     policyStatus.lElementIndex);
    ret = pCertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain,
     &policyPara, &policyStatus);
    ok(ret, "CertVerifyCertificateChainPolicy failed: %08lx\n", GetLastError());
    ok(policyStatus.dwError == CERT_E_UNTRUSTEDROOT ||
        policyStatus.dwError == TRUST_E_CERT_SIGNATURE, /* win7 + win8 */
        "Expected CERT_E_UNTRUSTEDROOT or TRUST_E_CERT_SIGNATURE, got %08lx\n", policyStatus.dwError);
    ok(policyStatus.lChainIndex == 0 && policyStatus.lElementIndex == 0,
     "Expected both indexes 0, got %ld, %ld\n", policyStatus.lChainIndex,
     policyStatus.lElementIndex);
    pCertFreeCertificateChain(chain);
    CertFreeCertificateContext(cert);

    check_base_policy();
    check_ssl_policy();
    check_authenticode_policy();
    CHECK_CHAIN_POLICY_STATUS_ARRAY(CERT_CHAIN_POLICY_BASIC_CONSTRAINTS, NULL,
     basicConstraintsPolicyCheck, &oct2007, NULL);
    check_msroot_policy();
}

START_TEST(chain)
{
    HMODULE hCrypt32 = GetModuleHandleA("crypt32.dll");
    pCertCreateCertificateChainEngine = (void*)GetProcAddress(hCrypt32, "CertCreateCertificateChainEngine");
    pCertGetCertificateChain = (void*)GetProcAddress(hCrypt32, "CertGetCertificateChain");
    pCertFreeCertificateChain = (void*)GetProcAddress(hCrypt32, "CertFreeCertificateChain");
    pCertFreeCertificateChainEngine = (void*)GetProcAddress(hCrypt32, "CertFreeCertificateChainEngine");
    pCertVerifyCertificateChainPolicy = (void*)GetProcAddress(hCrypt32, "CertVerifyCertificateChainPolicy");

    testCreateCertChainEngine();
    if (!pCertGetCertificateChain || !pCertFreeCertificateChain)
    {
        win_skip("Cert*CertificateChain functions not available\n");
    }
    else
    {
        testVerifyCertChainPolicy();
        testGetCertChain();
        test_CERT_CHAIN_PARA_cbSize();
    }
}
