/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <fstream>
#include <iostream>

#include <boost/lexical_cast.hpp>
#include <boost/tokenizer.hpp>

#include <Wt/WApplication>
#include <Wt/WDatePicker>
#include <Wt/WCalendar>
#include <Wt/WEnvironment>
#include <Wt/WIconPair>
#include <Wt/WImage>
#include <Wt/WLineEdit>
#include <Wt/WMenu>
#include <Wt/WPushButton>
#include <Wt/WStackedWidget>
#include <Wt/WTabWidget>
#include <Wt/WTable>
#include <Wt/WTableCell>
#include <Wt/WText>
#include <Wt/WTreeNode>
#include <Wt/WViewWidget>

#include "Home.h"
#include "TreeListExample.h"

/*
 * A utility container widget which defers creation of its single
 * child widget until the container is loaded (which is done on-demand
 * by a WMenu). The constructor takes the create function for the
 * widget as a parameter.
 *
 * We use this to defer widget creation until needed, which is used
 * for the Treelist example tab.
 */
template <typename Function>
class DeferredWidget : public WContainerWidget
{
public:
  DeferredWidget(Function f)
    : f_(f) { }

private:
  void load() {
    addWidget(f_());
  }

  Function f_;
};

template <typename Function>
DeferredWidget<Function> *deferCreate(Function f)
{
  return new DeferredWidget<Function>(f);
}

/* Shortcut for a <div id=""> */
class Div : public WContainerWidget
{
public:
  Div(WContainerWidget *parent, const std::string& id)
    : WContainerWidget(parent)
  {
    setId(id);
  }
};

Home::Home(WContainerWidget *parent)
  : WContainerWidget(parent)
{
  Div *topWrapper = new Div(this, "top_wrapper");
  Div *topContent = new Div(topWrapper, "top_content");
  new Div(topContent, "top_languages");

  /*
   * Later add languages
   * <a href="#">en</a> <a href="#">nl</a> 
   */

  WText *topWt = new WText(tr("top_wt"), topContent);
  topWt->setInline(false);
  topWt->setId("top_wt");

  WText *bannerWt = new WText(tr("banner_wrapper"), this);
  bannerWt->setId("banner_wrapper");

  Div *mainWrapper = new Div(this, "main_wrapper");
  Div *mainContent = new Div(mainWrapper, "main_content");
  Div *mainMenu = new Div(mainContent, "main_menu");

  WStackedWidget *contents = new WStackedWidget();
  contents->setId("main_page");

  WMenu *menu = new WMenu(contents, Vertical, mainMenu);
  menu->setRenderAsList(true);
  menu->enableBrowserHistory("main");

  menu->addItem("Introduction", introduction());
  menu->addItem("News", deferCreate(boost::bind(&Home::news, this)),
		WMenuItem::PreLoading);
  menu->addItem("Features", wrapViewOrDefer(&Home::features),
		WMenuItem::PreLoading);
  menu->addItem("Documentation", wrapViewOrDefer(&Home::documentation),
		WMenuItem::PreLoading);

  std::string s = wApp->state("main");
  wApp->setState("main",
		 boost::lexical_cast<std::string>(menu->items().size()));

  menu->addItem("Examples", examples(), WMenuItem::PreLoading);
  menu->addItem("Download", deferCreate(boost::bind(&Home::download, this)),
		WMenuItem::PreLoading);
  menu->addItem("Community", wrapViewOrDefer(&Home::community),
		WMenuItem::PreLoading);

  wApp->setState("main", s);

  changeTitle(menu->currentItem());

  menu->itemSelectRendered.connect(SLOT(this, Home::changeTitle));

  sideBarContent_ = new WContainerWidget(mainMenu);

  mainContent->addWidget(contents);
  WContainerWidget *clearAll = new WContainerWidget(mainContent);
  clearAll->setStyleClass("clearall");

  WText *footerWrapper = new WText(tr("footer_wrapper"), this);
  footerWrapper->setId("footer_wrapper");
}

void Home::changeTitle(WMenuItem *item)
{
  wApp->setTitle(L"Wt, C++ Web Toolkit - " + item->text().value());
}

WWidget *Home::introduction()
{
  return new WText(tr("home.intro"));
}

void Home::refresh()
{
  readNews(recentNews_, "latest-news.txt");
  readNews(historicalNews_, "historical-news.txt");
  readReleases(releases_, "releases.txt");

  WContainerWidget::refresh();
}

WWidget *Home::news()
{
  WContainerWidget *result = new WContainerWidget();

  result->addWidget(new WText("<h3><span>News</span></h3>"));

  result->addWidget(new WText("<h4><span>Latest News</span></h4>"));
  recentNews_ = new WTable();
  readNews(recentNews_, "latest-news.txt");
  result->addWidget(recentNews_);

  result->addWidget(new WText("<h4><span>Historical News</span></h4>"));
  historicalNews_ = new WTable();
  readNews(historicalNews_, "historical-news.txt");
  result->addWidget(historicalNews_);

  return result;
}

WWidget *Home::status()
{
  return new WText(tr("home.status"));
}

WWidget *Home::features()
{
  return new WText(tr("home.features"));
}

WWidget *Home::documentation()
{
  return new WText(tr("home.documentation"));
}

WWidget *Home::helloWorldExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.hello"), result);

  WTreeNode *tree = makeTreeMap("Hello world", 0);
  makeTreeFile("hello.C", tree);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::homepageExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.wt"), result);

  WTreeNode *tree = makeTreeMap("Wt Homepage", 0);
  WTreeNode *home = makeTreeMap("class Home", tree);
  makeTreeFile("Home.h", home);
  makeTreeFile("Home.C", home);
  WTreeNode *treeexample = makeTreeMap("class TreeListExample", tree);
  makeTreeFile("TreeListExample.h", treeexample);
  makeTreeFile("TreeListExample.C", treeexample);
  makeTreeFile("wt-home.xml", tree);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::chartExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.chart"), result);

  WTreeNode *tree = makeTreeMap("Chart example", 0);
  WTreeNode *chartsExample = makeTreeMap("class ChartsExample", tree);
  makeTreeFile("ChartsExample.h", chartsExample);
  makeTreeFile("ChartsExample.C", chartsExample);
  WTreeNode *chartConfig = makeTreeMap("class ChartConfig", tree);
  makeTreeFile("ChartConfig.h", chartConfig);
  makeTreeFile("ChartConfig.C", chartConfig);
  WTreeNode *panelList = makeTreeMap("class PanelList", tree);
  makeTreeFile("PanelList.h", panelList);
  makeTreeFile("PanelList.C", panelList);
  makeTreeFile("CsvUtil.C", tree);
  makeTreeFile("charts.xml", tree);
  makeTreeFile("charts.css", tree);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::treelistExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.treelist"), result);
  new TreeListExample(result);
  new WText(tr("home.examples.treelist-remarks"), result);

  return result;
}

WWidget *Home::hangmanExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.hangman"), result);

  WTreeNode *tree = makeTreeMap("Hangman game", 0);
  tree->setLoadPolicy(WTreeNode::PreLoading);

  WTreeNode *widgets = makeTreeMap("Widgets", tree);
  WTreeNode *loginwidget = makeTreeMap("class LoginWidget", widgets);
  makeTreeFile("LoginWidget.h", loginwidget);
  makeTreeFile("LoginWidget.C", loginwidget);
  WTreeNode *hangmanwidget = makeTreeMap("class HangmanWidget", widgets);
  makeTreeFile("HangmanWidget.h", hangmanwidget);
  makeTreeFile("HangmanWidget.C", hangmanwidget);
  WTreeNode *highscoreswidget = makeTreeMap("class HighScoresWidget", widgets);
  makeTreeFile("HighScoresWidget.h", highscoreswidget);
  makeTreeFile("HighScoresWidget.C", highscoreswidget);
  WTreeNode *hangmangame = makeTreeMap("class HangmanGame", widgets);
  makeTreeFile("HangmanGame.h", hangmangame);
  makeTreeFile("HangmanGame.C", hangmangame);
  WTreeNode *other = makeTreeMap("Other", tree);
  WTreeNode *hangmandb = makeTreeMap("class HangmanDb", other);
  makeTreeFile("HangmanDb.h", hangmandb);
  makeTreeFile("HangmanDb.C", hangmandb);
  WTreeNode *dictionary = makeTreeMap("class Dictionary", other);
  makeTreeFile("Dictionary.h", dictionary);
  makeTreeFile("Dictionary.C", dictionary);
  makeTreeFile("hangman.C", other);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::composerExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.composer"), result);

  WTreeNode *tree = makeTreeMap("Mail composer example", 0);

  WTreeNode *classMap;
  classMap = makeTreeMap("class AddresseeEdit", tree);
  makeTreeFile("AddresseeEdit.h", classMap);
  makeTreeFile("AddresseeEdit.C", classMap);
  classMap = makeTreeMap("class AttachmentEdit", tree);
  makeTreeFile("AttachmentEdit.h", classMap);
  makeTreeFile("AttachmentEdit.C", classMap);
  classMap = makeTreeMap("class ComposeExample", tree);
  makeTreeFile("ComposeExample.h", classMap);
  makeTreeFile("ComposeExample.C", classMap);
  classMap = makeTreeMap("class Composer", tree);
  makeTreeFile("Composer.h", classMap);
  makeTreeFile("Composer.C", classMap);
  classMap = makeTreeMap("class ContactSuggestions", tree);
  makeTreeFile("ContactSuggestions.h", classMap);
  makeTreeFile("ContactSuggestions.C", classMap);
  classMap = makeTreeMap("class Label", tree);
  makeTreeFile("Label.h", classMap);
  makeTreeFile("Label.C", classMap);
  classMap = makeTreeMap("class Option", tree);
  makeTreeFile("Option.h", classMap);
  makeTreeFile("Option.C", classMap);
  classMap = makeTreeMap("class OptionList", tree);
  makeTreeFile("OptionList.h", classMap);
  makeTreeFile("OptionList.C", classMap);
  makeTreeFile("Contact.h", tree);
  makeTreeFile("Attachment.h", tree);
  makeTreeFile("composer.xml", tree);
  makeTreeFile("composer.css", tree);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::dragdropExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.dragdrop"), result);

  WTreeNode *tree = makeTreeMap("DragDrop example", 0);

  WTreeNode *classMap;
  classMap = makeTreeMap("class Character", tree);
  makeTreeFile("Character.h", classMap);
  makeTreeFile("Character.C", classMap);
  makeTreeFile("DragExample.C", tree);
  makeTreeFile("dragdrop.css", tree);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::chatExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.chat"), result);

  WTreeNode *tree = makeTreeMap("Chat example", 0);

  WTreeNode *classMap;
  classMap = makeTreeMap("class SimpleChatWidget", tree);
  makeTreeFile("SimpleChatWidget.h", classMap);
  makeTreeFile("SimpleChatWidget.C", classMap);
  classMap = makeTreeMap("class SimpleChatServer", tree);
  makeTreeFile("SimpleChatServer.h", classMap);
  makeTreeFile("SimpleChatServer.C", classMap);
  makeTreeFile("simpleChat.C", tree);
  makeTreeFile("simplechat.css", tree);
  makeTreeFile("simplechat.xml", tree);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::fileExplorerExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.fileexplorer"), result);

  WTreeNode *tree = makeTreeMap("File explorer example", 0);

  WTreeNode *classMap;
  classMap = makeTreeMap("class FileTreeTableNode", tree);
  makeTreeFile("FileTreeTableNode.h", classMap);
  makeTreeFile("FileTreeTableNode.C", classMap);
  classMap = makeTreeMap("class FileTreeTable", tree);
  makeTreeFile("FileTreeTable.h", classMap);
  makeTreeFile("FileTreeTable.C", classMap);
  makeTreeFile("FileTreeExample.C", tree);
  makeTreeFile("filetree.css", tree);

  tree->expand();

  result->addWidget(tree);

  return result;
}

WWidget *Home::calendarExample()
{
  WContainerWidget *result = new WContainerWidget();

  new WText(tr("home.examples.calendar"), result);
  
  new WText("<p>A field for editing a date in conjunction "
	    "with a WDatePicker:</p>", result);

  WTable *tab = new WTable(result);
  new WText("Please enter your birth date: ", tab->elementAt(0, 0));
  WLineEdit *dateEdit = new WLineEdit(tab->elementAt(0, 1));
  dateEdit->setMargin(5, Right);

  new WDatePicker(new WPushButton("..."), dateEdit,
		  false, tab->elementAt(0, 1));
  tab->elementAt(0, 0)->setPadding(8);
  tab->elementAt(0, 1)->setPadding(8);

  new WText("<p>A plain calendar:</p>", result);
  WCalendar *cal = new WCalendar(false, result);
  cal->setMargin(8);

  new WText("<p>A calendar with multiple selection:</p>", result);
  cal = new WCalendar(false, result);
  cal->setMultipleSelection(true);
  cal->setMargin(8);


  return result;
}

WWidget *Home::wrapViewOrDefer(WWidget *(Home::*createWidget)())
{
  /*
   * We can only create a view if we have javascript for the client-side
   * tree manipulation -- otherwise we require server-side event handling
   * which is not possible with a view since the server-side widgets do
   * not exist. Otherwise, all we can do to avoid unnecessary server-side
   * resources is deferring creation until load time.
   */
  if (!wApp->environment().agentIEMobile()
      && wApp->environment().javaScript())
    return makeStaticModel(boost::bind(createWidget, this));
  else
    return deferCreate(boost::bind(createWidget, this));
}

WWidget *Home::examples()
{
  WContainerWidget *result = new WContainerWidget();

  result->addWidget(new WText(tr("home.examples")));

  WTabWidget *exampleTabs = new WTabWidget(result);
  exampleTabs->enableBrowserHistory("example");

  /*
   * The following code is functionally equivalent to:
   *
   *   exampleTabs->addTab(helloWorldExample(), "Hello world");
   *
   * However, we optimize here for memory consumption (it is a homepage
   * after all, and we hope to be slashdotted some day)
   *
   * Therefore, we wrap all the static content (including the tree
   * widgets), into WViewWidgets with static models. In this way the
   * widgets are not actually stored in memory on the server.
   *
   * For the tree list example (for which we cannot use a view with a
   * static model, since we allow the tree to be manipulated) we use
   * the defer utility function to defer its creation until it is
   * loaded.
   */
  exampleTabs->addTab(wrapViewOrDefer(&Home::helloWorldExample),
		      "Hello world");
  exampleTabs->addTab(wrapViewOrDefer(&Home::chartExample),
                      "Charts");
  exampleTabs->addTab(wrapViewOrDefer(&Home::homepageExample),
		      "Wt homepage");
  exampleTabs->addTab(deferCreate(boost::bind(&Home::treelistExample, this)),
		      "Treelist");
  exampleTabs->addTab(wrapViewOrDefer(&Home::hangmanExample),
		      "Hangman");
  exampleTabs->addTab(wrapViewOrDefer(&Home::chatExample),
		      "Chat");
  exampleTabs->addTab(wrapViewOrDefer(&Home::composerExample),
		      "Mail composer");
  exampleTabs->addTab(wrapViewOrDefer(&Home::dragdropExample),
		      "Drag &amp; Drop");
  exampleTabs->addTab(wrapViewOrDefer(&Home::fileExplorerExample),
		      "File explorer");
  exampleTabs->addTab(deferCreate(boost::bind(&Home::calendarExample, this)),
		      "Calendar");
  
  return result;
}

WWidget *Home::download()
{
  WContainerWidget *result = new WContainerWidget();
  result->addWidget(new WText("<h3><span>Download</span></h3>"));
  result->addWidget(new WText(tr("home.download.license")));
  result->addWidget(new WText(tr("home.download.requirements")));
  result->addWidget(new WText(tr("home.download.cvs")));
  result->addWidget(new WText("<h4>Available packages</h4>"));

  releases_ = new WTable();
  readReleases(releases_, "releases.txt");
  result->addWidget(releases_);

  result->addWidget
    (new WText("<p>Older releases are still available at "
	       + href("http://sourceforge.net/project/showfiles.php?"
		      "group_id=153710#files",
		      "sourceforge.net")
	       + "</p>"));

  return result;
}

std::string Home::href(const std::string url, const std::string description)
{
  return "<a href=\"" + url + "\" target=\"_blank\">" + description + "</a>";
}

WWidget *Home::community()
{
  return new WText(tr("home.community"));
}

void Home::readNews(WTable *newsTable, const std::string newsfile)
{
  std::ifstream f(newsfile.c_str());

  newsTable->clear();

  int row = 0;

  while (f) {
    std::string line;
    getline(f, line);

    if (f) {
      typedef boost::tokenizer<boost::escaped_list_separator<char> >
	CsvTokenizer;
      CsvTokenizer tok(line);

      CsvTokenizer::iterator i=tok.begin();

      newsTable->elementAt(row, 0)->
	addWidget(new WText("<p><b>" + *i + "</b></p>"));
      newsTable->elementAt(row, 0)
	->setContentAlignment(AlignCenter | AlignTop);
      newsTable->elementAt(row, 0)
	->resize(WLength(16, WLength::FontEx), WLength());
      newsTable
	->elementAt(row, 1)->addWidget(new WText("<p>" + *(++i) + "</p>"));

      ++row;
    }
  }
}

void Home::readReleases(WTable *releaseTable, const std::string releasefile)
{
  std::ifstream f(releasefile.c_str());

  releaseTable->clear();

  releaseTable->elementAt(0, 0)->addWidget(new WText("<b>Version</b>"));
  releaseTable->elementAt(0, 1)->addWidget(new WText("<b>Date</b>"));
  releaseTable->elementAt(0, 2)->addWidget(new WText("<b>Description</b>"));

  releaseTable->elementAt(0, 0)->resize(WLength(10, WLength::FontEx),
					WLength());
  releaseTable->elementAt(0, 1)->resize(WLength(15, WLength::FontEx),
					WLength());

  int row = 1;

  while (f) {
    std::string line;
    getline(f, line);

    if (f) {
      typedef boost::tokenizer<boost::escaped_list_separator<char> >
	CsvTokenizer;
      CsvTokenizer tok(line);

      CsvTokenizer::iterator i=tok.begin();

      std::string version = *i;
      releaseTable->elementAt(row, 0)->addWidget
	(new WText(href("http://prdownloads.sourceforge.net/witty/wt-"
			+ version + ".tar.gz?download", "Wt " + version)));
      releaseTable->elementAt(row, 1)->addWidget(new WText(*(++i)));
      releaseTable->elementAt(row, 2)->addWidget(new WText(*(++i)));

      ++row;
    }
  }
}

WTreeNode *Home::makeTreeMap(const std::string name, WTreeNode *parent)
{
  WIconPair *labelIcon
    = new WIconPair("icons/yellow-folder-closed.png",
		    "icons/yellow-folder-open.png", false);

  WTreeNode *node = new WTreeNode(name, labelIcon, parent);
  node->label()->setFormatting(WText::PlainFormatting);

  if (!parent) {
    node->setImagePack("icons/");
    node->expand();
    node->setLoadPolicy(WTreeNode::NextLevelLoading);
  }

  return node;
}

WTreeNode *Home::makeTreeFile(const std::string name, WTreeNode *parent)
{
  WIconPair *labelIcon
    = new WIconPair("icons/document.png",
		    "icons/yellow-folder-open.png", false);

  return new WTreeNode("<a href=\"" + name + "\" target=\"_blank\">"
		       + name + "</a>", labelIcon, parent);
}

WApplication *createApplication(const WEnvironment& env)
{
  WApplication *app = new WApplication(env);

  app->messageResourceBundle().use("wt-home", false);
  app->useStyleSheet("images/wt.css");
  app->useStyleSheet("images/wt_ie.css", "lt IE 7");
  app->useStyleSheet("home.css");
  app->setTitle("Wt, C++ Web Toolkit");

  new Home(app->root());
  return app;
}

int main(int argc, char **argv)
{
  return WRun(argc, argv, &createApplication);
}

