/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */
#include "Wt/WAnchor"
#include "Wt/WApplication"
#include "Wt/WEnvironment"
#include "Wt/WImage"
#include "Wt/WText"
#include "DomElement.h"

namespace Wt {

WAnchor::WAnchor(WContainerWidget *parent)
  : WContainerWidget(parent),
    resource_(0),
    text_(0),
    image_(0),
    target_(TargetSelf),
    refChanged_(false),
    targetChanged_(false)
{
  setInline(true);
}

WAnchor::WAnchor(const std::string& ref, WContainerWidget *parent)
  : WContainerWidget(parent),
    ref_(ref),
    resource_(0),
    text_(0),
    image_(0),
    target_(TargetSelf),
    refChanged_(false),
    targetChanged_(false)
{
  setInline(true);
}

WAnchor::WAnchor(WResource *resource, WContainerWidget *parent)
  : WContainerWidget(parent),
    resource_(0),
    text_(0),
    image_(0),
    target_(TargetSelf),
    refChanged_(false),
    targetChanged_(false)
{
  setInline(true);
  setResource(resource);
}

WAnchor::WAnchor(const std::string& ref, const WString& text,
		 WContainerWidget *parent)
  : WContainerWidget(parent),
    ref_(ref),
    resource_(0),
    text_(0),
    image_(0),
    target_(TargetSelf),
    refChanged_(false),
    targetChanged_(false)
{ 
  setInline(true);
  text_ = new WText(text, this);
}

WAnchor::WAnchor(WResource *resource, const WString& text,
		 WContainerWidget *parent)
  : WContainerWidget(parent),
    resource_(0),
    text_(0),
    image_(0),
    target_(TargetSelf),
    refChanged_(false),
    targetChanged_(false)
{ 
  setInline(true);
  text_ = new WText(text, this);
  setResource(resource);
}

WAnchor::WAnchor(const std::string& ref, WImage *image,
		 WContainerWidget *parent)
  : WContainerWidget(parent),
    ref_(ref),
    resource_(0),
    text_(0),
    image_(0),
    target_(TargetSelf),
    refChanged_(false),
    targetChanged_(false)
{ 
  setInline(true);
  image_ = image;

  if (image_)
    addWidget(image_);
}

WAnchor::WAnchor(WResource *resource, WImage *image,
		 WContainerWidget *parent)
  : WContainerWidget(parent),
    resource_(0),
    text_(0),
    image_(0),
    target_(TargetSelf),
    refChanged_(false),
    targetChanged_(false)
{ 
  setInline(true);
  image_ = image;

  if (image_)
    addWidget(image_);

  setResource(resource);
}

void WAnchor::setRef(const std::string& ref)
{
  ref_ = ref;
  refChanged_ = true;

  repaint(RepaintPropertyIEMobile);
}

void WAnchor::setResource(WResource *resource)
{
  resource_ = resource;

  if (resource_) {
    resource_->dataChanged.connect(SLOT(this, WAnchor::resourceChanged));
    setRef(resource_->generateUrl());
  }
}

void WAnchor::setTarget(AnchorTarget target)
{
  if (target_ != target) {
    target_ = target;
    targetChanged_ = true;
  }
}

const WString& WAnchor::text() const
{
  static WString empty("");
  if (text_)
    return text_->text();
  else
    return empty;
}

void WAnchor::setText(const WString& text)
{
  if (!text_) {
    text_ = new WText(text, this);
  } else
    if (!text.empty())
      text_->setText(text);
    else {
      delete text_;
      text_ = 0;
    }
}

void WAnchor::setImage(WImage *image)
{
  if (image_)
    delete image_;
  image_ = image;

  if (image_)
    addWidget(image_);
}

void WAnchor::resourceChanged()
{
  setRef(resource_->generateUrl());
}

void WAnchor::updateDom(DomElement& element, bool all)
{
  if (refChanged_ || all) {
    element.setAttribute("href", fixRelativeUrl(ref_));
    refChanged_ = false;
  }

  if (targetChanged_ || all) {
    switch (target_) {
    case TargetSelf:
      if (WApplication::instance()->environment().inFrame())
	element.setAttribute("target", "_parent");
      else if (!all)
	element.setAttribute("target", "_self");
      break;
    case TargetThisWindow:
      element.setAttribute("target", "_top");
      break;
    case TargetNewWindow:
      element.setAttribute("target", "_blank");
    }
    targetChanged_ = false;
  }

  WContainerWidget::updateDom(element, all);
}

DomElementType WAnchor::domElementType() const
{
  return DomElement_A;
}

}
