/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <boost/lexical_cast.hpp>

#include "Wt/WApplication"
#include "Wt/WContainerWidget"
#include "Wt/WWebWidget"
#include "Wt/WStringUtil"

#include "DomElement.h"
#include "WebController.h"
#include "Configuration.h"
#include "WebRenderer.h"
#include "WebRequest.h"
#include "WebSession.h"
#include "FileServe.h"
#include "WtRandom.h"
#include "EscapeOStream.h"
#ifdef WIN32
#include <process.h> // for getpid()
#ifdef min
#undef min
#endif
#endif

namespace skeletons {
  extern const char *Boot_html;
  extern const char *Ajax_html;
  extern const char *Plain_html;
  extern const char *JsNoAjax_html;
  extern const char *Wt_js;
  extern const char *CommAjax_js;
  extern const char *CommScript_js;
}

namespace Wt {

const int MESSAGE_COUNTER_SIZE = 5;

WebRenderer::WebRenderer(WebSession& session)
  : session_(session),
    visibleOnly_(true),
    twoPhaseThreshold_(5000),
    learning_(false)
{ }

void WebRenderer::setTwoPhaseThreshold(int bytes)
{
  twoPhaseThreshold_ = bytes;
}

//#define DEBUG_RENDER

void WebRenderer::needUpdate(WWebWidget *w)
{
  if (session_.env().ajax()) {
    UpdateMap& usedMap = updateMap_;
#ifdef DEBUG_RENDER
    std::cerr << "needUpdate: " << w->formName() << " (" << typeid(*w).name()
		  << ")" << std::endl;
#endif //DEBUG_RENDER

    usedMap.insert(w);
  }
}

void WebRenderer::doneUpdate(WWebWidget *w)
{
#ifdef DEBUG_RENDER
    std::cerr << "doneUpdate: " << w->formName() << " (" << typeid(*w).name()
		  << ")" << std::endl;
#endif //DEBUG_RENDER

  UpdateMap& usedMap = updateMap_;

  UpdateMap::iterator i = usedMap.find(w);
  if (i != usedMap.end())
    usedMap.erase(i);
}

const std::vector<WObject *>& WebRenderer::formObjects() const
{
  return currentFormObjects_;
}

void WebRenderer::saveChanges()
{
  collectJS(&collectedChanges_);
}

void WebRenderer::discardChanges()
{
  collectJS(0);
}

void WebRenderer::letReload(WebRequest& request)
{
  setHeaders(request, "text/plain; charset=UTF-8");

  if (   request.headerValue("User-Agent").find("Opera") != std::string::npos
      || request.headerValue("User-Agent").find("MSIE") != std::string::npos)
    request.out() << "window." << (session_.env().inFrame() ? "parent." : "")
		  << "location.reload(true);";
  else
    streamRedirectJS(request.out(), session_.bookmarkUrl());
}

void WebRenderer::streamRedirectJS(std::ostream& out,
				   const std::string& redirect)
{
  out << "if(document.images) window."
      << (session_.env().inFrame() ? "parent." : "")
      << "location.replace('" << redirect << "'); else window."
      << (session_.env().inFrame() ? "parent." : "")
      << "location.href='" + redirect + "';"; 
}

void WebRenderer::serveMainWidget(WebRequest& request,
				  ResponseType responseType)
{
  switch (responseType) {
  case UpdateResponse:
    serveJavaScriptUpdate(request);
    break;
  case FullResponse:
    switch (session_.type()) {
    case WebSession::Application:
      serveMainpage(request);
      break;
    case WebSession::WidgetSet:
      serveWidgetSet(request);
    }
  }
}

void WebRenderer::serveBootstrap(WebRequest& request)
{
  FileServe boot(skeletons::Boot_html);
  boot.setVar("RELATIVE_URL", session_.mostRelativeUrl());
  boot.setVar("RANDOMSEED",
    boost::lexical_cast<std::string>(WtRandom::getUnsigned() + getpid()));
  boot.setVar("RELOAD_IS_NEWSESSION",
	      WebController::conf().reloadIsNewSession()
	      ? "true" : "false");
  boot.setVar("USE_COOKIES",
	      WebController::conf().sessionTracking()
	      == Configuration::CookiesURL
	      ? "true" : "false");

  bool needsNoScriptWrap
    =  request.headerValue("User-Agent").find("Safari") != std::string::npos
    || request.headerValue("User-Agent").find("Opera") != std::string::npos
    || request.headerValue("User-Agent").find("Konqueror") != std::string::npos;

if (needsNoScriptWrap) {
    boot.setVar("NOSCRIPT_START", "<noscript>");
    boot.setVar("NOSCRIPT_END", "</noscript>");
  } else {
    boot.setVar("NOSCRIPT_START", "");
    boot.setVar("NOSCRIPT_END", "");
  }

  request.addHeader("Cache-Control", "max-age=0");
  request.setContentType("text/html");
  boot.stream(request.out());
}

void WebRenderer::serveError(WebRequest& request, const std::exception& e,
			     ResponseType responseType)
{
  serveError(request, std::string(e.what()), responseType);
}

void WebRenderer::serveError(WebRequest& request, const std::string& message,
			     ResponseType responseType)
{
  if (responseType == FullResponse
      && session_.type() == WebSession::Application) {
    request.setContentType("text/html");
    request.out()
      << "<title>Error occurred.</title>"
      << (session_.env().inFrame()
	  ? "<script>window.parent.document.title='Error occured.';</script>"
	  : "")
      << "<h2>Error occurred.</h2>"
      << WWebWidget::escapeText(WString(message), true).toUTF8()
      << std::endl;    
  } else {
    collectedChanges_ << "alert(";
    DomElement::jsStringLiteral(collectedChanges_,
				"Error occurred:\n" + message, '\'');
    collectedChanges_ << ");";
  }
}

void WebRenderer::setCookie(const std::string name, const std::string value,
			    int maxAge, const std::string domain,
			    const std::string path)
{
  cookiesToSet_.push_back(Cookie(name, value, path, domain, maxAge));
}

static void replace(std::string& s, char c, std::string r)
{
  std::string::size_type p = 0;

  while ((p = s.find(c, p)) != std::string::npos) {
    s.replace(p, 1, r);
    p += r.length();
  }
}

void WebRenderer::setHeaders(WebRequest& request, const std::string mimeType)
{
  std::string cookies;

  for (unsigned i = 0; i < cookiesToSet_.size(); ++i) {
    std::string value = cookiesToSet_[i].value;
    replace(value, '"', "\"");

    cookies += cookiesToSet_[i].name + "=\"" + value + "\";Version=\"1\";";
    if (cookiesToSet_[i].maxAge != -1)
      cookies += "Max-Age=\""
	+ boost::lexical_cast<std::string>(cookiesToSet_[i].maxAge) + "\";";
    if (!cookiesToSet_[i].domain.empty())
      cookies += "Domain=\"" + cookiesToSet_[i].domain + "\";";
    if (!cookiesToSet_[i].path.empty())
      cookies += "Path=\"" + cookiesToSet_[i].path + "\";";
  }
  cookiesToSet_.clear();

  if (!cookies.empty())
    request.addHeader("Set-Cookie", cookies);

  request.setContentType(mimeType);
}

void WebRenderer::serveJavaScriptUpdate(WebRequest& request)
{
  setHeaders(request, "text/plain; charset=UTF-8");
  streamJavaScriptUpdate(request.out(), request.id(), true);
}

void WebRenderer::streamJavaScriptUpdate(std::ostream& out, int id,
					 bool doTwoPhaze)
{
  WApplication *app = session_.app();

  std::string redirect = session_.getRedirect();

  if (!redirect.empty()) {
    streamRedirectJS(out, redirect);
    return;
  }

  if (   doTwoPhaze
      && !app->isQuited()
      && session_.state() != WebSession::ExpectLoad)
    visibleOnly_ = true;

  out << collectedChanges_.str();
  collectedChanges_.str("");

  declareInlineStyles(out, app, false);
  loadStyleSheets(out, app);
  loadScriptLibraries(out, app, true);

  out << app->newBeforeLoadJavaScript();

  if (app->autoJavaScriptChanged_) {
    out << app->javaScriptClass()
		  << ".private.autoJavaScript=function(){"
		  << app->autoJavaScript_ << "};";
    app->autoJavaScriptChanged_ = false;
  }

  if (app->domRoot2_)
    app->domRoot2_->rootAsJavaScript(app, out, false);

  collectJavaScriptUpdate(out);

  if (visibleOnly_) {
    visibleOnly_ = false;

    collectJavaScriptUpdate(collectedChanges_);

    if (collectedChanges_.rdbuf()->in_avail() < (int)twoPhaseThreshold_) {
      out << collectedChanges_.str();
      collectedChanges_.str("");
    } else {
      out << app->javaScriptClass()
		    << ".private.update(null, 'none', null, false);";
    }
  }

  app->domRoot_->doneRerender();
  if (app->domRoot2_)
    app->domRoot2_->doneRerender();

  if (app->ajaxMethod() == WApplication::DynamicScriptTag)
    out << app->javaScriptClass() << ".private.updateDone(" << id << ");";

  loadScriptLibraries(out, app, false);
}

void WebRenderer::streamCommJs(WApplication *app, std::ostream& out)
{
  FileServe js(app->ajaxMethod() == WApplication::XMLHttpRequest
	       ? skeletons::CommAjax_js
	       : skeletons::CommScript_js);

  js.setVar("APP_CLASS", app->javaScriptClass());
  js.setVar("CLOSE_CONNECTION",
	    WebController::conf().serverType()
	    == Configuration::WtHttpdServer ? "false" : "true");

  js.stream(out);
}

void WebRenderer::serveMainpage(WebRequest& request)
{
  WApplication *app = session_.app();

  std::string redirect = session_.getRedirect();

  if (!redirect.empty()) {
    request.setRedirect(redirect);
    return;
  }

  WWebWidget *mainWebWidget = app->domRoot_;

  mainWebWidget->prepareRerender();

  visibleOnly_ = true;

  /*
   * The element to render. This automatically creates loading stubs for
   * invisible widgets, which is excellent for both JavaScript and
   * non-JavaScript versions.
   */
  DomElement *mainElement = mainWebWidget->createSDomElement();

  collectedChanges_.str("");
  preLearnStateless(app);

  const bool xhtml = app->environment().contentType() == WEnvironment::XHTML1;

  std::string styleSheets;
  for (unsigned i = 0; i < app->styleSheets_.size(); ++i) {
    if (!app->styleSheets_[i].dependency.empty())
      styleSheets += "<!--[if " + app->styleSheets_[i].dependency
	+ "]>";
    styleSheets += "<link href='"
      + app->fixRelativeUrl(app->styleSheets_[i].uri) 
      + "' rel='stylesheet' type='text/css'" + (xhtml ? "/>" : ">");
    if (!app->styleSheets_[i].dependency.empty())
      styleSheets += "<![endif]-->";
  }
  app->styleSheetsAdded_ = 0;

  std::string scriptLibraries;
  for (unsigned i = 0; i < app->scriptLibraries_.size(); ++i) {
    scriptLibraries += "<script type=\"text/javascript\" src=\""
      + app->fixRelativeUrl(app->scriptLibraries_[i].uri)
      + "\"></script>";
  }
  app->scriptLibrariesAdded_ = 0;

  FileServe page(session_.env().ajax() ? skeletons::Ajax_html :
		 skeletons::Plain_html);

  page.setVar("XMLPREAMBLE", "");
  page.setVar("DOCTYPE", app->docType());

  if (xhtml) {
    page.setVar("HTMLATTRIBUTES",
		"xmlns=\"http://www.w3.org/1999/xhtml\""
		/*" xmlns:svg=\"http://www.w3.org/2000/svg\""*/);
    page.setVar("METACLOSE", "/>");
  } else {
    if (session_.env().agentIE())
      page.setVar("HTMLATTRIBUTES",
		  "xmlns:v=\"urn:schemas-microsoft-com:vml\""
		  " lang=\"en\" dir=\"ltr\"");
    else
      page.setVar("HTMLATTRIBUTES", "lang=\"en\" dir=\"ltr\"");
    page.setVar("METACLOSE", ">");
  }

  page.setVar("RELATIVE_URL", session_.mostRelativeUrl());
  page.setVar("STYLESHEET", app->styleSheet().cssText(true));
  page.setVar("STYLESHEETS", styleSheets);
  page.setVar("SCRIPTS", scriptLibraries);

  page.setVar("TITLE", app->title().toUTF8());
  page.setVar("APP_CLASS", app->javaScriptClass());
  app->titleChanged_ = false;

  std::string contentType = xhtml ? "application/xhtml+xml" : "text/html";

  contentType += "; charset=UTF-8";

  setHeaders(request, contentType);

  // Form objects, need in either case (Ajax or not)
  currentFormObjectsList_ = createFormObjectsList(app);

  if (page.contents() == skeletons::Ajax_html) {
    page.streamUntil(request.out(), "WTJS");

    FileServe js(skeletons::Wt_js);
    js.setVar("WT_CLASS", WT_CLASS);
    js.setVar("APP_CLASS", app->javaScriptClass());
    js.setVar("AUTO_JAVASCRIPT", app->autoJavaScript_);
    app->autoJavaScriptChanged_ = false;

    /*
     * In fact only Opera cannot use innerHTML in XHTML documents.
     * We could check for opera only since the workaround innerHTML is
     * substantially slower...
     */
    js.setVar("INNER_HTML", xhtml ? "false" : "true");

    std::string states;
    std::string historyKey;
    for (WApplication::StateMap::const_iterator i = app->state_.begin();
	 i != app->state_.end(); ++i) {
      if (!states.empty())
	states += ',';
      states += '"' + i->first + "\": \"" + i->second + '"';
    }

    js.setVar("STATES_INIT", "{" + states + "}");

    js.setVar("FORM_OBJECTS", '[' + currentFormObjectsList_ + ']');
    js.setVar("RELATIVE_URL", '"' + session_.mostRelativeUrl() + '"');
    js.setVar("KEEP_ALIVE",
	      boost::lexical_cast<std::string>
	      (WebController::conf().sessionTimeout() / 2));
    js.setVar("IN_FRAME", session_.env().inFrame() ? "true" : "false");
    js.setVar("INDICATOR_TIMEOUT", "500");
  
    std::string htmlJs = '{' + mainElement->javaScript() + '}';

    js.setVar("ONLOAD", app->afterLoadJavaScript()
	      + htmlJs + collectedChanges_.str());
    collectedChanges_.str("");

    app->statesChanged_.clear();

    js.stream(request.out());
    streamCommJs(app, request.out());
  }

  page.streamUntil(request.out(), "HTML");

  DomElement::TimeoutList timeouts;
  {
    EscapeOStream out(request.out());
    mainElement->asHTML(out, timeouts);
    delete mainElement;
  }

  std::stringstream onload;
  DomElement::createTimeoutJs(onload, timeouts, app);

  page.setVar("JAVASCRIPT",
	      onload.str() + app->beforeLoadJavaScript());

  int refresh = WebController::conf().sessionTimeout() / 3;
  for (unsigned i = 0; i < timeouts.size(); ++i)
    refresh = std::min(refresh, 1 + timeouts[i].msec/1000);
  if (app->isQuited())
    refresh = 100000; // ridiculously large
  page.setVar("REFRESH", boost::lexical_cast<std::string>(refresh));

  page.stream(request.out());

  visibleOnly_ = false;
}

void WebRenderer::serveWidgetSet(WebRequest& request)
{ 
  WApplication *app = session_.app();

  setHeaders(request, "text/javascript; charset=UTF-8");

  currentFormObjectsList_ = createFormObjectsList(app);

  FileServe script(skeletons::Wt_js);
  script.setVar("WT_CLASS", WT_CLASS);
  script.setVar("APP_CLASS", app->javaScriptClass());
  script.setVar("AUTO_JAVASCRIPT", app->autoJavaScript_);
  script.setVar("INNER_HTML", "true");
  script.setVar("STATES_INIT", "{}");
  script.setVar("FORM_OBJECTS", '[' + currentFormObjectsList_ + ']');
  script.setVar("RELATIVE_URL", '"' + session_.mostRelativeUrl() + '"');
  script.setVar("KEEP_ALIVE",
		boost::lexical_cast<std::string>
		(WebController::conf().sessionTimeout() / 2));
  script.setVar("IN_FRAME", "false");
  script.setVar("INDICATOR_TIMEOUT", "500");
  script.setVar("ONLOAD", "");
  script.stream(request.out());

  app->autoJavaScriptChanged_ = false;

  streamCommJs(app, request.out());

  declareInlineStyles(request.out(), app, true);

  app->styleSheetsAdded_ = app->styleSheets_.size();
  loadStyleSheets(request.out(), app);

  app->scriptLibrariesAdded_ = app->scriptLibraries_.size();
  loadScriptLibraries(request.out(), app, true);
 
  request.out() << std::endl << app->beforeLoadJavaScript();

  visibleOnly_ = true;

  /*
   * Render Root widgets (domRoot_ and children of domRoot2_) as
   * JavaScript
   */
  DomElement *c = app->domRoot_->webWidget()->createSDomElement();
  std::string cvar;
  {
    EscapeOStream sout(request.out()); 
    c->asJavaScript(sout, DomElement::Create);
    cvar = c->asJavaScript(sout, DomElement::Update);
  }
  delete c;

  request.out() << "document.body.insertBefore("
		<< cvar << ",document.body.firstChild);" << std::endl;

  app->domRoot2_->rootAsJavaScript(app, request.out(), true);

  collectedChanges_.str("");
  preLearnStateless(app);

  visibleOnly_ = false;

  request.out() << collectedChanges_.str();
  collectedChanges_.str("");

  request.out() << app->afterLoadJavaScript()
		<< app->javaScriptClass() << ".private.load();";

  loadScriptLibraries(request.out(), app, false);
}

void WebRenderer::loadScriptLibraries(std::ostream& out,
				      WApplication *app, bool start)
{
  int first = app->scriptLibraries_.size() - app->scriptLibrariesAdded_;

  if (start) {
    for (unsigned i = first; i < app->scriptLibraries_.size(); ++i) {
      std::string uri = app->fixRelativeUrl(app->scriptLibraries_[i].uri);

      out << app->scriptLibraries_[i].beforeLoadJS
	  << app->javaScriptClass()
	  << ".private.loadScript('" << uri << "',";
      DomElement::jsStringLiteral(out, app->scriptLibraries_[i].symbol, '\'');
      out << ");"
	  << app->javaScriptClass() << ".private.onJsLoad(\""
	  << uri << "\",function() {";
    }
  } else {
    for (unsigned i = first; i < app->scriptLibraries_.size(); ++i) {
      out << "});";
    }
    app->scriptLibrariesAdded_ = 0;
  }
}

void WebRenderer::declareInlineStyles(std::ostream& out, WApplication *app,
				      bool all)
{
  std::string styles = app->styleSheet().cssText(all);

  if (!styles.empty()) {
    out << app->javaScriptClass()
	<< ".private.addStyle(";
    DomElement::jsStringLiteral(out, styles, '\'');
    out << ");" << std::endl;
  }
}

void WebRenderer::loadStyleSheets(std::ostream& out, WApplication *app)
{
  int first = app->styleSheets_.size() - app->styleSheetsAdded_;

  for (unsigned i = first; i < app->styleSheets_.size(); ++i)
    out << app->javaScriptClass()
	<< ".private.addStyleSheet('"
	<< app->fixRelativeUrl(app->styleSheets_[i].uri) << "');" << std::endl;

  app->styleSheetsAdded_ = 0;
}

void WebRenderer::collectChanges(std::vector<DomElement *>& changes)
{
  WApplication *app = session_.app();

  std::multimap<int, WWebWidget *> depthOrder;

  for (UpdateMap::const_iterator i = updateMap_.begin();
       i != updateMap_.end(); ++i) {
    int depth = 1;

    WWidget *w = *i;
    for (; w->parent(); w = w->parent(), ++depth) ;

    if (w != app->domRoot_ && w != app->domRoot2_) {
      /*
      std::cerr << "ignoring: " << (*i)->formName()
                << " (" << typeid(**i).name()
                << ") " << w->formName()
                << " (" << typeid(*w).name()
                << ")" << std::endl; */
      // not in displayed widgets
      depth = 0;
    }

    depthOrder.insert(std::make_pair(depth, *i));
  }

  for (std::multimap<int, WWebWidget *>::const_iterator i = depthOrder.begin();
       i != depthOrder.end(); ++i) {
    UpdateMap::iterator j = updateMap_.find(i->second);
    if (j != updateMap_.end()) {
      WWebWidget *w = *j;

      // depth == 0: remove it from the update list
      if (i->first == 0) {
	w->propagateRenderOk();
	continue;
      }

      //std::cerr << learning_ << " " << loading_ 
      //          << " updating: " << w->formName() << std::endl;

#ifdef DEBUG_RENDER
        std::cerr << "updating: " << w->formName()
		  << " (" << typeid(*w).name() << ")" << std::endl;
#endif

      if (!learning_ && visibleOnly_) {
	if (!w->isStubbed()) {
	  w->getSDomChanges(changes);

	  /* if (!w->isVisible()) {
	    // We should postpone rendering the changes -- but
	    // at the same time need to propageRenderOk() now for stateless
	    // slot learning to work properly.
	    w->getSDomChanges(changes);
	  } else
	  w->getSDomChanges(changes); */
	} else
	  ; //std::cerr << "Ignoring: " << w->formName() << std::endl;
      } else
	w->getSDomChanges(changes);
    }
  }
}

void WebRenderer::collectJavaScriptUpdate(std::ostream& out)
{
  WApplication *app = session_.app();

  if (&out != &collectedChanges_) {
    out << collectedChanges_.str();
    collectedChanges_.str("");
  }

  collectJS(&out);

  /*
   * Now, as we have cleared and recorded all JavaScript changes that were
   * caused by the actual code, we can learn stateless code and collect
   * changes that result.
   */

  preLearnStateless(app);

  if (&out != &collectedChanges_) {
    out << collectedChanges_.str();
    collectedChanges_.str("");
  }

  collectJS(&out);

  if (formObjectsChanged_) {
    std::string formObjectsList = createFormObjectsList(app);
    if (formObjectsList != currentFormObjectsList_) {
      currentFormObjectsList_ = formObjectsList;
      out << app->javaScriptClass()
	  << ".private.setFormObjects([" << currentFormObjectsList_ << "]);";
    }
  }

  out << app->afterLoadJavaScript();

  if (app->isQuited()) {
    out << app->javaScriptClass() << ".private.quit();";

    WContainerWidget *timers = app->timerRoot();
    DomElement *d = DomElement::getForUpdate(timers, DomElement_DIV);
    d->setProperty(PropertyInnerHTML, "");
    EscapeOStream sout(out);
    d->asJavaScript(sout, DomElement::Update);

    delete d;
  }
}

void WebRenderer::updateFormObjects(WWebWidget *source, bool checkDescendants)
{
  formObjectsChanged_ = true;
}

std::string WebRenderer::createFormObjectsList(WApplication *app)
{
  currentFormObjects_.clear();

  app->domRoot_->getFormObjects(currentFormObjects_);
  if (app->domRoot2_)
    app->domRoot2_->getFormObjects(currentFormObjects_);

  std::string result;
  for (unsigned i = 0; i < currentFormObjects_.size(); ++i) {
    if (i != 0)
      result += ',';
    result += "'" + currentFormObjects_[i]->formName() + "'";
  }

  formObjectsChanged_ = false;

  return result;
}

void WebRenderer::collectJS(std::ostream* js)
{
  std::vector<DomElement *> changes;

  collectChanges(changes);

  if (js) {
    *js << '{';
    EscapeOStream sout(*js);

    for (unsigned i = 0; i < changes.size(); ++i)
      changes[i]->asJavaScript(sout, DomElement::Delete);

    for (unsigned i = 0; i < changes.size(); ++i)
      changes[i]->asJavaScript(sout, DomElement::Create);

    for (unsigned i = 0; i < changes.size(); ++i) {
      changes[i]->asJavaScript(sout, DomElement::Update);
      delete changes[i];
    }
  } else {
    for (unsigned i = 0; i < changes.size(); ++i)
      delete changes[i];
  }

  if (session_.type() == WebSession::WidgetSet) {
    if (js)
      *js << '}';
    return;
  }

  WApplication *app = session_.app();

  if (js) { 
    if (app->titleChanged_) {
      *js << app->javaScriptClass()
	  << ".private.setTitle(";
      DomElement::jsStringLiteral(*js, app->title().toUTF8(), '\'');
      *js << ");";
    }

    for (std::set<std::string>::const_iterator i = app->statesChanged_.begin();
	 i != app->statesChanged_.end(); ++i) {
      std::string k = *i;
      std::string v = app->state_[k];

      *js << app->javaScriptClass()
	  << ".private.historyChangeState(";
      DomElement::jsStringLiteral(*js, k, '\'');
      *js << ",";
      DomElement::jsStringLiteral(*js, v, '\'');
      *js << ");";
    }

    *js << '}';
  }

  app->titleChanged_ = false;
  app->statesChanged_.clear();
}

void WebRenderer::preLearnStateless(WApplication *app)
{
  bool isIEMobile = app->environment().agentIEMobile();

  if (isIEMobile || !session_.env().ajax())
    return;

  const WApplication::SignalMap& ss = session_.app()->exposedSignals();

  for (WApplication::SignalMap::const_iterator i = ss.begin();
       i != ss.end(); ++i) {
    WWidget *ww = dynamic_cast<WWidget *>(i->second->sender());

    if (ww && !ww->isStubbed()) {
      WWidget *a = ww->adam();
      if (a == app->domRoot_ || a == app->domRoot2_)
	i->second->processPreLearnStateless(this);
    }
  }
}

std::string WebRenderer::learn(WStatelessSlot* slot)
{
  collectJS(&collectedChanges_);

  if (slot->type() == WStatelessSlot::PreLearnStateless)
    learning_ = true;

  slot->trigger();

  std::stringstream js;
  collectJS(&js);

  std::string result = js.str();

  if (slot->type() == WStatelessSlot::PreLearnStateless) {
    slot->undoTrigger();
    collectJS(0);

    learning_ = false;
  } else { // AutoLearnStateless
    collectedChanges_ << result;
  }

  slot->setJavaScript(result);

  return result;
}

}
