#!/usr/bin/env python

################################################################################
# Copyright (c) 2005 Georg Neis <gn@oglaroon.de>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
################################################################################


# USAGE:
# Click on the small arrow to expand a directory tree.  Double click on a file
# to get its content.


from optparse import OptionParser
import os
import sys

import pygtk
pygtk.require('2.0')
import gtk

import ixplib


class Directory:
    def __init__(self, client, pathname):
        assert pathname.endswith('/')

        self.pathname = pathname
        self.subdirs = []
        self.files = []

        fd = client.open(pathname)
        content = client.read(fd).splitlines()
        client.close(fd)
        for entry in content:
            if entry.endswith('/'):
                # entry is a directory, step deeper
                dir = Directory(client, pathname + entry)
                self.subdirs.append(dir)
            else:
                # entry is a file
                self.files.append(pathname + entry)


class DirectoryTree:

    def __init__(self, ixp, dir):
        self.ixp = ixp
        self.window = gtk.Window()
        self.window.set_title('IXP Browser')
        self.window.set_size_request(400, 800)
        self.window.connect("delete_event", self.delete_event)
        self.scrolled_window = gtk.ScrolledWindow()
        self.treestore = gtk.TreeStore(str)
        self.fill_treestore(dir)
        self.treeview = gtk.TreeView(self.treestore)
        self.treeview.connect("row-activated", self.row_activated)
        self.cell = gtk.CellRendererText()
        self.column = gtk.TreeViewColumn()
        self.column.pack_start(self.cell, True)
        self.column.add_attribute(self.cell, 'text', 0)
        self.column.set_sort_column_id(0)
        self.treeview.append_column(self.column)
        self.treeview.set_search_column(0)
        # causes segfault on delete_event (!?), thus disabled:
        #self.treeview.set_search_equal_func(self.search_equal_func)
        self.scrolled_window.add_with_viewport(self.treeview)
        self.scrolled_window.set_policy(gtk.POLICY_AUTOMATIC,
                                        gtk.POLICY_AUTOMATIC)
        self.exit_button = gtk.Button('Exit')
        self.exit_button.connect('clicked', self.clicked)
        self.expand_button = gtk.Button('Expand all')
        self.expand_button.connect('clicked', self.clicked)
        self.collapse_button = gtk.Button('Collapse all')
        self.collapse_button.connect('clicked', self.clicked)
        self.hbox = gtk.HBox()
        self.hbox.pack_start(self.exit_button)
        self.hbox.pack_start(self.expand_button)
        self.hbox.pack_start(self.collapse_button)
        self.vbox = gtk.VBox()
        self.vbox.pack_start(self.scrolled_window)
        self.vbox.pack_start(self.hbox, expand=False)
        self.window.add(self.vbox)
        self.window.show_all()

    def fill_treestore(self, dir, parent=None):
        child = self.treestore.append(parent, [dir.pathname])
        for subdir in dir.subdirs:
            self.fill_treestore(subdir, child)
        for file in dir.files:
            self.treestore.append(child, [file])

    def clicked(self, button):
        if button == self.exit_button:
            gtk.main_quit()
        elif button == self.expand_button:
            self.treeview.expand_all()
        elif button == self.collapse_button:
            self.treeview.collapse_all()

    def delete_event(self, window, event):
        gtk.main_quit()

    def row_activated(self, row, path, col):
        filename = self.treestore.get_value(self.treestore.get_iter(path), 0)
        FileContent(self.ixp, filename)

    def search_equal_func(self, model, col, key, iter):
        filename = self.treestore.get_value(iter, col)
        return key not in filename


class FileContent:

    def __init__(self, ixp, filename):
        self.ixp  = ixp
        fd = self.ixp.open(filename)
        content = self.ixp.read(fd)
        ixp.close(fd)

        self.window = gtk.Window()
        self.window.set_title(filename)
        self.window.set_size_request(500, 300)
        self.scrolled_window = gtk.ScrolledWindow()
        self.textview = gtk.TextView()
        self.textview.get_buffer().set_text(content)
        self.textview.set_editable(False)
        self.scrolled_window.add_with_viewport(self.textview)
        self.scrolled_window.set_policy(gtk.POLICY_AUTOMATIC,
                                        gtk.POLICY_AUTOMATIC)
        self.back_button = gtk.Button('Back')
        self.back_button.connect('clicked', self.clicked)
        self.vbox = gtk.VBox()
        self.vbox.pack_start(self.scrolled_window)
        self.vbox.pack_start(self.back_button, expand=False)
        self.window.add(self.vbox)
        self.window.show_all()

    def clicked(self, button):
        self.window.destroy()


def main():
    def_socket = os.getenv('WMIR_SOCKET',
        '%s/.wmii/ixp/wmifs-%s' % (os.environ['HOME'], os.environ['DISPLAY']))
    dir = '/'

    cli = OptionParser(usage='%prog [<options>] [<directory>]\nbrowse an IXP '
                             'directory tree',
                       version='%prog 0.4')
    cli.add_option('-s', '--socket', action='store',
                   default=os.getenv('WMIR_SOCKET'),
                   help='socket file [default: '
                        'WMIR_SOCKET environment variable]')
    options, args = cli.parse_args()

    program = os.path.basename(sys.argv[0])

    if options.socket is None:
        cli.print_usage(sys.stderr)
        print >>sys.stderr, '%s: error: no socket file specified' % program
        sys.exit(1)

    if len(args) > 0:
        dir = args[0].rstrip('/') + ('/')

    try:
        ixp = ixplib.Client(options.socket)
    except ixplib.ClientError, e:
        print >>sys.stderr, '%s: error:' % program, e
        sys.exit(2)

    DirectoryTree(ixp, Directory(ixp, dir))
    gtk.main()


if __name__ == '__main__':
    main()
