/*
 * (C)opyright MMIV-MMV Anselm R. Garbe <garbeam at gmail dot com>
 * See LICENSE file for license details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "blitz.h"

#include <cairo/cairo-xlib.h>

static void
draw_bg(Display * dpy, cairo_t * cr, Draw * d)
{
    /* draw bg */
    cairo_set_rgb_color(cr, d->bg.r, d->bg.g, d->bg.b);
    cairo_set_alpha(cr, 1.0);
    if(!d->notch) {
        cairo_rectangle(cr, d->rect.x, d->rect.y, d->rect.width,
                        d->rect.height);
        cairo_fill(cr);
        return;
    }
    /* NW -> NE */
    cairo_rectangle(cr, d->rect.x, d->rect.y, d->rect.width, d->notch->y);
    cairo_fill(cr);
    cairo_rectangle(cr, d->rect.x, d->notch->y, d->notch->x,
                    d->notch->height);
    cairo_fill(cr);
    /* NE -> SE */
    cairo_rectangle(cr, d->notch->x + d->notch->width, d->notch->y,
                    d->rect.width - (d->notch->x + d->notch->width),
                    d->notch->height);
    cairo_fill(cr);
    /* SW -> SE */
    cairo_rectangle(cr, d->rect.x, d->notch->y + d->notch->height,
                    d->rect.width,
                    d->rect.height - (d->notch->y + d->notch->height));
    cairo_fill(cr);
}

static void
_draw_border(cairo_t * cr, Draw * d)
{
    /* draw bg border */
    cairo_set_line_width(cr, 1);
    cairo_set_line_join(cr, CAIRO_LINE_JOIN_MITER);
    /* default */
    /* NW->SW */
    cairo_set_alpha(cr, 1.0);
    cairo_set_rgb_color(cr, d->border[3].r, d->border[3].g,
                        d->border[3].b);
    cairo_move_to(cr, d->rect.x, d->rect.y);
    cairo_rel_line_to(cr, 0, d->rect.height);
    cairo_stroke(cr);
    /* NW->NE */
    cairo_set_rgb_color(cr, d->border[0].r, d->border[0].g,
                        d->border[0].b);
    cairo_move_to(cr, d->rect.x, d->rect.y);
    cairo_rel_line_to(cr, d->rect.width, 0);
    cairo_stroke(cr);
    /* SW->SE */
    cairo_set_rgb_color(cr, d->border[2].r, d->border[2].g,
                        d->border[2].b);
    cairo_move_to(cr, d->rect.x, d->rect.y + d->rect.height);
    cairo_rel_line_to(cr, d->rect.width, 0);
    cairo_stroke(cr);
    /* SE->NE */
    cairo_set_rgb_color(cr, d->border[1].r, d->border[1].g,
                        d->border[1].b);
    cairo_move_to(cr, d->rect.x + d->rect.width, d->rect.y);
    cairo_rel_line_to(cr, 0, d->rect.height);
    cairo_stroke(cr);
}

static void
draw_text(Display * dpy, cairo_t * cr, Draw * d)
{
    cairo_text_extents_t textext;
    cairo_font_extents_t fext;
    int offx, offy;
    int i, shortened = 0;
    size_t len = 0;
    static char text[2048];

    if(!d->text)
        return;
    STRLCPY(text, d->text, sizeof(text));
    len = strlen(text);

    /* query font */
    cairo_select_font(cr, d->fnt.font, CAIRO_FONT_SLANT_NORMAL,
                      CAIRO_FONT_WEIGHT_NORMAL);
    cairo_scale_font(cr, d->fnt.scale);

    /* query text extends */
    cairo_current_font_extents(cr, &fext);
    cairo_text_extents(cr, (unsigned char *) text, &textext);

    /* calculate y offset, should be baseline */
    offy = d->rect.y + d->rect.height / 2
        + ((d->rect.height - 2) - (fext.ascent + fext.descent)) / 2;
    /* check if text has to be shrinked */


    /* shorten text if necessary */
    for(i = 1; (len - (i * 3) > 0) && (textext.width > d->rect.width); i++) {
        len -= i * 3;
        text[len - 1] = '\0';
        cairo_text_extents(cr, (unsigned char *) text, &textext);
        shortened = 1;
    }
    if(textext.width > d->rect.width)
        return;
    if(shortened && (len > 3)) {
        text[len - 4] = '.';
        text[len - 3] = '.';
        text[len - 2] = '.';
    }

    /* calculate x offset */
    switch (d->fnt.align) {
    case WEST:
        offx = d->rect.x + fext.max_x_advance;
        break;
    case EAST:
        offx =
            d->rect.x + d->rect.width - (fext.max_x_advance +
                                         textext.width);
        break;
    default:                   /* CENTER */
        offx = d->rect.x + (d->rect.width - textext.width) / 2;
        break;
    }

    /* draw fg text */
    cairo_set_alpha(cr, 1.0);
    cairo_set_rgb_color(cr, d->fg.r, d->fg.g, d->fg.b);
    cairo_move_to(cr, offx, offy);
    cairo_show_text(cr, (unsigned char *) text);
}

/* draws meter */
void
draw_meter(Display * dpy, Draw * d)
{
    cairo_t *cr;
    int w = d->text ? d->rect.height / 2 : d->rect.width - 4;
    int offy, mh;

    cr = cairo_create();
    if(!cr) {
        fprintf(stderr, "%s", "wmii: cannot create cairo context");
        exit(1);
    }

    cairo_set_target_drawable(cr, dpy, d->drawable);
    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
    draw_bg(dpy, cr, d);
    _draw_border(cr, d);

    /* draw bg */
    mh = ((d->rect.height - 4) * d->val) / 100;
    offy = d->rect.y + d->rect.height - 2 - mh;
    cairo_set_rgb_color(cr, d->fill.r, d->fill.g, d->fill.b);
    cairo_set_alpha(cr, 1.0);
    cairo_rectangle(cr, d->rect.x + 2, offy, w, mh);
    cairo_fill(cr);

    if(d->text) {
        d->rect.x += w + 4;
        d->rect.width -= (w + 4);
        draw_text(dpy, cr, d);
        /* restore */
        d->rect.x -= w + 4;
        d->rect.width += (w + 4);
    }

    /* get rid of the cairo overhead */
    cairo_destroy(cr);
}

/* draws label */
static void
_draw_label(Display * dpy, Draw * d, int border)
{
    cairo_t *cr = cairo_create();
    if(!cr) {
        fprintf(stderr, "%s", "wmii: cannot create cairo context");
        exit(1);
    }

    cairo_set_target_drawable(cr, dpy, d->drawable);
    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
    draw_bg(dpy, cr, d);
    if(border)
        _draw_border(cr, d);
    draw_text(dpy, cr, d);

    /* get rid of the cairo overhead */
    cairo_destroy(cr);
}

void
draw_label(Display * dpy, Draw * d)
{
    _draw_label(dpy, d, 1);
}

void
draw_label_noborder(Display * dpy, Draw * d)
{
    _draw_label(dpy, d, 0);
}

int
text_width(Display * dpy, char *font, double size, unsigned char *utf8txt)
{
    cairo_text_extents_t text;
    cairo_t *cr;

    if(!utf8txt)
        return 0;
    cr = cairo_create();

    if(!cr) {
        fprintf(stderr, "%s", "wmii: cannot create cairo context");
        exit(1);
    }

    /* query font */
    cairo_select_font(cr, font, CAIRO_FONT_SLANT_NORMAL,
                      CAIRO_FONT_WEIGHT_NORMAL);
    cairo_scale_font(cr, size);

    /* query text extends */
    cairo_text_extents(cr, utf8txt, &text);
    cairo_destroy(cr);
    return (int) text.width;
}
