/*
 * (C)opyright MMIV-MMV Anselm R. Garbe <garbeam at gmail dot com>
 * See LICENSE file for license details.
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/Xproto.h>
#include <X11/Xutil.h>

#include "wmii.h"

/* array indexes for file pointers */
typedef enum { B_CTL,
    B_LOOKUP,
    B_SIZE,
    B_LAST
} BarIndexes;

typedef enum { LABEL, SPACER, METER } IType;

typedef struct {
    File *root;
    Draw d;
    IType type;
} Item;

IXPServer *ixps = 0;
Display *dpy;
GC gc;
Window win;
XRectangle rect;
XRectangle brect;
int screen_num;
int displayed = 0;
char sockfile[256];
File *files[B_LAST];
Item *items = 0;
unsigned int size = 0;
Pixmap pmap;

static Draw zero_draw = { 0 };

static void draw_bar();
static void new(void *obj, char *arg);
static void quit(void *obj, char *arg);
static void display(void *obj, char *arg);
static void handle_after_write(IXPServer * s, File * f);

Action acttbl[] = {
    {"quit", quit},
    {"display", display},
    {"new", new},
    {0, 0}
};

char *version[] = {
    "wmibar - window manager improved bar - " VERSION " (" DRAW ")\n"
        "  (C)opyright MMIV-MMV Anselm R. Garbe\n", 0
};

static void
usage()
{
    fprintf(stderr, "%s",
            "usage: wmibar [-s <socket file>] [-v] [<x>,<y>,<width>,<height>]\n"
            "      -s    socket file (default: /tmp/.ixp-$USER/wmibar-$WMII_IDENT)\n"
            "      -v    version info\n");
    exit(1);
}

/**
 * <path>/meter/value               "<int>"
 * <path>/meter/style/color       "#RRGGBBAA"
 * <path>/meter/style/bg-color     "#RRGGBBAA"
 * <path>/meter/style/border-color   "#RRGGBBAA #RRGGBBAA #RRGGBBAA #RRGGBBAA"
 * <path>/meter/event/b1press       "<command>"
 * <path>/meter/event/b2press       "<command>"
 * <path>/meter/event/b3press       "<command>"
 * <path>/meter/event/b4press       "<command>"
 * <path>/meter/event/b5press       "<command>"
 * <path>/meter/event/b1release     "<command>"
 * <path>/meter/event/b2release     "<command>"
 * <path>/meter/event/b3release     "<command>"
 * <path>/meter/event/b4release     "<command>"
 * <path>/meter/event/b5release     "<command>"
 */
static void
create_meter(char *path)
{
    File *f;
    char file[MAX_BUF];
    int i;

    snprintf(file, MAX_BUF, "%s/meter/value", path);
    f = ixp_create(ixps, file);
    f->after_write = handle_after_write;
    snprintf(file, MAX_BUF, "%s/meter/style/color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_F_BG);
    snprintf(file, MAX_BUF, "%s/meter/style/bg-color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_N_BG);
    snprintf(file, MAX_BUF, "%s/meter/style/border-color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_N_BORDER);
    for(i = 1; i < 6; i++) {    /* 5 buttons events */
        snprintf(file, MAX_BUF, "%s/meter/event/b%dpress", path, i);
        ixp_create(ixps, file);
        snprintf(file, MAX_BUF, "%s/meter/event/b%drelease", path, i);
        ixp_create(ixps, file);
    }
}

/**
 * <path>/spacer/style/color         "#RRGGBBAA"
 * <path>/spacer/style/bg-color   "#RRGGBBAA"
 * <path>/spacer/style/border-color  "#RRGGBBAA #RRGGBBAA #RRGGBBAA #RRGGBBAA"
 * <path>/spacer/event/b1press     "<command>"
 * <path>/spacer/event/b2press     "<command>"
 * <path>/spacer/event/b3press     "<command>"
 * <path>/spacer/event/b4press     "<command>"
 * <path>/spacer/event/b5press     "<command>"
 * <path>/spacer/event/b1release       "<command>"
 * <path>/spacer/event/b2release       "<command>"
 * <path>/spacer/event/b3release       "<command>"
 * <path>/spacer/event/b4release       "<command>"
 * <path>/spacer/event/b5release       "<command>"
 */
static void
create_spacer(char *path)
{
    char file[MAX_BUF];
    int i;

    snprintf(file, MAX_BUF, "%s/spacer/style/bg-color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_N_BG);
    snprintf(file, MAX_BUF, "%s/spacer/style/border-color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_N_BORDER);
    for(i = 1; i < 6; i++) {    /* 5 buttons events */
        snprintf(file, MAX_BUF, "%s/spacer/event/b%dpress", path, i);
        ixp_create(ixps, file);
        snprintf(file, MAX_BUF, "%s/spacer/event/b%drelease", path, i);
        ixp_create(ixps, file);
    }
}

/**
 * <path>/label/text                 "<txt value>"
 * <path>/label/style/text-align     "<align>"
 * <path>/label/style/text-font   "<value>"
 * <path>/label/style/text-size   "<int>"
 * <path>/label/style/text-color     "#RRGGBBAA"
 * <path>/label/style/bg-color     "#RRGGBBAA"
 * <path>/label/style/border-color   "#RRGGBBAA #RRGGBBAA #RRGGBBAA #RRGGBBAA"
 * <path>/label/event/b1press       "<command>"
 * <path>/label/event/b2press       "<command>"
 * <path>/label/event/b3press       "<command>"
 * <path>/label/event/b4press       "<command>"
 * <path>/label/event/b5press       "<command>"
 * <path>/label/event/b1release     "<command>"
 * <path>/label/event/b2release     "<command>"
 * <path>/label/event/b3release     "<command>"
 * <path>/label/event/b4release     "<command>"
 * <path>/label/event/b5release     "<command>"
 */
static void
create_label(char *path)
{
    File *f;
    char file[MAX_BUF];
    int i;

    snprintf(file, MAX_BUF, "%s/label/text", path);
    f = ixp_create(ixps, file);
    f->after_write = handle_after_write;
    snprintf(file, MAX_BUF, "%s/label/style/text-align", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_FT_ALIGN);
    snprintf(file, MAX_BUF, "%s/label/style/text-font", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_FT_FAM);
    snprintf(file, MAX_BUF, "%s/label/style/text-size", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_FT_SIZE);
    snprintf(file, MAX_BUF, "%s/label/style/text-color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_N_TEXT);
    snprintf(file, MAX_BUF, "%s/label/style/bg-color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_N_BG);
    snprintf(file, MAX_BUF, "%s/label/style/border-color", path);
    wmii_create_ixpfile(ixps, file, DEFAULT_N_BORDER);
    for(i = 1; i < 6; i++) {    /* 5 buttons events */
        snprintf(file, MAX_BUF, "%s/label/event/b%dpress", path, i);
        ixp_create(ixps, file);
        snprintf(file, MAX_BUF, "%s/label/event/b%drelease", path, i);
        ixp_create(ixps, file);
    }
}

/* bar actions ------------------------------------------------------ */

static void
quit(void *obj, char *arg)
{
    ixps->runlevel = SHUTDOWN;
}

static void
display(void *obj, char *arg)
{
    if(!arg)
        return;
    displayed = int_val(arg);
    if(displayed) {
        XMapRaised(dpy, win);
        draw_bar();
    } else {
        XUnmapWindow(dpy, win);
        XSync(dpy, False);
    }
}

static void
new(void *obj, char *arg)
{
    if(!strncmp(arg, "label", 5))
        create_label(&arg[6]);
    else if(!strncmp(arg, "meter", 5))
        create_meter(&arg[6]);
    else if(!strncmp(arg, "spacer", 6))
        create_spacer(&arg[7]);
}

/* {{{ bar functions */

static void
init_draw_meter(char *path, Draw * d)
{
    char buf[MAX_BUF];
    File *f;

    /* style stuff */
    snprintf(buf, MAX_BUF, "%s/meter/value", path);
    f = ixp_walk(ixps, buf);
    d->val = int_val(f->content);
    snprintf(buf, MAX_BUF, "%s/meter/style/color", path);
    f = ixp_walk(ixps, buf);
    str_to_color(&d->fill, f->content);
    snprintf(buf, MAX_BUF, "%s/meter/style/bg-color", path);
    f = ixp_walk(ixps, buf);
    str_to_color(&d->bg, f->content);
    snprintf(buf, MAX_BUF, "%s/meter/style/border-color", path);
    f = ixp_walk(ixps, buf);
    str_to_color4(d->border, f->content);
    d->draw = draw_meter;
}

static void
init_draw_spacer(char *path, Draw * d)
{
    char buf[MAX_BUF];
    File *f;

    /* style stuff */
    snprintf(buf, MAX_BUF, "%s/spacer/style/bg-color", path);
    f = ixp_walk(ixps, buf);
    str_to_color(&d->bg, f->content);
    snprintf(buf, MAX_BUF, "%s/spacer/style/border-color", path);
    f = ixp_walk(ixps, buf);
    str_to_color4(d->border, f->content);
    d->draw = draw_label;
}

static void
init_draw_label(char *path, Draw * d)
{
    char buf[MAX_BUF];
    File *f;

    /* text stuff */
    snprintf(buf, MAX_BUF, "%s/label/text", path);
    f = ixp_walk(ixps, buf);
    d->text = f->content;
    /* style stuff */
    snprintf(buf, MAX_BUF, "%s/label/style/text-align", path);
    f = ixp_walk(ixps, buf);
    str_to_align(&(d->fnt.align), f->content);
    snprintf(buf, MAX_BUF, "%s/label/style/text-font", path);
    f = ixp_walk(ixps, buf);
    d->fnt.font = f->content;
    snprintf(buf, MAX_BUF, "%s/label/style/text-size", path);
    f = ixp_walk(ixps, buf);
    d->fnt.scale = (double) int_val(f->content);
    snprintf(buf, MAX_BUF, "%s/label/style/text-color", path);
    f = ixp_walk(ixps, buf);
    str_to_color(&d->fg, f->content);
    snprintf(buf, MAX_BUF, "%s/label/style/bg-color", path);
    f = ixp_walk(ixps, buf);
    str_to_color(&d->bg, f->content);
    snprintf(buf, MAX_BUF, "%s/label/style/border-color", path);
    f = ixp_walk(ixps, buf);
    str_to_color4(d->border, f->content);
    d->draw = draw_label;
}

static void
init_item(char *path, Item * i)
{
    i->d = zero_draw;
    i->d.screen_num = screen_num;
    i->d.gc = gc;
    i->d.drawable = pmap;
    i->d.rect = brect;
    i->d.rect.y = 0;
    /* determine type */
    i->root = ixp_walk(ixps, path)->content;
    if(!strncmp(i->root->name, "label", 6)) {
        i->type = LABEL;
        init_draw_label(path, &i->d);
    } else if(!strncmp(i->root->name, "spacer", 7)) {
        i->type = SPACER;
        init_draw_spacer(path, &i->d);
    } else if(!strncmp(i->root->name, "meter", 6)) {
        i->type = METER;
        init_draw_meter(path, &i->d);
    }
}

static int
comp_str(const void *s1, const void *s2)
{
    return strcmp(*(char **) s1, *(char **) s2);
}

static void
draw()
{
    unsigned int i, xoff = 0;
    unsigned int spacer = 0;
    unsigned int usedw = 0;
    unsigned int spacew = 0;

    if(!items)
        return;

    /* precalc */
    for(i = 0; i < size; i++) {
        switch (items[i].type) {
        case LABEL:
            items[i].d.rect.width = brect.height + (items[i].d.text ?
                                                    text_width(dpy,
                                                               items[i].d.
                                                               fnt.font,
                                                               items[i].d.
                                                               fnt.scale,
                                                               (unsigned
                                                                char *)
                                                               items[i].d.
                                                               text) : 0);
            break;
        case METER:
            items[i].d.rect.width = brect.height / 2 + 2;
            break;
        case SPACER:
            items[i].d.rect.width = 0;
            spacer++;
            break;
        }
        usedw += items[i].d.rect.width;
    }

    /* seek largest label and shrink it */
    if(usedw > brect.width) {
        Draw *largest = &items[i].d;
        for(i = 1; i < size; i++) {
            if(items[i].d.rect.width > largest->rect.width)
                largest = &items[i].d;
        }
        largest->rect.width -= ((usedw - brect.width) + spacer);
        usedw = brect.width - spacer;
    }

    /* calc spacer widths, if any */
    spacew = spacer ? (brect.width - usedw) / spacer : 1;
    if(spacew < 1)
        spacew = 1;

    /* do x offset stuff */
    for(i = 0; i < size; i++) {
        if(items[i].type == SPACER)
            items[i].d.rect.width = spacew;
        items[i].d.rect.x = xoff;
        xoff += items[i].d.rect.width;
    }

    /* off-by-1 fix */
    items[i - 1].d.rect.width = brect.width - items[i - 1].d.rect.x;

    /* now draw */
    for(i = 0; i < size; i++)
        items[i].d.draw(dpy, &items[i].d);
    XCopyArea(dpy, pmap, win, gc, 0, 0, brect.width, brect.height, 0, 0);
    XSync(dpy, False);
}

static void
draw_bar()
{
    File *label = 0;
    File *f;

    if(!displayed)
        return;
    if(items)
        free(items);
    items = 0;
    if(files[B_LOOKUP]->content)
        label = ixp_walk(ixps, files[B_LOOKUP]->content);
    if(!label || !label->content) {
        Draw d = { 0 };

        /* default stuff */
        d.screen_num = screen_num;
        d.gc = gc;
        d.drawable = pmap;
        d.rect.width = brect.width;
        d.rect.height = brect.height;
        d.bg.r = d.bg.g = d.bg.b = 0.1;
        d.fg.r = d.fg.g = d.fg.b = 0.1;
        STRLCPY(d.bg.colstr, DEFAULT_F_BG, sizeof(d.bg.colstr));
        STRLCPY(d.fg.colstr, DEFAULT_F_TEXT, sizeof(d.fg.colstr));
        draw_label_noborder(dpy, &d);
    } else {
        unsigned int i = 0;
        char buf[512];
        char **paths = 0;
        /* take order into account, directory names are used in
         * alphabetical order */
        size = 0;
        for(f = label->content; f; f = f->next)
            size++;
        paths = malloc(sizeof(char *) * size);
        items = malloc(sizeof(Item) * size);
        for(f = label->content; f; f = f->next)
            paths[i++] = f->name;
        qsort(paths, size, sizeof(char *), comp_str);
        for(i = 0; i < size; i++) {
            snprintf(buf, sizeof(buf), "%s/%s",
                     (char *) files[B_LOOKUP]->content, paths[i]);
            init_item(buf, &items[i]);
        }
        draw();
        free(paths);
    }
}

static Item *
get_item_for_file(File * f)
{
    int i;
    for(i = 0; i < size; i++)
        if(items[i].root == f)
            return &items[i];
    return 0;
}

static void
handle_buttonrelease(XButtonReleasedEvent * e)
{
    File *p;
    char buf[MAX_BUF];
    char path[512];
    int i;

    for(i = 0; i < size; i++) {
        if(is_point_in_rect(e->x, e->y, &items[i].d.rect)) {
            path[0] = '\0';
            wmii_get_ixppath(items[i].root, path, sizeof(path));
            snprintf(buf, MAX_BUF, "%s/event/b%drelease", path, e->button);
            if((p = ixp_walk(ixps, buf)))
                if(p->content)
                    spawn(dpy, p->content);
            return;
        }
    }
}

static void
handle_buttonpress(XButtonPressedEvent * e)
{
    File *p;
    char buf[MAX_BUF];
    char path[512];
    int i;

    for(i = 0; i < size; i++) {
        if(is_point_in_rect(e->x, e->y, &items[i].d.rect)) {
            path[0] = '\0';
            wmii_get_ixppath(items[i].root, path, sizeof(path));
            snprintf(buf, MAX_BUF, "%s/event/b%dpress", path, e->button);
            if((p = ixp_walk(ixps, buf)))
                if(p->content)
                    spawn(dpy, p->content);
            return;
        }
    }
}

static void
check_event(Connection * e)
{
    XEvent ev;

    while(XPending(dpy)) {
        XNextEvent(dpy, &ev);
        switch (ev.type) {
        case ButtonPress:
            handle_buttonpress(&ev.xbutton);
            break;
        case ButtonRelease:
            handle_buttonrelease(&ev.xbutton);
            break;
        case Expose:
            if(ev.xexpose.count == 0) {
                /*XRaiseWindow(dpy, win); */
                draw_bar();
            }
            break;
        default:
            break;
        }
    }
}

static void
handle_after_write(IXPServer * s, File * f)
{
    int i;
    size_t len;
    Item *item;
    char buf[512];

    buf[0] = '\0';
    if(!strncmp(f->name, "text", 5)) {
        if((item = get_item_for_file(f->parent))) {
            wmii_get_ixppath(f->parent->parent, buf, sizeof(buf));
            init_draw_label(buf, &item->d);
            draw();
        }
    } else if(!strncmp(f->name, "value", 6)) {
        if((item = get_item_for_file(f->parent))) {
            wmii_get_ixppath(f->parent->parent, buf, sizeof(buf));
            init_draw_meter(buf, &item->d);
            draw();
        }
    } else if(files[B_LOOKUP] == f) {
        draw_bar();
    } else if(files[B_SIZE] == f) {
        char *size = files[B_SIZE]->content;
        if(size && strrchr(size, ',')) {
            str_to_rect(dpy, &rect, &brect, size);
            XFreePixmap(dpy, pmap);
            XMoveResizeWindow(dpy, win, brect.x, brect.y,
                              brect.width, brect.height);
            XSync(dpy, False);
            pmap = XCreatePixmap(dpy, win, brect.width, brect.height,
                                 DefaultDepth(dpy, screen_num));
            XSync(dpy, False);
            draw_bar();
        }
    } else if(files[B_CTL] == f) {
        for(i = 0; acttbl[i].name; i++) {
            len = strlen(acttbl[i].name);
            if(!strncmp(acttbl[i].name, (char *) f->content, len)) {
                if(strlen(f->content) > len) {
                    acttbl[i].func(0, &((char *) f->content)[len + 1]);
                } else {
                    acttbl[i].func(0, 0);
                }
                break;
            }
        }
    }
    check_event(0);
}

static void
handle_before_read(IXPServer * s, File * f)
{
    char buf[64];
    if(f != files[B_SIZE])
        return;
    snprintf(buf, sizeof(buf), "%d,%d,%d,%d", brect.x, brect.y,
             brect.width, brect.height);
    if(f->content)
        free(f->content);
    f->content = strdup(buf);
    f->size = strlen(buf);
}

/* }}} */

static void
run(char *size)
{
    XSetWindowAttributes wa;
    XGCValues gcv;

    /* init */
    if(!(files[B_CTL] = ixp_create(ixps, "/ctl"))) {
        perror("wmibar: cannot connect IXP server");
        exit(1);
    }
    files[B_CTL]->after_write = handle_after_write;
    files[B_LOOKUP] = ixp_create(ixps, "/lookup");
    files[B_LOOKUP]->after_write = handle_after_write;
    files[B_SIZE] = ixp_create(ixps, "/size");
    files[B_SIZE]->before_read = handle_before_read;
    files[B_SIZE]->after_write = handle_after_write;

    wa.override_redirect = 1;
    wa.background_pixmap = ParentRelative;
    wa.event_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask
        | SubstructureRedirectMask | SubstructureNotifyMask;

    brect.x = brect.y = brect.width = brect.height = 0;
    rect.x = rect.y = 0;
    rect.width = DisplayWidth(dpy, screen_num);
    rect.height = DisplayHeight(dpy, screen_num);
    str_to_rect(dpy, &rect, &brect, size);
    if(!brect.width) {
        brect.width = DisplayWidth(dpy, screen_num);
    }
    if(!brect.height) {
        brect.height = 20;
    }

    win = XCreateWindow(dpy, RootWindow(dpy, screen_num), brect.x, brect.y,
                        brect.width, brect.height, 0, DefaultDepth(dpy,
                                                                   screen_num),
                        CopyFromParent, DefaultVisual(dpy, screen_num),
                        CWOverrideRedirect | CWBackPixmap | CWEventMask,
                        &wa);
    XDefineCursor(dpy, win, XCreateFontCursor(dpy, XC_left_ptr));
    XSync(dpy, False);

    gcv.function = GXcopy;
    gcv.graphics_exposures = False;
    gc = XCreateGC(dpy, win, 0, 0);

    pmap =
        XCreatePixmap(dpy, win, brect.width, brect.height,
                      DefaultDepth(dpy, screen_num));

    /* main event loop */
    run_server_with_fd_support(ixps, ConnectionNumber(dpy),
                               check_event, 0);
    deinit_server(ixps);
    XFreePixmap(dpy, pmap);
    XFreeGC(dpy, gc);
    XCloseDisplay(dpy);
}

/* }}} */

int
main(int argc, char *argv[])
{
    char size[64];
    int i;

    sockfile[0] = '\0';
    /* command line args */
    for(i = 1; (i < argc) && (argv[i][0] == '-'); i++) {
        switch (argv[i][1]) {
        case 'v':
            fprintf(stdout, "%s", version[0]);
            exit(0);
            break;
        case 's':
            if(i + 1 < argc) {
                STRLCPY(sockfile, argv[++i], sizeof(sockfile));
            } else {
                usage();
            }
            break;
        default:
            usage();
            break;
        }
    }

    dpy = XOpenDisplay(0);
    if(!dpy) {
        fprintf(stderr, "%s", "wmibar: cannot open display\n");
        exit(1);
    }

    screen_num = DefaultScreen(dpy);

    size[0] = '\0';
    if(argc > i)
        STRLCPY(size, argv[i], sizeof(size));

    ixps = wmii_setup_server(sockfile, sizeof(sockfile), "wmibar");
    run(size);

    return 0;
}
