/*
 * (C)opyright MMIV-MMV Anselm R. Garbe <garbeam at gmail dot com>
 * See LICENSE file for license details.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/Xproto.h>
#include <X11/Xutil.h>

#include "wmii.h"

/* array indexes for file pointers */
typedef enum { K_CTL,
    K_LOOKUP,
    K_SIZE,
    K_GRAB_KB,
    K_TEXT_ALIGN,
    K_TEXT_FONT,
    K_TEXT_SIZE,
    K_TEXT_COLOR,
    K_BG_COLOR,
    K_BORDER_COLOR,
    K_LAST
} KeyIndexes;

typedef struct Shortcut Shortcut;

struct Shortcut {
    File *file;
    unsigned long mod;
    KeyCode key;
    unsigned int button;
    Shortcut *next;
};

IXPServer *ixps = 0;
Display *dpy;
GC gc;
Window win;
Window root;
XRectangle krect;
XRectangle rect;
int screen_num;
char sockfile[256];
Shortcut *shortcut = 0;
File *files[K_LAST];
int grabkb = 0;
unsigned int num_lock_mask, valid_mask;

Shortcut zero_shortcut = { 0 };

static void grab_shortcut(Shortcut * s);
static void ungrab_shortcut(Shortcut * s);
static void draw_shortcut_box(char *text);
static void quit(void *obj, char *arg);


Action acttbl[] = {
    {"quit", quit},
    {0, 0}
};

char *version[] = {
    "wmikeys - window manager improved keys - " VERSION " (" DRAW ")\n"
        "  (C)opyright MMIV-MMV Anselm R. Garbe\n", 0
};

static void
center()
{
    krect.x = rect.width / 2 - krect.width / 2;
    krect.y = rect.height / 2 - krect.height / 2;
}

static void
usage()
{
    fprintf(stderr, "%s",
            "usage: wmikeys [-s <socket file>] [-v] [<x>,<y>,<width>,<height>]\n"
            "      -s     socket file (default: /tmp/.ixp-$USER/wmikeys-$WMII_IDENT)\n"
            "      -v     version info\n");
    exit(1);
}

/* shortcut functions ----------------------------------------------- */

static Shortcut *
alloc_shortcut(File * f)
{
    char *k;
    Shortcut *s = malloc(sizeof(Shortcut));
    *s = zero_shortcut;
    s->file = f;
    k = strrchr(s->file->name, '+');
    if(k) {
        k++;
    } else {
        k = s->file->name;
    }
    s->button = str_to_button(k);
    if(!s->button)
        s->key = XKeysymToKeycode(dpy, XStringToKeysym(k));
    s->mod = str_to_mod(s->file->name);
    s->next = 0;

    return s;
}

static void
free_shortcut(Shortcut * s)
{
    ungrab_shortcut(s);
    free(s);
}

/* grabs shortcut on all screens */
static void
grab_shortcut(Shortcut * s)
{
    if(s->button) {
        XGrabButton(dpy, s->button, s->mod, root,
                    False, ButtonPressMask, GrabModeAsync, GrabModeAsync,
                    None, None);
        if(num_lock_mask) {
            XGrabButton(dpy, s->button, s->mod | num_lock_mask, root,
                        False, ButtonPressMask, GrabModeAsync,
                        GrabModeAsync, None, None);
            XGrabButton(dpy, s->button, s->mod | num_lock_mask | LockMask,
                        root, False, ButtonPressMask, GrabModeAsync,
                        GrabModeAsync, None, None);
        }
    } else if(s->key) {
        XGrabKey(dpy, s->key, s->mod, root,
                 True, GrabModeAsync, GrabModeAsync);
        if(num_lock_mask) {
            XGrabKey(dpy, s->key, s->mod | num_lock_mask, root,
                     True, GrabModeAsync, GrabModeAsync);
            XGrabKey(dpy, s->key, s->mod | num_lock_mask | LockMask, root,
                     True, GrabModeAsync, GrabModeAsync);
        }
    }
    XSync(dpy, False);
}

/* don't handle evil keys anymore, just define more shortcuts if you
   cannot live without evil key handling */
static void
ungrab_shortcut(Shortcut * s)
{
    if(s->button) {
        XUngrabButton(dpy, s->button, s->mod, root);
        if(num_lock_mask) {
            XUngrabButton(dpy, s->button, s->mod | num_lock_mask, root);
            XUngrabButton(dpy, s->button, s->mod | num_lock_mask |
                          LockMask, root);
        }
    } else if(s->key) {
        XUngrabKey(dpy, s->key, s->mod, root);
        if(num_lock_mask) {
            XUngrabKey(dpy, s->key, s->mod | num_lock_mask, root);
            XUngrabKey(dpy, s->key, s->mod | num_lock_mask |
                       LockMask, root);
        }
    }
    XSync(dpy, False);
}

static void
create_shortcut(File * f)
{
    Shortcut *p = shortcut;
    Shortcut *s = alloc_shortcut(f);

    if(!shortcut) {
        shortcut = s;
    } else {
        while(p->next)
            p = p->next;
        p->next = s;
    }
    grab_shortcut(s);
}

static void
destroy_shortcut(Shortcut * s)
{
    Shortcut *p = shortcut;

    if(s == shortcut) {
        shortcut = shortcut->next;
    } else {
        while(p->next != s)
            p = p->next;
        p->next = s->next;
    }
    free_shortcut(s);
}

static void
next_keystroke(unsigned long *mod, KeyCode * key)
{
    XEvent e;
    KeySym sym;
    *mod = 0;
    do {
        XMaskEvent(dpy, KeyPressMask, &e);
        *mod |= e.xkey.state & valid_mask;
        *key = e.xkey.keycode;
        sym = XKeycodeToKeysym(dpy, e.xkey.keycode, 0);
    } while(IsModifierKey(sym));
}

static void
emulate_key_press(unsigned long mod, KeyCode key)
{
    XEvent e;
    Window client_win;
    int revert;

    XGetInputFocus(dpy, &client_win, &revert);

    e.xkey.type = KeyPress;
    e.xkey.time = CurrentTime;
    e.xkey.window = client_win;
    e.xkey.display = dpy;
    e.xkey.state = mod;
    e.xkey.keycode = key;
    XSendEvent(dpy, client_win, True, KeyPressMask, &e);
    e.xkey.type = KeyRelease;
    XSendEvent(dpy, client_win, True, KeyReleaseMask, &e);
    XSync(dpy, False);
}

static void
handle_shortcut_chain(Window w, Shortcut * processed, File * first,
                      char *prefix, int grab)
{
    unsigned long mod;
    KeyCode key;
    File *f = first;
    Shortcut *s;
    char buf[MAX_BUF];

    if(grab) {
        XGrabKeyboard(dpy, w, True, GrabModeAsync,
                      GrabModeAsync, CurrentTime);
        XMapRaised(dpy, win);
    }
    draw_shortcut_box(prefix);
    next_keystroke(&mod, &key);

    if((processed->mod == mod) && (processed->key == key)) {
        /* double shortcut */
        emulate_key_press(mod, key);
        f = 0;                  /* don't proceed in next loop */
    }
    for(; f; f = f->next) {

        s = alloc_shortcut(f);
        if((s->mod == mod) && (s->key == key)) {
            if(f->size && f->content)
                spawn(dpy, f->content);
            else {
                snprintf(buf, MAX_BUF, "%s/%s", prefix, f->name);
                handle_shortcut_chain(w, s, f->content, buf, 0);
            }
            break;
        }
        free_shortcut(s);
    }
    if(grab) {
        XUngrabKeyboard(dpy, CurrentTime);
        XUnmapWindow(dpy, win);
        XSync(dpy, False);
    }
}

static void
handle_shortcut_gkb(Window w, unsigned long mod, KeyCode key)
{
    Shortcut *s;
    File *f, *p;
    if(!files[K_LOOKUP]->content)
        return;
    f = ixp_walk(ixps, files[K_LOOKUP]->content);
    if(!f || !is_directory(f))
        return;
    for(p = f->content; p; p = p->next) {
        s = alloc_shortcut(p);
        if((s->mod == mod) && (s->key == key)) {
            if(p->size && p->content)
                spawn(dpy, p->content);
            break;
        }
        free_shortcut(s);
        if(!p->next)
            XBell(dpy, 0);
    }
}

static void
handle_shortcut(Window w, unsigned long mod, KeyCode key,
                unsigned int button)
{
    Shortcut *s;
    if(!files[K_LOOKUP]->content)
        return;
    for(s = shortcut; s; s = s->next) {
        if((s->mod == mod) && (s->key == key) && (s->button == button)) {
            if(s->file->size && s->file->content) {
                spawn(dpy, s->file->content);
                return;
            }
            break;
        }
    }
    if(s)
        handle_shortcut_chain(w, s, s->file->content, s->file->name, 1);
}

/* keys actions ---------------------------------------------------- */

static void
quit(void *obj, char *arg)
{
    ixps->runlevel = SHUTDOWN;
}

static void
update()
{
    File *f, *p;
    Shortcut *s;
    if(!files[K_LOOKUP]->content)
        return;

    f = ixp_walk(ixps, files[K_LOOKUP]->content);

    if(!f || !is_directory(f))
        return;                 /* cannot update */

    /* destroy existing shortcuts if any */
    while((s = shortcut))
        destroy_shortcut(s);

    if(grabkb) {
        XGrabKeyboard(dpy, root, True, GrabModeAsync,
                      GrabModeAsync, CurrentTime);
        return;
    }

    /* create new shortcuts */
    for(p = f->content; p; p = p->next)
        create_shortcut(p);
}

/* draw stuff ----------------------------------------------------- */

/*
 * Function assumes following fs-structure:
 *
 * /box/style/text-align     "<align>"
 * /box/style/text-font   "<value>"
 * /box/style/text-size   "<int>"
 * /box/style/text-color     "#RRGGBBAA"
 * /box/style/bg-color     "#RRGGBBAA"
 */
static void
draw_shortcut_box(char *text)
{
    Draw d = { 0 };

    krect.width = text_width(dpy, files[K_TEXT_FONT]->content,
                             int_val(files[K_TEXT_SIZE]->content),
                             (unsigned char *) text) + krect.height;
    center();
    XMoveResizeWindow(dpy, win, krect.x, krect.y, krect.width,
                      krect.height);

    /* default stuff */
    d.screen_num = screen_num;
    d.gc = gc;
    d.drawable = win;
    str_to_align(&(d.fnt.align), files[K_TEXT_ALIGN]->content);
    d.fnt.font = (char *) files[K_TEXT_FONT]->content;
    d.fnt.scale = int_val((char *) files[K_TEXT_SIZE]->content);
    d.text = text;
    d.rect.y = 0;
    d.rect.width = krect.width;
    d.rect.height = krect.height;
    str_to_color(&d.bg, files[K_BG_COLOR]->content);
    str_to_color(&d.fg, files[K_TEXT_COLOR]->content);
    str_to_color4(d.border, files[K_BORDER_COLOR]->content);
    draw_label(dpy, &d);
}

static void
check_event(Connection * c)
{
    XEvent ev;

    while(XPending(dpy)) {
        XNextEvent(dpy, &ev);
        switch (ev.type) {
        case ButtonPress:
            ev.xbutton.state &= valid_mask;
            if(ev.xbutton.state != 0) {
                handle_shortcut(root, ev.xbutton.state, 0,
                                ev.xbutton.button);
            }
            break;
        case KeyPress:
            ev.xkey.state &= valid_mask;
            if(grabkb)
                handle_shortcut_gkb(root, ev.xkey.state, ev.xkey.keycode);
            else
                handle_shortcut(root, ev.xkey.state, ev.xkey.keycode, 0);
            break;
        default:
            break;
        }
    }
}

static void
handle_after_write(IXPServer * s, File * f)
{
    int i;
    size_t len;

    if(f == files[K_CTL]) {
        for(i = 0; acttbl[i].name; i++) {
            len = strlen(acttbl[i].name);
            if(!strncmp(acttbl[i].name, (char *) f->content, len)) {
                if(strlen(f->content) > len) {
                    acttbl[i].func(0, &((char *) f->content)[len + 1]);
                } else {
                    acttbl[i].func(0, 0);
                }
                break;
            }
        }
    } else if(files[K_SIZE] == f) {
        char *size = files[K_SIZE]->content;
        if(size && strrchr(size, ','))
            str_to_rect(dpy, &rect, &krect, size);
    } else if(f == files[K_GRAB_KB]) {
        grabkb = int_val(files[K_GRAB_KB]->content);
        if(!grabkb) {
            XUngrabKeyboard(dpy, CurrentTime);
            XUnmapWindow(dpy, win);
            XSync(dpy, False);
        } else
            update();
    } else if(f == files[K_LOOKUP]) {
        update();
    }
    check_event(0);
}

static void
handle_before_read(IXPServer * s, File * f)
{
    char buf[64];
    if(f != files[K_SIZE])
        return;
    snprintf(buf, sizeof(buf), "%d,%d,%d,%d", krect.x, krect.y,
             krect.width, krect.height);
    if(f->content)
        free(f->content);
    f->content = strdup(buf);
    f->size = strlen(buf);
}

static void
run(char *size)
{
    XSetWindowAttributes wa;
    XGCValues gcv;

    /* init */
    if(!(files[K_CTL] = ixp_create(ixps, "/ctl"))) {
        perror("wmikeys: cannot connect IXP server");
        exit(1);
    }
    files[K_CTL]->after_write = handle_after_write;
    files[K_LOOKUP] = ixp_create(ixps, "/lookup");
    files[K_LOOKUP]->after_write = handle_after_write;
    files[K_SIZE] = ixp_create(ixps, "/size");
    files[K_SIZE]->before_read = handle_before_read;
    files[K_SIZE]->after_write = handle_after_write;
    files[K_GRAB_KB] = wmii_create_ixpfile(ixps, "/grab-keyb", "0");
    files[K_GRAB_KB]->after_write = handle_after_write;
    files[K_TEXT_ALIGN] =
        wmii_create_ixpfile(ixps, "/box/style/text-align",
                            DEFAULT_FT_ALIGN);
    files[K_TEXT_FONT] =
        wmii_create_ixpfile(ixps, "/box/style/text-font", DEFAULT_FT_FAM);
    files[K_TEXT_SIZE] =
        wmii_create_ixpfile(ixps, "/box/style/text-size", DEFAULT_FT_SIZE);
    files[K_TEXT_COLOR] =
        wmii_create_ixpfile(ixps, "/box/style/text-color", DEFAULT_F_TEXT);
    files[K_BG_COLOR] =
        wmii_create_ixpfile(ixps, "/box/style/bg-color", DEFAULT_F_BG);
    files[K_BORDER_COLOR] =
        wmii_create_ixpfile(ixps, "/box/style/border-color",
                            DEFAULT_F_BORDER);

    wa.override_redirect = 1;
    wa.background_pixmap = ParentRelative;
    wa.event_mask =
        ExposureMask | SubstructureRedirectMask | SubstructureNotifyMask;

    root = RootWindow(dpy, screen_num);
    rect.x = rect.y = 0;
    rect.width = DisplayWidth(dpy, screen_num);
    rect.height = DisplayHeight(dpy, screen_num);
    krect.x = krect.y = -1;
    krect.width = krect.height = 0;
    str_to_rect(dpy, &rect, &krect, size);
    /* default is center position */
    if(!krect.width) {
        krect.width = 200;
    }
    if(!krect.height) {
        krect.height = 20;
    }
    center();

    init_lock_modifiers(dpy, &valid_mask, &num_lock_mask);

    win = XCreateWindow(dpy, RootWindow(dpy, screen_num), krect.x, krect.y,
                        krect.width, krect.height, 0, DefaultDepth(dpy,
                                                                   screen_num),
                        CopyFromParent, DefaultVisual(dpy, screen_num),
                        CWOverrideRedirect | CWBackPixmap | CWEventMask,
                        &wa);
    XDefineCursor(dpy, win, XCreateFontCursor(dpy, XC_left_ptr));
    XSync(dpy, False);

    gcv.function = GXcopy;
    gcv.graphics_exposures = False;

    gc = XCreateGC(dpy, win, 0, 0);

    /* main event loop */
    run_server_with_fd_support(ixps, ConnectionNumber(dpy),
                               check_event, 0);
    deinit_server(ixps);
    XFreeGC(dpy, gc);
    XCloseDisplay(dpy);
}

/* }}} */

static int
dummy_error_handler(Display * dpy, XErrorEvent * err)
{
    return 0;
}

int
main(int argc, char *argv[])
{
    char size[64];
    int i;

    sockfile[0] = '\0';
    /* command line args */
    for(i = 1; (i < argc) && (argv[i][0] == '-'); i++) {
        switch (argv[i][1]) {
        case 'v':
            fprintf(stdout, "%s", version[0]);
            exit(0);
            break;
        case 's':
            if(i + 1 < argc) {
                STRLCPY(sockfile, argv[++i], sizeof(sockfile));
            } else {
                usage();
            }
            break;
        default:
            usage();
            break;
        }
    }

    dpy = XOpenDisplay(0);
    if(!dpy) {
        fprintf(stderr, "%s", "wmikeys: cannot open display\n");
        exit(1);
    }

    XSetErrorHandler(dummy_error_handler);
    screen_num = DefaultScreen(dpy);
    size[0] = '\0';
    if(argc > i)
        STRLCPY(size, argv[i], sizeof(size));

    ixps = wmii_setup_server(sockfile, sizeof(sockfile), "wmikeys");

    run(size);

    return 0;
}
