/*  Wordtrans, front-end for several dictionaries, for the Qt toolkit.
    Copyright (C) 2000-2004 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "QDictDiccionario.h"
#include "intern.h"
#include "QConfig.h"
#include "qfunciones.h"
#include <qfile.h>
#include <qtextstream.h>
#include <qglobal.h> //Nmero versin qt


QDictDiccionario::QDictDiccionario() : DiccionarioBase() {
	setTipo( DiccionarioBase::Dict_dict );

	setEstrategia("."); //Estrategia por defecto del servidor
	setDatabase("*");	//Todas las bases de datos
	setServidor("localhost");
	setPuerto("2628");

	setIdioma1( _("N/A") );
    setIdioma2( _("N/A") );

	//QStrList lista_databases= getAllDatabases();
	//QStrList lista_estrategias= getAllStrategies();
}

QDictDiccionario::~QDictDiccionario() {
}

QString QDictDiccionario::getFullName() {
	QString nombre= getAlias(); // + " (dictd)";
	return nombre;
}

bool QDictDiccionario::busca(const QString& palabra, QString & resultado,
                             bool con_formato) {
	resultado="";
	n_matches=0;
	bestMatch="";

	QString word=palabra;

	word=word.stripWhiteSpace();

	if (word.isEmpty()) {
		setError( _("Nothing to look for") );
		return false;
	}

	if (word.contains( "\"", FALSE)!=0 ) {
		setError( _("Quotation marks not allowed") );
		return false;
	}

	if ( !QDictDiccionario::esValidoServidor(getServidor()) ) {
		setError( _("Server no valid") );
		return false;
	}

	if ( !QDictDiccionario::esValidoPuerto(getPuerto()) ) {
		setError( _("Port no valid") );
		return false;
	}

	QCString comando;
	comando="dict -h ";
	comando+= getServidor().latin1();
	comando+=" -p ";
	comando+= getPuerto().latin1();
	comando+=" -d \"";
	comando+= getDatabase().latin1();
	comando+="\" -s \"";
	comando+= getEstrategia().latin1();
	comando+="\" \"";
	if ( getUTF8()==true ) {
		comando+= word.utf8();
	} else {
		comando+= word.latin1();
	}
	comando+="\"";

	QStringList lista;
	bool ok=ejecuta_orden( comando, lista, false );
	
	if (lista.count() >= 3)
		bestMatch=lista[3];

	//if (con_formato) 
	formateaTexto( lista, resultado, con_formato );

	return ok;
}


bool QDictDiccionario::ejecuta_orden(QCString orden, QStringList & lista, bool otra_cosa) {
	QString output="";
	//qDebug("Orden: %s\n", orden.data());
	//printf("Orden: %s\n", orden.data());
	DiccionarioBase::exec_command( orden, output );

	extraeDatos( output, lista, otra_cosa );

	return true;
}

void QDictDiccionario::extraeDatos(QString& output, QStringList & lista, bool otra_cosa) {
	lista.clear();

	if (!output.isEmpty()) {
		QTextStream t( &output, IO_ReadOnly);
        t.setEncoding( QTextStream::Latin1 );
		QString linea;

		int n = 0;
		while ( !t.eof() ) {        // until end of file...
			linea = t.readLine();       // line of text excluding '\n'

			if (!otra_cosa) {
				//Miramos si la primera lnea acaba con "found"
				if (n==0) {
					QString last= linea.right(5);
					//if (strcmp(last.latin1(), "found")==0)
					if ( last == "found" )
						continue;
				}

				//Buscamos un From justo al principio de la lnea.
				//Si lo hay es que hemos encontrado una definicin nueva.
				int pos=linea.find( "From", 0, true );
				//debug("Pos: %d", pos);
				if ( pos == 0 ) n_matches++;
			}

			linea = linea.stripWhiteSpace();
			if (linea.isNull()) linea=" ";
			lista.append(linea);
			n++;
		}

	}
}

void QDictDiccionario::formateaTexto(const QStringList & lista, 
                                     QString & resultado, bool con_formato) {
	resultado="";

	QString titulo_inicio;
	QString titulo_fin;
	if (DiccionarioBase::html_simple_header) { 
		titulo_inicio="<b>";
		titulo_fin="</b>";
	} else {
		/*
		titulo_inicio="<table width=\"100%\" cellpadding=\"1\" border=\"0\"> "
            	"<tr><td align=\"left\" bgcolor=\"#52799e\"> "
	            "<font color=\"#ffffff\" face=\"Helvetica, Arial\"><b>";
		*/
		titulo_inicio="<table width=\"100%\" cellpadding=\"1\" border=\"0\"> "
            	"<tr><td align=\"left\" bgcolor=\"#52799e\"> "
	            "<font color=\"#ffffff\"><b>";
		titulo_fin="</b></font></td></tr></table>";
	}

	//Metemos todo en el QString
	for (QStringList::ConstIterator it = lista.begin(); it != lista.end(); ++it) {
		resultado += (*it);
		if (con_formato)
			resultado += "\r"; //Luego lo sustituimos por <br>
		else
			resultado += "\n";
	}


	if ( getUTF8()==true ) {
		resultado = QString::fromUtf8( resultado.latin1() );
	}

	if (con_formato) {
		//Los smbolos < y > hay que sustituirlos por &lt; y &gt;
		while ( resultado.find('<',0) > -1) {
			int pos=resultado.find('<',0);
			resultado.replace( pos, 1, "&lt;" );
		}

		while ( resultado.find('>',0) > -1) {
			int pos=resultado.find('>',0);
			resultado.replace( pos, 1, "&gt;" );
		}

		//while ( resultado.find('"',0) > -1) {
		//	int pos=resultado.find('"',0);
		//	resultado=resultado.replace( pos, 1, "&dquot;" );
		//}

		//Enlaces. Los enlaces van marcados entre llaves {}
		//Miramos a ver la lnea tiene algn corchete:
		while ( resultado.find('{',0) > -1) {
			int pos=resultado.find('{',0);
			if ( pos > -1 ) {
				resultado.remove( pos, 1 );
				//Miramos si justo a continuacin hay otro ms
				if ( resultado[ pos ] == '{' )
					resultado.remove( pos, 1 ); // Lo borramos

				//Avanzamos hasta el final del enlace
				int initial_pos=pos;
				QString enlace="";
				//QString ref="";
				bool eol=false;
				bool hay_error=false;
				while ( (!hay_error) && (resultado[pos]!='}') && 
                         (pos < resultado.length()) ) 
				{
					if ( resultado[pos] == QChar('\r') )
						resultado.replace( pos, 1, " " );

					//Si hay comillas en el enlace se sustituye por 
					//un apostrofe. Por supuesto esto rompe el enlace,
					//pero de otro modo las comillas hara un lo
					//a QTextView y ni siquieran mostraran el
					//enlace. He probado a expandir las comillas con \"
					//pero no funciona...
					if ( resultado[pos] == QChar('"') )
						//ref += "\\";
						resultado.replace( pos, 1, "'");

					//Ignoramos posibles '{' dentro del enlace.
					//Probablemente se deba a enlaces rotos, como
					//el que hay en la definicin "unix" del foldoc.
					if ( resultado[pos] == QChar('{') ) {
						qDebug( "warning: '{' inside a link. Maybe a broken link?");
						//resultado=resultado.remove( pos, 1 );
						hay_error=true;
						continue;
					}
					enlace += resultado[pos];
					//ref += resultado[pos];
					pos++;
				}
				if (pos>=resultado.length()) eol=true;
				//if (eol) continue;

				int final_pos=pos-1;
				//Borramos el } final
				if ( resultado[pos] == '}' ) resultado.remove( pos, 1);
				if ( resultado[pos] == '}' ) resultado.remove( pos, 1);

				//debug("Lnea: '%s'", linea.data() );
				//debug("Enlace: '%s'", enlace.latin1() );
				//debug("initial_pos: %d  final_pos: %d", initial_pos, final_pos);

				QString link;
				link="<a href=\"";
				if (hay_error) {
					link += "about: broken link";
				}
				else {
					if (DiccionarioBase::no_internal_links) {						
						link += DiccionarioBase::base_page
                             + DiccionarioBase::additional_options
                             + "dict="
                             + reemplazaEspacios(getAlias()) + "&amp;word=" 
                             + reemplazaEspacios(enlace);
					} else {
						link += "dict:" + enlace + "@" + getAlias();
						/*
						link += getAlias() + ":";
						link += enlace;
						*/
					}
				}
				link += "\">";
				link += enlace;
				link += "</a>";
				//debug("Link: '%s'", link.latin1() );
				int len= (final_pos-initial_pos)+1;
				resultado.replace(initial_pos, len, link );
			}
		}

		//Buscamos el ttulo
		int pos2= resultado.find("\rFrom",0);
		while (pos2 > -1) {
			resultado.insert( pos2 + 1, titulo_inicio );
			pos2 = resultado.find('\r', pos2 + 1);
			if (pos2 > -1 ) resultado.insert( pos2, titulo_fin );
			pos2= resultado.find("\rFrom",pos2 + 1);
		}
	
		//Si es con formato reemplazamos todos los \n por <br>.
		while ( resultado.find('\r',0) > -1) {
			int pos=resultado.find('\r',0);
			resultado.replace( pos, 1, "<br>\n" );
		}
	}

	//debug("Resultado='%s'", resultado.latin1());
}

QString QDictDiccionario::reemplazaEspacios( const QString & cadena ) {
	QString resultado(cadena);
	int pos;
	//Espacios
	pos=resultado.find(' ',0);
	while (pos > -1) {
		resultado.replace( pos, 1, "%20" );
		pos=resultado.find(' ',0);
	}

	//Smbolo &
	pos=resultado.find('&',0);
	while (pos > -1) {
		resultado.replace( pos, 1, "%26" );
		pos=resultado.find(' ',0);
	}

	return resultado;
}

/*
void QDictDiccionario::formateaTexto( QStringList & lista ) {
	QStringList temp;

	QString linea;

	for (QStringList::Iterator it = lista.begin(); it != lista.end(); ++it) {
		linea=(*it);
		//debug("Lnea: '%s' (%d)", linea.data(), linea.length() );
		if ( getUTF8()==true ) {
			//QString s= QString::fromUtf8( linea, linea.length() );
			//linea= QTextCodec::codecForLocale()->fromUnicode(s);
			linea = QString::fromUtf8( linea );
			//debug("Latin1: '%s'", linea.data() );
		}

		//Los smbolos < y > hay que sustituirlos por &lt; y &gt;
		while ( linea.find('<',0) > -1) {
			int pos=linea.find('<',0);
			linea=linea.replace( pos, 1, "&lt;" );
		}

		while ( linea.find('>',0) > -1) {
			int pos=linea.find('>',0);
			linea=linea.replace( pos, 1, "&gt;" );
		}

		//Enlaces. Los enlaces van marcados entre corchetes {}
		//Miramos a ver la lnea tiene algn corchete:
		//debug("Analizando lnea '%s'", linea.data() );
		while ( linea.find('{',0) > -1) {
			int pos=linea.find('{',0);
			if ( pos > -1 ) {
				linea=linea.remove( pos, 1 );
				//Miramos si justo a continuacin hay otro ms
				if ( linea[ pos ] == '{' )
					linea=linea.remove( pos, 1 ); // Lo borramos

				//Avanzamos hasta el final del enlace
				int initial_pos=pos;
				QString enlace="";
				bool eol=false;
				while ( (linea[pos]!='}') && (pos < linea.length()) ) {
					enlace += linea[pos];
					pos++;
				}
				if (pos>=linea.length()) eol=true;
				//if (eol) continue;

				int final_pos=pos-1;
				//Borramos el } final
				if ( linea[pos] == '}' ) linea=linea.remove( pos, 1);
				if ( linea[pos] == '}' ) linea=linea.remove( pos, 1);

				//debug("Lnea: '%s'", linea.data() );
				//debug("Enlace: '%s'", enlace.data() );
				//debug("initial_pos: %d  final_pos: %d", initial_pos, final_pos);

				QString link;
				link="<a href=\"";
				link += enlace;
				link += "\">";
				link += enlace;
				link += "</a>";
				int len= (final_pos-initial_pos)+1;
				linea=linea.replace(initial_pos, len, link );
			}
		}

		temp.append( linea );
	}

	lista=temp;
}
*/

bool QDictDiccionario::aprende(const QString& texto_idioma1, const QString& texto_idioma2) {
	return false;
}

void QDictDiccionario::save() {
	conf->setGroup("General");

	conf->writeEntry("alias", alias.latin1() );
	conf->writeEntry("idioma1", idioma1.latin1() );
	conf->writeEntry("idioma2", idioma2.latin1() );
	conf->writeBoolEntry("deleted",getDeleted() );

	conf->writeEntry("strategy", getEstrategia().latin1() );
	conf->writeEntry("database", getDatabase().latin1() );
	conf->writeEntry("server", getServidor().latin1() );
	conf->writeEntry("port", getPuerto().latin1() );

	conf->writeBoolEntry("utf8", getUTF8() );

	conf->setGroup("Iconos");
	conf->writeEntry("idioma1", getIconoIdioma1().latin1() );
    conf->writeEntry("idioma2", getIconoIdioma2().latin1() );

	save_desc();
	save_font();

	conf->sync();
}

void QDictDiccionario::load(const QString& fichero) {
	conf->setFileName(fichero.latin1());

	conf->setGroup("General");

	alias = conf->readEntry("alias", "sin_alias");
	idioma1 = conf->readEntry("idioma1", "language1");
	idioma2 = conf->readEntry("idioma2", "language2");
	setDeleted( conf->readBoolEntry("deleted",false) );

	setEstrategia( conf->readEntry("strategy", "exact") );
	setDatabase( conf->readEntry("database", "*") );
	setServidor( conf->readEntry("server", "localhost") );
	setPuerto( conf->readEntry("port", "2628") );

	setUTF8( conf->readBoolEntry("utf8", false ) );

	conf->setGroup("Iconos");
	setIconoIdioma1( conf->readEntry("idioma1","/usr/share/wordtrans/i2e.xpm") );
	setIconoIdioma2( conf->readEntry("idioma2","/usr/share/wordtrans/e2i.xpm") );

	load_desc();
	load_font();
}

bool QDictDiccionario::update() {
	return true;
}

QString QDictDiccionario::getGeneralInfo(bool con_formato) {
	QCString comando;
	comando="dict -I -h ";
	comando+= getServidor().latin1();
	comando+=" -p ";
	comando+= getPuerto().latin1();

	QStringList lista;
	tempo_info="";

	ejecuta_orden( comando, lista, true );

	//tempo_info.append("</pre>");
	//tempo_info.insert( tempo_info.begin(),"<pre>"); 

	formateaTexto( lista, tempo_info, false );

	if (con_formato)
		tempo_info="<qt title=\"General Info\">\n<pre>\n" + tempo_info + "</pre></qt>";

	return tempo_info;
}

QString QDictDiccionario::getDictionaryInfo(bool con_formato) {
	//if (strcmp( getDatabase().latin1(), "*" )==0) {
	if ( getDatabase() == "*" ) {
		tempo_info=
		_("This dictionary is using the database \"*\" (which means \"all\"), \n"
		  "but the server cannot give information of all dictionaries at the same time.\n"
		  "Select only one dictionary and try again.\n");
		return tempo_info;
	}

	QCString comando;
	comando="dict -h ";
	comando+= getServidor().latin1();
	comando+=" -p ";
	comando+= getPuerto().latin1();
	comando+=" --info \"";
	comando+= getDatabase().latin1();
	comando+="\"";

	QStringList lista;
	tempo_info="";

	ejecuta_orden( comando, lista, true );


	formateaTexto( lista, tempo_info, con_formato );

	if (con_formato)
		tempo_info="<qt title=\"Dict Info\">\n" + tempo_info + "</qt>";

	return tempo_info;
}

QStringList QDictDiccionario::getAllDatabases() {
	QStringList temp;
	QStringList lista_databases;

	QCString comando;
	comando="dict -D -h ";
	comando+= getServidor().latin1();
	comando+=" -p ";
	comando+= getPuerto().latin1();

	ejecuta_orden( comando, temp, true );

	if (temp.isEmpty()) return lista_databases;

	QString linea;
	
	for (QStringList::Iterator it = temp.at(1); it != temp.end(); ++it) {
		linea= (*it);
		linea= linea.stripWhiteSpace();
		int pos = linea.find( ' ' );
		//debug("Linea: '%s' Pos: %d", linea.data(), pos );
		if (pos>0) linea= linea.left( pos );
		lista_databases.append(linea);
		//debug("Lnea: '%s'", linea.data() );
	}

	return lista_databases;
}

QStringList QDictDiccionario::getAllStrategies() {
	QStringList temp;
	QStringList lista_estrategias;

	QCString comando;
	comando="dict -S -h ";
	comando+= getServidor().latin1();
	comando+=" -p ";
	comando+= getPuerto().latin1();

	ejecuta_orden( comando, temp, true );

	if (temp.isEmpty()) return lista_estrategias;

	QString linea;
	
	for (QStringList::Iterator it = temp.at(1); it != temp.end(); ++it) {
		linea= (*it);
		linea= linea.stripWhiteSpace();
		int pos = linea.find( ' ' );
		//debug("Linea: '%s' Pos: %d", linea.data(), pos );
		if (pos>0) linea= linea.left( pos );
		lista_estrategias.append(linea);
		//debug("Lnea: '%s'", linea.data() );
	}

	return lista_estrategias;
}


bool QDictDiccionario::esValidoServidor( const QString& server ) {
	for (unsigned int n=0; n < server.length(); n++ ) {
		QChar c= server[n];
		if ( (!c.isLetter()) && (!c.isNumber()) && (c!=QChar(':')) &&
             (c!=QChar('.')) && (c!=QChar('/')) ) return false;
	}
	return true;
}

bool QDictDiccionario::esValidoPuerto( const QString& port ) {
	for (unsigned int n=0; n < port.length(); n++ ) {
		QChar c= port[n];
		if (!c.isNumber()) return false;
	}
	return true;
}

