/* fieldlistview.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2002-2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: fieldlistview.cc,v 1.58 2005/05/15 13:45:43 ralf Exp $ */

#include "fieldlistview.h"
#include <stdlib.h>
#include "awindow.h"
#include "guielement.h"

const char *FieldListView::type="FieldListView";

FieldListView::FieldListView( AGUIX *caguix, int cx, int cy, int cwidth, int cheight, int cdata ):GUIElement( caguix )
{
  int i;

  fields = 1;
  fieldwidth = new int[fields];
  for ( i = 0; i < fields; i++ )
    fieldwidth[i] = -1;
  used_width = new int[fields];
  for ( i = 0; i < fields; i++ )
    used_width[i] = 0;
  maxrow = new int[fields];
  for ( i = 0; i < fields; i++ )
    maxrow[i] = -1;

  elements = 0;
  arraysize = 100;
  elementarray = new Content*[arraysize];
  yoffset = 0;
  xoffset = 0;

  fieldalign = new align_t[fields];
  for ( i = 0; i < fields; i++ )
    fieldalign[i] = ALIGN_LEFT;

  fieldtext = new std::string[fields];
  for ( i = 0; i < fields; i++ )
    fieldtext[i] = "";

  fieldtextmerged = new bool[fields];
  for ( i = 0; i < fields; i++ )
    fieldtextmerged[i] = false;

  fieldtextclicked = -1;

  activerow = -1;

  this->data = cdata;
  _x = cx;
  _y = cy;
  if ( cwidth > 6 )
    _w = cwidth;
  else
    _w = 6;
  if ( cheight > 6 )
    _h = cheight;
  else
    _h = 6;
  bgset = false;
  hbar = 1;
  vbar = 1;
  vbar_width = 10;
  hbar_height = 10;
  hbar_dir = 0;
  vbar_dir = 0;
  vbar_pressed = false;
  hbar_pressed = false;
  hbarwin = 0;
  vbarwin = 0;
  selecthandler=NULL;
  font = NULL;
  gettimeofday( &lastwheel, NULL );
  mbg = 0;
  elementHeight = 0;
  ox = 0;
  oy = 0;

  buffer = 0;

  displayFocus = false;
  setCanHandleFocus();

  cond_changed = false;
  showHeader = false;

  vbarMode = VBAR_IDLE;
  hbarMode = HBAR_IDLE;
  grabs = 0;
  scrollDelta = 0;
  thirdMode = THIRD_IDLE;
  scrollXDir = 0;
  scrollYDir = 0;
  selectMode = SELECT_IDLE;
  clickLastSelectedRow = clickFirstSelectedRow = -1;
  clickState = false;
  clickSelected = 0;
  middleMode = MIDDLE_IDLE;

  headerCols[0] = 1;
  headerCols[1] = 0;
}

FieldListView::~FieldListView()
{
  int i;

  destroy();
  
  delete [] fieldwidth;
  delete [] used_width;
  delete [] fieldalign;
  delete [] fieldtext;
  delete [] fieldtextmerged;
  for ( i = 0; i < elements; i++ ) {
    delete elementarray[i];
  }
  delete [] elementarray;
  delete [] maxrow;
}

FieldListView::Content::Content()
{
  int i;

  cur_fields = 1;
  texts = new std::string*[cur_fields];
  for ( i = 0; i < cur_fields; i++)
    texts[i] = new std::string("");

  // default colors
  fg[ 0 ] = 1;
  fg[ 1 ] = 2;
  fg[ 2 ] = 1;
  fg[ 3 ] = 2;
  bg[ 0 ] = 0;
  bg[ 1 ] = 1;
  bg[ 2 ] = 7;
  bg[ 3 ] = 7;

  select = false;
  data = 0;
  dataExt = NULL;
  mark = false;
}

FieldListView::Content::~Content()
{
  int i;

  for ( i = 0; i < cur_fields; i++)
    delete texts[i];
  delete [] texts;
  
  if ( dataExt != NULL ) delete dataExt;
}

void FieldListView::Content::setText( int field, std::string text )
{
  if ( field < 0 ) return;
  if ( field >= cur_fields ) trim( field + 1 );
  *texts[ field ] = text;
}

void FieldListView::Content::trim( int newfields )
{
  std::string **nt;
  int i;

  if ( newfields < 1 ) return;
  if ( newfields == cur_fields ) return;

  nt = new std::string*[ newfields ];
  if ( newfields < cur_fields ) {
    for ( i = 0; i < newfields; i++ ) {
      nt[i] = texts[i];
    }
    for ( i = newfields; i < cur_fields; i++ ) {
      delete texts[i];
    }
  } else {
    for ( i = 0; i < cur_fields; i++ ) {
      nt[i] = texts[i];
    }
    for ( i = cur_fields; i < newfields; i++ ) {
      nt[i] = new std::string( "" );
    }
  }
  cur_fields = newfields;
  delete [] texts;
  texts = nt;
}

std::string FieldListView::Content::getText( int field ) const
{
  if ( ( field < 0 ) || ( field >= cur_fields ) ) return "";
  return *texts[field];

}

void FieldListView::setFieldWidthQ( int field, int width )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  if ( width < -1 ) {
    width = -1;
    used_width[ field ] = -1;
    maxrow[ field ] = -1;
  }
  fieldwidth[ field ] = width;
}

void FieldListView::setFieldWidth( int field, int width )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  setFieldWidthQ( field, width );

  redraw();
}

int FieldListView::Content::getLength( int field ) const
{
  if ( ( field < 0 ) || ( field >= cur_fields ) ) return 0;
  return texts[field]->length();

}

/*
 * setText
 *
 * sets the text for the specified row and field
 *
 * this will not redraw the lv because of the performance
 */
void FieldListView::setText( int row, int field, std::string text )
{
  Content *te;

  if ( ( row < 0 ) || ( row >= elements ) ) return;
  if ( field < 0 ) return;

  te = elementarray[ row ];
  // make changes to te
  
  // test if field already exists
  if ( field >= fields ) {
    // increase fieldwidth
    setNrOfFields( field + 1 );
  }

  // test if new text will change dynamic width
  if ( fieldwidth[ field ] < 0 ) {
    // dynamic field width
    if ( used_width[ field ] >= 0 ) {
      // only change if not unknown
      if ( ( (int)text.length() ) > used_width[ field ] ) {
        // text is longer so only change used_width
        used_width[ field ] = text.length();
	maxrow[ field ] = row;
	cond_changed = true;
      } else if ( ( (int)text.length() ) < used_width[ field ] ) {
        // text is shorter
	if ( ( maxrow[ field ] < 0 ) || ( maxrow[ field ] == row ) ) {
	  // we don't know the max row or we are the max row
	  // we don't know new width so mark it as unknown
	  used_width[ field ] = -1;
	  maxrow[ field ] = -1;
	  cond_changed = true;
	}
      }
    }
  }
  elementarray[ row ]->setText( field, text );
  if ( isRowVisible( row ) == true ) cond_changed = true;
}

void FieldListView::increaseArray( int newsize )
{
  Content **newarray;
  int i;

  if ( newsize < arraysize ) return;

  newarray = new Content*[newsize];
  for ( i = 0; i < elements; i++ ) {
    newarray[i] = elementarray[i];
  }
  delete [] elementarray;
  elementarray = newarray;
  arraysize = newsize;
}

void FieldListView::checkShrink()
{
  double loadfactor;
  int newsize, i;
  Content **newarray;

  if ( arraysize <= 100 ) return;  // never less then 100 (possible) entires
  
  loadfactor = (double)elements;
  loadfactor /= arraysize;
  if ( loadfactor < .1 ) {
    // only if less the 10 percent used
    newsize = arraysize / 5;  // decrease to 20%
    if ( newsize < 100 ) newsize = 100;
    newarray = new Content*[newsize];
    for ( i = 0; i < elements; i++ ) {
      newarray[i] = elementarray[i];
    }
    delete [] elementarray;
    elementarray = newarray;
    arraysize = newsize;
  }
}

void FieldListView::setNrOfFields( int nv )
{
  int *nw, *nu, i, *nmr;
  align_t *na;
  std::string *nt;
  bool *ntm;

  if ( nv < 1 ) return;

  nw = new int[nv];
  nu = new int[nv];
  na = new align_t[nv];
  nt = new std::string[nv];
  ntm = new bool[nv];
  nmr = new int[nv];
  if ( nv < fields ) {
    for ( i = 0; i < nv; i++ ) {
      nw[i] = fieldwidth[i];
      nu[i] = used_width[i];
      na[i] = fieldalign[i];
      nt[i] = fieldtext[i];
      ntm[i] = fieldtextmerged[i];
      nmr[i] = maxrow[i];
    }
  } else {
    for ( i = 0; i < fields; i++ ) {
      nw[i] = fieldwidth[i];
      nu[i] = used_width[i];
      na[i] = fieldalign[i];
      nt[i] = fieldtext[i];
      ntm[i] = fieldtextmerged[i];
      nmr[i] = maxrow[i];
    }
    for ( i = fields; i < nv; i++ ) {
      nw[i] = -1;
      nu[i] = -1;
      na[i] = ALIGN_LEFT;
      nt[i] = "";
      ntm[i] = false;
      nmr[i] = -1;
    }
  }
  delete [] fieldwidth;
  delete [] used_width;
  delete [] fieldalign;
  delete [] fieldtext;
  delete [] fieldtextmerged;
  delete [] maxrow;
  fieldwidth = nw;
  used_width = nu;
  fieldalign = na;
  fieldtext = nt;
  fieldtextmerged = ntm;
  maxrow = nmr;
  fields = nv;

  fieldtextclicked = -1;  // when changing fields deactivate pressed
  redraw();
}

int FieldListView::getUsedWidth( int field )
{
  int i, nw, tw, r;

  if ( ( field < 0 ) || ( field >= fields ) ) return -1;
  
  if ( used_width[ field ] < 0 ) {
    nw = 0;
    if ( elements > 0 ) {
      r = (int)( (double)elements * rand() / ( RAND_MAX + 1.0 ) );
      if ( r >= elements ) r = 0;  // will not happen but just to be sure
      maxrow[ field ] = r;
      nw = elementarray[ r ]->getLength( field );
      for ( i = 0; i < elements; i++ ) {
	tw = elementarray[ i ]->getLength( field );
	if ( tw > nw ) {
	  nw = tw;
	  maxrow[ field ] = i;
	} else if ( tw == nw ) {
	  // the same length as current max
	  // but because I don't want that always the first is the max
	  // I will get a random value and decide if I use this as the new max
	  r = (int)( (double)elements* rand() / ( RAND_MAX + 1.0 ) );
	  if ( ( r >= ( ( elements / 2 ) - 2 ) ) && ( r <= ( ( elements / 2 ) - 2 ) ) ) {
	    maxrow[ field ] = i;
	  }
	}
      }
    }
    used_width[ field ] = nw;
  }
  return used_width[ field ];
}

int FieldListView::addRow()
{
  int newrow;

  if ( elements == arraysize ) increaseArray( arraysize * 2 );
  newrow = elements++;
  elementarray[ newrow ] = new Content();

  // deactivate select/middle mode
  selectModeIgnore();
  middleModeIgnore();

  return newrow;
}

int FieldListView::deleteRow( int row )
{
  int i;

  if ( ( row < 0 ) || ( row >= elements ) ) return -1;
  delete elementarray[ row ];
  for ( i = row + 1; i < elements; i++ ) {
    elementarray[ i - 1 ] = elementarray[ i ];
  }
  elements--;
  if ( activerow == row ) activerow = -1;
  else if ( activerow > row ) {
    // decrease activerow so the same entry is active
    activerow--;
  }
  checkShrink();

  for ( i = 0; i < fields; i++ ) {
    // invalid dynamic fields
    if ( fieldwidth[i] == -1 ) {
      if ( ( maxrow[ i ] < 0 ) || ( maxrow[i] == row ) ) {
	used_width[i] = -1;
	maxrow[i] = -1;
      }
    }
  }

  // check yoffset
  if ( ( yoffset + maxdisplayv ) >= elements ) yoffset = elements - maxdisplayv;
  if ( yoffset < 0 ) yoffset = 0;

  if ( elements < maxdisplayv ) {
    // not all lines used so reset the unused elements
    clearNonElements();
  }

  // deactivate select/middle mode
  selectModeIgnore();
  middleModeIgnore();
  return 0;
}

void FieldListView::setXOffset( int nv )
{
  int t;

  t = getMaxTextWidth();
  if ( ( nv + maxdisplayh ) >= t ) nv = t - maxdisplayh;
  if ( nv < 0 ) nv = 0;
  if ( nv != xoffset ) {
    xoffset = nv;
    
    redrawHeader();
    redrawContent();
    hbarredraw();
  }
}

int FieldListView::getMaxTextWidth()
{
  //TODO: Vielleicht zwischenspeichern und nur bei deleteRow und setText aktualisieren
  int i, wi;

  wi = 0;
  for ( i = 0; i < fields; i++ ) {
    if ( fieldwidth[i] < 0 ) {
      wi += getUsedWidth( i );
    } else {
      wi += fieldwidth[i];
    }
  }
  return wi;
}

void FieldListView::redrawContent()
{
  char *line;
  int i;
  
  line = (char*)malloc( sizeof(char) * ( maxdisplayh + 1 ) );
  
  for ( i = 0; i < maxdisplayv; i++ ) {
    if ( ( i + yoffset ) >= elements ) break;
    redraw( yoffset + i, line );
  }
  
  free( line );
  drawBuffer();
}

void FieldListView::setYOffset( int nv )
{
  if ( ( nv + maxdisplayv ) >= elements ) nv = elements - maxdisplayv;
  if ( nv < 0 ) nv = 0;
  if ( nv != yoffset ) {
    yoffset = nv;

    clearNonElements();
    redrawContent();
    vbarredraw();
  }
}

void FieldListView::setFieldAlignQ( int field, align_t nv )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  fieldalign[ field ] = nv;
}

void FieldListView::setFieldAlign( int field, align_t nv )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  setFieldAlignQ( field, nv );

  redraw();
}

void FieldListView::setFieldTextQ( int field, const std::string nv )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  fieldtext[ field ] = nv;
}

void FieldListView::setFieldText( int field, const std::string nv )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  setFieldTextQ( field, nv );

  redraw();
}

void FieldListView::setFieldTextMergedQ( int field, bool nv )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  fieldtextmerged[ field ] = nv;
}

void FieldListView::setFieldTextMerged( int field, bool nv )
{
  if ( ( field < 0 ) || ( field >= fields ) ) return;
  setFieldTextMergedQ( field, nv );

  redraw();
}

void FieldListView::setSize( int rows )
{
  int i;
  if ( rows < elements ) {
    for ( i = rows; i < elements; i++ )
      delete elementarray[ i ];
    elements = rows;
    checkShrink();
  } else if ( rows > elements ) {
    if ( rows > arraysize ) increaseArray( rows );
    for ( i = elements; i < rows; i++ ) {
      elementarray[ i ] = new Content();
    }
    elements = rows;
  }
  if ( activerow >= elements ) activerow = -1;

  for ( i = 0; i < fields; i++ ) {
    // invalid dynamic fields
    if ( fieldwidth[i] == -1 ) {
      used_width[i] = -1;
      maxrow[i] = -1;
    }
  }

  // check yoffset
  if ( ( yoffset + maxdisplayv ) >= elements ) yoffset = elements - maxdisplayv;
  if ( yoffset < 0 ) yoffset = 0;

  if ( elements < maxdisplayv ) {
    // not all lines used so reset the unused elements
    clearNonElements();
  }

  // deactivate select/middle mode
  selectModeIgnore();
  middleModeIgnore();
}

void FieldListView::redraw( int element )
{
  char *buf;

  if ( ( element < 0 ) || ( element >= elements ) ) return;
  if ( ( element >= yoffset ) && ( element < ( yoffset + maxdisplayv ) ) ) {
    buf = (char*)malloc( sizeof(char) * ( maxdisplayh + 1 ) );
    redraw( element, buf );
    free( buf );
    drawBuffer( element );
  }
}

/*
 * redraw( element, line )
 *
 * draws element with buffer line
 * No buffer->window copying
 */
void FieldListView::redraw( int element, char *line )
{
  std::string s;
  const char *cstr;
  int sx, cpos, ex, vi_offset_s, vi_offset_c, f, len, tlen, ssx, toff;
  int vis;
  int tfg, tbg;

  if ( isCreated() == false ) return;
  if ( ( element < 0 ) || ( element >= elements ) ) return;
  if ( ( element >= yoffset ) && ( element < ( yoffset + maxdisplayv ) ) ) {
    memset( line, ' ', maxdisplayh );
    
    sx = 0;
    cpos = 0;
    for ( f = 0; f < fields; f++ ) {
      if ( fieldwidth[ f ] < 0)
	ex = sx + getUsedWidth( f );
      else
	ex = sx + fieldwidth[ f ];
      
      if ( ( sx < ( xoffset + maxdisplayh ) ) && ( ex > xoffset ) ) {
	// field visible
	vi_offset_c = a_min( xoffset + maxdisplayh, ex ) - a_max( sx, xoffset );
	if ( fieldalign[ f ] == ALIGN_RIGHT ) {
	  if ( ex < ( xoffset + maxdisplayh ) )
	    vi_offset_s = 0;
	  else
	    vi_offset_s = ex - ( xoffset + maxdisplayh ); 
	  // remember vi_offset_s is now from right
	  
	  s = elementarray[ element ]->getText( f );
	  len = s.length();
	  if ( len > 0 ) {
	    cstr = s.c_str();
	    if ( len > vi_offset_s ) {
	      // something visible from this entry
	      tlen = len - vi_offset_s;
	      // only up to tlen chars
	      if ( tlen <= vi_offset_c ) {
		// everything from beginning visible
		ssx = 0;
		toff = vi_offset_c - tlen;
	      } else {
		ssx = tlen - vi_offset_c;
		toff = 0;
	      }
	      // toff + bytes copyied are always <= vi_offset-c
	      // if toff is 0 then because of the min we only copy up to vi_offset_c chars
	      // otherwise toff is vi_offset_c - tlen and because tlen<= vi_offset_c we copy
	      // only tlen bytes => toff + bytes copyied = vi_offset_c - tlen + tlen
	      // 
	      // done to show we do not copy more bytes then available
	      memcpy( line + cpos + toff, cstr + ssx, a_min( tlen, vi_offset_c ) );
	    }
	  }	  
	} else {
	  if ( sx >= xoffset )
	    vi_offset_s = 0;
	  else
	    vi_offset_s = xoffset - sx;
	  
	  // append at cpos from field beginning with vi_offset_s vi_offset_c chars
	  s = elementarray[ element ]->getText( f );
	  len = s.length();
	  if ( len > 0 ) {
	    cstr = s.c_str();
	    if ( len > vi_offset_s ) {
	      // something visible from this entry
	      memcpy( line + cpos, cstr + vi_offset_s, a_min( vi_offset_c, len - vi_offset_s ) );
	    }
	  }
	}
	cpos += vi_offset_c;
      }
      
      sx = ex;
    }
    line[ cpos ] = '\0';

    vis = element - yoffset;

    if ( element == activerow ) {
      if ( elementarray[ element ]->getSelect() == true ) {
	tfg = elementarray[ element ]->getFG( CC_SELACT );
	tbg = elementarray[ element ]->getBG( CC_SELACT );
      } else {
	tfg = elementarray[ element ]->getFG( CC_ACTIVE );
	tbg = elementarray[ element ]->getBG( CC_ACTIVE );
      }
    } else {
      if ( elementarray[ element ]->getSelect() == true ) {
	tfg = elementarray[ element ]->getFG( CC_SELECT );
	tbg = elementarray[ element ]->getBG( CC_SELECT );
      } else {
	tfg = elementarray[ element ]->getFG( CC_NORMAL );
	tbg = elementarray[ element ]->getBG( CC_NORMAL );
      }
    }
    _aguix->setFG( tbg );
    _aguix->FillRectangle( buffer,
			  ox,
			  vis * elementHeight + oy,
			  maxdisplayh * elementWidth,
			  elementHeight );
    if ( font == NULL )
      _aguix->setFG( tfg );
    else
      _aguix->setFG( font->getGC(), tfg );
    if ( font == NULL )
      _aguix->DrawText( buffer, line, ox + 1, vis * elementHeight + oy );
    else
      _aguix->DrawText( buffer, font, line, ox + 1, vis * elementHeight + oy );
    if ( getVisMark( element ) == true ) {
#if 0
      _aguix->DrawDottedXorRectangle( buffer, 
				     ox,
				     vis * elementHeight + oy,
				     maxdisplayh * elementWidth,
				     elementHeight );
#else
      _aguix->setDashDFG( 1 );
      _aguix->setDashDBG( 0 );
      _aguix->DrawDashDRectangle( buffer, 
				 ox,
				 vis * elementHeight + oy,
				 maxdisplayh * elementWidth,
				 elementHeight );
#endif
    }
  }
}

void FieldListView::Content::setFG( colorclass_t cc, int col )
{
  fg[ cc ] = col;
}

int FieldListView::Content::getFG( colorclass_t cc ) const
{
  return fg[ cc ];
}

void FieldListView::Content::setBG( colorclass_t cc, int col )
{
  bg[ cc ] = col;
}

int FieldListView::Content::getBG( colorclass_t cc ) const
{
  return bg[ cc ];
}

void FieldListView::setFG( int row, colorclass_t cc, int col )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setFG( cc, col );
}

int FieldListView::getFG( int row, colorclass_t cc ) const
{
  if ( ( row < 0 ) || ( row >= elements ) ) return 0;
  return elementarray[ row ]->getFG( cc );
}

void FieldListView::setBG( int row, colorclass_t cc, int col )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setBG( cc, col );
}

int FieldListView::getBG( int row, colorclass_t cc ) const
{
  if ( ( row < 0 ) || ( row >= elements ) ) return 0;
  return elementarray[ row ]->getBG( cc );
}

void FieldListView::Content::setSelect( bool nv )
{
  select = nv;
}

bool FieldListView::Content::getSelect() const
{
  return select;
}

void FieldListView::Content::setMark( bool nv )
{
  mark = nv;
}

bool FieldListView::Content::getMark() const
{
  return mark;
}

/*
 * setSelectQ: set select status
 * only disadvantage to setSelect is that the row isn't redrawed
 */
void FieldListView::setSelectQ( int row, bool nv )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setSelect( nv );
}

void FieldListView::setSelect( int row, bool nv )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  setSelectQ( row, nv );
  redraw( row );
}

bool FieldListView::getSelect( int row ) const
{
  if ( ( row < 0 ) || ( row >= elements ) ) return false;
  return elementarray[ row ]->getSelect();
}

/*
 * setActiveRowQ
 * sets the active row without redrawing of affected rows
 */
void FieldListView::setActiveRowQ( int nv )
{
  if ( ( nv < -1 ) || ( nv >= elements ) ) return;
  activerow = nv;
}

void FieldListView::setActiveRow( int nv )
{
  int oldact;

  if ( ( nv < -1 ) || ( nv >= elements ) ) return;

  // remember old activerow for redraw
  oldact = activerow;

  setActiveRowQ( nv );

  redraw( activerow );
  if ( oldact >= 0 ) redraw( oldact );  // redraw old active element
}

int FieldListView::getActiveRow() const
{
  return activerow;
}

/*
 * setVisMaskQ: set visual mark status
 * only disadvantage to setVisMask is that the row isn't redrawed
 */
void FieldListView::setVisMarkQ( int row, bool nv )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setMark( nv );
}

void FieldListView::setVisMark( int row, bool nv )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  setVisMarkQ( row, nv );
  redraw( row );
}

bool FieldListView::getVisMark( int row ) const
{
  if ( ( row < 0 ) || ( row >= elements ) ) return false;
  return elementarray[ row ]->getMark();
}

void FieldListView::resize( int tw, int th )
{
  if ( ( tw < 6 ) || ( th < 6 ) ) return;
  _w = tw;
  _h = th;
  if ( isCreated() == true ) {
    _parent->resizeSubWin( win, tw, th );
    _aguix->freePixmap( buffer );
    buffer = _aguix->createPixmap( win, tw, th );
    recalcMaxValues();
    setupBars();
    if ( ( yoffset + maxdisplayv ) >= elements ) {
      setYOffset( yoffset );
    }
    if ( ( xoffset + maxdisplayh ) >= getMaxLen() ) {
      setXOffset( xoffset );
    }
    resetWin( true );
  }
}

int FieldListView::getData() const
{
  return data;
}

void FieldListView::setData( int tdata )
{
  this->data = tdata;
}

void FieldListView::redraw()
{
  mredraw();
  hbarredraw();
  vbarredraw();
  redrawHeader();
  redrawContent();
}

void FieldListView::flush()
{
  drawBuffer();
}

void FieldListView::handleExpose( Window msgwin, int ex, int ey, int ew, int eh )
{
  if ( isCreated() == false ) return;
  if ( msgwin == win ) {
    drawBuffer( ex, ey, ew, eh );
  } else if ( msgwin == hbarwin ) hbarredraw();
  else if ( msgwin == vbarwin ) vbarredraw();
}

bool FieldListView::handleMessage( XEvent *E, Message *msg )
{
  bool returnvalue;
  struct timeval t2;
  int dt, s, us, scrollspeed, f;

  if ( isCreated() == false ) return false;

  returnvalue = false;
  if ( msg->type == Expose ) {
    handleExpose( msg->window,
		  msg->x,
		  msg->y,
		  msg->width,
		  msg->height );
  }
  if ( vbarMode != VBAR_IDLE ) handleVBar( msg );
  else if ( hbarMode != HBAR_IDLE ) handleHBar( msg );
  else if ( thirdMode != THIRD_IDLE ) handleThird( msg );
  else if ( selectMode != SELECT_IDLE ) handleSelect( msg );
  else if ( middleMode != MIDDLE_IDLE ) handleMiddle( msg );
  else {
    if ( msg->type == ButtonPress ) {
      // jetzt men wir was machen
      bool isc = false;
      if ( msg->window == hbarwin ) {
	takeFocus();
	handleHBar( msg );
	isc = true;
      } else if ( msg->window == vbarwin ) {
	takeFocus();
	handleVBar( msg );
	isc = true;
      } else if ( msg->window == win ) {
	// handleSelect, handleThird
	takeFocus();
	
	f = getFieldForPoint( msg->mousex, msg->mousey );
	
	if ( ( f >= 0 ) && ( f < fields ) ) {
	  // field clicked
	  handleHeader( msg, f );
	} else {
	  if ( msg->button == Button3 ) handleThird( msg );
	  else if ( ( msg->button == Button4 ) || ( msg->button == Button5 ) ) {
	    gettimeofday( &t2, NULL );
	    s = abs( (int)( t2.tv_sec - lastwheel.tv_sec ) );
	    us = t2.tv_usec - lastwheel.tv_usec;
	    if ( us < 0 ) {
	      us += 1000000;
	      s--;
	    }
	    dt = us + s * 1000000;
	    dt /= 100000;
	    if ( dt < 2 ) scrollspeed = 5;
	    else if ( dt < 4 ) scrollspeed = 2;
	    else scrollspeed = 1;
	    if ( msg->button == Button4 ) scrollV1( -scrollspeed );
	    else if ( msg->button == Button5 ) scrollV1( scrollspeed );
	    lastwheel = t2;
	  } else if ( msg->button == Button2 ) {
	    handleMiddle( msg );
	  } else {
	    handleSelect( msg );
	  }
	}
	isc = true;
      }
      if ( isc == true ) {
	AGMessage *agmsg = AGUIX_allocAGMessage();
	agmsg->fieldlv.lv = this;
	agmsg->fieldlv.row = -1;
	agmsg->fieldlv.time = msg->time;
	agmsg->fieldlv.mouse = true;
	agmsg->type = AG_FIELDLV_PRESSED;
	_aguix->putAGMsg( agmsg );
      }
    } else if ( msg->type == ButtonRelease ) {
      // handle button release only for header
      if ( msg->window == win ) {
	handleHeader( msg, -1 );
      }
    } else if ( msg->type == KeyPress ) {
      if ( ( getAcceptFocus() == true ) && ( getHasFocus() == true ) ) {
	if ( isVisible() == true ) {
	  if ( _parent->isTopParent( msg->window ) == true ) {
	    // we have the focus so let's react to some keys
	    // lets call an extra handler so it can be overwritten
	    returnvalue = handleKeys( msg );
	  }
	}
      }
    }
  }
  if ( msg->lockElement == this ) msg->ack = true;
  return returnvalue;
}

void FieldListView::hbarredraw()
{
  if ( hbar == 0 ) return;
  if ( isCreated() == false ) return;
  int tx, ty, tw, th;
  int maxlen = getMaxLen();

  tw = _w - 2 - ( ( displayFocus == true ) ? 2 : 0 );
  if ( vbar != 0 ) tw -= vbar_width;
  th = hbar_height;
  ty = 0;
  tx = 0;

  _aguix->SetWindowBG( hbarwin, _parent->getBG() );
  _aguix->ClearWin( hbarwin );
  if ( hbar_dir == 1 ) {
    _aguix->setFG( 1 );
  } else {
    _aguix->setFG( 2 );
  }
  _aguix->DrawLine( hbarwin, tx, ty + th - 1, tx, ty );
  _aguix->DrawLine( hbarwin, tx, ty, tx + th - 1, ty );
  if ( hbar_dir == 1 ) {
    _aguix->setFG( 2 );
  } else {
    _aguix->setFG( 1 );
  }
  _aguix->DrawLine( hbarwin, tx + 1, ty + th - 1, tx + th - 1, ty + th - 1 );
  _aguix->DrawLine( hbarwin, tx + th - 1, ty + th - 1, tx + th - 1, ty + 1 );
  // Left-Arrow
  _aguix->setFG( 1 );
  _aguix->DrawLine( hbarwin, tx + th * 4 / 5, ty + th / 5, tx + th / 5, ty + th / 2 );
  _aguix->DrawLine( hbarwin, tx + th / 5, ty + th / 2, tx + th * 4 / 5, ty + th * 4 / 5 );

  _aguix->DrawLine( hbarwin, tx + th, ty + th - 1, tx + tw - th - 1, ty + th - 1 );
  _aguix->setFG( 2 );
  _aguix->DrawLine( hbarwin, tx + th, ty, tx + tw - th - 1, ty );

  if ( hbar_dir == 2 ) {
    _aguix->setFG( 1 );
  } else {
    _aguix->setFG( 2 );
  }
  _aguix->DrawLine( hbarwin,
		   tx + tw - th,
		   ty + th - 1,
		   tx + tw - th,
		   ty );
  _aguix->DrawLine( hbarwin,
		   tx + tw - th,
		   ty,
		   tx + tw - 1 ,
		   ty );
  if ( hbar_dir == 2 ) {
    _aguix->setFG( 2 );
  } else {
    _aguix->setFG( 1 );
  }
  _aguix->DrawLine( hbarwin,
		   tx + tw - th + 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + th - 1 );
  _aguix->DrawLine( hbarwin,
		   tx + tw - 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + 1 );
  // Right-Arrow
  _aguix->setFG( 1 );
  _aguix->DrawLine( hbarwin,
		   tx + tw - th * 4 / 5,
		   ty + th / 5,
		   tx + tw - th / 5,
		   ty + th / 2 );
  _aguix->DrawLine( hbarwin,
		   tx + tw - th / 5,
		   ty + th / 2,
		   tx + tw - th * 4 / 5,
		   ty + th * 4 / 5 );
  // jetzt noch die eigentliche Leiste
  tw -= 2 * th;  // Breite wenn alles anzeigbar ist
  th -= 2;
  ty++;
  if ( maxlen > maxdisplayh ) {
    int dw = tw * maxdisplayh;
    dw /= maxlen;

    if ( dw < 10 ) dw = 10;

    int a = maxlen - maxdisplayh;
    double b = tw - dw;  // das was brig bleibt
    b /= a;            // verteilt sich auch die nichtsichtbaren Zeichen
    tx = tx + hbar_height + (int)( xoffset * b );
    tw = dw;
  } else {
    tx = tx + hbar_height;
  }
  if ( hbar_pressed == false ) {
    _aguix->setFG( 2 );
  } else {
    _aguix->setFG( 1 );
  }
  _aguix->DrawLine( hbarwin,
		   tx,
		   ty + th - 1,
		   tx,
		   ty );
  _aguix->DrawLine( hbarwin,
		   tx,
		   ty,
		   tx + tw - 1,
		   ty );
  if ( hbar_pressed == false ) {
    _aguix->setFG( 1 );
  } else {
    _aguix->setFG( 2 );
  }
  _aguix->DrawLine( hbarwin,
		   tx + 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + th - 1 );
  _aguix->DrawLine( hbarwin,
		   tx + tw - 1,
		   ty + th - 1,
		   tx + tw - 1,
		   ty + 1 );  
}

void FieldListView::vbarredraw()
{
  if(vbar==0) return;
  if ( isCreated() == false ) return;
  int tx,ty,tw,th;

  th=_h-2 - ( ( displayFocus == true ) ? 2 : 0 );
  if(hbar!=0) th-=hbar_height;
  tw=vbar_width;
  tx=0;
  ty=0;

  _aguix->SetWindowBG(vbarwin,_parent->getBG());
  _aguix->ClearWin(vbarwin);
  if(vbar_dir==1) {
    _aguix->setFG(1);
  } else {
    _aguix->setFG(2);
  }
  _aguix->DrawLine(vbarwin,tx,ty+tw-1,tx,ty);
  _aguix->DrawLine(vbarwin,tx,ty,tx+tw-1,ty);
  if(vbar_dir==1) {
    _aguix->setFG(2);
  } else {
    _aguix->setFG(1);
  }
  _aguix->DrawLine(vbarwin,tx+1,ty+tw-1,tx+tw-1,ty+tw-1);
  _aguix->DrawLine(vbarwin,tx+tw-1,ty+tw-1,tx+tw-1,ty+1);
  // Up-Arrow
  _aguix->setFG(1);
  _aguix->DrawLine(vbarwin,tx+tw/5,ty+tw*4/5,tx+tw/2,ty+tw/5);
  _aguix->DrawLine(vbarwin,tx+tw/2,ty+tw/5,tx+tw*4/5,ty+tw*4/5);

  _aguix->DrawLine(vbarwin,tx+tw-1,ty+tw,tx+tw-1,ty+th-tw-1);
  _aguix->setFG(2);
  _aguix->DrawLine(vbarwin,tx,ty+tw,tx,ty+th-tw-1);

  if(vbar_dir==2) {
    _aguix->setFG(1);
  } else {
    _aguix->setFG(2);
  }
  _aguix->DrawLine(vbarwin,tx,ty+th-1,tx,ty+th-tw);
  _aguix->DrawLine(vbarwin,tx,ty+th-tw,tx+tw-1,ty+th-tw);
  if(vbar_dir==2) {
    _aguix->setFG(2);
  } else {
    _aguix->setFG(1);
  }
  _aguix->DrawLine(vbarwin,tx+1,ty+th-1,tx+tw-1,ty+th-1);
  _aguix->DrawLine(vbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+th-tw);
  // Down-Arrow
  _aguix->setFG(1);
  _aguix->DrawLine(vbarwin,tx+tw/5,ty+th-tw*4/5,tx+tw/2,ty+th-tw/5);
  _aguix->DrawLine(vbarwin,tx+tw/2,ty+th-tw/5,tx+tw*4/5,ty+th-tw*4/5);
  // jetzt noch die eigentliche Leiste
  th-=2*tw;  // Hhe wenn alles anzeigbar ist
  tw-=2;
  tx++;
  int curelements=getElements();
  if ( curelements > maxdisplayv ) {
    int dh=th*maxdisplayv;
    dh /= curelements;

    if ( dh < 10 ) dh = 10;

    int a = curelements - maxdisplayv;
    double b=th-dh;  // das was brig bleibt
    b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
    ty=ty+vbar_width+(int)(yoffset*b);
    th=dh;
  } else {
    ty=ty+vbar_width;
  }
  if(vbar_pressed==false) {
    _aguix->setFG(2);
  } else {
    _aguix->setFG(1);
  }
  _aguix->DrawLine(vbarwin,tx,ty+th-1,tx,ty);
  _aguix->DrawLine(vbarwin,tx,ty,tx+tw-1,ty);
  if(vbar_pressed==false) {
    _aguix->setFG(1);
  } else {
    _aguix->setFG(2);
  }
  _aguix->DrawLine(vbarwin,tx+1,ty+th-1,tx+tw-1,ty+th-1);
  _aguix->DrawLine(vbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+1);  
}

void FieldListView::mredraw()
{
  if ( isCreated() == false ) return;
  _aguix->setFG( mbg );
  _aguix->FillRectangle( buffer, 0, 0, _w, _h );
  _aguix->setFG( 2 );
  _aguix->DrawLine( buffer, 0, _h - 1, 0, 0 );
  _aguix->DrawLine( buffer, 0, 0, _w - 1, 0 );
  _aguix->setFG( 1 );
  _aguix->DrawLine( buffer, 1, _h - 1, _w - 1, _h - 1 );
  _aguix->DrawLine( buffer, _w - 1, _h - 1, _w - 1, 1 );
  if ( ( hbar != 0 ) && ( vbar != 0 ) ) {
    int tx, ty;
    tx = ( vbar == 1 ) ? ( ( displayFocus == true ) ? 2 : 1 ) : ( _w - ( ( displayFocus == true ) ? 2 : 1 ) - vbar_width );
    ty = ( hbar == 1 ) ? ( ( displayFocus == true ) ? 2 : 1 ) : ( _h - ( ( displayFocus == true ) ? 2 : 1 ) - hbar_height );
    _aguix->setFG( 0 );
    _aguix->FillRectangle( buffer, tx, ty, vbar_width, hbar_height );
  }
  if ( displayFocus == true ) {
    if ( getHasFocus() == true ) {
      _aguix->setFG( 1 );
      _aguix->DrawLine( buffer, 1, _h - 2, 1, 1 );
      _aguix->DrawLine( buffer, 1, 1, _w - 2, 1 );
      _aguix->setFG( 2 );
      _aguix->DrawLine( buffer, 2, _h - 2, _w - 2, _h - 2 );
      _aguix->DrawLine( buffer, _w - 2, _h - 2, _w - 2, 2 );
    } else {
      _aguix->setFG( 2 );
      _aguix->DrawLine( buffer, 1, _h - 2, 1, 1 );
      _aguix->DrawLine( buffer, 1, 1, _w - 2, 1 );
      _aguix->setFG( 1 );
      _aguix->DrawLine( buffer, 2, _h - 2, _w - 2, _h - 2 );
      _aguix->DrawLine( buffer, _w - 2, _h - 2, _w - 2, 2 );
    }
  } 
}

void FieldListView::recalcMaxValues()
{
  int th, tw;

  if ( font == NULL ) elementHeight = _aguix->getCharHeight();
  else elementHeight = font->getCharHeight();

  if ( font == NULL ) elementWidth = _aguix->getCharWidth();
  else elementWidth = font->getCharWidth();

  /* this is for maxdisplayv/h ... */
  if ( hbar != 0 ) {
    th = _h - 2 - hbar_height;
  } else {
    th = _h - 2;
  }

  if ( displayFocus == true )
    th -= 2;

  if ( showHeader == true )
    th -= getHeaderHeight();

  if ( th < 0 ) th = 0;
  maxdisplayv = th / elementHeight;

  if ( vbar != 0 ) {
    tw = _w - 2 - vbar_width;
  } else {
    tw = _w - 2;
  }

  if ( displayFocus == true )
    tw -= 2;

  if ( tw < 0 ) tw = 0;
  maxdisplayh = tw / elementWidth;

  /* ... and this for the drawing of the elements */

  if ( hbar == 1 ) oy = 1 + hbar_height;
  else oy = 1;
  if ( vbar == 1 ) ox = 1 + vbar_width;
  else ox = 1;

  if ( displayFocus == true ) {
    oy++;
    ox++;
  }
  if ( showHeader == true ) {
    oy += getHeaderHeight();
  }
}

int FieldListView::getElements() const
{
  return elements;
}

int FieldListView::getXOffset() const
{
  return xoffset;
}

int FieldListView::getYOffset() const
{
  return yoffset;
}

int FieldListView::getMaxLen()
{
  return getMaxTextWidth();
}

int FieldListView::getMaxDisplayV() const
{
  return maxdisplayv;
}

int FieldListView::getMaxDisplayH() const
{
  return maxdisplayh;
}

void FieldListView::setHBarState( int state )
{
  if ( hbar == state ) return;
  if ( ( state == 0 ) && ( isCreated() == true ) ) _parent->removeSubWin( hbarwin );
  if ( ( hbar == 0 ) && ( state > 0 ) && ( isCreated() == true ) ) hbarwin = _parent->getSubWindow( win, 0, 0, 5, 5 );
  hbar = state;
  recalcMaxValues();
  setupBars();
  resetWin( true );
}

void FieldListView::setVBarState(int state)
{
  if ( state == vbar ) return;
  if ( ( state == 0 ) && ( isCreated() == true ) ) _parent->removeSubWin( vbarwin );
  if ( ( vbar == 0 ) && ( state > 0 ) && ( isCreated() == true ) ) vbarwin = _parent->getSubWindow( win, 0, 0, 5, 5 );
  vbar = state;
  recalcMaxValues();
  setupBars();
  resetWin( true );
}

int FieldListView::getHBarState() const
{
  return hbar;
}

int FieldListView::getVBarState() const
{
  return vbar;
}

void FieldListView::setVBarWidth( int new_width )
{
  if ( ( new_width < 0 ) || ( ( new_width + 4 ) >= _w ) ) return;
  vbar_width = new_width;
  recalcMaxValues();
  setupBars();
  vbarredraw();
  resetWin( true );
}

int FieldListView::getVBarWidth() const
{
  return vbar_width;
}

void FieldListView::setHBarHeight( int new_height )
{
  if ( ( new_height < 0 ) || ( ( new_height + 4 ) >= _h ) ) return;
  hbar_height = new_height;
  recalcMaxValues();
  setupBars();
  hbarredraw();
  resetWin( true );
}

int FieldListView::getHBarHeight() const
{
  return hbar_height;
}

/*
 * clearNonElements
 *
 * will reset background for unused entries
 *
 */
void FieldListView::clearNonElements()
{
  int sy,e;

  if ( isCreated() == false ) return;
  if ( ( yoffset + maxdisplayv ) > elements ) {
    sy = elements - yoffset;
    sy = sy * elementHeight + oy;
    e = maxdisplayv - ( elements - yoffset );
    _aguix->setFG( mbg );
    _aguix->FillRectangle( buffer, ox, sy, maxdisplayh * elementWidth, e * elementHeight );
  }
}

int FieldListView::selectRowsRange( int startRow, int endRow, bool state )
{
  int dire, selected = 0;

  int curRow = -1;

  if ( startRow >= elements ) startRow = elements - 1;
  if ( endRow >= elements ) endRow = elements - 1;
  if ( ( startRow < 0 ) || ( endRow < 0 ) ) return 0;

  // skip first entry because it's selected already
  if ( startRow < endRow ) {
    startRow++;
    dire = 1;
  } else if ( startRow > endRow ) {
    startRow--;
    dire = -1;
  } else dire = 0;
  
  for ( curRow = startRow; ; curRow += dire ) {
    if ( curRow >= elements ) break;
    if ( curRow < 0 ) break;

    setSelect( curRow, state );
    if ( curRow == endRow ) setActiveRow( curRow );
    runSelectHandler( curRow );
    selected++;

    if ( curRow == endRow ) break;
  }
  return selected;
}

void FieldListView::handleSelect( Message *msg )
{
  int j;
  Window root, child;
  int root_x, root_y, win_x, win_y;
  unsigned int keys_buttons;
  int newelement;
  AGMessage *agmsg;
  
  if ( isCreated() == false ) return;

  if ( msg->type == ButtonPress ) {
    if ( msg->window != win ) return;
    if ( selectMode != SELECT_IDLE ) return;

    j = getRowForMouseY( msg->mousey );
    if ( j < 0 ) return;
    if ( ( j + yoffset ) >= elements ) return;
    if ( j >= maxdisplayv ) return;

    clickLastSelectedRow = j + yoffset;
    clickState = ( getSelect( clickLastSelectedRow ) == false ) ? true : false;
    clickFirstSelectedRow = clickLastSelectedRow;
    clickSelected = 1;

    setSelect( clickLastSelectedRow, clickState );
    setActiveRow( clickLastSelectedRow );
    runSelectHandler( clickLastSelectedRow );
    
    selectMode = SELECT_HOLD;

    _aguix->Flush();
    _aguix->msgLock( this );
  } else if ( msg->type == MotionNotify ) {
    if ( ( msg->window == win ) &&
	 ( ( selectMode == SELECT_HOLD ) ||
	   ( selectMode == SELECT_SCROLL_DOWN ) ||
	   ( selectMode == SELECT_SCROLL_UP ) ) ) {
      //    _aguix->queryPointer( win, &mx, &my );
      XQueryPointer( _aguix->getDisplay(),
		     msg->window,
		     &root,
		     &child,
		     &root_x,
		     &root_y,
		     &win_x,
		     &win_y,
		     &keys_buttons );
      if ( ( keys_buttons & Button1Mask ) != 0 ) {
	j = getRowForMouseY( win_y );
	if ( ( j < 0 ) && ( yoffset > 0 ) ) {
	  newelement = yoffset - 1;
	  if ( newelement >= 0 ) {
	    if ( selectMode != SELECT_SCROLL_UP ) {
	      scrollV1( -1 );
	      clickSelected += selectRowsRange( clickLastSelectedRow, newelement, clickState );
	      clickLastSelectedRow = newelement;
	      selectMode = SELECT_SCROLL_UP;
	      _aguix->enableTimer();
	    }
	  }
	} else if ( ( j >= maxdisplayv ) && ( ( yoffset + maxdisplayv ) < elements ) ) {
	  newelement = yoffset + maxdisplayv;
	  if ( newelement < elements ) {
	    if ( selectMode != SELECT_SCROLL_DOWN ) {
	      scrollV1( 1 );
	      clickSelected += selectRowsRange( clickLastSelectedRow, newelement, clickState );
	      clickLastSelectedRow = newelement;
	      selectMode = SELECT_SCROLL_DOWN;
	      _aguix->enableTimer();
	    }
	  }
	} else {
	  if ( j < 0 ) j = 0;
	  if ( ( j + yoffset ) >= elements ) {
	    j = elements - 1 - yoffset;
	  }
	  if ( ( j + yoffset ) != clickLastSelectedRow ) {
	    newelement = j + yoffset;
	    clickSelected += selectRowsRange( clickLastSelectedRow, newelement, clickState );
	    clickLastSelectedRow = newelement;
	  }
	  if ( selectMode != SELECT_HOLD ) {
	    selectMode = SELECT_HOLD;
	    _aguix->disableTimer();
	  }
	}
      }
    }
  } else if ( msg->type == ButtonRelease ) {
    if ( ( msg->window == win ) &&
	 ( msg->button == Button1 ) ) {
      switch ( selectMode ) {
	case SELECT_IGNORE:
	  _aguix->msgUnlock( this );
	  selectMode = SELECT_IDLE;
	  break;
	case SELECT_SCROLL_UP:
	case SELECT_SCROLL_DOWN:
	  _aguix->disableTimer();
	case SELECT_HOLD:
	  agmsg = AGUIX_allocAGMessage();
	  agmsg->fieldlv.lv = this;
	  agmsg->fieldlv.row = clickFirstSelectedRow;
	  agmsg->fieldlv.time = msg->time;
	  agmsg->fieldlv.mouse = true;
	  if ( clickSelected == 1 ) {
	    agmsg->type = AG_FIELDLV_ONESELECT;
	  } else {
	    agmsg->type = AG_FIELDLV_MULTISELECT;
	  }
	  _aguix->putAGMsg( agmsg );
	  _aguix->msgUnlock( this );
	  selectMode = SELECT_IDLE;
	  break;
	default:
	  break;
      }
    }
  } else if ( msg->type == ClientMessage ) {
    if ( msg->specialType == Message::TIMEREVENT ) {
      if ( ( msg->time % 1 ) == 0 ) { //HARDCODED
	if ( selectMode == SELECT_SCROLL_UP ) {
	  newelement = clickLastSelectedRow - 1;
	  if ( newelement >= 0 ) {
	    scrollV1( -1 );
	    clickSelected += selectRowsRange( clickLastSelectedRow, newelement, clickState );
	    clickLastSelectedRow = newelement;
	  }
	} else if ( selectMode == SELECT_SCROLL_DOWN ) {
	  newelement = clickLastSelectedRow + 1;
	  if ( newelement < elements ) {
	    scrollV1( 1 );
	    clickSelected += selectRowsRange( clickLastSelectedRow, newelement, clickState );
	    clickLastSelectedRow = newelement;
	  }
	}
	_aguix->Flush();
      }
    }
  }
}

void FieldListView::updateScrollValues( int mx, int my )
{
  int midx, midy, dsx, dsy;

  midx = _w - 2;
  if ( vbar != 0 ) midx -= vbar_width;
  dsx = midx;
  midx /= 2;
  dsx /= 6;

  midy = _h - 2;
  if ( hbar != 0 ) midy -= hbar_height;
  if ( showHeader == true ) midy -= getHeaderHeight();
  dsy = midy;
  midy /= 2;
  dsy /= 6;

  if ( mx < ( midx - dsx ) ) scrollXDir = -1;
  else if ( mx > ( midx + dsx ) ) scrollXDir = 1;
  else scrollXDir = 0;
  
  if ( my < ( midy - dsy ) ) scrollYDir = -1;
  else if ( my > ( midy + dsy ) ) scrollYDir = 1;
  else scrollYDir = 0;
  
  //HARDCODED
  if ( ( my < dsy ) || ( my > ( midy + 2 * dsy ) ) ) scrollDelta = 1;
  else if ( ( mx < dsx ) || ( mx > ( midx + 2 * dsx ) ) ) scrollDelta = 1;
  else scrollDelta = 3;
}

void FieldListView::handleThird(Message *msg)
{
  int mx, my;
  int j;
  
  if ( isCreated() == false ) return;

  if ( msg->type == ButtonPress ) {
    if ( msg->window != win ) return;
    if ( thirdMode != THIRD_IDLE ) return;

    j = getRowForMouseY( msg->mousey );
    if ( j < 0 ) return;
    if ( j >= maxdisplayv ) return;

    mx = msg->mousex;
    my = msg->mousey;
    my -= oy;
    mx -= ox;

    updateScrollValues( mx, my );

    thirdMode = THIRD_SCROLL;

    if ( scrollXDir < 0 ) setXOffset( xoffset - 1 );
    else if ( scrollXDir > 0 ) setXOffset( xoffset + 1 );
    if ( scrollYDir < 0 ) scrollV1( -1 );
    else if ( scrollYDir > 0 ) scrollV1( 1 );
    _aguix->Flush();
    _aguix->msgLock( this );
    _aguix->enableTimer();
  } else if ( msg->type == MotionNotify ) {
    if ( thirdMode == THIRD_SCROLL ) {
      _aguix->queryPointer( win, &mx, &my );
      my -= oy;
      mx -= ox;
      
      updateScrollValues( mx, my );
    }
  } else if ( msg->type == ButtonRelease ) {
    if ( ( msg->window == win ) &&
	 ( msg->button == Button3 ) ) {
      switch ( thirdMode ) {
	case THIRD_SCROLL:
	  _aguix->disableTimer();
	  _aguix->msgUnlock( this );
	  thirdMode = THIRD_IDLE;
	  break;
	default:
	  break;
      }
    }
  } else if ( msg->type == ClientMessage ) {
    if ( ( msg->specialType == Message::TIMEREVENT ) &&
	 ( thirdMode == THIRD_SCROLL ) ) {
      if ( ( msg->time % scrollDelta ) == 0 ) {
	if ( scrollXDir < 0 ) setXOffset( xoffset - 1 );
	else if ( scrollXDir > 0 ) setXOffset( xoffset + 1 );
	if ( scrollYDir < 0 ) scrollV1( -1 );
	else if ( scrollYDir > 0 ) scrollV1( 1 );
	_aguix->Flush();
      }
    }
  }
}

void FieldListView::handleHBar( Message *msg )
{
  if ( hbar == 0 ) return;
  if ( isCreated() == false ) return;
  int mx, my;
  int dir;
  int hx, hy, hw, hh;
  int maxlen = getMaxLen();
  int tx, ty, tw, th;
  double b = 0;

  mx = msg->mousex;
  my = msg->mousey;

  hw = _w - 2 - ( ( displayFocus == true ) ? 2 : 0 );
  if ( vbar != 0 ) hw -= vbar_width;
  hh = hbar_height;
  hy = 0;
  hx = 0;

  tw = hw;
  th = hbar_height;
  ty = 0;
  tx = 0;
  tw -= 2 * th;  // Breite wenn alles anzeigbar ist
  th -= 2;
  ty++;
  if ( maxlen > maxdisplayh ) {
    int dw = tw * maxdisplayh;
    dw /= maxlen;
    
    if ( dw < 10 ) dw = 10;
    
    int a = maxlen - maxdisplayh;
    b = tw - dw;  // das was brig bleibt
    b /= a;            // verteilt sich auch die nichtsichtbaren Zeichen
    tx += hbar_height + (int)( xoffset * b );
    tw = dw;
  } else {
    tx += hbar_height;
  }
  
  if ( msg->type == ButtonPress ) {
    if ( hbarMode != HBAR_IDLE ) return;
    if ( msg->window == hbarwin ) {
      dir = 0;
      if ( msg->button == Button1 ) {
	if ( ( mx > hx ) && ( mx <= ( hx + hh ) ) ) {
	  // left
	  dir = -1;
	  hbar_dir = 1;
	  hbarMode = HBAR_SCROLL_LEFT;
	}
	if ( ( mx > ( hx + hw - hh ) ) && ( mx <= ( hx + hw ) ) ) {
	  // right
	  dir = 1;
	  hbar_dir = 2;
	  hbarMode = HBAR_SCROLL_RIGHT;
	}
      }

      if ( dir != 0 ) {
	vbarredraw();
	setXOffset( xoffset + dir );
	_aguix->Flush();
	_aguix->msgLock( this );
	_aguix->enableTimer();
      } else if ( msg->button == Button4 ) {
	setXOffset( xoffset + ( -maxdisplayh + 1 ) / 4 );
	_aguix->Flush();
      } else if ( msg->button == Button5 ) {
	setXOffset( xoffset + ( maxdisplayh - 1 ) / 4 );
	_aguix->Flush();
      } else if ( ( mx >= tx ) &&
		  ( mx <= ( tx + tw ) ) &&
		  ( my >= ty ) &&
		  ( my <= ( ty + th ) ) &&
		  ( msg->button == Button1 ) ) {
	// the hbar-scroller is pressed
	if ( maxlen > maxdisplayh ) {
	  grabs = XGrabPointer( _aguix->getDisplay(),
				hbarwin,
				False,
				Button1MotionMask | ButtonReleaseMask | EnterWindowMask | LeaveWindowMask,
				GrabModeAsync,
				GrabModeAsync,
				None,
				None,
				CurrentTime );
	  _aguix->setCursor( hbarwin, AGUIX::SCROLLH_CURSOR );
	  scrollDelta = mx - tx;
	  hbar_pressed = true;
	  hbarredraw();
	  _aguix->Flush();
	  hbarMode = HBAR_SCROLL;
	  _aguix->msgLock( this );
	}
      } else {
	if ( msg->button == Button1 ) {
	  if ( mx < tx ) {
	    setXOffset( xoffset - maxdisplayh + 1 );
	  } else {
	    setXOffset( xoffset + maxdisplayh - 1 );
	  }
	  _aguix->Flush();
	}
      }
    }
  } else if ( msg->type == MotionNotify ) {
    if ( ( msg->window == hbarwin ) &&
	 ( hbarMode == HBAR_SCROLL ) ) {
      int tmx;
      double f1;
      tmx = msg->mousex;
      tmx -= hbar_height;
      tmx -= scrollDelta;
      f1 = tmx / b;
      setXOffset( (int)f1 );
      _aguix->Flush();
    }
  } else if ( msg->type == ButtonRelease ) {
    if ( ( msg->window == hbarwin ) &&
	 ( msg->button == Button1 ) ) {
      switch ( hbarMode ) {
	case HBAR_SCROLL_LEFT:
	case HBAR_SCROLL_RIGHT:
	  hbar_dir = 0;
	  hbarredraw();
	  _aguix->Flush();
	  _aguix->disableTimer();
	  _aguix->msgUnlock( this );
	  break;
	case HBAR_SCROLL:
	  _aguix->unsetCursor( hbarwin );
	  if ( grabs == GrabSuccess ) XUngrabPointer( _aguix->getDisplay(), CurrentTime );
	  hbar_pressed=false;
	  hbarredraw();
	  _aguix->Flush();
	  _aguix->msgUnlock( this );
	  break;
	default:
	  break;
      }
      hbarMode = HBAR_IDLE;
    }
  } else if ( msg->type == ClientMessage ) {
    if ( ( msg->specialType == Message::TIMEREVENT ) &&
	 ( hbarMode != HBAR_IDLE ) ) {
      if ( ( msg->time % 2 ) == 0 ) { //HARDCODED
	switch ( hbarMode ) {
	  case HBAR_SCROLL_LEFT:
	    setXOffset( xoffset - 1 );
	    break;
	  case HBAR_SCROLL_RIGHT:
	    setXOffset( xoffset + 1 );
	    break;
	default:
	  break;
	}
	_aguix->Flush();
      }
    }
  }
  return;
}

void FieldListView::handleVBar(Message *msg)
{
  if ( vbar == 0 ) return;
  if ( isCreated() == false ) return;
  int mx, my;
  int dir;
  int vx, vy, vw, vh;
  int tx, ty, tw, th;
  double b = 1;
  int curelements = getElements();

  mx = msg->mousex;
  my = msg->mousey;

  vh = _h - 2 - ( ( displayFocus == true ) ? 2 : 0 );
  if ( hbar != 0 ) vh -= hbar_height;
  vw = vbar_width;
  vx = 0;
  vy = 0;

  th = vh;
  tw = vbar_width;
  tx = 0;
  ty = 0;
  th -= 2 * tw;  // Hhe wenn alles anzeigbar ist
  tw -= 2;
  tx++;
  ty++;
  if ( curelements > maxdisplayv ) {
    int dh = th * maxdisplayv;
    dh /= curelements;
    
    if ( dh < 10 ) dh = 10;
    
    int a = curelements - maxdisplayv;
    b = th - dh;     // das was brig bleibt
    b /= a;          // verteilt sich auch die nichtsichtbaren Zeichen
    ty += vbar_width + (int)( yoffset * b );
    th = dh;
  } else {
    ty += vbar_width;
  }

  if ( msg->type == ButtonPress ) {
    if ( vbarMode != VBAR_IDLE ) return;
    if ( msg->window == vbarwin ) {
      dir = 0;
      if ( msg->button == Button1 ) {
	if ( ( my > vy ) && ( my <= ( vy + vw ) ) ) {
	  // up
	  dir = -1;
	  vbar_dir = 1;
	  vbarMode = VBAR_SCROLL_UP;
	}
	if ( ( my > ( vy + vh - vw ) ) && ( my <= ( vy + vh ) ) ) {
	  // down
	  dir = 1;
	  vbar_dir = 2;
	  vbarMode = VBAR_SCROLL_DOWN;
	}
      }

      if ( dir != 0 ) {
	vbarredraw();
	scrollV1( dir );
	_aguix->Flush();
	_aguix->msgLock( this );
	_aguix->enableTimer();
      } else if ( msg->button == Button4 ) {
	scrollV( ( -maxdisplayv + 1 ) / 4 );
	_aguix->Flush();
      } else if ( msg->button == Button5 ) {
	scrollV(  ( maxdisplayv - 1 ) / 4 );
	_aguix->Flush();
      } else if ( ( mx >= tx ) &&
		  ( mx <= ( tx + tw ) ) &&
		  ( my >= ty ) &&
		  ( my <= ( ty + th ) ) &&
		  ( msg->button == Button1 ) ) {
	// vbar-scroller pressed
	if ( curelements > maxdisplayv ) {
	  grabs = XGrabPointer( _aguix->getDisplay(),
				vbarwin,
				False,
				Button1MotionMask | ButtonReleaseMask | EnterWindowMask | LeaveWindowMask,
				GrabModeAsync,
				GrabModeAsync,
				None,
				None,
				CurrentTime );
	  _aguix->setCursor( vbarwin, AGUIX::SCROLLV_CURSOR );
	  scrollDelta = my - ty;
	  vbar_pressed = true;
	  vbarredraw();
	  _aguix->Flush();
	  vbarMode = VBAR_SCROLL;
	  _aguix->msgLock( this );
	}
      } else {
	if (msg->button == Button1 ) {
	  if ( my < ty ) {
	    scrollV( - maxdisplayv + 1 );
	  } else {
	    scrollV( maxdisplayv - 1 );
	  }
	  _aguix->Flush();
	}
      }
    }
  } else if ( msg->type == MotionNotify ) {
    if ( ( msg->window == vbarwin ) &&
	 ( vbarMode == VBAR_SCROLL ) ) {
      int tmy;
      double f1;
      tmy = msg->mousey;
      tmy -= vbar_width;
      tmy -= scrollDelta;
      f1 = tmy / b;
      setYOffset( (int)f1 );
      _aguix->Flush();
    }
  } else if ( msg->type == ButtonRelease ) {
    if ( ( msg->window == vbarwin ) &&
	 ( msg->button == Button1 ) ) {
      switch ( vbarMode ) {
	case VBAR_SCROLL_UP:
	case VBAR_SCROLL_DOWN:
	  vbar_dir = 0;
	  vbarredraw();
	  _aguix->Flush();
	  _aguix->disableTimer();
	  _aguix->msgUnlock( this );
	  break;
	case VBAR_SCROLL:
	  _aguix->unsetCursor(vbarwin);
	  if ( grabs == GrabSuccess ) XUngrabPointer( _aguix->getDisplay(), CurrentTime );
	  vbar_pressed=false;
	  vbarredraw();
	  _aguix->Flush();
	  _aguix->msgUnlock( this );
	  break;
	default:
	  break;
      }
      vbarMode = VBAR_IDLE;
    }
  } else if ( msg->type == ClientMessage ) {
    if ( ( msg->specialType == Message::TIMEREVENT ) &&
	 ( vbarMode != VBAR_IDLE ) ) {
      if ( ( msg->time % 2 ) == 0 ) { //HARDCODED
	switch ( vbarMode ) {
	  case VBAR_SCROLL_UP:
	    scrollV1( -1 );
	    break;
	  case VBAR_SCROLL_DOWN:
	    scrollV1( 1 );
	    break;
	default:
	  break;
	}
	_aguix->Flush();
      }
    }
  }
  return;
}

void FieldListView::scrollV(int delta)
{
  setYOffset(yoffset+delta);
}

void FieldListView::scrollV1(int dir)
{
  setYOffset(yoffset+dir);
}

void FieldListView::setSelectHandler( void (*nsh)( FieldListView*, int p ) )
{
  selecthandler = nsh;
}

/*
 * showActive zeigt den aktiven Eintrag
 * sie tut nichts, wenn Eintrag sichtbar
 * ist der oberhalb, wird er oben als erstes Element sichtbar
 * ist er unterhalb, wird er als letztes sichtbar
 */
void FieldListView::showActive()
{
  if ( activerow >= 0 ) {
    if ( activerow < yoffset ) setYOffset( activerow );
    else if ( ( activerow - yoffset ) >= maxdisplayv ) setYOffset( activerow - maxdisplayv + 1 );
  }
}

/*
 * centerActive zentriert Ansicht auf aktiven Eintrag
 * egal, ob sichtbar oder nicht
 */
void FieldListView::centerActive()
{
  if ( activerow >= 0 ) {
    setYOffset( activerow - maxdisplayv / 2 );
  }
}

/*
 * setupBars
 *
 * will move/resize hbar/vbar
 */
void FieldListView::setupBars()
{
  if ( isCreated() == false ) return;

  if ( hbar != 0 ) {
    int tx, ty, tw, th;
    tw = _w - 2 - ( ( displayFocus == true ) ? 2 : 0 );
    if ( vbar != 0 ) tw -= vbar_width;
    th = hbar_height;
    if ( hbar == 1 ) ty = 1 + ( ( displayFocus == true ) ? 1 : 0 );
    else ty = _h - 1 - th - ( ( displayFocus == true ) ? 1 : 0 );
    if ( vbar == 1 ) tx = 1 + vbar_width + ( ( displayFocus == true ) ? 1 : 0 );
    else tx = 1 + ( ( displayFocus == true ) ? 1 : 0 );
    
    // atleast 10 pixel width
    if ( tw < 10 ) tw = 10;
    _parent->resizeSubWin( hbarwin, tw, th );
    _parent->moveSubWin( hbarwin, tx, ty );
  }
  if ( vbar != 0 ) {
    int tx, ty, tw, th;
    th = _h - 2 - ( ( displayFocus == true ) ? 2 : 0 );
    if ( hbar != 0 ) th -= hbar_height;
    tw = vbar_width;
    if ( vbar == 1 ) tx = 1 + ( ( displayFocus == true ) ? 1 : 0 );
    else tx = _w - 1 - tw - ( ( displayFocus == true ) ? 1 : 0 );
    if ( hbar == 1 ) ty = 1 + hbar_height + ( ( displayFocus == true ) ? 1 : 0 );
    else ty = 1 + ( ( displayFocus == true ) ? 1 : 0 );
    
    // atleast 10 pixel height
    if ( th < 10 ) th = 10;
    _parent->resizeSubWin( vbarwin, tw, th );
    _parent->moveSubWin( vbarwin, tx, ty );
  }
}

int FieldListView::setFont(char *fontname)
{
  font=_aguix->getFont(fontname);
  if(font==NULL) return -1;
  recalcMaxValues();
  setupBars();
  resetWin( true );
  return 0;
}

const char *FieldListView::getType() const
{
  return type;
}

bool FieldListView::isType(const char *qtype) const
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

bool FieldListView::isParent(Window child) const
{
  if ( isCreated() == true ) {
    if(child==win) return true;
    if(child==hbarwin) return true;
    if(child==vbarwin) return true;
  }
  return false;
}

void FieldListView::runSelectHandler( int p )
{
  if ( selecthandler != NULL ) ( selecthandler )( this, p );
}

void FieldListView::showRow( int row )
{
  if ( ( row >=0 ) && ( row < elements ) ){
    if ( row < yoffset ) {
      if ( abs( row - yoffset ) == 1 ) scrollV1( -1 );
      else scrollV( row - yoffset );
    } else if ( ( row - yoffset ) >= maxdisplayv ) {
      if ( abs( row - yoffset - maxdisplayv + 1 ) == 1 ) scrollV1( 1 );
      else scrollV( row - yoffset - maxdisplayv + 1 );
    }
  }
}

int FieldListView::getRowHeight() const
{
  if ( font == NULL ) return _aguix->getCharHeight();
  else return font->getCharHeight();
}

int FieldListView::getMBG() const
{
  return mbg;
}

void FieldListView::setMBG(int color)
{
  mbg = color;
  if ( isCreated() == true ) {
    /* TODO: eigentlich brauche ich hier nicht mehr SetWindowBG */
    //_aguix->SetWindowBG( win, mbg );
    clearNonElements();
  }
}

int FieldListView::maximizeX()
{
  int elem_width, s;

  if ( font == NULL ) elem_width = _aguix->getCharWidth();
  else elem_width = font->getCharWidth();
  s = elem_width * getMaxLen() + ( ( vbar != 0 ) ? vbar_width : 0 ) + 2 + ( ( displayFocus == true ) ? 2 : 0 );
  resize( s, getHeight() );
  return s;
}

int FieldListView::maximizeY()
{
  int elem_height, s;

  if ( font == NULL ) elem_height = _aguix->getCharHeight();
  else elem_height = font->getCharHeight();
  s = elem_height * getElements() +
    ( ( hbar != 0 ) ? hbar_height : 0 ) + 2 +
    ( ( displayFocus == true ) ? 2 : 0 ) +
    ( ( showHeader == true ) ? getHeaderHeight() : 0 );
  resize( getWidth(), s );
  return s;
}

void FieldListView::handleMiddle(Message *msg)
{
  int j;
  int win_x, win_y;
  int newelement;
  AGMessage *agmsg;
  
  if ( isCreated() == false ) return;

  if ( msg->type == ButtonPress ) {
    if ( msg->window != win ) return;
    if ( middleMode != MIDDLE_IDLE ) return;

    j = getRowForMouseY( msg->mousey );
    if ( j < 0 ) return;
    if ( ( j + yoffset ) >= elements ) return;
    if ( j >= maxdisplayv ) return;

    clickLastSelectedRow = j + yoffset;
    clickFirstSelectedRow = clickLastSelectedRow;

    setActiveRow( clickLastSelectedRow );
    runSelectHandler( clickLastSelectedRow );
    
    middleMode = MIDDLE_HOLD;

    _aguix->Flush();
    _aguix->msgLock( this );
  } else if ( msg->type == MotionNotify ) {
    if ( ( msg->window == win ) &&
	 ( ( middleMode == MIDDLE_HOLD ) ||
	   ( middleMode == MIDDLE_SCROLL_DOWN ) ||
	   ( middleMode == MIDDLE_SCROLL_UP ) ) ) {
      _aguix->queryPointer( win, &win_x, &win_y );
      j = getRowForMouseY( win_y );
      if ( ( j < 0 ) && ( yoffset > 0 ) ) {
	newelement = yoffset - 1;
	if ( newelement >= 0 ) {
	  if ( middleMode != MIDDLE_SCROLL_UP ) {
	    scrollV1( -1 );
	    setActiveRow( newelement );
	    runSelectHandler( newelement );
	    clickLastSelectedRow = newelement;
	    middleMode = MIDDLE_SCROLL_UP;
	    _aguix->enableTimer();
	  }
	}
      } else if ( ( j >= maxdisplayv ) && ( ( yoffset + maxdisplayv ) < elements ) ) {
	newelement = yoffset + maxdisplayv;
	if ( newelement < elements ) {
	  if ( middleMode != MIDDLE_SCROLL_DOWN ) {
	    scrollV1( 1 );
	    setActiveRow( newelement );
	    runSelectHandler( newelement );
	    clickLastSelectedRow = newelement;
	    middleMode = MIDDLE_SCROLL_DOWN;
	    _aguix->enableTimer();
	  }
	}
      } else {
	if ( j < 0 ) j = 0;
	if ( ( j + yoffset ) >= elements ) {
	  j = elements - 1 - yoffset;
	}
	if ( ( j + yoffset ) != clickLastSelectedRow ) {
	  newelement = j + yoffset;
	  setActiveRow( newelement );
	  runSelectHandler( newelement );
	  clickLastSelectedRow = newelement;
	}
	if ( middleMode != MIDDLE_HOLD ) {
	  middleMode = MIDDLE_HOLD;
	  _aguix->disableTimer();
	}
      }
    }
  } else if ( msg->type == ButtonRelease ) {
    if ( ( msg->window == win ) &&
	 ( msg->button == Button2 ) ) {
      switch ( middleMode ) {
	case MIDDLE_IGNORE:
	  _aguix->msgUnlock( this );
	  middleMode = MIDDLE_IDLE;
	  break;
	case MIDDLE_SCROLL_UP:
	case MIDDLE_SCROLL_DOWN:
	  _aguix->disableTimer();
	case MIDDLE_HOLD:
	  agmsg = AGUIX_allocAGMessage();
	  agmsg->fieldlv.lv = this;
	  agmsg->fieldlv.row = clickFirstSelectedRow;
	  agmsg->fieldlv.time = msg->time;
	  agmsg->fieldlv.mouse = true;
	  agmsg->type = AG_FIELDLV_ONESELECT;
	  _aguix->putAGMsg( agmsg );
	  _aguix->msgUnlock( this );
	  middleMode = MIDDLE_IDLE;
	  break;
	default:
	  break;
      }
    }
  } else if ( msg->type == ClientMessage ) {
    if ( msg->specialType == Message::TIMEREVENT ) {
      if ( ( msg->time % 1 ) == 0 ) { //HARDCODED
	if ( middleMode == MIDDLE_SCROLL_UP ) {
	  newelement = clickLastSelectedRow - 1;
	  if ( newelement >= 0 ) {
	    scrollV1( -1 );
	    setActiveRow( newelement );
	    runSelectHandler( newelement );
	    clickLastSelectedRow = newelement;
	  }
	} else if ( middleMode == MIDDLE_SCROLL_DOWN ) {
	  newelement = clickLastSelectedRow + 1;
	  if ( newelement < elements ) {
	    scrollV1( 1 );
	    setActiveRow( newelement );
	    runSelectHandler( newelement );
	    clickLastSelectedRow = newelement;
	  }
	}
	_aguix->Flush();
      }
    }
  }
}

bool FieldListView::handleKeys(Message *msg)
{
  bool returnvalue=false;
  bool sendmsg = false;

  if ( isCreated() == false ) return false;
  if ( msg != NULL ) {
    //TODO2:
    // most important features:
    // 1.up/down
    //   should react like handleMiddle
    // 2.return for simulate doubleclick
    //   problem: there is no doubleclick-feature at all at the moment
    // 3.space for selecting/unselecting
    //   just like handleSelect
    if ( msg->type == KeyPress ) {
      if ( ( msg->key == XK_Up ) ||
	   ( msg->key == XK_Down ) ||
	   ( msg->key == XK_Home ) ||
	   ( msg->key == XK_End ) ||
	   ( msg->key == XK_Prior ) ||
	   ( msg->key == XK_Next ) ) {
	int pos;

	switch ( msg->key ) {
	case XK_Up:
	  if ( activerow > 0 )
	    pos = activerow - 1;
	  else
	    pos = 0;
	  break;
	case XK_Down:
	  if ( activerow < 0 )
	    pos = 0;
	  else if ( activerow >= ( elements - 1 ) )
	    pos = elements - 1;
	  else
	    pos = activerow + 1;
	  break;
	case XK_Home:
	  pos = 0;
	  break;
	case XK_End:
	  pos = elements - 1;
	  break;
	case XK_Prior:
	  if ( activerow < 0 )
	    pos = 0;
	  else {
	    pos = activerow - maxdisplayv + 1;
	    if ( pos < 0 )
	      pos = 0;
	  }
	  break;
	case XK_Next:
	  if ( activerow < 0 )
	    pos = 0;
	  else {
	    pos = activerow + maxdisplayv - 1;
	    if ( pos >= ( elements - 1 ) )
	      pos = elements - 1;
	  }
	  break;
	default:
	  pos = -1;
	  break;
	}
	if ( ( pos >= 0 ) && ( pos < elements ) ){
	  // now activate entry pos
	  setActiveRow( pos );
	  if ( ( ( pos - yoffset ) < 0 ) || ( ( pos - yoffset ) >= maxdisplayv ) )
	    showActive();
	  runSelectHandler( pos );
	  _aguix->Flush();
          sendmsg = true;
	}
      } else if ( msg->key == XK_space ) {
	if ( activerow >= 0 ) {
	  bool state = getSelect( activerow );
	  state = ( state == false ) ? true : false;
	  setSelect( activerow, state );
	  runSelectHandler( activerow );
	  _aguix->Flush();
          sendmsg = true;
	}
      } else if ( msg->key == XK_Return ) {
      }
    }
  }
  
  if ( sendmsg == true ) {
    AGMessage *agmsg = AGUIX_allocAGMessage();
    agmsg->fieldlv.lv = this;
    agmsg->fieldlv.row = activerow;  // since we only change things for
                                     // activerow with keys send this
    agmsg->fieldlv.time = time( NULL );
    agmsg->fieldlv.mouse = false;
    agmsg->type = AG_FIELDLV_ONESELECT;
    _aguix->putAGMsg( agmsg );
  }
  return returnvalue;
}

void FieldListView::setData( int row, int ndata )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setData( ndata );
}

int FieldListView::getData( int row ) const
{
  if ( ( row < 0 ) || ( row >= elements ) ) return 0;
  return elementarray[ row ]->getData();
}

void FieldListView::setDataExt( int row, FieldLVRowData *ndata )
{
  if ( ( row < 0 ) || ( row >= elements ) ) return;
  elementarray[ row ]->setDataExt( ndata );
}

FieldLVRowData *FieldListView::getDataExt( int row ) const
{
  if ( ( row < 0 ) || ( row >= elements ) ) return NULL;
  return elementarray[ row ]->getDataExt();
}

void FieldListView::Content::setData( int ndata )
{
  data = ndata;
}

int FieldListView::Content::getData() const
{
  return data;
}

void FieldListView::Content::setDataExt(FieldLVRowData *ndata )
{
  if ( dataExt != NULL ) delete dataExt;
  dataExt = ndata;
}

FieldLVRowData *FieldListView::Content::getDataExt() const
{
  return dataExt;
}

bool FieldListView::isValidRow( int row ) const
{
  if ( ( row >= 0 ) && ( row < elements ) ) return true;
  return false;
}

int FieldListView::insertRow( int row )
{
  int i;

  if ( row < 0 ) return -1;
  if ( row >= elements ) return addRow();

  if ( elements == arraysize ) increaseArray( arraysize * 2 );

  for ( i = elements - 1; i >= row; i-- ) {
    elementarray[ i + 1 ] = elementarray[ i ];
  }

  elements++;
  elementarray[ row ] = new Content();

  if ( activerow >= row ) {
    // since we insert at row, activerow must be raised
    activerow++;
  }

  for ( i = 0; i < fields; i++ ) {
    if ( maxrow[i] >= row ) {
      // correct maximum row
      maxrow[i]++;
    }
  }

  // deactivate select/middle mode
  selectModeIgnore();
  middleModeIgnore();

  return row;
}

void FieldListView::swapRows( int row1, int row2 )
{
  Content *te;
  int i;

  if ( ( isValidRow( row1 ) != true ) || ( isValidRow( row2 ) != true ) ) return;
  
  te = elementarray[ row1 ];
  elementarray[ row1 ] = elementarray[ row2 ];
  elementarray[ row2 ] = te;

  if ( row1 == activerow ) {
    // row1 was active => make row2 active
    activerow = row2;
  } else if ( row2 == activerow ) {
    activerow = row1;
  }
  
  for ( i = 0; i < fields; i++ ) {
    if ( maxrow[i] == row1 ) {
      // row1 is the row with maximum length
      maxrow[i] = row2;
    } else if ( maxrow[i] == row2 ) {
      maxrow[i] = row1;
    }
  }

  // deactivate select/middle mode
  selectModeIgnore();
  middleModeIgnore();

  return;
}

void FieldListView::setPreColors( int row, precolor_t mode )
{
  if ( isValidRow( row ) == false ) return;

  switch( mode ) {
    case PRECOLOR_ONLYSELECT:
      setFG( row, CC_NORMAL, 1 );
      setFG( row, CC_SELECT, 2 );
      setFG( row, CC_ACTIVE, 1 );
      setFG( row, CC_SELACT, 2 );
      setBG( row, CC_NORMAL, 0 );
      setBG( row, CC_SELECT, 1 );
      setBG( row, CC_ACTIVE, 0 );
      setBG( row, CC_SELACT, 1 );
      break;
    case PRECOLOR_ONLYACTIVE:
      setFG( row, CC_NORMAL, 1 );
      setFG( row, CC_SELECT, 1 );
      setFG( row, CC_ACTIVE, 2 );
      setFG( row, CC_SELACT, 2 );
      setBG( row, CC_NORMAL, 0 );
      setBG( row, CC_SELECT, 0 );
      setBG( row, CC_ACTIVE, 1 );
      setBG( row, CC_SELACT, 1 );
      break;
    case PRECOLOR_NOTSELORACT:
      setFG( row, CC_NORMAL, 1 );
      setFG( row, CC_SELECT, 1 );
      setFG( row, CC_ACTIVE, 1 );
      setFG( row, CC_SELACT, 1 );
      setBG( row, CC_NORMAL, 0 );
      setBG( row, CC_SELECT, 0 );
      setBG( row, CC_ACTIVE, 0 );
      setBG( row, CC_SELACT, 0 );
      break;
    default:
      setFG( row, CC_NORMAL, 1 );
      setFG( row, CC_SELECT, 2 );
      setFG( row, CC_ACTIVE, 1 );
      setFG( row, CC_SELACT, 2 );
      setBG( row, CC_NORMAL, 0 );
      setBG( row, CC_SELECT, 1 );
      setBG( row, CC_ACTIVE, 7 );
      setBG( row, CC_SELACT, 7 );
      break;
  }
}

/*
 * resetWin
 *
 * this will clear the window (only win, not the bars)
 * and if wanted a redraw is done
 */
void FieldListView::resetWin( bool doRedraw )
{
  if ( isCreated() == false ) return;
  if ( doRedraw == true ) {
    mredraw();
    redrawHeader();
    redrawContent();
  } else {
    _aguix->ClearWin( win );
  }
}

int FieldListView::getRowForMouseY( int my ) const
{
  int r;
  
  r = ( my - oy ) / elementHeight;
  if ( ( my - oy ) < 0 ) {
    r--;
  }
  return r;
}

void FieldListView::drawBuffer()
{
  drawBuffer( 0, 0, _w, _h );
}

void FieldListView::drawBuffer( int dx, int dy, int dw, int dh )
{
  if ( isCreated() == true ) {
    if ( ( dx < 0 ) ||
	 ( dy < 0 ) ||
	 ( dx >= _w ) ||
	 ( dy >= _h ) ||
	 ( dw < 1 ) ||
	 ( dh < 1 ) ||
	 ( dw > ( _w - dx ) ) ||
	 ( dh > ( _h - dy ) ) ) return;

    _aguix->copyArea( buffer, win, dx, dy, dw, dh, dx, dy );
  }
}

void FieldListView::drawBuffer( int element )
{
  if ( ( element < 0 ) || ( element >= elements ) ) return;
  if ( ( element >= yoffset ) && ( element < ( yoffset + maxdisplayv ) ) ) {
    drawBuffer( ox, oy + ( element - yoffset ) * elementHeight, maxdisplayh * elementWidth, elementHeight );
  }
}

void FieldListView::setDisplayFocus( bool nv )
{
  displayFocus = nv;
  recalcMaxValues();
  setupBars();
  resetWin( true );
}

bool FieldListView::getDisplayFocus() const
{
  return displayFocus;
}

int FieldListView::getInnerWidth() const
{
  int tw;

  if ( vbar != 0 ) {
    tw = _w - 2 - vbar_width;
  } else {
    tw = _w - 2;
  }

  if ( displayFocus == true )
    tw -= 2;

  return tw;
}

int FieldListView::getInnerHeight() const
{
  //TODO: Muss ich showHeader beachten?
  int th;

  if ( hbar != 0 ) {
    th = _h - 2 - hbar_height;
  } else {
    th = _h - 2;
  }

  if ( displayFocus == true )
    th -= 2;

  return th;
}

void FieldListView::doCreateStuff()
{
  GUIElement::doCreateStuff();
  if ( buffer == 0 ) {
    buffer = _aguix->createPixmap( win, _w, _h );
    mredraw();
  }
  recalcMaxValues();
  if ( hbarwin == 0 ) {
    if ( hbar != 0 ) hbarwin = _parent->getSubWindow( win, 0, 0, 5, 5 );
  }
  if ( vbarwin == 0 ) {
    if ( vbar != 0 ) vbarwin = _parent->getSubWindow( win, 0, 0, 5, 5 );  
  }
  setupBars();
}

void FieldListView::doDestroyStuff()
{
  if ( buffer != 0 ) {
    _aguix->freePixmap( buffer );
    buffer = 0;
  }
  if ( hbarwin != 0 ) {
    if ( hbar != 0 ) {
      _parent->removeSubWin( hbarwin );
    }
    hbarwin = 0;
  }
  if ( vbarwin != 0 ) {
    if ( vbar != 0 ) {
      _parent->removeSubWin( vbarwin );
    }
    vbarwin = 0;
  }
  GUIElement::doDestroyStuff();
}

/*
 * cond_redraw()
 *
 * only redraw when column width or visible entry changed
 * at the moment this is only set be setText !
 */
void FieldListView::cond_redraw()
{
  if ( cond_changed == true ) {
    redraw();
    cond_changed = false;
  }
}

bool FieldListView::isRowVisible( int row ) const
{
  if ( ( row >=0 ) && ( row < elements ) ){
    if ( ( row >= yoffset ) && ( ( row - yoffset ) < maxdisplayv ) ) {
      return true;
    }
  }
  return false;
}

std::string FieldListView::getText( int row, int field ) const
{
  if ( isValidRow( row ) == false ) return "";
  if ( ( field < 0 ) || ( field >= fields ) ) return "";
  return elementarray[ row ]->getText( field );
}

void FieldListView::setShowHeader( bool nv )
{
  showHeader = nv;
  recalcMaxValues();

  // check yoffset
  if ( ( yoffset + maxdisplayv ) >= elements ) yoffset = elements - maxdisplayv;
  if ( yoffset < 0 ) yoffset = 0;

  if ( elements < maxdisplayv ) {
    // not all lines used so reset the unused elements
    clearNonElements();
  }

  resetWin( true );
  vbarredraw();
}

bool FieldListView::getShowHeader() const
{
  return showHeader;
}

int FieldListView::getHeaderHeight() const
{
  return elementHeight + 4 + 1;
}

void FieldListView::redrawHeader()
{
  //TODO: ausloesen innerhalb von Content?
  std::string s;
  const char *cstr;
  int sx, ex, vi_offset_s, vi_offset_c, f, len, tlen, fakefield;
  int hh, vsx, vex;
  char *tstr;
  align_t usedAlign;

  if ( isCreated() == false ) return;
  if ( showHeader != true ) return;

  hh = getHeaderHeight();
  _aguix->setFG( headerCols[1] );
  _aguix->FillRectangle( buffer,
			ox,
			oy - hh,
			maxdisplayh * elementWidth,
			hh - 1 );
  
  sx = ex = 0;
  for ( f = 0; f < fields; f++ ) {
    // merge all fields with fieldtextmerged == true
    fakefield = f;
    ex = sx;
    for (;;) {
      if ( fieldwidth[ fakefield ] < 0)
	ex += getUsedWidth( fakefield );
      else
	ex += fieldwidth[ fakefield ];
      if ( fieldtextmerged[ fakefield ] == false ) break;
      if ( ( fakefield + 1 ) >= fields ) break;
      fakefield++;
    }
    
    if ( ex > sx ) {
      // only draw header for non-empty fields
      if ( ( sx < ( xoffset + maxdisplayh ) ) && ( ex > xoffset ) ) {
	// field visible
	vsx = a_max( sx, xoffset );
	vex = a_min( xoffset + maxdisplayh, ex );
	_aguix->setFG( ( fieldtextclicked == f ) ? 1 : 2 );
	_aguix->DrawLine( buffer,
			 ox + ( vsx - xoffset ) * elementWidth,
			 oy - hh,
			 ox + ( vex - xoffset ) * elementWidth - 1,
			 oy - hh );
	if ( sx >= xoffset ) {
	  // left start visible
	  _aguix->DrawLine( buffer,
			   ox + ( vsx - xoffset ) * elementWidth,
			   oy - hh + 1,
			   ox + ( vsx - xoffset ) * elementWidth,
			   oy - 2 );
	}
	_aguix->setFG( ( fieldtextclicked == f ) ? 2 : 1 );
	_aguix->DrawLine( buffer,
			 ox + ( vsx - xoffset ) * elementWidth,
			 oy - 2,
			 ox + ( vex - xoffset ) * elementWidth - 1,
			 oy - 2 );
	if ( ex <= ( xoffset + maxdisplayh ) ) {
	  // right end visible
	  _aguix->DrawLine( buffer,
			   ox + ( vex - xoffset ) * elementWidth - 1,
			   oy - hh + 1,
			   ox + ( vex - xoffset ) * elementWidth - 1,
			   oy - 2 );
	}
	// this many characters are visible in the header
	vi_offset_c = ( ( ( vex - vsx ) * elementWidth ) - 4 ) / elementWidth;
	if ( vi_offset_c > 0 ) {
	  s = fieldtext[f];
	  len = s.length();
	  usedAlign = fieldalign[ f ];
	  if ( ( ( ( ( ex - sx ) * elementWidth ) - 4 ) / elementWidth ) <= len ) {
	    // column width smaller than text width
	    // so always use left alignment for better reading
	    usedAlign = ALIGN_LEFT;
	  }
	  if ( usedAlign == ALIGN_RIGHT ) {
	    if ( ex < ( xoffset + maxdisplayh ) )
	      vi_offset_s = 0;
	    else
	      vi_offset_s = ex - ( xoffset + maxdisplayh ); 
	    // remember vi_offset_s is now from right
	    
	    if ( len > 0 ) {
	      tstr = dupstring( s.c_str() );
	      if ( len > vi_offset_s ) {
		// something visible from this entry
		tlen = len - vi_offset_s;
		// only up to tlen chars
		tstr[tlen] = '\0';
		if ( tlen <= vi_offset_c ) {
		  // everything from beginning visible
		  cstr = tstr;
		} else {
		  cstr = tstr + tlen - vi_offset_c;
		  tlen = vi_offset_c;
		}
		if ( font == NULL )
		  _aguix->setFG( headerCols[0] );
		else
		  _aguix->setFG( font->getGC(), headerCols[0] );
		if ( font == NULL )
		  _aguix->DrawText( buffer, cstr, ox + ( vex - xoffset - tlen ) * elementWidth - 2, oy - hh + 2 );
		else
		  _aguix->DrawText( buffer, font, cstr, ox + ( vex - xoffset - tlen ) * elementWidth - 2, oy - hh + 2 );
	      }
	      _freesafe( tstr );
	    }	  
	  } else {
	    if ( sx >= xoffset )
	      vi_offset_s = 0;
	    else
	      vi_offset_s = xoffset - sx;
	    
	    if ( len > 0 ) {
	      tstr = dupstring( s.c_str() );
	      if ( len > vi_offset_s ) {
		// something visible from this entry
		cstr = tstr + vi_offset_s;
		tlen = len - vi_offset_s;
		if ( tlen > vi_offset_c ) {
		  tstr[ vi_offset_s + vi_offset_c ] = '\0';
		}
		if ( font == NULL )
		  _aguix->setFG( headerCols[0] );
		else
		  _aguix->setFG( font->getGC(), headerCols[0] );
		if ( font == NULL )
		  _aguix->DrawText( buffer, cstr, ox + ( vsx - xoffset ) * elementWidth + 2, oy - hh + 2 );
		else
		  _aguix->DrawText( buffer, font, cstr, ox + ( vsx - xoffset ) * elementWidth + 2, oy - hh + 2 );
	      }
	      _freesafe( tstr );
	    }
	  }
	}
      }
    }
    
    sx = ex;
    f = fakefield;
  }
  // now draw a dummy field if there is space left
  // there is no real need for this but it looks better
  if ( ex < ( maxdisplayh + xoffset ) ) {
    // sx is now start pixel!
    sx = ( ex - xoffset ) * elementWidth;
    if ( sx < 0 ) sx = 0;
    ex = getInnerWidth() - 1;
    if ( ex < 0 ) ex = 0;
    if ( sx < ex ) {
      _aguix->setFG( headerCols[1] );
      _aguix->FillRectangle( buffer,
			    ox + sx,
			    oy - hh,
			    ex - sx,
			    hh - 1 );
      _aguix->setFG( 2 );
      _aguix->DrawLine( buffer,
		       ox + sx,
		       oy - hh,
		       ox + ex,
		       oy - hh );
      _aguix->DrawLine( buffer,
		       ox + sx,
		       oy - hh + 1,
		       ox + sx,
		       oy - 2 );
      _aguix->setFG( 1 );
      _aguix->DrawLine( buffer,
		       ox + sx,
		       oy - 2,
		       ox + ex,
		       oy - 2 );
      _aguix->DrawLine( buffer,
		       ox + ex,
		       oy - hh + 1,
		       ox + ex,
		       oy - 2 );
    }
  }
}

/*
 * returns number of field clicked
 * returnvalue:
 *  -1 when outside field area
 *  fieldnumber for clicked field
 *    for merged fields it's the first one
 */
int FieldListView::getFieldForPoint( int mx, int my)
{
  int sx, ex, f, fakefield;
  int hh, vsx, vex;
  int found = -1;

  if ( isCreated() == false ) return -1;
  if ( showHeader != true ) return -1;

  hh = getHeaderHeight();

  if ( ( my < ( oy - hh ) ) || ( my >= oy ) ) return -1;
  if ( ( mx < ox ) || ( mx >= ( ox + maxdisplayh * elementWidth ) ) ) return -1;

  sx = 0;
  for ( f = 0; f < fields; f++ ) {
    // merge all fields with fieldtextmerged == true
    fakefield = f;
    ex = sx;
    for (;;) {
      if ( fieldwidth[ fakefield ] < 0)
	ex += getUsedWidth( fakefield );
      else
	ex += fieldwidth[ fakefield ];
      if ( fieldtextmerged[ fakefield ] == false ) break;
      if ( ( fakefield + 1 ) >= fields ) break;
      fakefield++;
    }
    
    if ( ( sx < ( xoffset + maxdisplayh ) ) && ( ex >= xoffset ) ) {
      // field visible
      vsx = a_max( sx, xoffset );
      vex = a_min( xoffset + maxdisplayh, ex );
      if ( ( mx >= ( ox + ( vsx - xoffset ) * elementWidth ) ) &&
	   ( mx < ( ox + ( vex - xoffset ) * elementWidth ) ) ) {
	found = f;
	break;
      }
    }
  
    sx = ex;
    f = fakefield;
  }
  return found;
}

void FieldListView::handleHeader( Message *msg, int f )
{
  int hh;

  if ( msg == NULL ) return;

  if ( ( msg->type != ButtonPress ) && ( msg->type != ButtonRelease ) ) return;
  if ( ( msg->button != Button1 ) && ( msg->button != Button2 ) && ( msg->button != Button3 ) ) return;
  
  if ( ( f < 0 ) || ( f >= fields ) ) {
    f = getFieldForPoint( msg->mousex, msg->mousey );
  }
  
  if ( msg->type == ButtonPress ) {
    fieldtextclicked = f;
  } else if ( msg->type == ButtonRelease ) {
    if ( fieldtextclicked == f ) {
      if ( ( f >= 0 ) && ( f < fields ) ) {
        AGMessage *agmsg = AGUIX_allocAGMessage();
        agmsg->fieldlv.lv = this;
        agmsg->fieldlv.row = f;
        agmsg->fieldlv.time = msg->time;
        agmsg->fieldlv.mouse = true;
        agmsg->fieldlv.button = msg->button;
        agmsg->type = AG_FIELDLV_HEADERCLICKED;
        _aguix->putAGMsg( agmsg );
      }
    }
    fieldtextclicked = -1;
  }
  redrawHeader();

  hh = getHeaderHeight();
  drawBuffer( ox, oy - hh, maxdisplayh * elementWidth, hh );
}

void FieldListView::setHeaderFG( int col )
{
  headerCols[0] = col;
  redrawHeader();
}

void FieldListView::setHeaderBG( int col )
{
  headerCols[1] = col;
  redrawHeader();
}

int FieldListView::getHeaderFG() const
{
  return headerCols[0];
}

int FieldListView::getHeaderBG() const
{
  return headerCols[1];
}

void FieldListView::selectModeIgnore()
{
  if ( selectMode == SELECT_IDLE ) return;

  switch ( selectMode ) {
    case SELECT_SCROLL_DOWN:
    case SELECT_SCROLL_UP:
      _aguix->disableTimer();
      break;
    default:
      break;
  }
  selectMode = SELECT_IGNORE;
}

void FieldListView::middleModeIgnore()
{
  if ( middleMode == MIDDLE_IDLE ) return;

  switch ( middleMode ) {
    case MIDDLE_SCROLL_DOWN:
    case MIDDLE_SCROLL_UP:
      _aguix->disableTimer();
      break;
    default:
      break;
  }
  middleMode = MIDDLE_IGNORE;
}
