/* lister.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: lister.cc,v 1.31 2005/10/17 19:07:45 ralf Exp $ */

#include "lister.h"
#include "worker.h"
#include "normalmode.h"
#include "worker_locale.h"
#include "wconfig.h"
#include "configheader.h"
#include "configparser.tab.h"

Lister::Lister(Worker *tparent)
{
  this->parent = tparent;
  aguix = parent->getAGUIX();
  win = parent->getListerWin( this );

  w = win->getWidth();
  h = win->getHeight();
  if ( w < 50 ) w = 50;
  if ( h < 50 ) h = 50;
  lvb=(Button*)win->add(new Button(aguix,0,0,w,"","",1,1,0,0,1));
  lvb->setAcceptFocus( false );
  aguix->Flush();
  activemode=NULL;
  activemodeid=-1;
  hasFocus=false;
  int j=parent->getMaxModeNr();
  modes=new ListerMode*[j];
  for(int i=0;i<j;i++) {
    modes[i]=parent->getMode4ID(i,this);
  }
  timerclear( &lastlvbclick );
}

Lister::~Lister()
{
  if(activemode!=NULL) activemode->off();
  int j=parent->getMaxModeNr();
  for(int i=0;i<j;i++) {
    delete modes[i];
  }
  delete [] modes;
  delete win;
}

void Lister::messageHandler(AGMessage *msg)
{
  bool subHandle=true;
  struct timeval acttime;
  if(msg!=NULL) {
    switch(msg->type) {
      case AG_SIZECHANGED:
        int bh, tw, th;
        tw = win->getWidth();
        th = win->getHeight();
        if ( tw < 50 ) tw = 50;
        if ( th < 50 ) th = 50;
        bh=aguix->getCharHeight()+4;
        lvb->resize(tw,bh);
        this->w=tw;
        this->h=th;
        break;
      case AG_BUTTONCLICKED:
        if(msg->button.button==lvb) {
          if(msg->button.state==2) {
            configure();
          } else {
            makeActive();
            
            gettimeofday( &acttime, NULL );
            if ( aguix->isDoubleClick( &acttime, &lastlvbclick ) == true ) {
              if ( activemode != NULL ) activemode->lvbDoubleClicked();
              timerclear( &lastlvbclick );
            } else {
              lastlvbclick.tv_sec = acttime.tv_sec;
              lastlvbclick.tv_usec = acttime.tv_usec;
            }
          }
        }
        break;
      case AG_MOUSECLICKED:
        if(msg->mouse.window==win->getWindow()) {
          makeActive();
        }
        break;
    }
  }
  if(subHandle==true) if(activemode!=NULL) activemode->messageHandler(msg);
}

AGUIX *Lister::getAGUIX()
{
  return aguix;
}

AWindow *Lister::getAWindow()
{
  return win;
}
void Lister::setActiveMode(ListerMode *lm)
{
  activemode=lm;
}

void Lister::getGeometry(int *x2,int *y2,int *w2,int *h2)
{
  int tw,th;
  *x2=0;
  lvb->getSize(&tw,&th);
  *y2=th;
  *w2=this->w;
  *h2=this->h-th;
  if ( *h2 < 0 ) *h2 = 0;
}

ListerMode *Lister::getActiveMode()
{
  return activemode;
}

void Lister::setFocus(bool state)
{
  int fg,bg;
  hasFocus=state;
  if(state==true) {
    fg=wconfig->getSelLVB(0);
    bg=wconfig->getSelLVB(1);
    if(activemode!=NULL) activemode->activate();
  } else {
    fg=wconfig->getUnselLVB(0);
    bg=wconfig->getUnselLVB(1);
    if(activemode!=NULL) activemode->deactivate();
  }
  lvb->setFG(0,fg);
  lvb->setBG(0,bg);
  lvb->setFG(1,fg);
  lvb->setBG(1,bg);
}

bool Lister::getFocus()
{
  return hasFocus;
}

void Lister::switch2Mode(int nr)
{
  if(activemodeid==nr) return;
  if(activemode!=NULL) activemode->off();
  int j=parent->getMaxModeNr();
  if((nr>=0)&&(nr<j)) {
    activemode=modes[nr];
    activemodeid=nr;
    modes[nr]->on();
  } else {
    activemode=NULL;
    activemodeid=-1;
  }
}

void Lister::setName(const char *str)
{
  lvb->setText(0,str);
  lvb->setText(1,str);
}

void Lister::setStatebarText(const char *str)
{
  parent->setStatebarText(str);
}

void Lister::makeActive()
{
  if(hasFocus==true) return; // Wir sind schon aktiv
  Lister *ol=parent->getOtherLister(this);
  if(ol==NULL) return; // Hoppala, da lief was schief
  ol->setFocus(false);
  setFocus(true);
}

int Lister::getSide()
{
  return parent->getSide(this);
}

bool Lister::isActive()
{
  return hasFocus;
}

void Lister::configure()
{
  int nr,j,i;
  const char *cstr;
  ChooseButton **cbs;
  Button **bs;
  AWindow *win2 = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 16 ) );
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  const int cfixnr = cfix +
                     AContainer::ACONT_NORESIZE;

  win2->create();

  AContainer *ac1 = win2->setContainer( new AContainer( win2, 1, 3 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 166 ), 1 ), 0, 0, cincwnr );

  nr = parent->getMaxModeNr();

  AContainer *ac1_1 = ac1->add( new AContainer( win2, 2, nr ), 0, 1 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );

  cbs=(ChooseButton**)_allocsafe(sizeof(ChooseButton*)*nr);
  bs=(Button**)_allocsafe(sizeof(Button*)*nr);
  for(i=0;i<nr;i++) {
    cbs[i] = new ChooseButton( aguix, 0, 0, ( ( modes[i] == activemode ) ? 1 : 0 ), "", LABEL_RIGHT, 0, 0 );
    ac1_1->add( cbs[i], 0, i, cfixnr );
    cstr = modes[i]->getLocaleName();
    bs[i] = new Button( aguix, 0, 0, cstr, 1, 0, 0 );
    ac1_1->add( bs[i], 1, i, cincw );
  }

  AContainer *ac1_2 = ac1->add( new AContainer( win2, 2, 1 ), 0, 2 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, cfix );

  okb->takeFocus();
  win2->setDoTabCycling( true );
  win2->contMaximize( true );
  win2->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win2);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win2->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
        else {
          for(j=0;j<nr;j++) if(bs[j]==msg->button.button) break;
          if(j<nr) {
            modes[j]->configure();
          }
        }
      } else if(msg->type==AG_KEYPRESSED) {
        if(msg->key.key==XK_Return) {
          if ( cancelb->getHasFocus() == false )
            ende=1;
        } else if(msg->key.key==XK_Escape) {
          ende=-1;
        } else {
          j=msg->key.key-XK_1;
          if((j>=0)&&(j<nr)) {
            if((msg->key.keystate&ShiftMask)==ShiftMask) {
              modes[j]->configure();
            } else {
              for(i=0;i<nr;i++) {
                if ( i != j ) cbs[i]->setState( false );
                else cbs[i]->setState( ( cbs[i]->getState() == false ) ? true : false );
              }
            }
          }
        }
      } else if(msg->type==AG_CHOOSECLICKED) {
        for(j=0;j<nr;j++) if(cbs[j]==msg->choose.button) break;
        if(j<nr) {
          for(i=0;i<nr;i++) {
            if ( i != j ) cbs[i]->setState( false );
          }
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    // Bei OK den Mode aendern
    // TODO: Den config-Zustand uebernehmen, dazu vielleicht alle modis duplizieren und
    //       diese bearbeiten
    j=-1;
    for(i=0;i<nr;i++) {
      if ( cbs[i]->getState() == true ) {
        j=i;
        break;
      }
    }
    switch2Mode(j);
  }
  delete win2;
  _freesafe(cbs);
  _freesafe(bs);
}

Worker *Lister::getWorker()
{
  return parent;
}

void Lister::cyclicfunc(cyclicfunc_mode_t mode)
{
  if(activemode!=NULL) activemode->cyclicfunc(mode);
}

int Lister::saveState(Datei *fh)
{
  std::string str1;
  int i, j;

  if ( fh == NULL ) return 1;
  
  if ( activemode != NULL ) {
    fh->configPutPairString( "activemode", Worker::getNameOfMode( Worker::getID4Mode( activemode ) ) );
  }
  // Ueber alle Modes: ID speichern und dann den Mode speichern ueberlassen
  j = parent->getMaxModeNr();
  for ( i = 0; i < j; i++ ) {
    // put mode name in quotes so the scanner will return STRING
    str1 = "\"";
    str1 += Worker::getNameOfMode( Worker::getID4Mode( modes[i] ) );
    str1 += "\"";
    fh->configOpenSection( str1.c_str() );
    modes[i]->save( fh );
    fh->configCloseSection();
  }
  return 0;
}

int Lister::loadState()
{
  int i, j;
  int found_error = 0;
  int amode = -1;

  j = parent->getMaxModeNr();
  for (;;) {
    if ( worker_token == ACTIVEMODE_WCP ) {
      readtoken();
      
      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();
      
      if ( worker_token == STRING_WCP ) {
        for ( i = 0; i < j; i++ ) {
          if ( modes[i]->isType( yylval.strptr ) == true ) {
            amode = i;
          }
        }
      } else {
        found_error = 1;
        break;
      }
      readtoken();
      
      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();     
    } else if ( worker_token == STRING_WCP ) {
      for ( i = 0; i < j; i++ ) {
        if ( modes[i]->isType( yylval.strptr ) == true ) {
          break;
        }
      }
      readtoken();
      
      if ( i < j ) {
        if ( worker_token != LEFTBRACE_WCP ) {
          found_error = 1;
          break;
        }
        readtoken();
        modes[i]->load();
        
        if ( worker_token != RIGHTBRACE_WCP ) {
          found_error = 1;
          break;
        }
        readtoken();
      } else {
        found_error = 1;
        break;
      }
    } else {
      break;
    }
  }
  if ( amode >= 0 ) switch2Mode( amode );
  return found_error;
}

int Lister::loadBinState(Datei *fh)
{
  int j=parent->getMaxModeNr();
  int rmode, i;

  do {
    rmode=fh->getInt();
    if(rmode>=0) { // Noch kein Ende
      if ( rmode == 0 ) {
        for ( i = 0; i < j; i++ ) {
          if ( modes[i]->isType( "NormalMode" ) == true ) {
            modes[i]->loadBin( fh );
          }
        }
      } else if ( rmode == 1 ) {
        for ( i = 0; i < j; i++ ) {
          if ( modes[i]->isType( "ShowImageMode" ) == true ) {
            modes[i]->loadBin( fh );
          }
        }
      } else if ( rmode == 2 ) {
        for ( i = 0; i < j; i++ ) {
          if ( modes[i]->isType( "InformationMode" ) == true ) {
            modes[i]->loadBin( fh );
          }
        }
      } else {
        // Mode unbekannt => Chunk ueberlesen
        fh->overreadChunk();
      }
    }
  } while((rmode>=0)&&(fh->errors()==0));
  int chunksize=fh->getInt();
  if(chunksize>=(Datei::getIntSize()+
                 Datei::getUShortSize())) {
    rmode=fh->getInt();
    chunksize-=Datei::getIntSize();
    bool rhf=(fh->getUShort()==1)?true:false;
    chunksize-=Datei::getUShortSize();
    switch ( rmode ) {
      case 1:
        for ( i = 0; i < j; i++ ) {
          if ( modes[i]->isType( "ShowImageMode" ) == true ) {
            switch2Mode( i );
          }
        }
        break;
      case 2:
        for ( i = 0; i < j; i++ ) {
          if ( modes[i]->isType( "InformationMode" ) == true ) {
            switch2Mode( i );
          }
        }
        break;
      default:
        for ( i = 0; i < j; i++ ) {
          if ( modes[i]->isType( "NormalMode" ) == true ) {
            switch2Mode( i );
          }
        }
        break;
    }
    setFocus(rhf);
  }
  while(chunksize>0) {
    fh->getUChar();
    chunksize--;
  }
  return 0;
}

void  Lister::reconfig()
{
  int fg,bg;
  if(hasFocus==true) {
    fg=wconfig->getSelLVB(0);
    bg=wconfig->getSelLVB(1);
  } else {
    fg=wconfig->getUnselLVB(0);
    bg=wconfig->getUnselLVB(1);
  }
  lvb->setFG(0,fg);
  lvb->setBG(0,bg);
  lvb->setFG(1,fg);
  lvb->setBG(1,bg);
  if(activemode!=NULL) activemode->reconfig();
}

const char *Lister::getNameOfMode(int nr)
{
  if((nr<0)||(nr>=parent->getMaxModeNr())) nr=0;
  return modes[nr]->getType();
}

bool Lister::startdnd(DNDMsg *dm)
{
  int j=parent->getMaxModeNr(),i;
  for(i=0;i<j;i++) {
    if(modes[i]->startdnd(dm)==true) break;
  }
  return (i<j)?true:false;
}

