/* textstoragefile.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2005-2007 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "textstoragefile.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "worker_locale.h"
#include "pdatei.h"

TextStorageFile::TextStorageFile( std::string filename,
                                  AWidth &lencalc,
                                  int initial_size ) : m_tss( "", lencalc ),
                                                       m_filename( filename ),
                                                       m_current_size( 64 * 1024 ),
                                                       m_lencalc( lencalc ),
                                                       m_incomplete_file( true )
{
    if ( initial_size > 0 ) {
        m_current_size = initial_size;
    }
    
    readFile();
}

int TextStorageFile::getNrOfLines() const
{
    return m_tss.getNrOfLines();
}

int TextStorageFile::getLine( int line_nr, unsigned int offset, int len, std::string &return_line ) const
{
    return m_tss.getLine( line_nr, offset, len, return_line );
}

int TextStorageFile::getLine( int line_nr, unsigned int offset, std::string &return_line ) const
{
    return getLine( line_nr, offset, -1, return_line );
}

int TextStorageFile::getMaxLineWidth() const
{
    return m_tss.getMaxLineWidth();
}

int TextStorageFile::getLineWidth( int line_nr ) const
{
    return m_tss.getLineWidth( line_nr );
}

void TextStorageFile::setLineLimit( int new_limit )
{
    m_tss.setLineLimit( new_limit );
}

std::pair<int,int> TextStorageFile::getRealLinePair( int line_nr ) const
{
    return m_tss.getRealLinePair( line_nr );
}

AWidth &TextStorageFile::getAWidth()
{
    return m_tss.getAWidth();
}

int TextStorageFile::findLineNr( std::pair<int,int> real_line ) const
{
    return m_tss.findLineNr( real_line );
}

void TextStorageFile::readFile()
{
    char *buffer;
    PDatei pd;

    if ( m_current_size < 0 ) return;

    buffer = (char*)_allocsafe( m_current_size );

    if ( pd.open( m_filename.c_str() ) == 0 ) {
        int last_read = 0;
        int read_bytes = 0;
        do {
            last_read = pd.read( buffer + read_bytes, m_current_size - read_bytes );
            if ( last_read > 0 )
                read_bytes += last_read;
        } while ( last_read > 0 && read_bytes < m_current_size );

        if ( read_bytes >= 0 && last_read >= 0 ) {

            // tab replacing
            std::string notab_buffer;
      
            for ( int i = 0; i < read_bytes; i++ ) {
                if ( buffer[i] != '\t' ) {
                    // ignore \r from dos files
                    if ( buffer[i] != '\r' ) {
                        notab_buffer += buffer[i];
                    }
                } else {
                    //TODO currently I replace tabs with 8 spaces
                    //  In the future I hope I can make some GUI improvements
                    //  so it will not draw tabs but align the next character
                    //  at the next tab stop
                    notab_buffer.append( 8, ' ' );
                }
            }

            //      m_tss = TextStorageString( buffer, read_bytes );
            m_tss = TextStorageString( notab_buffer, m_lencalc );

            // try to read another byte to check for EOF
            // (it could be done by asking for the actual size
            //  but that wouldn't work with pipes...)
            if ( pd.read( buffer, 1 ) < 1 ) {
                m_incomplete_file = false;
            } else {
                m_incomplete_file = true;
            }
        } else {
            char *tstr = (char*)_allocsafe( strlen( catalog.getLocale( 281 ) ) +
                                            strlen( m_filename.c_str() ) + 1 );
            sprintf( tstr, catalog.getLocale( 281 ), m_filename.c_str() );
            m_tss = TextStorageString( tstr, m_lencalc );
            _freesafe( tstr );
        }
    }

    _freesafe( buffer );
}

void TextStorageFile::readMore( int size )
{
    if ( size > 0 ) {
        m_current_size = size;
    } else {
        m_current_size *= 2;
    }
    
    readFile();
}

bool TextStorageFile::incompleteFile()
{
    return m_incomplete_file;
}
