/* searchop.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2006-2007 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "searchop.hh"
#include "listermode.h"
#include "normalmode.h"
#include "worker.h"
#include "execlass.h"
#include <iostream>
#include "nwc_virtualdir.hh"
#include "fileviewer.hh"
#include "flagreplacer.hh"
#include <aguix/acontainerbb.h>
#include <typeinfo>

const char *SearchOp::name = "SearchOp";
std::list<SearchOp::ResultStore*> SearchOp::_stored_results;
int SearchOp::_stored_results_ctr = 0;
MutEx SearchOp::_stored_results_lock;

#define DEFAULT_EDIT_COMMAND "{scripts}/xeditor -line {l} {f}"

SearchOp::SearchOp() : FunctionProto(),
                       _start_ac( NULL ),
                       _startdir_text( NULL ),
                       _startdir_sg( NULL ),
                       _storelv( NULL ),
                       _last_startcont_state( false ),
                       _last_result( "" ),
                       _rst( NULL ),
                       _resultlv( NULL ),
                       _sth( NULL ),
                       _matchname_sg( NULL ),
                       _matchname_regexp_cb( NULL ),
                       _matchname_fullpath_cb( NULL ),
                       _matchname_case_cb( NULL ),
                       _matchcontent_sg( NULL ),
                       _matchcontent_case_cb( NULL ),
                       _follow_symlinks_cb( NULL ),
                       _search_vfs_cb( NULL ),
                       _search_started( false )
{
  _edit_command = DEFAULT_EDIT_COMMAND;
  _show_prev_results = false;
  hasConfigure = true;
  _stored_results_lock.lock();
  _stored_results_ctr++;
  _stored_results_lock.unlock();
}

SearchOp::~SearchOp()
{
  _stored_results_lock.lock();
  _stored_results_ctr--;
  if ( _stored_results_ctr < 1 ) {
    clearStoredResults();
  }
  _stored_results_lock.unlock();
}

SearchOp*
SearchOp::duplicate() const
{
  SearchOp *ta = new SearchOp();
  ta->setEditCommand( _edit_command );
  ta->setShowPrevResults( _show_prev_results );
  return ta;
}

bool
SearchOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *
SearchOp::getName()
{
  return name;
}

int
SearchOp::run( WPUContext *wpu, ActionMessage *msg )
{
  AGUIX *aguix = Worker::getAGUIX();
  AGMessage *agmsg;
  AWindow *win;
  int endmode=-1;
  std::string str1;
  int w, h, sw, sh;

  _stored_results_lock.lock();

  _initial_start_dir = "/";
  
  if ( msg->mode != msg->AM_MODE_DNDACTION ) {
    Lister *l1;
    ListerMode *lm1;
    NormalMode *nm1;
    l1 = msg->getWorker()->getActiveLister();
    if ( l1 != NULL ) {
      lm1 = l1->getActiveMode();
      if ( lm1 != NULL ) {
        if ( lm1->isType( "NormalMode" ) == true ) {
          nm1 = (NormalMode*)lm1;
          if ( nm1->getCurrentDir() != NULL )
            _initial_start_dir = nm1->getCurrentDir();
        }
      }
    }
  }

  initColorDefs();

  win = new AWindow( aguix, 10, 10, 10, 10, 0, getDescription() );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 5 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainerBB *ac1_1 = dynamic_cast<AContainerBB*>( ac1->add( new AContainerBB( win, 1, 2 ), 0, 0 ) );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 5 );

  _start_ac = ac1_1->add( new AContainer( win, 2, 3 ), 0, 0 );
  _start_ac->setMinSpace( 5 );
  _start_ac->setMaxSpace( 5 );
  _start_ac->setBorderWidth( 0 );

  _start_ac->add( new Text( aguix, 0, 0, catalog.getLocale( 722 ), 1 ), 0, 0, AContainer::CFIX );
  _startdir_text = (Text*)_start_ac->add( new Text( aguix, 0, 0, catalog.getLocale( 723 ), 1 ), 1, 0, AContainer::CINCW );
  _startdir_text->hide();
  _startdir_sg = (StringGadget*)_start_ac->add( new StringGadget( aguix, 0, 0, 100, _initial_start_dir.c_str(), 0 ), 1, 0, AContainer::CINCW );
  _startdir_text->resize( _startdir_text->getWidth(), _startdir_sg->getHeight() );

  AContainer *ac1_2 = _start_ac->add( new AContainer( win, 4, 1 ), 1, 1 );
  ac1_2->setMinSpace( 0 );
  ac1_2->setMaxSpace( 0 );
  ac1_2->setBorderWidth( 0 );

  _start_ac->add( new Text( aguix, 0, 0, catalog.getLocale( 724 ), 1 ), 0, 1, AContainer::CFIX );
  _matchname_sg = (StringGadget*)ac1_2->add( new StringGadget( aguix, 0, 0, 100, "*", 0 ), 0, 0, AContainer::CINCW );
  _matchname_regexp_cb = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0, false,
                                                                      catalog.getLocale( 502 ), LABEL_RIGHT, 1, 0 ), 1, 0, AContainer::CFIXNR );
  _matchname_fullpath_cb = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0, false,
                                                                        catalog.getLocale( 570 ), LABEL_RIGHT, 1, 0 ), 2, 0, AContainer::CFIXNR );
  _matchname_case_cb = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0, false,
                                                                    catalog.getLocale( 725 ), LABEL_RIGHT, 1, 0 ), 3, 0, AContainer::CFIXNR );
  _matchname_regexp_cb->resize( _matchname_sg->getHeight(), _matchname_sg->getHeight() );
  _matchname_fullpath_cb->resize( _matchname_sg->getHeight(), _matchname_sg->getHeight() );
  _matchname_case_cb->resize( _matchname_sg->getHeight(), _matchname_sg->getHeight() );
  ac1_2->readLimits();
  
  AContainer *ac1_3 = _start_ac->add( new AContainer( win, 2, 1 ), 1, 2 );
  ac1_3->setMinSpace( 0 );
  ac1_3->setMaxSpace( 0 );
  ac1_3->setBorderWidth( 0 );

  _start_ac->add( new Text( aguix, 0, 0, catalog.getLocale( 726 ), 1 ), 0, 2, AContainer::CFIX );
  _matchcontent_sg = (StringGadget*)ac1_3->add( new StringGadget( aguix, 0, 0, 100, "", 0 ), 0, 0, AContainer::CINCW );
  _matchcontent_case_cb = (ChooseButton*)ac1_3->add( new ChooseButton( aguix, 0, 0, false,
                                                                       catalog.getLocale( 725 ), LABEL_RIGHT, 1, 0 ), 1, 0, AContainer::CFIXNR );
  _matchcontent_case_cb->resize( _matchcontent_sg->getHeight(), _matchcontent_sg->getHeight() );
  ac1_3->readLimits();

  AContainer *ac1_globsets = ac1_1->add( new AContainer( win, 4, 1 ), 0, 1 );
  ac1_globsets->setMinSpace( 5 );
  ac1_globsets->setMaxSpace( 5 );
  ac1_globsets->setBorderWidth( 0 );

  _follow_symlinks_cb = (ChooseButton*)ac1_globsets->add( new ChooseButton( aguix, 0, 0, false,
                                                                            catalog.getLocale( 298 ), LABEL_RIGHT, 1, 0 ), 0, 0, AContainer::CFIXNR );
  _search_vfs_cb = (ChooseButton*)ac1_globsets->add( new ChooseButton( aguix, 0, 0, false,
                                                                       catalog.getLocale( 727 ), LABEL_RIGHT, 1, 0 ), 1, 0, AContainer::CFIXNR );
  Button *newsearchb = (Button*)ac1_globsets->add( new Button( aguix,
                                                               0,
                                                               0,
                                                               catalog.getLocale( 728 ),
                                                               1,
                                                               0,
                                                               0 ), 3, 0, AContainer::CFIX );

  AContainer *ac1_res = ac1->add( new AContainer( win, 1, 2 ), 0, 1 );
  ac1_res->setMinSpace( 5 );
  ac1_res->setMaxSpace( 5 );
  ac1_res->setBorderWidth( 0 );
  ac1->setHWeight( 2, 0, 1 );

  _resultlv = (FieldListView*)ac1_res->add( new FieldListView( aguix, 0, 0, 100, 100, 0 ), 0, 0, AContainer::CMIN );
  _resultlv->setNrOfFields( 3 );
  _resultlv->setFieldWidth( 1, 5 );
  _resultlv->setFieldText( 0, catalog.getLocale( 718 ) );
  _resultlv->setFieldText( 2, catalog.getLocale( 729 ) );
  _resultlv->setFieldTextMerged( 0, true );
  _resultlv->setHBarState( 2 );
  _resultlv->setVBarState( 2 );
  _resultlv->setShowHeader( true );
  _resultlv->setAcceptFocus( true );
  _resultlv->setDisplayFocus( true );
  _resultlv->setMBG( wconfig->getListerBG() );

  AContainer *ac1_4 = ac1_res->add( new AContainer( win, 4, 1 ), 0, 1 );
  ac1_4->setMinSpace( 0 );
  ac1_4->setMaxSpace( -1 );
  ac1_4->setBorderWidth( 0 );

  Button *enterb =(Button*)ac1_4->add( new Button( aguix, 0, 0, catalog.getLocale( 730 ), 1, 0, 0 ), 0, 0, AContainer::CFIX );
  Button *viewb = (Button*)ac1_4->add( new Button( aguix, 0, 0, catalog.getLocale( 731 ), 1, 0, 0 ), 1, 0, AContainer::CFIX );
  Button *editb = (Button*)ac1_4->add( new Button( aguix, 0, 0, catalog.getLocale( 732 ), 1, 0, 0 ), 2, 0, AContainer::CFIX );
  Button *removeb = (Button*)ac1_4->add( new Button( aguix, 0, 0, catalog.getLocale( 733 ), 1, 0, 0 ),
                                         3, 0, AContainer::CFIX );
  
  AContainerBB *ac1_res_store = dynamic_cast<AContainerBB*>( ac1->add( new AContainerBB( win, 1, 3 ), 0, 2 ) );
  ac1_res_store->setMinSpace( 5 );
  ac1_res_store->setMaxSpace( 5 );
  ac1_res_store->setBorderWidth( 5 );

  ac1_res_store->add( new Text( aguix, 0, 0, catalog.getLocale( 734 ), 1 ), 0, 0, AContainer::CINCW );
  _storelv = (FieldListView*)ac1_res_store->add( new FieldListView( aguix, 0, 0, 100, 5 * aguix->getCharHeight(), 0 ), 0, 1, AContainer::CMIN );
  _storelv->setHBarState( 0 );
  _storelv->setVBarState( 2 );
  _storelv->setNrOfFields( 7 );
  _storelv->setFieldWidth( 1, 5 );
  _storelv->setFieldWidth( 3, 5 );
  _storelv->setFieldWidth( 5, 5 );
  _storelv->setFieldText( 0, catalog.getLocale( 735 ) );
  _storelv->setFieldWidth( 0, aguix->getTextWidth( catalog.getLocale( 735 ) ) + 20 );
  _storelv->setFieldText( 2, catalog.getLocale( 729 ) );
  _storelv->setFieldWidth( 2, aguix->getTextWidth( catalog.getLocale( 729 ) ) + 20 );
  _storelv->setFieldText( 4, catalog.getLocale( 736 ) );
  _storelv->setFieldWidth( 4, aguix->getTextWidth( catalog.getLocale( 736 ) ) + 20 );
  _storelv->setFieldText( 6, catalog.getLocale( 737 ) );
  _storelv->setFieldWidth( 6, aguix->getTextWidth( catalog.getLocale( 737 ) ) + 100 );
  _storelv->setFieldTextMerged( 0, true );
  _storelv->setFieldTextMerged( 2, true );
  _storelv->setFieldTextMerged( 4, true );
  _storelv->setShowHeader( true );
  _storelv->setAcceptFocus( true );
  _storelv->setDisplayFocus( true );
  _storelv->setMBG( wconfig->getListerBG() );

  Button *clearstoreb =(Button*)ac1_res_store->add( new Button( aguix, 0, 0, catalog.getLocale( 738 ), 1, 0, 0 ), 0, 2, AContainer::CINCW );

  AContainer *ac1_6 = ac1->add( new AContainer( win, 2, 1 ), 0, 3 );
  ac1_6->setMinSpace( 5 );
  ac1_6->setMaxSpace( 5 );
  ac1_6->setBorderWidth( 0 );

  ac1_6->add( new Text( aguix, 0, 0, catalog.getLocale( 739 ), 1 ), 0, 0, AContainer::CFIX );
  Text *state_text = (Text*)ac1_6->add( new Text( aguix, 0, 0, catalog.getLocale( 740 ), 1 ), 1, 0, AContainer::CINCW );

  AContainer *ac1_5 = ac1->add( new AContainer( win, 4, 1 ), 0, 4 );
  ac1_5->setMinSpace( 5 );
  ac1_5->setMaxSpace( 5 );
  ac1_5->setBorderWidth( 0 );
  Button *startb =(Button*)ac1_5->add( new Button( aguix,
                                                   0,
                                                   0,
                                                   catalog.getLocale( 741 ),
                                                   1,
                                                   0,
                                                   0 ), 0, 0, AContainer::CFIX );
  Button *stopb = (Button*)ac1_5->add( new Button( aguix,
                                                   0,
                                                   0,
                                                   catalog.getLocale( 742 ),
                                                   1,
                                                   0,
                                                   0 ), 1, 0, AContainer::CFIX );
  Button *closeb = (Button*)ac1_5->add( new Button( aguix,
                                                    0,
                                                    0,
                                                    catalog.getLocale( 633 ),
                                                    1,
                                                    0,
                                                    0 ), 3, 0, AContainer::CFIX );
  
  int startb_maxwidth = aguix->getTextWidth( catalog.getLocale( 741 ) ), temp_width;
  temp_width = aguix->getTextWidth( catalog.getLocale( 743 ) );
  if ( temp_width > startb_maxwidth ) startb_maxwidth = temp_width;
  temp_width = aguix->getTextWidth( catalog.getLocale( 744 ) );
  if ( temp_width > startb_maxwidth ) startb_maxwidth = temp_width;

  startb_maxwidth += aguix->getTextWidth( "  " ) + 10;
  startb->resize( startb_maxwidth, startb->getHeight() );
  ac1_5->readLimits();

  win->setDoTabCycling( true );
  win->contMaximize( true );

  w = win->getWidth();
  h = win->getHeight();

  sw = aguix->getRootWindowWidth();
  sh = aguix->getRootWindowHeight();
  sw = (int)( (double)sw * 0.6 );
  sh = (int)( (double)sh * 0.7 );
  if ( sw < 200 ) sw = 200;
  if ( sh < 100 ) sh = 100;
  if ( w < sw ) w = sw;
  if ( h < sh ) h = sh;
  win->resize( w, h );

  win->centerScreen();

  _sth = NULL;
  if ( _show_prev_results == true ) {
    setCurrentResultStore( _stored_results.size() - 1 );
  } else {
    //    setCurrentResultStore( -1 );
    newSearch();
  }

  prepareStartCont( true );
  buildLV();
  buildStoreLV();

  if ( _show_prev_results == true ) {
    _resultlv->takeFocus();
  } else {
    _matchname_sg->takeFocus();
  }
  win->useStippleBackground();

  _startdir_sg->selectAll();
  _matchname_sg->selectAll();
  _matchcontent_sg->selectAll();

  win->show();
  msg->getWorker()->setWaitCursor();

  int update_ctr = 0;

  for (; endmode == -1 ; ) {
    agmsg = aguix->GetMessage( win );
    if ( agmsg != NULL ) {
      switch ( agmsg->type ) {
        case AG_CLOSEWINDOW:
          if ( agmsg->closewindow.window == win->getWindow() ) endmode = 1;
          break;
        case AG_BUTTONCLICKED:
          if ( agmsg->button.button == closeb ) endmode = 1;
          else if ( agmsg->button.button == startb ) {
            if ( _sth == NULL || _sth->getCurrentMode() == SearchThread::FINISHED ) {
              startSearch();
            } else {
              if ( _sth->getCurrentMode() == SearchThread::SEARCHING ) {
                _sth->setOrder( SearchThread::SUSPEND );
              } else {
                _sth->setOrder( SearchThread::SEARCH );
              }
            }
          } else if ( agmsg->button.button == newsearchb ) {
            newSearch();
          } else if ( agmsg->button.button == stopb ) {
            if ( _sth != NULL )
              _sth->setOrder( SearchThread::STOP );
          } else if ( agmsg->button.button == enterb ) {
            endmode = 2;
          } else if ( agmsg->button.button == clearstoreb ) {
            clearStoredResults();
            newSearch();
          } else if ( agmsg->button.button == removeb ) {
            removeSelectedResults();
          } else if ( agmsg->button.button == viewb ) {
            viewResult();
          } else if ( agmsg->button.button == editb ) {
            editResult();
          }
          break;
        case AG_FIELDLV_ONESELECT:
        case AG_FIELDLV_MULTISELECT:
          if ( agmsg->fieldlv.lv == _storelv ) {
            int row = _storelv->getActiveRow();
            if ( _storelv->isValidRow( row ) == true ) {
              stopSearch();
              setCurrentResultStore( row );
              buildLV();
            }
          } else if ( agmsg->fieldlv.lv == _resultlv ) {
            int row = _resultlv->getActiveRow();
            if ( _resultlv->isValidRow( row ) == true ) {
              storeActiveRow( row );
            }
          }
          break;
        case AG_FIELDLV_DOUBLECLICK:
          if ( agmsg->fieldlv.lv == _resultlv ) {
            endmode = 2;
          }
          break;
        case AG_STRINGGADGET_OK:
          if ( agmsg->stringgadget.sg == _startdir_sg ) {
              if ( agmsg->stringgadget.keystate == ControlMask ) {
                  startSearch();
              } else {
                  _matchname_sg->takeFocus();
              }
          } else if ( agmsg->stringgadget.sg == _matchname_sg ) {
              if ( agmsg->stringgadget.keystate == ControlMask ) {
                  startSearch();
              } else {
                  _matchcontent_sg->takeFocus();
              }
          } else if ( agmsg->stringgadget.sg == _matchcontent_sg ) {
            startSearch();
          }
          break;
        case AG_KEYPRESSED:
          if ( agmsg->key.key == XK_Return && _resultlv->getHasFocus() == true ) {
            int row = _resultlv->getActiveRow();
            if ( _resultlv->isValidRow( row ) == true ) {
              endmode = 2;
            }
          } else if ( agmsg->key.key == XK_Return && KEYSTATEMASK( agmsg->key.keystate ) == ControlMask ) {
              startSearch();
          } else if ( agmsg->key.key == XK_Escape ) {
            endmode = 1;
          } else if ( agmsg->key.key == XK_F3 ) {
            viewResult();
          } else if ( agmsg->key.key == XK_F4 ) {
            editResult();
          } else if ( agmsg->key.key == XK_BackSpace && _resultlv->getHasFocus() == true ) {
            removeSelectedResults();
          } else if ( agmsg->key.key == XK_F5 ) {
            newSearch();
          }
          break;
      }
      aguix->ReplyMessage( agmsg );
    } else {
      if ( _sth == NULL || _sth->getCurrentMode() == SearchThread::FINISHED ) {
        prepareButton( *startb, BUTTON_IDLE );

        /* the thread is finished so update store to show total nr of results
         * but only after all results are gathered */
        if ( _search_started == true &&
             ( _sth == NULL || _sth->getNrOfResults() == 0 ) ) {
            buildStoreLV();
            _search_started = false;
        }
      } else if ( _sth->getCurrentMode() == SearchThread::SEARCHING )
        prepareButton( *startb, BUTTON_SUSPEND );
      else if ( _sth->getCurrentMode() == SearchThread::SUSPENDED )
        prepareButton( *startb, BUTTON_CONTINUE );

      if ( (++update_ctr % 10 ) == 0 )
        updateState( *state_text );

      if ( _sth != NULL && _sth->getNrOfResults() > 0 ) {
        // don't get more than 10 results at once to keep Worker responsive
        gatherResults( 10 );
      } else waittime( 10 );
    }
  }

  stopSearch();

  if ( endmode == 2 ) {
    //enter active entry
    int row = _resultlv->getActiveRow();
    if ( _resultlv->isValidRow( row ) ) {
      enterResult( msg, _resultlv->getText( row, 2 ) );
    }
  }
  
  delete win;

  cleanupPtrs();

  _stored_results_lock.unlock();

  msg->getWorker()->unsetWaitCursor();
  
  return 0;
}

const char *
SearchOp::getDescription()
{
  return catalog.getLocaleCom( 50 );
}

void SearchOp::updateState( Text &state_text )
{
  std::string statetxt;

  if ( _sth != NULL && _sth->getCurrentMode() == SearchThread::SEARCHING ) {
    statetxt = _sth->getCurrentActionInfo();
  } else if ( _sth != NULL && _sth->getCurrentMode() == SearchThread::SUSPENDED ) {
    statetxt = catalog.getLocale( 745 );
  } else {
    statetxt = catalog.getLocale( 740 );
  }
  state_text.setText( statetxt.c_str() );
}

void SearchOp::prepareButton( Button &startb, buttonmode_t mode )
{
  std::string newtext;
  if ( mode == BUTTON_CONTINUE ) {
    newtext = catalog.getLocale( 743 );
  } else if ( mode == BUTTON_SUSPEND ) {
    newtext = catalog.getLocale( 744 );
  } else {
    newtext = catalog.getLocale( 741 );
  }

  if ( newtext != startb.getText( 0 ) )
    startb.setText( 0, newtext.c_str() );
}

void SearchOp::prepareStartCont( bool force )
{
  if ( _rst != NULL ) {
    bool result_available = ( _rst->getDir()->empty() == true ) ? false : true;
    if ( result_available != _last_startcont_state || force == true ) {
      if ( result_available == true ) {
        _startdir_sg->hide();
        _startdir_text->show();

        int minw = _start_ac->getMinWidth( 1, 0 );
        _start_ac->add( _startdir_text, 1, 0, AContainer::CINCW );
        _start_ac->setMinWidth( minw, 1, 0 );
        
        _start_ac->rearrange();
      } else {
        _startdir_sg->show();
        _startdir_text->hide();

        int minw = _start_ac->getMinWidth( 1, 0 );
        _start_ac->add( _startdir_sg, 1, 0, AContainer::CINCW );
        _start_ac->setMinWidth( minw, 1, 0 );
        
        _start_ac->rearrange();
      }
      _last_startcont_state = result_available;
    }
  }
}


bool SearchOp::consumeResult( SearchThread::SearchResult &result, FieldListView &lv )
{
  bool new_result = false;

  std::string fullname = result.getFullname();

  int row = lv.addRow();
  
  if ( result.getLineNr() >= 0 ) {
    char linestr[ A_BYTESFORNUMBER( int ) ];
    sprintf( linestr, "%d", result.getLineNr() + 1);
    
    lv.setText( row, 0, linestr );
  } else {
    lv.setText( row, 0, "" );
  }

  if ( result.getIsDir() == true && fullname != "/" )
    fullname += "/";
  
  lv.setText( row, 2, fullname );

  if ( result.getIsDir() == true ) {
    lv.setColor( row, dircol );
  } else {
    lv.setColor( row, filecol );
  }
  
  if ( _last_result != fullname ) {
    new_result = true;
    _last_result = fullname;
  }

  if ( row == 0 ) {
    lv.setActiveRow( row );
    storeActiveRow( row );
  }

  return new_result;
}

void SearchOp::enterResult( ActionMessage *msg, const std::string &fullname )
{
  Lister *l1;
  ListerMode *lm1;
  NormalMode *nm1;

  l1 = msg->getWorker()->getActiveLister();
  if ( l1 != NULL ) {
    l1->switch2Mode( 0 );
    lm1 = l1->getActiveMode();
    if ( lm1 != NULL ) {
      if ( lm1->isType( "NormalMode" ) == true ) {
        nm1 = (NormalMode*)lm1;
        nm1->enterPath( fullname );
      }
    }
  }
}

void SearchOp::clearStoredResults()
{
  stopSearch();
  while ( _stored_results.empty() == false ) {
    ResultStore *rs = _stored_results.front();
    _stored_results.pop_front();
    delete rs;
  }
  
  _rst = NULL;
  
  if ( _storelv != NULL )
    buildStoreLV();
}

void SearchOp::buildLV()
{
  std::list<SearchThread::SearchResult> *res = _rst->getRes();
  SearchSettings sets = _rst->getSearchSettings();
  int active_row = sets.getActiveRow();

  _resultlv->setSize( 0 );
  for ( std::list<SearchThread::SearchResult>::iterator it = res->begin();
        it != res->end();
        it++ ) {
    consumeResult( *it, *_resultlv );
  }
  _resultlv->setActiveRow( active_row );
  _resultlv->redraw();
  
  prepareStartCont();
}

void SearchOp::prepareNewSearch()
{
  _last_result = "";
  _resultlv->setSize( 0 );
  _resultlv->redraw();

  prepareStartCont();
}

void SearchOp::buildStoreLV()
{
  if ( _storelv != NULL ) {
    _storelv->setSize( 0 );
    int row = -1;
    for ( std::list<ResultStore*>::iterator it = _stored_results.begin();
          it != _stored_results.end();
          it++ ) {
      row = _storelv->addRow();
      _storelv->setColor( row, resultcol );

      SearchSettings sets = (*it)->getSearchSettings();
      std::string resstr;
      char buf[A_BYTESFORNUMBER(int)];

      sprintf( buf, "%d", (int)( (*it)->getRes()->size() ) );
      _storelv->setText( row, 0, buf );
      _storelv->setText( row, 2, sets.getMatchName() );
      _storelv->setText( row, 4, sets.getMatchContent() );
      _storelv->setText( row, 6, sets.getBaseDir() );
    }
    _storelv->showRow( row );
    _storelv->redraw();
  }
}

SearchOp::ResultStore *SearchOp::getResultStore( int pos, const SearchSettings *sets )
{
  ResultStore *erg = NULL;

  // try to find position
  if ( pos >= 0 ) {
    for ( std::list<ResultStore*>::iterator it = _stored_results.begin();
          it != _stored_results.end();
          it++, pos-- ) {
      if ( pos == 0 ) {
        erg = *it;
        break;
      } 
    }
  }

  if ( erg == NULL ) {
    // nothing found so create a new one with settings from
    // the current or latest entry
    erg = new ResultStore( "search1" );

    std::list<ResultStore*>::iterator it;
    for ( it = _stored_results.begin();
          it != _stored_results.end();
          it++ ) {
      if ( *it == _rst ) {
        break;
      }
    }

    if ( sets != NULL ) {
      erg->setSearchSettings( *sets );
    } else {
      if ( it != _stored_results.end() ) {
        erg->setSearchSettings( (*it)->getSearchSettings() );
      } else {
        if ( _stored_results.size() > 0 ) {
          erg->setSearchSettings( _stored_results.back()->getSearchSettings() );
        } else {
          erg->setSearchSettings( getCurrentSearchSettings() );
        }
      }
    }
    
    _stored_results.push_back( erg );
    
    buildStoreLV();
  }

  return erg;
}

class DW_SetFollowSymlinks : public NWC::DirWalkCallBack
{
public:
  DW_SetFollowSymlinks( bool follow_symlinks, bool search_vfs ) : _follow_symlinks( follow_symlinks ),
                                                                  _search_vfs( search_vfs ) {}

  int visit( NWC::File &file, NWC::Dir::WalkControlObj &cobj ) { return 0; }
  int visitEnterDir( NWC::Dir &dir, NWC::Dir::WalkControlObj &cobj )
  {
    dir.setFollowSymlinks( _follow_symlinks );

    try {
      NWC::VirtualDir &vdir = dynamic_cast<NWC::VirtualDir&>( dir );
      vdir.setSearchVFS( _search_vfs );
    } catch ( std::bad_cast ) {
    }
    return 0;
  }
  int visitLeaveDir( NWC::Dir &dir, NWC::Dir::WalkControlObj &cobj ) { return 0; }
private:
  bool _follow_symlinks;
  bool _search_vfs;
};

void SearchOp::startSearch()
{
  stopSearch();

  SearchSettings sets = getCurrentSearchSettings();;

  std::auto_ptr<NWC::VirtualDir> use_dir( _rst->getClonedDir() );
  
  if ( use_dir->size() < 1 ) {
    std::string d = sets.getBaseDir();
    if ( d == "." || d.length() < 1 )
      d = _initial_start_dir;
    use_dir->add( NWC::VirtualDir( d ) );
  } else {
    setCurrentResultStore( -1, &sets );
  }
  prepareNewSearch();
  _resultlv->takeFocus();
  
  use_dir->setFollowSymlinks( sets.getFollowSymlinks() );
  use_dir->setSearchVFS( sets.getSearchVFS() );
  
  DW_SetFollowSymlinks dw1( sets.getFollowSymlinks(), sets.getSearchVFS() );
  use_dir->walk( dw1 );

  _sth = new SearchThread( std::auto_ptr<NWC::Dir>( use_dir ) );
  _sth->setMatchName( sets.getMatchName() );
  _sth->setMatchFullpath( sets.getMatchNameFullPath() );
  _sth->setMatchRegExp( sets.getMatchNameRE() );
  _sth->setMatchCaseSensitive( sets.getMatchNameCase() );
  _sth->setMatchContent( sets.getMatchContent() );
  _sth->setMatchContentCaseSensitive( sets.getMatchContentCase() );
  _sth->start();

  storeSearchSettings();
  buildStoreLV();
  
  _search_started = true;
}

void SearchOp::stopSearch()
{
  // stop thread, gather remaining results into rst and delete thread
  if ( _sth != NULL ) {
    _sth->setOrder( SearchThread::STOP );
    _sth->join();
    gatherResults();
    delete _sth;
    _sth = NULL;
  }
}

void SearchOp::gatherResults( int get_number_of_results )
{
  // get results from thread and put them into rst
  //TODO wait for a couple of results or idling search thread
  //     to reduce frequency of update
  if ( _sth != NULL && _sth->getNrOfResults() > 0 ) {
    _sth->lockResults();
    while ( _sth->getNrOfResults() > 0 ) {
      SearchThread::SearchResult erg = _sth->getResultLocked();
      _rst->getRes()->push_back( erg );
      if ( consumeResult( erg, *_resultlv ) == true ) {
        _rst->getDir()->add( NWC::FSEntry( erg.getFullname() ) );
      }

      if ( get_number_of_results > 0 ) {
        get_number_of_results--;
        if ( get_number_of_results == 0 )
          break;
      }
    }
    _sth->unlockResults();
    _resultlv->redraw();
    
    prepareStartCont();
  }
}

void SearchOp::newSearch()
{
  stopSearch();

  _startdir_sg->setText( _initial_start_dir.c_str() );
  SearchSettings sets = getCurrentSearchSettings();

  if ( _rst == NULL ||_rst->getDir()->size() > 0 ) {
    // first try last result in store whether it's empty or not
    setCurrentResultStore( _stored_results.size() - 1 );
    if ( _rst->getDir()->size() > 0 ) {
      setCurrentResultStore( -1, &sets );
      restoreSearchSettings();
    } else {
      _rst->setSearchSettings( sets );
      restoreSearchSettings();
    }
  }

  prepareNewSearch();
  _matchname_sg->takeFocus();
}

SearchOp::SearchSettings SearchOp::getCurrentSearchSettings()
{
  SearchSettings sets( _startdir_sg->getText(),
                       _matchname_sg->getText(),
                       _matchname_regexp_cb->getState(),
                       _matchname_fullpath_cb->getState(),
                       _matchname_case_cb->getState(),
                       _matchcontent_sg->getText(),
                       _matchcontent_case_cb->getState(),
                       _follow_symlinks_cb->getState(),
                       _search_vfs_cb->getState() );
  return sets;
}

void SearchOp::storeSearchSettings()
{
  _rst->setSearchSettings( getCurrentSearchSettings() );
}

void SearchOp::restoreSearchSettings()
{
  if ( _rst == NULL ) return;
  
  SearchSettings sets = _rst->getSearchSettings();
  if ( sets.getBaseDir().length() < 1 ) {
    _startdir_sg->setText( _initial_start_dir.c_str() );
  } else {
    _startdir_sg->setText( sets.getBaseDir().c_str() );
  }
  _matchname_sg->setText( sets.getMatchName().c_str() );
  _matchname_regexp_cb->setState( sets.getMatchNameRE() );
  _matchname_fullpath_cb->setState( sets.getMatchNameFullPath() );
  _matchname_case_cb->setState( sets.getMatchNameCase() );
  _matchcontent_sg->setText( sets.getMatchContent().c_str() );
  _matchcontent_case_cb->setState( sets.getMatchContentCase() );
  _follow_symlinks_cb->setState( sets.getFollowSymlinks() );
  _search_vfs_cb->setState( sets.getSearchVFS() );
}

void SearchOp::setCurrentResultStore( int pos, const SearchSettings *sets )
{
  _rst = getResultStore( pos, sets );
  if ( pos >= 0 ) {
    restoreSearchSettings();
  }
}

void SearchOp::storeActiveRow( int row )
{
  if ( _rst != NULL ) {
    _rst->setActiveRow( row );
  }
}

void SearchOp::cleanupPtrs()
{
  _start_ac = NULL;
  _startdir_text = NULL;
  _startdir_sg = NULL;
  _storelv = NULL;
  _rst = NULL;
  _resultlv = NULL;
  _sth = NULL;
  _matchname_sg = NULL;
  _matchname_regexp_cb = NULL;
  _matchname_fullpath_cb = NULL;
  _matchname_case_cb = NULL;
  _matchcontent_sg = NULL;
  _matchcontent_case_cb = NULL;
  _follow_symlinks_cb = NULL;
  _search_vfs_cb = NULL;
}

void SearchOp::removeSelectedResults()
{
  // first get all selected unique entries
  // for each entry search all results which uses the
  // same entry (for different lines)
  // then remove it from searchdir

  std::auto_ptr<std::list<std::string> > selected_entries = getSelectedResults();
  
  _rst->removeEntries( *( selected_entries.get() ) );
  buildLV();
  _resultlv->centerActive();
}

std::auto_ptr<std::list<std::string> > SearchOp::getSelectedResults()
{
  std::list<SearchThread::SearchResult> *res = _rst->getRes();
  std::list<SearchThread::SearchResult>::iterator active_entry = res->end();

  std::auto_ptr<std::list<std::string> > selected_entries( new std::list<std::string> );

  int lvpos = 0;
  for ( std::list<SearchThread::SearchResult>::iterator it = res->begin();
        it != res->end();
        it++, lvpos++ ) {
    if ( _resultlv->getSelect( lvpos ) == true ) {
      if ( selected_entries->empty() == true ) {
        selected_entries->push_back( (*it).getFullname() );
      } else {
        if ( selected_entries->back() != (*it).getFullname() ) {
          selected_entries->push_back( (*it).getFullname() );
        }
      }
    } else if ( lvpos == _resultlv->getActiveRow() ) {
      active_entry = it;
    }
  }

  if ( selected_entries->empty() == true && active_entry != res->end() ) {
    selected_entries->push_back( (*active_entry).getFullname() );
  }

  return selected_entries;
}

SearchThread::SearchResult SearchOp::getResultForRow( int row )
{
  std::list<SearchThread::SearchResult> *res = _rst->getRes();
  SearchThread::SearchResult r( "", -1 );

  int lvpos = 0;
  for ( std::list<SearchThread::SearchResult>::iterator it = res->begin();
        it != res->end();
        it++, lvpos++ ) {
    if ( lvpos == row ) {
      r = *it;
      break;
    }
  }

  return r;
}

void SearchOp::viewResult()
{
  int row = _resultlv->getActiveRow();
  if ( _resultlv->isValidRow( row ) ) {
    SearchThread::SearchResult r = getResultForRow( row );
    if ( r.getFullname().length() > 0 ) {
      std::list<std::string> l;
      //TODO check whether file system exceeds basic textstorage size and ask for increase
      l.push_back( r.getFullname() );

      bool old_sln = FileViewer::getShowLineNumbers();
      FileViewer::setShowLineNumbers( true );
      FileViewer fv;
      fv.view( l, r.getLineNr(), true );
      FileViewer::setShowLineNumbers( old_sln );
    }
  }
}

void SearchOp::editResult()
{
  int row = _resultlv->getActiveRow();
  if ( _resultlv->isValidRow( row ) ) {
    SearchThread::SearchResult r = getResultForRow( row );
    if ( r.getFullname().length() > 0 ) {
      editFile( r.getFullname(), r.getLineNr() );
    }
  }
}

void SearchOp::editFile( const std::string &filename, int line_number )
{
  FlagReplacer::FlagProducerMap fl;
  NWC::FSEntry fse( filename );
  char linestr[ A_BYTESFORNUMBER( int ) ];

  sprintf( linestr, "%d", line_number + 1 );

  fl.registerFlag( "f", fse.getBasename() );
  fl.registerFlag( "F", filename );
  fl.registerFlag( "l", linestr );

  std::string scriptdir = Worker::getDataDir();
  scriptdir += "/scripts";
  fl.registerFlag( "scripts", scriptdir );
  
  FlagReplacer rpl( fl );
  
  std::string exestr = rpl.replaceFlags( _edit_command );

  std::auto_ptr<ExeClass> ec( new ExeClass( ExeClass::EXECUTE_NORMAL ) );

  ec->cdDir( fse.getDirname().c_str() );
  ec->addCommand( "%s", exestr.c_str() );
  ec->execute();
}

bool SearchOp::save( Datei *fh )
{
  if ( fh == NULL ) return false;
  fh->configPutPairString( "editcommand", _edit_command.c_str() );
  fh->configPutPairBool( "showprevresults", _show_prev_results );
  return true;
}

int SearchOp::configure()
{
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  StringGadget *sg;
  AGMessage *msg;
  int endmode=-1;

  char *tstr;
  tstr = (char*)_allocsafe( strlen( catalog.getLocale( 293 ) ) + strlen( catalog.getLocaleCom( 50 ) ) + 1 );
  sprintf( tstr, catalog.getLocale( 293 ), catalog.getLocaleCom( 50 ) );
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
  win->create();
  _freesafe(tstr);

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 4 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 3, 1 ), 0, 0 );
  ac1_1->setMinSpace( 0 );
  ac1_1->setMaxSpace( 0 );
  ac1_1->setBorderWidth( 0 );

  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 719 ), 1 ),
              0, 0, AContainer::CFIX );
  sg = (StringGadget*)ac1_1->add( new StringGadget( aguix, 0, 0, 200, _edit_command.c_str(), 0 ),
                                  1, 0, AContainer::CINCW );
  Button *flagb = (Button*)ac1_1->add( new Button( aguix,
                                                   0,
                                                   0,
                                                   "F",
                                                   1,
                                                   0,
                                                   0 ), 2, 0, AContainer::CFIX );
  flagb->resize( flagb->getWidth(), sg->getHeight() );
  ac1_1->readLimits();

  Button *resetb = (Button*)ac1->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 196 ),
                                                  1,
                                                  0,
                                                  0 ), 0, 1, AContainer::CINCW );

  ChooseButton *sprcb = (ChooseButton*)ac1->add( new ChooseButton( aguix,
                                                                   0,
                                                                   0,
                                                                   _show_prev_results,
                                                                   catalog.getLocale( 746 ),
                                                                   LABEL_RIGHT, 1, 0 ),
                                                 0, 2, AContainer::CINCWNR );

  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 3 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, AContainer::CFIX );
  Button *cb = (Button*)ac1_2->add( new Button( aguix,
						0,
						0,
						catalog.getLocale( 8 ),
						1,
						0,
						0 ), 1, 0, AContainer::CFIX );
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();

  FlagReplacer::FlagHelp flags;

  flags.registerFlag( "{f}", catalog.getLocale( 589 ) );
  flags.registerFlag( "{F}", catalog.getLocale( 590 ) );
  flags.registerFlag( "{l}", catalog.getLocale( 718 ) );
  flags.registerFlag( "{scripts}", catalog.getLocaleFlag( 53 ) );

  for ( ;endmode == -1; ) {
    msg = aguix->WaitMessage( win );
    if ( msg != NULL ) {
      switch ( msg->type ) {
        case AG_CLOSEWINDOW:
          if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
          break;
        case AG_BUTTONCLICKED:
          if ( msg->button.button == okb ) endmode = 0;
          else if ( msg->button.button == cb) endmode = 1;
          else if ( msg->button.button == flagb ) {
            std::string str1 = FlagReplacer::requestFlag( flags );
            if ( str1.length() > 0 ) {
              sg->insertAtCursor( str1.c_str() );
            }
          } else if ( msg->button.button == resetb ) {
            sg->setText( DEFAULT_EDIT_COMMAND );
          }
          break;
      }
      aguix->ReplyMessage( msg );
    }
  }
  
  if ( endmode == 0 ) {
    // ok
    setEditCommand( sg->getText() );
    setShowPrevResults( sprcb->getState() );
  }
  
  delete win;

  return endmode;
}

void SearchOp::setEditCommand( const std::string &str )
{
  _edit_command = str;
}

void SearchOp::setShowPrevResults( bool nv )
{
  _show_prev_results = nv;
}

void SearchOp::initColorDefs()
{
  dircol.setFG( FieldListView::CC_NORMAL, wconfig->getUnselDir( 0 ) );
  dircol.setBG( FieldListView::CC_NORMAL, wconfig->getUnselDir( 1 ) );
  dircol.setFG( FieldListView::CC_SELECT, wconfig->getSelDir( 0 ) );
  dircol.setBG( FieldListView::CC_SELECT, wconfig->getSelDir( 1 ) );
  dircol.setFG( FieldListView::CC_ACTIVE, wconfig->getUnselDirAct( 0 ) );
  dircol.setBG( FieldListView::CC_ACTIVE, wconfig->getUnselDirAct( 1 ) );
  dircol.setFG( FieldListView::CC_SELACT, wconfig->getSelDirAct( 0 ) );
  dircol.setBG( FieldListView::CC_SELACT, wconfig->getSelDirAct( 1 ) );
  
  filecol.setFG( FieldListView::CC_NORMAL, wconfig->getUnselFile( 0 ) );
  filecol.setBG( FieldListView::CC_NORMAL, wconfig->getUnselFile( 1 ) );
  filecol.setFG( FieldListView::CC_SELECT, wconfig->getSelFile( 0 ) );
  filecol.setBG( FieldListView::CC_SELECT, wconfig->getSelFile( 1 ) );
  filecol.setFG( FieldListView::CC_ACTIVE, wconfig->getUnselFileAct( 0 ) );
  filecol.setBG( FieldListView::CC_ACTIVE, wconfig->getUnselFileAct( 1 ) );
  filecol.setFG( FieldListView::CC_SELACT, wconfig->getSelFileAct( 0 ) );
  filecol.setBG( FieldListView::CC_SELACT, wconfig->getSelFileAct( 1 ) );

  resultcol.setFG( FieldListView::CC_NORMAL, wconfig->getUnselFile( 0 ) );
  resultcol.setBG( FieldListView::CC_NORMAL, wconfig->getUnselFile( 1 ) );
  resultcol.setFG( FieldListView::CC_SELECT, wconfig->getUnselFile( 0 ) );
  resultcol.setBG( FieldListView::CC_SELECT, wconfig->getUnselFile( 1 ) );
  resultcol.setFG( FieldListView::CC_ACTIVE, wconfig->getUnselFileAct( 0 ) );
  resultcol.setBG( FieldListView::CC_ACTIVE, wconfig->getUnselFileAct( 1 ) );
  resultcol.setFG( FieldListView::CC_SELACT, wconfig->getUnselFileAct( 0 ) );
  resultcol.setBG( FieldListView::CC_SELACT, wconfig->getUnselFileAct( 1 ) );
}
