/* bookmarkdbproxy.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "bookmarkdbproxy.hh"
#include "bookmarkdb.hh"
#include "bookmarkdbentry.hh"
#include "worker.h"

BookmarkDBProxy::BookmarkDBProxy( std::auto_ptr<BookmarkDB> bookmarks ) : m_serial_nr( 0 ),
                                                                          m_bookmarks( bookmarks )
{
}

BookmarkDBProxy::~BookmarkDBProxy()
{
}

void BookmarkDBProxy::addEntry( const BookmarkDBEntry &entry )
{
    m_bookmarks->addEntry( entry );
    bookmarksChanged();
}

void BookmarkDBProxy::delEntry( const BookmarkDBEntry &entry )
{
    m_bookmarks->delEntry( entry );
    bookmarksChanged();
}

void BookmarkDBProxy::updateEntry( const BookmarkDBEntry &entry,
                                   const BookmarkDBEntry &newentry )
{
    m_bookmarks->updateEntry( entry, newentry );
    bookmarksChanged();
}

std::auto_ptr<BookmarkDBEntry> BookmarkDBProxy::getEntry( const std::string &name )
{
    return m_bookmarks->getEntry( name );
}

std::list<std::string> BookmarkDBProxy::getCats()
{
    return m_bookmarks->getCats();
}

std::list<BookmarkDBEntry> BookmarkDBProxy::getEntries( const std::string &cat )
{
    return m_bookmarks->getEntries( cat );
}

int BookmarkDBProxy::read()
{
    int res = m_bookmarks->read();
    if ( res == 0 ) {
        bookmarksChanged();
    }
    return res;
}

int BookmarkDBProxy::write()
{
    return m_bookmarks->write();
}

void BookmarkDBProxy::registerChangeCallback( GenericCallback<int> *cb )
{
    m_change_callbacks.push_back( cb );
}

void BookmarkDBProxy::unregisterChangeCallback( GenericCallback<int> *cb )
{
    m_change_callbacks.remove( cb );
}

void BookmarkDBProxy::bookmarksChanged()
{
    m_serial_nr++;

    m_cached_entries = m_bookmarks->getEntries( "" );

    for ( std::list< GenericCallback<int>* >::iterator it1 = m_change_callbacks.begin();
          it1 != m_change_callbacks.end();
          ++it1 ) {
        (*it1)->callback();
    }
}

BookmarkDBProxy::check_path_res_t BookmarkDBProxy::checkPath( const std::string &path,
                                                              std::list<BookmarkDBEntry> &return_entries )
{
    //TODO use a state machine to speedup the checks
    std::list< BookmarkDBEntry >::const_iterator it1;
    bool prefix_hit = false;
    check_path_res_t res = NO_HIT;

    for ( it1 = m_cached_entries.begin();
          it1 != m_cached_entries.end();
          ++it1 ) {
        const std::string bm_name = it1->getName();

        if ( path.length() == bm_name.length() ) {
            if ( bm_name == path ) {
                return_entries.push_front( *it1 );
                res = FULL_HIT;
            }
        } else if ( path.length() < bm_name.length() ) {
            if ( bm_name.compare( 0, path.length(), path ) == 0 ) {
                if ( bm_name[path.length()] == '/' ) {
                    // store it as candidate but there might be other candidates or full hits
                    return_entries.push_back( *it1 );
                    prefix_hit = true;
                }
            }
        }
    }

    if ( res != NO_HIT ) return res;

    if ( prefix_hit == true ) return PREFIX_HIT;

    return NO_HIT;
}
